/*
** Copyright (c) Massachusetts Institute of Technology 1994-1998.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. Works derived from this code are not distributed for
**        commercial gain without explicit permission from MIT 
**        (for permission contact lclint-request@sds.lcs.mit.edu).
*/
/*
** ltokenList.c
**
** based on list_template.c
**
** where T has T_equal (or change this) and T_unparse
*/

# include "lclintMacros.nf"
# include "llbasic.h"

/*@notnull@*/ /*@only@*/ ltokenList
ltokenList_new ()
{
  ltokenList s = (ltokenList) dmalloc (sizeof (*s));
  
  s->nelements = 0;
  s->nspace = ltokenListBASESIZE;
  s->elements = (ltoken *) 
    dmalloc (sizeof (*s->elements) * ltokenListBASESIZE);
  s->current = 0;

  return (s);
}

/*@notnull@*/ /*@only@*/ ltokenList
ltokenList_singleton (ltoken l)
{
  ltokenList s = (ltokenList) dmalloc (sizeof (*s));

  s->nelements = 1;
  s->nspace = ltokenListBASESIZE - 1;
  s->elements = (ltoken *) dmalloc (sizeof (*s->elements) * ltokenListBASESIZE);
  s->elements[0] = l;
  s->current = 0;

  return (s);
}

static void
ltokenList_grow (/*@notnull@*/ ltokenList s)
{
  int i;
  ltoken *newelements;

  s->nspace += ltokenListBASESIZE;

  newelements = (ltoken *) dmalloc (sizeof (*newelements)
				    * (s->nelements + s->nspace));

  for (i = 0; i < s->nelements; i++)
    {
      newelements[i] =  s->elements[i]; 
    }

  sfree (s->elements); 
  s->elements = newelements;
}

ltokenList 
ltokenList_push (/*@returned@*/ ltokenList s, ltoken el)
{
  ltokenList_addh (s, el);
  return s;
}

void 
ltokenList_addh (ltokenList s, ltoken el)
{
  llassert (ltokenList_isDefined (s));

  if (s->nspace <= 0)
    ltokenList_grow (s);

  s->nspace--;
  s->elements[s->nelements] = el;
  s->nelements++;
}

void 
ltokenList_reset (ltokenList s)
{
  if (ltokenList_isDefined (s))
    {
      s->current = 0;
    }
}

bool
ltokenList_isFinished (ltokenList s)
{
  return (ltokenList_isUndefined(s) || (s->current == s->nelements));
}

void 
ltokenList_advance (ltokenList s)
{
  if (ltokenList_isDefined (s))
    {
      s->current++;
      llassert (s->current <= s->nelements);
    }
}

ltoken 
ltokenList_head (ltokenList s)
{
  llassert (ltokenList_isDefined (s) && s->nelements > 0);
  return (s->elements[0]);
}

bool 
ltokenList_equal (ltokenList s1, ltokenList s2)
{
  if (ltokenList_isUndefined (s1))
    {
      return (ltokenList_isEmpty (s2));
    }
  else
    {
      if (ltokenList_isUndefined (s2))
	{
	  return ltokenList_isEmpty (s1);
	}
      else
	{
	  int i;
	  int size = s1->nelements;
	  
	  if (s2->nelements != size)
	    return FALSE;
	  
	  for (i = 0; i < size; i++)
	    {
	      if (!ltoken_similar (s1->elements[i], s2->elements[i]))
		return FALSE;
	    }
	  return TRUE;
	}
    }
}

/*@only@*/ ltokenList 
ltokenList_copy (ltokenList s)
{
  ltokenList r = ltokenList_new ();

  ltokenList_elements (s, x)
  {
    ltokenList_addh (r, ltoken_copy (x));
  } end_ltokenList_elements;

  return r;
}

void
ltokenList_removeCurrent (ltokenList s)
{
  int i;
  llassert (ltokenList_isDefined (s) && s->current >= 0 && s->current < s->nelements);

  for (i = s->current; i < s->nelements - 1; i++)
    {
      s->elements[i] = s->elements[i+1];
    }

  s->nelements--;
  s->nspace++;
}

ltoken 
ltokenList_current (ltokenList s)
{
  llassert (ltokenList_isDefined (s) && s->current >= 0 && s->current < s->nelements);
  return (s->elements[s->current]);
}

/*@only@*/ cstring
ltokenList_unparse (ltokenList s)
{
  int i;
  cstring st = cstring_undefined;

  if (ltokenList_isDefined (s))
    {
      for (i = 0; i < s->nelements; i++)
	{
	  if (i == 0)
	    {
	      st = cstring_copy (ltoken_unparse (s->elements[i]));
	    }
	  else
	    st = message ("%q, %s", st, ltoken_unparse (s->elements[i]));
	}
    }

  return st;
}

void
ltokenList_free (ltokenList s)
{
  if (ltokenList_isDefined (s))
    {
      sfree (s->elements);
      sfree (s);
    }
}
