;==============================================================================

; file: "_num2.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(##include "header.scm")

;------------------------------------------------------------------------------

; Some utilities

(define (##exact-int.width x)
  (if (##fixnum? x)
    (##fixnum.width x)
    (##bignum.width x)))

(define (##fixnum.width x)
  (if (##fixnum.negative? x)
    (let loop1 ((w 0) (x x))
      (if (##fixnum.< x -1) (loop1 (##fixnum.+ w 1) (##fixnum.ashr x 1)) w))
    (let loop2 ((w 0) (x x))
      (if (##fixnum.< 0 x) (loop2 (##fixnum.+ w 1) (##fixnum.ashr x 1)) w))))

(define (##bignum.width x)
  (if (bignum-negative? x)
    (##bignum.width (##- -1 x)) ; lazy...
    (let ((len (bignum-length x)))
      (##fixnum.+ (##fixnum.* (##fixnum.- len 2) (radix-width))
                  (##fixnum.width (bignum-digit-ref x (##fixnum.- len 1)))))))

(define (##exact-int.root x y)
  (if (##eq? x 0)
    0
    (let loop ((g (##exact-int.expt2
                   (##quotient (##+ (##exact-int.width x) (##- y 1))
                               y))))
      (let ((a (##expt g (##- y 1))))
        (let ((b (##* a y)))
          (let ((c (##* a (##- y 1))))
            (let ((d (##quotient (##+ x (##* g c)) b)))
              (if (##< d g) (loop d) g))))))))

(define (##exact-int.div x y)

  (define (div x y)
    (let ((z (##bignum.div x y)))
      (##set-car! z (##bignum.normalize (##car z)))
      (##set-cdr! z (##bignum.normalize (##cdr z)))
      z))

  (if (##fixnum? x)
    (if (##fixnum? y)
      (##cons (##fixnum.quotient x y) (##fixnum.remainder x y))
      (div (##bignum.<-fixnum x) y))
    (if (##fixnum? y)
      (div x (##bignum.<-fixnum y))
      (div x y))))

;------------------------------------------------------------------------------

; Fixnum operations
; -----------------

(define-system (##fixnum.zero? x)
  (##eq? x 0))

(define-system (##fixnum.positive? x)
  (##fixnum.< 0 x))

(define-system (##fixnum.negative? x)
  (##fixnum.< x 0))

(define-system (##fixnum.odd? x)
  (##fixnum.= (##fixnum.logand x 1) 1))

(define-system (##fixnum.even? x)
  (##fixnum.= (##fixnum.logand x 1) 0))

(define-nary-bool (##fixnum.= x y)
  #t
  #t
  (##fixnum.= x y)
  no-force
  no-check)

(define-nary-bool (##fixnum.< x y)
  #t
  #t
  (##fixnum.< x y)
  no-force
  no-check)

(define-nary-bool (##fixnum.> x y)
  #t
  #t
  (##fixnum.< y x)
  no-force
  no-check)

(define-nary-bool (##fixnum.<= x y)
  #t
  #t
  (##not (##fixnum.< y x))
  no-force
  no-check)

(define-nary-bool (##fixnum.>= x y)
  #t
  #t
  (##not (##fixnum.< x y))
  no-force
  no-check)

(define-nary (##fixnum.max x y)
  ()
  x
  (##fixnum.max x y)
  no-force
  no-check)

(define-nary (##fixnum.min x y)
  ()
  x
  (##fixnum.min x y)
  no-force
  no-check)

(define-nary (##fixnum.+ x y)
  0
  x
  (##fixnum.+ x y)
  no-force
  no-check)

(define-nary (##fixnum.* x y)
  1
  x
  (##fixnum.* x y)
  no-force
  no-check)

(define-nary (##fixnum.- x y)
  ()
  (##fixnum.- x)
  (##fixnum.- x y)
  no-force
  no-check)

(define-system (##fixnum.quotient x y))

(define-system (##fixnum.remainder x y)
  (##fixnum.- x (##fixnum.* (##fixnum.quotient x y) y)))

(define-system (##fixnum.modulo x y)
  (let ((r (##fixnum.remainder x y)))
    (if (##fixnum.zero? r)
      r
      (if (##fixnum.negative? x)
        (if (##fixnum.negative? y) r (##fixnum.+ r y))
        (if (##fixnum.negative? y) (##fixnum.+ r y) r)))))

(define-nary (##fixnum.logior x y)
  0
  x
  (##fixnum.logior x y)
  no-force
  no-check)

(define-nary (##fixnum.logxor x y)
  0
  x
  (##fixnum.logxor x y)
  no-force
  no-check)

(define-nary (##fixnum.logand x y)
  -1
  x
  (##fixnum.logand x y)
  no-force
  no-check)

(define-system (##fixnum.lognot x) (##fixnum.- -1 x))
(define-system (##fixnum.ashr x y))
(define-system (##fixnum.lshr x y))
(define-system (##fixnum.shl x y))

(define-system (##fixnum.->char x))
(define-system (##fixnum.<-char x))

;------------------------------------------------------------------------------

; Bignum operations
; -----------------

; Bignums are represented with 'word' vectors:
;
; assuming that the bignum 'n' is represented by the word vector 'v' of
; length 'l', we have
;
;                       l-2
;                      -----
;                      \                   i
; n  =  (v[0]*2-1)  *   >   v[i+1] * radix
;                      /
;                      -----
;                      i = 0
;
; note: v[0] = 0 if number is negative, v[0] = 1 if number is positive.
;
; 'radix' must be less than or equal to sqrt(max fixnum)+1.  This guarantees
; that the result of an arithmetic operation on bignum digits will be a fixnum
; (this includes the product of two digits).

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Bignum comparison

(define (##bignum.= x y)
  (if (##not (##fixnum.= (bignum-sign x) (bignum-sign y)))
    #f
    (let ((lx (bignum-length x)))
      (if (##not (##fixnum.= lx (bignum-length y)))
        #f
        (let loop ((i (##fixnum.- lx 1)))
          (if (##fixnum.< 0 i)
            (if (##not (##fixnum.= (bignum-digit-ref x i)
                                   (bignum-digit-ref y i)))
              #f
              (loop (##fixnum.- i 1)))
            #t))))))

(define (##bignum.< x y)
  (if (##not (##fixnum.= (bignum-sign x) (bignum-sign y)))
    (bignum-negative? x)
    (let ((lx (bignum-length x))
          (ly (bignum-length y)))
      (cond ((##fixnum.< lx ly)
             (bignum-positive? x))
            ((##fixnum.< ly lx)
             (bignum-negative? x))
            (else
             (let loop ((i (##fixnum.- lx 1)))
               (if (##fixnum.< 0 i)
                 (let ((dx (bignum-digit-ref x i))
                       (dy (bignum-digit-ref y i)))
                   (cond ((##fixnum.< dx dy) (bignum-positive? x))
                         ((##fixnum.< dy dx) (bignum-negative? x))
                         (else               (loop (##fixnum.- i 1)))))
                 #f)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Operations on fixnums that might result in a bignum

(define (##bignum.+/fixnum-fixnum x y)
  (if (##fixnum.negative? x)
    (if (##fixnum.negative? y)
      (let ((r (##fixnum.+ x y)))
        (if (##fixnum.negative? r)
          r
          (##bignum.+/bignum-fixnum ##bignum.2*min-fixnum32 r)))
      (##fixnum.+ x y))
    (if (##fixnum.negative? y)
      (##fixnum.+ x y)
      (let ((r (##fixnum.+ x y)))
        (if (##fixnum.negative? r)
          (##bignum.-/fixnum-bignum r ##bignum.2*min-fixnum32)
          r)))))

(define (##bignum.-/fixnum-fixnum x y)
  (if (##fixnum.negative? x)
    (if (##fixnum.negative? y)
      (##fixnum.- x y)
      (let ((r (##fixnum.- x y)))
        (if (##fixnum.negative? r)
          r
          (##bignum.+/bignum-fixnum ##bignum.2*min-fixnum32 r))))
    (if (##fixnum.negative? y)
      (let ((r (##fixnum.- x y)))
        (if (##fixnum.negative? r)
          (##bignum.-/fixnum-bignum r ##bignum.2*min-fixnum32)
          r))
      (##fixnum.- x y))))

(define (##bignum.*/fixnum-fixnum x y)
  (cond ((and (##not (##fixnum.< x (min-fixnum32-div-radix)))
              (##not (##fixnum.< (max-fixnum32-div-radix) x))
              (##fixnum.< (minus-radix) y)
              (##not (##fixnum.< (radix) y)))
         (##fixnum.* x y))
        ((or (##fixnum.zero? x) (##fixnum.zero? y))
         0)
        ((##fixnum.= x 1)
         y)
        ((##fixnum.= y 1)
         x)
        (else
         (##bignum.* (##bignum.<-fixnum x) (##bignum.<-fixnum y)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Mixed representation operations

(define (##bignum.+/bignum-fixnum x y)
  (##bignum.+ x (##bignum.<-fixnum y)))

(define (##bignum.-/bignum-fixnum x y)
  (##bignum.- x (##bignum.<-fixnum y)))

(define (##bignum.-/fixnum-bignum x y)
  (##bignum.- (##bignum.<-fixnum x) y))

(define (##bignum.*/bignum-fixnum x y)
  (cond ((##fixnum.zero? y)
         0)
        ((##fixnum.= y 1)
         x)
        (else
         (##bignum.* x (##bignum.<-fixnum y)))))

(define (##bignum.quotient/bignum-fixnum x y)
  (##bignum.quotient x (##bignum.<-fixnum y)))

(define (##bignum.quotient/fixnum-bignum x y)
  (##bignum.quotient (##bignum.<-fixnum x) y))

(define (##bignum.remainder/bignum-fixnum x y)
  (##bignum.remainder x (##bignum.<-fixnum y)))

(define (##bignum.remainder/fixnum-bignum x y)
  (##bignum.remainder (##bignum.<-fixnum x) y))

(define (##bignum.modulo/bignum-fixnum x y)
  (##bignum.modulo x (##bignum.<-fixnum y)))

(define (##bignum.modulo/fixnum-bignum x y)
  (##bignum.modulo (##bignum.<-fixnum x) y))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Operations where arguments are in bignum format

; Addition and substraction

(define (##bignum.+ x y)
  (##bignum.normalize (##bignum.sum x y (bignum-sign x) (bignum-sign y))))

(define (##bignum.- x y)
  (##bignum.normalize (##bignum.sum x y (bignum-sign x) (bignum-sign* y))))

(define (##bignum.sum x y sign-x sign-y)

  (define (adjust-sign! x s)
    (if (##fixnum.= (bignum-sign x) s)
      (bignum-set-positive! x)
      (bignum-set-negative! x))
    x)

  (cond ((##fixnum.= sign-x sign-y) ; same sign
         (adjust-sign! (##bignum.add x y) sign-x))
        ((##fixnum.< (bignum-length x) (bignum-length y))
         (adjust-sign! (##bignum.sub y x) sign-y))
        (else
         (adjust-sign! (##bignum.sub x y) sign-x))))

(define (##bignum.add x y)

  (define (add x y lx ly)
    (let ((r (bignum-make (##fixnum.+ lx 1))))

      (bignum-set-positive! r)

      (let loop1 ((i 1) (c 0)) ; add digits in y
        (if (##fixnum.< i ly)

          (let ((w (##fixnum.+ (##fixnum.+ (bignum-digit-ref x i)
                                           (bignum-digit-ref y i))
                               c)))
            (if (##fixnum.< w (radix))
              (begin
                (bignum-digit-set! r i w)
                (loop1 (##fixnum.+ i 1) 0))
              (begin
                (bignum-digit-set! r i (##fixnum.- w (radix)))
                (loop1 (##fixnum.+ i 1) 1))))

          (let loop2 ((i i) (c c)) ; propagate carry
            (if (##fixnum.< i lx)

              (let ((w (##fixnum.+ (bignum-digit-ref x i) c)))
                (if (##fixnum.< w (radix))
                  (begin
                    (bignum-digit-set! r i w)
                    (loop2 (##fixnum.+ i 1) 0))
                  (begin
                    (bignum-digit-set! r i (##fixnum.- w (radix)))
                    (loop2 (##fixnum.+ i 1) 1))))

              (if (##fixnum.zero? c)
                (bignum-shrink! r lx)
                (bignum-digit-set! r lx c))))))

      r))

  (let ((lx (bignum-length x))
        (ly (bignum-length y)))
    (if (##fixnum.< lx ly)
      (add y x ly lx)
      (add x y lx ly))))

(define (##bignum.sub x y)

  (define (sub x y lx ly)
    (let ((r (bignum-make lx)))

      (let loop1 ((i 1) (b 0)) ; substract digits in y
        (if (##fixnum.< i ly)

          (let ((w (##fixnum.- (##fixnum.- (bignum-digit-ref x i)
                                           (bignum-digit-ref y i))
                               b)))
            (if (##fixnum.negative? w)
              (begin
                (bignum-digit-set! r i (##fixnum.+ w (radix)))
                (loop1 (##fixnum.+ i 1) 1))
              (begin
                (bignum-digit-set! r i w)
                (loop1 (##fixnum.+ i 1) 0))))

          (let loop2 ((i i) (b b)) ; propagate borrow
            (if (##fixnum.< i lx)

              (let ((w (##fixnum.- (bignum-digit-ref x i) b)))
                (if (##fixnum.negative? w)
                  (begin
                    (bignum-digit-set! r i (##fixnum.+ w (radix)))
                    (loop2 (##fixnum.+ i 1) 1))
                  (begin
                    (bignum-digit-set! r i w)
                    (loop2 (##fixnum.+ i 1) 0))))

              (if (##fixnum.zero? b)
                (bignum-set-positive! r)
                (begin
                  (bignum-set-negative! r)
                  (##bignum.complement! r)))))))

      (##bignum.remove-leading-0s! r)

      r))
    
  (sub x y (bignum-length x) (bignum-length y)))

; Multiplication

(define (##bignum.* x y)

  (define (mul x y lx ly)
    (let ((r (bignum-make (##fixnum.- (##fixnum.+ lx ly) 1))))

      (if (##fixnum.= (bignum-sign x) (bignum-sign y))
        (bignum-set-positive! r)
        (bignum-set-negative! r))

      (let loop1 ((j 1)) ; for each digit in y
        (if (##fixnum.< j ly)

          (let ((d (bignum-digit-ref y j)))
            (let loop2 ((i 1) (k j) (c 0)) ; multiply and add
              (if (##fixnum.< i lx)

                (let ((w (##fixnum.+ (##fixnum.+ (bignum-digit-ref r k) c)
                                     (##fixnum.* (bignum-digit-ref x i) d))))
                  (bignum-digit-set! r k (##fixnum.logand w (radix-minus-1)))
                  (loop2 (##fixnum.+ i 1)
                         (##fixnum.+ k 1)
                         (##fixnum.ashr w (radix-width))))

                (begin
                  (bignum-digit-set! r k c)
                  (loop1 (##fixnum.+ j 1))))))))

      (##bignum.remove-leading-0s! r)

      r))

  (##bignum.normalize (mul x y (bignum-length x) (bignum-length y))))

; Division

(define (##bignum.quotient x y)
  (##bignum.normalize (##car (##bignum.div x y))))

(define (##bignum.remainder x y)
  (##bignum.normalize (##cdr (##bignum.div x y))))

(define (##bignum.modulo x y)
  (let ((r (##cdr (##bignum.div x y))))
    (if (bignum-zero? r)
      0
      (if (bignum-negative? x)
        (if (bignum-negative? y) (##bignum.normalize r) (##bignum.+ r y))
        (if (bignum-negative? y) (##bignum.+ r y) (##bignum.normalize r))))))

(define (##bignum.div x y)

  (define (single-digit-divisor-div x y lx ly r)

    ; simple algo for single digit divisor

    (let ((d (bignum-digit-ref y 1)))
      (let loop1 ((i (##fixnum.- lx 1)) (k 0))
        (if (##fixnum.< 0 i)
          (let ((w (##fixnum.+ (##fixnum.* k (radix)) (bignum-digit-ref x i))))
            (bignum-digit-set! r i (##fixnum.quotient w d))
            (loop1 (##fixnum.- i 1) (##fixnum.remainder w d)))
          (begin
            (##bignum.remove-leading-0s! r)
            (##cons r (##bignum.<-fixnum
                        (if (bignum-negative? x) (##fixnum.- k) k))))))))

  (define (multi-digit-divisor-div x y lx ly r)

    ; general algo from knuth

    ; STEP 1: normalize x and y

    (let loop2 ((shift 0)
                (n (##fixnum.* (bignum-digit-ref y (##fixnum.- ly 1)) 2)))
      (if (##fixnum.< n (radix))
        (loop2 (##fixnum.+ shift 1) (##fixnum.* n 2))

        (let ((nx (bignum-make (##fixnum.+ lx 1)))
              (ny (bignum-make ly)))

          (bignum-sign-set! nx (bignum-sign x))

          (let loop3 ((i 1) (c 0))
            (if (##fixnum.< i lx)
              (let ((w (##fixnum.+ (##fixnum.shl (bignum-digit-ref x i) shift) c)))
                (bignum-digit-set! nx i (##fixnum.logand w (radix-minus-1)))
                (loop3 (##fixnum.+ i 1) (##fixnum.ashr w (radix-width))))
              (bignum-digit-set! nx i c)))

          (let loop4 ((i 1) (c 0))
            (if (##fixnum.< i ly)
              (let ((w (##fixnum.+ (##fixnum.shl (bignum-digit-ref y i) shift) c)))
                (bignum-digit-set! ny i (##fixnum.logand w (radix-minus-1)))
                (loop4 (##fixnum.+ i 1) (##fixnum.ashr w (radix-width))))))

          (let loop5 ((i lx))
            (if (##not (##fixnum.< i ly))

              ; STEP 2: calculate next digit in quotient

              (let ((msd-of-ny
                     (bignum-digit-ref ny (##fixnum.- ly 1)))
                    (next-msd-of-ny
                     (bignum-digit-ref ny (##fixnum.- ly 2)))
                    (msd-of-nx
                     (bignum-digit-ref nx i))
                    (next-msd-of-nx
                     (bignum-digit-ref nx (##fixnum.- i 1)))
                    (next-next-msd-of-nx
                     (bignum-digit-ref nx (##fixnum.- i 2))))

                (define (next-digit q u)
                  (if (##fixnum.< u (radix))
                    (let* ((temp1 (##fixnum.* q next-msd-of-ny))
                           (temp2 (##fixnum.ashr temp1 (radix-width))))
                      (if (or (##fixnum.< u temp2)
                              (and (##fixnum.= temp2 u)
                                   (##fixnum.<
                                     next-next-msd-of-nx
                                     (##fixnum.logand temp1 (radix-minus-1)))))
                        (next-digit (##fixnum.- q 1) (##fixnum.+ u msd-of-ny))
                        q))
                    q))

                (let ((q (if (##fixnum.= msd-of-nx msd-of-ny)
                           (next-digit
                             (radix-minus-1)
                             (##fixnum.+ msd-of-ny next-msd-of-nx))
                           (let ((temp (##fixnum.+
                                         (##fixnum.* msd-of-nx (radix))
                                         next-msd-of-nx)))
                             (next-digit
                               (##fixnum.quotient temp msd-of-ny)
                               (##fixnum.modulo temp msd-of-ny))))))

                  ; STEP 3: multiply and substract

                  (let loop7 ((j 1)
                              (k (##fixnum.- i (##fixnum.- ly 1)))
                              (b 0))
                    (if (##fixnum.< j ly)

                      (let ((w (##fixnum.-
                                 (##fixnum.+ (bignum-digit-ref nx k) b)
                                 (##fixnum.* (bignum-digit-ref ny j) q))))
                        (bignum-digit-set! nx k
                          (##fixnum.logand w (radix-minus-1)))
                        (loop7 (##fixnum.+ j 1)
                               (##fixnum.+ k 1)
                               (##fixnum.ashr w (radix-width))))

                      (let ((w (##fixnum.+ (bignum-digit-ref nx k) b)))
                        (bignum-digit-set! nx k
                          (##fixnum.logand w (radix-minus-1)))
                        (if (##fixnum.negative? w)
                          (begin
                            (bignum-digit-set!
                              r
                              (##fixnum.- i (##fixnum.- ly 1))
                              (##fixnum.- q 1))
                            (let loop8 ((j 1)
                                        (k (##fixnum.- i (##fixnum.- ly 1)))
                                        (c 0))
                              (if (##fixnum.< j ly)

                                (let ((w (##fixnum.+
                                           (##fixnum.+
                                             (bignum-digit-ref nx k)
                                             (bignum-digit-ref ny j))
                                           c)))
                                  (bignum-digit-set!
                                    nx
                                    k
                                    (##fixnum.logand w (radix-minus-1)))
                                  (loop8 (##fixnum.+ j 1)
                                         (##fixnum.+ k 1)
                                         (##fixnum.ashr w (radix-width))))
                                (bignum-digit-set!
                                  nx
                                  k
                                  (##fixnum.logand
                                    (##fixnum.+ (bignum-digit-ref nx k) c)
                                    (radix-minus-1))))))
                            (bignum-digit-set!
                              r
                              (##fixnum.- i (##fixnum.- ly 1))
                              q))
                        (loop5 (##fixnum.- i 1)))))))))

          (let loop9 ((i (##fixnum.- ly 1)) (k 0))
            (if (##fixnum.< 0 i)
              (let ((w (##fixnum.+ (##fixnum.* k (radix))
                                   (bignum-digit-ref nx i))))
                (bignum-digit-set! nx i (##fixnum.ashr w shift))
                (loop9 (##fixnum.- i 1)
                       (##fixnum.logand
                         w
                         (##fixnum.- (##fixnum.shl 1 shift) 1))))))

          (##bignum.remove-leading-0s! nx)
          (##bignum.remove-leading-0s! r)
          (##cons r nx)))))

  (define (div x y lx ly)
    (if (##fixnum.< lx ly)

      (##cons ##bignum.0 x)

      (let ((r (bignum-make (##fixnum.+ (##fixnum.- lx ly) 2))))

        (if (##fixnum.= (bignum-sign x) (bignum-sign y))
          (bignum-set-positive! r)
          (bignum-set-negative! r))

        (if (##fixnum.= ly 2)
          (single-digit-divisor-div x y lx ly r)
          (multi-digit-divisor-div x y lx ly r)))))

  (div x y (bignum-length x) (bignum-length y)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Utilities:

(define (##bignum.copy x)
  (let ((len (bignum-length x)))
    (let ((y (bignum-make len)))
      (let loop ((i (##fixnum.- len 1)))
        (if (##fixnum.< i 0)
          y
          (begin
            (bignum-digit-set! y i (bignum-digit-ref x i))
            (loop (##fixnum.- i 1))))))))

(define (##bignum.remove-leading-0s! x)
  (let ((sign (bignum-sign x)))
    (bignum-sign-set! x 1) ; set to something different than 0
    (let loop ((i (##fixnum.- (bignum-length x) 1)))
      (if (##fixnum.zero? (bignum-digit-ref x i))
        (loop (##fixnum.- i 1))
        (bignum-shrink! x (##fixnum.+ i 1))))
    (bignum-sign-set! x sign)))

(define (##bignum.complement! r)
  (let ((lr (bignum-length r)))
    (let loop ((i 1) (c 0))
      (if (##fixnum.< i lr)
        (let ((w (##fixnum.+ (bignum-digit-ref r i) c)))
          (if (##fixnum.< 0 w)
            (begin
              (bignum-digit-set! r i (##fixnum.- (radix) w))
              (loop (##fixnum.+ i 1) 1))
            (begin
              (bignum-digit-set! r i 0)
              (loop (##fixnum.+ i 1) 0))))))))

(define (##bignum.normalize x)
  (let ((lx-minus-1 (##fixnum.- (bignum-length x) 1)))
    (if (##fixnum.< (max-digits-for-fixnum32) lx-minus-1)
      x
      (let loop ((n 0) (i lx-minus-1))
        (cond ((##fixnum.< 0 i)
               (if (##fixnum.< n (min-fixnum32-div-radix))
                 x
                 (let ((y (##fixnum.- (##fixnum.* n (radix))
                                      (bignum-digit-ref x i))))
                   (if (##fixnum.< y 0)
                     (loop y (##fixnum.- i 1))
                     x))))
              ((bignum-negative? x)
               n)
              (else
               (let ((n (##fixnum.- n)))
                 (if (##fixnum.negative? n) x n))))))))

(define (##bignum.<-fixnum n)
  (if (or (##fixnum.< n -16) (##fixnum.< 16 n))
    (##bignum.<-fixnum* n)
    (##vector-ref ##bignum.constants (##fixnum.+ n 16))))

(define (##bignum.<-fixnum* n)
  (let ((neg-n (if (##fixnum.negative? n) n (##fixnum.- n))))
    (let loop1 ((nb-digits 0) (x neg-n))
      (if (##not (##fixnum.zero? x))
        (loop1 (##fixnum.+ nb-digits 1)
               (##fixnum.ashr (##fixnum.+ x (radix-minus-1)) (radix-width)))
        (let ((r (bignum-make (##fixnum.+ nb-digits 1))))
          (if (##fixnum.negative? n)
            (bignum-set-negative! r)
            (bignum-set-positive! r))
          (let loop2 ((i 1) (x neg-n))
            (if (##not (##fixnum.zero? x))
              (begin
                (bignum-digit-set!
                  r
                  i
                  (##fixnum.logand (##fixnum.- (minus-radix) x)
                                   (radix-minus-1)))
                (loop2 (##fixnum.+ i 1)
                       (##fixnum.ashr (##fixnum.+ x (radix-minus-1))
                                      (radix-width))))
              r)))))))

(define ##bignum.constants (bignum-constants))

(define ##bignum.0
  (##bignum.<-fixnum 0))

(define ##bignum.2*min-fixnum32
  (##bignum.* (##bignum.<-fixnum (min-fixnum32)) (##bignum.<-fixnum 2)))

;------------------------------------------------------------------------------

; Ratnum operations
; -----------------

(define (##ratnum.= x y)
  (and (##= (ratnum-numerator x) (ratnum-numerator y))
       (##= (ratnum-denominator x) (ratnum-denominator y))))

(define (##ratnum.< x y)
  (##< (##* (ratnum-numerator x) (ratnum-denominator y))
       (##* (ratnum-denominator x) (ratnum-numerator y))))

(define (##ratnum.+ x y)
  (##ratnum.normalize
    (##+ (##* (ratnum-numerator x) (ratnum-denominator y))
         (##* (ratnum-denominator x) (ratnum-numerator y)))
    (##* (ratnum-denominator x) (ratnum-denominator y))))

(define (##ratnum.* x y)
  (##ratnum.normalize
    (##* (ratnum-numerator x) (ratnum-numerator y))
    (##* (ratnum-denominator x) (ratnum-denominator y))))

(define (##ratnum.- x y)
  (##ratnum.normalize
    (##- (##* (ratnum-numerator x) (ratnum-denominator y))
         (##* (ratnum-denominator x) (ratnum-numerator y)))
    (##* (ratnum-denominator x) (ratnum-denominator y))))

(define (##ratnum./ x y)
  (##ratnum.normalize
    (##* (ratnum-numerator x) (ratnum-denominator y))
    (##* (ratnum-denominator x) (ratnum-numerator y))))

(define (##ratnum.floor x)
  (let ((num (ratnum-numerator x))
        (den (ratnum-denominator x)))
    (if (##negative? num)
      (##quotient (##- num (##- den 1)) den)
      (##quotient num den))))

(define (##ratnum.ceiling x)
  (let ((num (ratnum-numerator x))
        (den (ratnum-denominator x)))
    (if (##negative? num)
      (##quotient num den)
      (##quotient (##+ num (##- den 1)) den))))

(define (##ratnum.truncate x)
  (##quotient (ratnum-numerator x) (ratnum-denominator x)))

(define (##ratnum.round x)
  (let ((num (ratnum-numerator x))
        (den (ratnum-denominator x)))
    (if (##eq? den 2)
      (if (##negative? num)
        (##* (##quotient (##- num 1) 4) 2)
        (##* (##quotient (##+ num 1) 4) 2))
      (##floor (##ratnum.normalize (##+ (##* num 2) den) (##* den 2))))))

(define (##ratnum.normalize num den)
  (let ((x (##gcd num den)))
    (let ((y (if (##negative? den) (##negate x) x)))
      (let ((num (##quotient num y))
            (den (##quotient den y)))
        (if (##eq? den 1)
          num
          (ratnum-make num den))))))

(define (##ratnum.<-exact-int x)
  (ratnum-make x 1))

;------------------------------------------------------------------------------

; Flonum operations
; -----------------

(define-system (##flonum.->fixnum x))

(define-system (##flonum.<-fixnum x))

(define-nary (##flonum.max x y)
  ()
  x
  (##flonum.max x y)
  no-force
  no-check)

(define-nary (##flonum.min x y)
  ()
  x
  (##flonum.min x y)
  no-force
  no-check)

(define-nary (##flonum.+ x y)
  (inexact-0)
  x
  (##flonum.+ x y)
  no-force
  no-check)

(define-nary (##flonum.* x y)
  (inexact-+1)
  x
  (##flonum.* x y)
  no-force
  no-check)

(define-nary (##flonum.- x y)
  ()
  (##flonum.- x)
  (##flonum.- x y)
  no-force
  no-check)

(define-nary (##flonum./ x y)
  ()
  (##flonum./ (inexact-+1) x)
  (##flonum./ x y)
  no-force
  no-check)

(define-system (##flonum.abs x))

(define-system (##flonum.floor x))
(define-system (##flonum.ceiling x))

(define-system (##flonum.truncate x)
  (let ((y (##flonum.floor x)))
    (if (and (##flonum.negative? x) (##not (##flonum.= x y)))
      (##flonum.+ y (inexact-+1))
      y)))

(define-system (##flonum.round x))

(define-system (##flonum.exp x))
(define-system (##flonum.log x))
(define-system (##flonum.sin x))
(define-system (##flonum.cos x))
(define-system (##flonum.tan x))
(define-system (##flonum.asin x))
(define-system (##flonum.acos x))
(define (##flonum.atan x #!optional (y (absent-obj)))
  (if (##eq? y (absent-obj))
    (##flonum.atan x)
    (##flonum.atan x y)))
(define-system (##flonum.sqrt x))
(define-system (##flonum.copysign x y))

(define-system (##flonum.zero? x)
  (##flonum.= x (inexact-0)))

(define-system (##flonum.positive? x)
  (##flonum.< (inexact-0) x))

(define-system (##flonum.negative? x)
  (##flonum.< x (inexact-0)))

(define-system (##flonum.finite? x)
  (##not (or (##flonum.nan? x)
             (##flonum.= x +inf.)
             (##flonum.= x -inf.))))

(define-system (##flonum.nan? x))

(define-nary-bool (##flonum.= x y)
  #t
  #t
  (##flonum.= x y)
  no-force
  no-check)

(define-nary-bool (##flonum.< x y)
  #t
  #t
  (##flonum.< x y)
  no-force
  no-check)

(define-nary-bool (##flonum.> x y)
  #t
  #t
  (##flonum.> x y)
  no-force
  no-check)

(define-nary-bool (##flonum.<= x y)
  #t
  #t
  (##flonum.<= x y)
  no-force
  no-check)

(define-nary-bool (##flonum.>= x y)
  #t
  #t
  (##flonum.>= x y)
  no-force
  no-check)

(define (##flonum.<-ratnum x)
  (let* ((num (ratnum-numerator x))
         (n (##abs num))
         (d (ratnum-denominator x))
         (wn (##exact-int.width n)) ; 2^(wn-1) <= n < 2^wn
         (wd (##exact-int.width d)) ; 2^(wd-1) <= d < 2^wd
         (p (##fixnum.- wn wd)))

    (define (f1 sn sd)
      (if (##< sn sd) ; n/(d*2^p) < 1 ?
        (f2 (##* sn 2) sd (##fixnum.- p 1))
        (f2 sn sd p)))

    (define (f2 a b p)
      ; 1 <= a/b < 2  and  n/d = (2^p*a)/b  and  n/d < 2^(p+1)
      (let* ((max-shift
              (##fixnum.- p (flonum-e-min)))
             (shift
              (if (##fixnum.< (flonum-m-bits) max-shift)
                (flonum-m-bits)
                max-shift))
             (abs-result
              (##flonum.*
               (##flonum.<-exact-int
                (##round
                 (##ratnum.normalize (##* a (##exact-int.expt2 shift)) b)))
               (##flonum.expt2 (##fixnum.- p shift)))))
        (if (##negative? num)
          (##flonum.copysign abs-result (inexact--1))
          abs-result)))

    ; 2^(p-1) <= n/d < 2^(p+1)
    ; 1/2 <= n/(d*2^p) < 2  or equivalently  1/2 <= (n*2^-p)/d < 2

    (if (##fixnum.negative? p)
      (f1 (##* n (##exact-int.expt2 (##fixnum.- p))) d)
      (f1 n (##* d (##exact-int.expt2 p))))))

(define (##flonum.<-bignum x)

  (define (f1 x)
    (let ((lx (bignum-length x)))
      (let loop ((i (##fixnum.- lx 1)) (res (inexact-0)))
        (if (##fixnum.< 0 i)
          (loop (##fixnum.- i 1)
                (##flonum.+ (##flonum.* res (inexact-radix))
                            (##flonum.<-fixnum (bignum-digit-ref x i))))
          res))))

  (define (f2 x)
    (let* ((w (##bignum.width x))                     ; 2^(w-1) <= x < 2^w
           (p (##fixnum.- w (flonum-m-bits-plus-1)))) ; 2^52 <= x/2^p < 2^53
      (if (##fixnum.< 0 p)
        (let* ((2^p (##exact-int.expt2 p))
               (a (##exact-int.div x 2^p))
               (q (##car a))
               (r (##cdr a))
               (r*2 (##* r 2)))
          (##flonum.*
           (##flonum.expt2 p)
           (f1 (if (or (and (##exact-int.even? q)
                            (##= r*2 2^p))
                       (##< r*2 2^p))
                 q
                 (##+ q 1)))))
          (f1 x))))

  (if (bignum-negative? x)
    (let ((abs-x (##bignum.copy x)))
      (bignum-set-positive! abs-x)
      (##flonum.copysign (f2 abs-x) (inexact--1)))
    (f2 x)))

(define (##flonum.<-exact-int x)
  (if (##fixnum? x)
    (##flonum.<-fixnum x)
    (##flonum.<-bignum x)))

(define (##exact-int.expt2 n) ; n >= 0
  (##expt 2 n))

(define (##exact-int.even? x)
  (if (##fixnum? x)
    (##fixnum.even? x)
    (bignum-even? x)))

(define (##flonum.expt2 n)
  (cond ((##fixnum.zero? n)
         (inexact-+1))
        ((##fixnum.negative? n)
         (##expt (inexact-+1/2) (##fixnum.- n)))
        (else
         (##expt (inexact-+2) n))))

(define (##flonum.->exact-int x)
  (let loop1 ((z (##flonum.abs x)) (n 1))
    (if (##flonum.< (inexact-radix) z)
      (loop1 (##flonum./ z (inexact-radix)) (##fixnum.+ n 1))
      (let loop2 ((res 0) (z z) (n n))
        (if (##fixnum.< 0 n)
          (let ((floor-of-z (##flonum.floor z)))
            (loop2 (##+ (##flonum.->fixnum floor-of-z) (##* res (radix)))
                   (##flonum.* (##flonum.- z floor-of-z) (inexact-radix))
                   (##fixnum.- n 1)))
          (if (##flonum.negative? x)
            (##negate res)
            res))))))

(define (##flonum.->inexact-exponential-format x)

  (define (exp-form-pos x y i)
    (let ((i*2 (##fixnum.+ i i)))
      (let ((z (if (and (##not (##fixnum.< (flonum-e-bias) i*2))
                        (##not (##flonum.< x y)))
                 (exp-form-pos x (##flonum.* y y) i*2)
                 (##vector x 0 1))))
        (let ((a (##vector-ref z 0)) (b (##vector-ref z 1)))
          (let ((i+b (##fixnum.+ i b)))
            (if (and (##not (##fixnum.< (flonum-e-bias) i+b))
                     (##not (##flonum.< a y)))
              (begin
                (##vector-set! z 0 (##flonum./ a y))
                (##vector-set! z 1 i+b)))
            z)))))

  (define (exp-form-neg x y i)
    (let ((i*2 (##fixnum.+ i i)))
      (let ((z (if (and (##fixnum.< i*2 (flonum-e-bias-minus-1))
                        (##flonum.< x y))
                 (exp-form-neg x (##flonum.* y y) i*2)
                 (##vector x 0 1))))
        (let ((a (##vector-ref z 0)) (b (##vector-ref z 1)))
          (let ((i+b (##fixnum.+ i b)))
            (if (and (##fixnum.< i+b (flonum-e-bias-minus-1))
                     (##flonum.< a y))
              (begin
                (##vector-set! z 0 (##flonum./ a y))
                (##vector-set! z 1 i+b)))
            z)))))

  (define (exp-form x)
    (if (##flonum.< x (inexact-+1))
      (let ((z (exp-form-neg x (inexact-+1/2) 1)))
        (##vector-set! z 0 (##flonum.* (inexact-+2) (##vector-ref z 0)))
        (##vector-set! z 1 (##fixnum.- -1 (##vector-ref z 1)))
        z)
      (exp-form-pos x (inexact-+2) 1)))

  (if (##flonum.negative? (##flonum.copysign (inexact-+1) x))
    (let ((z (exp-form (##flonum.copysign x (inexact-+1)))))
      (##vector-set! z 2 -1)
      z)
    (exp-form x)))

(define (##flonum.->exact-exponential-format x)
  (let ((z (##flonum.->inexact-exponential-format x)))
    (let ((y (##vector-ref z 0)))
      (if (##not (##flonum.< y (inexact-+2))) ; +inf. or +nan.?
        (begin
          (if (##flonum.< (inexact-0) y)
            (##vector-set! z 0 (flonum-+m-min))  ; +inf.
            (##vector-set! z 0 (flonum-+m-max))) ; +nan.
          (##vector-set! z 1 (flonum-e-bias-plus-1)))
        (##vector-set! z 0
          (##flonum.->exact-int
           (##flonum.* (##vector-ref z 0) (flonum-m-min)))))
      (##vector-set! z 1 (##fixnum.- (##vector-ref z 1) (flonum-m-bits)))
      z)))

(define (##flonum.inexact->exact x)
  (let* ((z (##flonum.->exact-exponential-format x))
         (y (##* (##vector-ref z 0) (##exact-int.expt2 (##vector-ref z 1)))))
    (if (##fixnum.negative? (##vector-ref z 2))
      (##negate y)
      y)))

(define (##flonum.->bits x)

  (define (bits a b)
    (if (##< a (flonum-+m-min))
      a
      (##+ (##- a (flonum-+m-min))
           (##* (##fixnum.+ (##fixnum.+ b (flonum-m-bits)) (flonum-e-bias))
                (flonum-+m-min)))))

  (let* ((z (##flonum.->exact-exponential-format x))
         (y (bits (##vector-ref z 0) (##vector-ref z 1))))
    (if (##fixnum.negative? (##vector-ref z 2))
      (##+ (flonum-sign-bit) y)
      y)))

(define (##flonum.->ratnum x)
  (let ((y (##flonum.inexact->exact x)))
    (if (exact-int? y)
      (##ratnum.<-exact-int y)
      y)))

;------------------------------------------------------------------------------

; Cpxnum operations
; -----------------

(define (##cpxnum.= x y)
  (and (##= (cpxnum-real x) (cpxnum-real y))
       (##= (cpxnum-imag x) (cpxnum-imag y))))

(define (##cpxnum.+ x y)
  (let ((a (cpxnum-real x)) (b (cpxnum-imag x))
        (c (cpxnum-real y)) (d (cpxnum-imag y)))
    (##make-rectangular (##+ a c) (##+ b d))))

(define (##cpxnum.* x y)
  (let ((a (cpxnum-real x)) (b (cpxnum-imag x))
        (c (cpxnum-real y)) (d (cpxnum-imag y)))
    (##make-rectangular (##- (##* a c) (##* b d)) (##+ (##* a d) (##* b c)))))

(define (##cpxnum.- x y)
  (let ((a (cpxnum-real x)) (b (cpxnum-imag x))
        (c (cpxnum-real y)) (d (cpxnum-imag y)))
    (##make-rectangular (##- a c) (##- b d))))

(define (##cpxnum./ x y)
  (let ((a (cpxnum-real x)) (b (cpxnum-imag x))
        (c (cpxnum-real y)) (d (cpxnum-imag y)))
    (let ((q (##+ (##* c c) (##* d d))))
      (##make-rectangular (##/ (##+ (##* a c) (##* b d)) q)
                          (##/ (##- (##* b c) (##* a d)) q)))))

(define (##cpxnum.<-non-cpxnum x)
  (cpxnum-make x 0))

;------------------------------------------------------------------------------

; number->string, string->number

(define (##number->string x rad)
  (##number->string* x rad #f))

(define (##number->string* x rad force-sign?)
  (number-dispatch x '()
    (##fixnum.number->string* x rad force-sign?)
    (##bignum.number->string* x rad force-sign?)
    (##ratnum.number->string* x rad force-sign?)
    (##flonum.number->string* x rad force-sign?)
    (##cpxnum.number->string* x rad force-sign?)))

(define (##exact-int.number->string* x rad force-sign?)
  (if (##fixnum? x)
    (##fixnum.number->string* x rad force-sign?)
    (##bignum.number->string* x rad force-sign?)))

(define ##digits "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ")

(define (##fixnum.number->string* n rad force-sign?)

  (define (loop k n i)
    (let ((x (##fixnum.quotient n rad)))
      (let ((s (if (##fixnum.zero? x)
                 (##make-string (##fixnum.+ i k) #\space)
                 (loop k x (##fixnum.+ i 1)))))
        (##string-set!
         s
         (##fixnum.- (##string-length s) i)
         (##string-ref ##digits
                       (##fixnum.- (##fixnum.remainder n rad))))
        s)))

  (if (##fixnum.negative? n)
    (let ((s (loop 1 n 1)))
      (##string-set! s 0 #\-)
      s)
    (if force-sign?
      (let ((s (loop 1 (##fixnum.- n) 1)))
        (##string-set! s 0 #\+)
        s)
      (loop 0 (##fixnum.- n) 1))))

(define (##bignum.number->string* n rad force-sign?)

  (define (bignum->string n rad r r-log-rad radix-log-r-num)
    (let ((len (##fixnum.* (##fixnum.quotient
                             (##fixnum.+
                               (##fixnum.* (##fixnum.- (bignum-length n) 1)
                                           radix-log-r-num)
                               (##fixnum.- (radix-log-den) 1))
                             (radix-log-den))
                           r-log-rad)))
      (let ((n (##bignum.copy n))
            (s (##make-string (##fixnum.+ len 1) #\space)))

        (define (put-digits k i)
          (let loop1 ((k k) (i i) (j r-log-rad) (last-non-zero i))
            (if (##fixnum.< 0 j)
              (let ((d (##fixnum.remainder k rad)))
                (##string-set! s i (##string-ref ##digits d))
                (loop1 (##fixnum.quotient k rad)
                       (##fixnum.- i 1)
                       (##fixnum.- j 1)
                       (if (##fixnum.zero? d) last-non-zero i)))
              last-non-zero)))

        (define (move-digits i j)
          (let loop2 ((i i) (j j))
            (if (##fixnum.< len i)
              (##string-shrink! s j)
              (begin
                (##string-set! s j (##string-ref s i))
                (loop2 (##fixnum.+ i 1) (##fixnum.+ j 1))))))

        (let loop3 ((i len))

          (let ((k
                 ; k = next digit in base `r'
                 ; use simple algo for dividing in place by `r'
                 ; (which is known to be less than or equal to radix)

                 (let loop4 ((j (##fixnum.- (bignum-length n) 1)) (k 0))
                   (if (##fixnum.< 0 j)
                     (let ((x (##fixnum.+ (##fixnum.* k (radix))
                                          (bignum-digit-ref n j))))
                       (bignum-digit-set! n j (##fixnum.quotient x r))
                       (loop4 (##fixnum.- j 1) (##fixnum.remainder x r)))
                     k))))

            (let ((last-non-zero (put-digits k i)))
              (##bignum.remove-leading-0s! n)
              (if (##not (bignum-zero? n))
                (loop3 (##fixnum.- i r-log-rad))
                (if (bignum-negative? n)
                  (begin
                    (##string-set! s 0 #\-)
                    (move-digits last-non-zero 1))
                  (if force-sign?
                    (begin
                      (##string-set! s 0 #\+)
                      (move-digits last-non-zero 1))
                    (move-digits last-non-zero 0))))))))))

  (cond ((##fixnum.= rad 2)
         (bignum->string n rad (r.2) (r-log-rad.2) (radix-log-r-num.2)))
        ((##fixnum.= rad 8)
         (bignum->string n rad (r.8) (r-log-rad.8) (radix-log-r-num.8)))
        ((##fixnum.= rad 10)
         (bignum->string n rad (r.10) (r-log-rad.10) (radix-log-r-num.10)))
        (else
         (bignum->string n rad (r.16) (r-log-rad.16) (radix-log-r-num.16)))))

(define (##ratnum.number->string* x rad force-sign?)
  (##string-append
   (##exact-int.number->string* (ratnum-numerator x) rad force-sign?)
   "/"
   (##exact-int.number->string* (ratnum-denominator x) rad #f)))

(##define-macro (chez-fp-syntax) #f)

(define ##10^-constants (10^-constants))

(define (##flonum.printout v sign-prefix)

  ; This algorithm is derived from the paper "Printing Floating-Point
  ; Numbers Quickly and Accurately" by Robert G. Burger and R. Kent Dybvig,
  ; SIGPLAN'96 Conference on Programming Language Design an Implementation.

  ; v is a flonum
  ; f is an exact integer (fixnum or bignum)
  ; e is an exact integer (fixnum only)

  (##define-macro (exactintquotient x y)
    `(##quotient ,x ,y))

  (##define-macro (exactintremainder x y)
    `(##remainder ,x ,y))

  (##define-macro (exactint* x y)
    `(##* ,x ,y))

  (##define-macro (exactint+ x y)
    `(##+ ,x ,y))

  (##define-macro (exactint= x y)
    `(let ((x ,x) (y ,y))
       (if (##fixnum? x)
         (if (##fixnum? y)
           (##fixnum.= x y)
           #f)
         (if (##fixnum? y)
           #f
           (##bignum.= x y)))))

  (##define-macro (exactint< x y)
    `(let ((x ,x) (y ,y))
       (if (##fixnum? x)
         (if (##fixnum? y)
           (##fixnum.< x y)
           (bignum-positive? y))
         (if (##fixnum? y)
           (bignum-negative? x)
           (##bignum.< x y)))))

  (##define-macro (exactinteven? x)
    `(let ((x ,x))
       (if (##fixnum? x)
         (##fixnum.even? x)
         (bignum-even? x))))

  (##define-macro (2^ n)
    `(##exact-int.expt2 ,n))

  (define (10^ n) ; 0 <= n < 326
    (##vector-ref ##10^-constants n))

  (define (base-10-log x)
    (##define-macro (1/log10) `',(/ (log 10)))
    (##flonum.* (##flonum.log x) (1/log10)))

  (##define-macro (epsilon)
    1e-10)

  (define (scale r s m+ m- round? v)

    ; r is an exact integer (fixnum or bignum)
    ; s is an exact integer (fixnum or bignum)
    ; m+ is an exact integer (fixnum or bignum)
    ; m- is an exact integer (fixnum or bignum)
    ; round? is a boolean
    ; v is a flonum

    (let ((est
           (##flonum.->fixnum
            (##flonum.ceiling (##flonum.- (base-10-log v) (epsilon))))))
      (if (##fixnum.negative? est)
        (let ((factor (10^ (##fixnum.- est))))
          (fixup (exactint* r factor)
                 s
                 (exactint* m+ factor)
                 (exactint* m- factor)
                 est
                 round?))
        (fixup r
               (exactint* s (10^ est))
               m+
               m-
               est
               round?))))

  (define (fixup r s m+ m- k round?)
    (if (if round?
          (##not (exactint< (exactint+ r m+) s))
          (exactint< s (exactint+ r m+)))
      (##cons (##fixnum.+ k 1)
              (generate r
                        s
                        m+
                        m-
                        round?
                        0))
      (##cons k
              (generate (exactint* r 10)
                        s
                        (exactint* m+ 10)
                        (exactint* m- 10)
                        round?
                        0))))

  (define (generate r s m+ m- round? n)
    (let* ((d (exactintquotient r s))
           (r (exactintremainder r s))
           (tc (if round?
                 (##not (exactint< (exactint+ r m+) s))
                 (exactint< s (exactint+ r m+)))))
      (if (if round? (##not (exactint< m- r)) (exactint< r m-))
        (let ((last-digit
               (if tc
                 (let ((r*2 (exactint* r 2)))
                   (if (or (and (##fixnum.even? d)
                                (exactint= r*2 s)) ; tie -> round d to even
                           (exactint< r*2 s))
                     d
                     (##fixnum.+ d 1)))
                 d)))
          (##make-string (##fixnum.+ n 1)
                         (##string-ref ##digits last-digit)))
        (if tc
          (##make-string (##fixnum.+ n 1)
                         (##string-ref ##digits (##fixnum.+ d 1)))
          (let ((str
                 (generate (exactint* r 10)
                           s
                           (exactint* m+ 10)
                           (exactint* m- 10)
                           round?
                           (##fixnum.+ n 1))))
            (##string-set! str n (##string-ref ##digits d))
            str)))))

  (define (flonum->exponent-and-digits v)
    (let* ((x (##flonum.->exact-exponential-format v))
           (f (##vector-ref x 0))
           (e (##vector-ref x 1))
           (round? (exactinteven? f)))
      (if (##fixnum.negative? e)
        (if (and (##not (##fixnum.= e (flonum-e-min)))
                 (exactint= f (flonum-+m-min)))
          (scale (exactint* f 4)
                 (2^ (##fixnum.- 2 e))
                 2
                 1
                 round?
                 v)
          (scale (exactint* f 2)
                 (2^ (##fixnum.- 1 e))
                 1
                 1
                 round?
                 v))
        (let ((2^e (2^ e)))
          (if (exactint= f (flonum-+m-min))
            (scale (exactint* (exactint* f 2^e) 4)
                   4
                   (exactint* 2^e 2)
                   2^e
                   round?
                   v)
            (scale (exactint* (exactint* f 2^e) 2)
                   2
                   2^e
                   2^e
                   round?
                   v))))))

  (let* ((x (flonum->exponent-and-digits v))
         (e (##car x))
         (d (##cdr x))            ; d = digits
         (n (##string-length d))) ; n = number of digits

    (cond ((and (##not (##fixnum.< e 0)) ; 0<=e<=10
                (##not (##fixnum.< 10 e)))

           (cond ((##fixnum.= e 0) ; e=0

                  ; Format 1: .DDD    (0.DDD in chez-fp-syntax)

                  (##string-append sign-prefix
                                   (if (chez-fp-syntax) "0." ".")
                                   d))

                 ((##fixnum.< e n) ; e<n

                  ; Format 2: D.DDD up to DDD.D

                  (##string-append sign-prefix
                                   (##substring d 0 e)
                                   "."
                                   (##substring d e n)))

                 ((##fixnum.= e n) ; e=n

                  ; Format 3: DDD.    (DDD.0 in chez-fp-syntax)

                  (##string-append sign-prefix
                                   d
                                   (if (chez-fp-syntax) ".0" ".")))

                 (else ; e>n

                  ; Format 4: DDD000000.    (DDD000000.0 in chez-fp-syntax)

                  (##string-append sign-prefix
                                   d
                                   (##make-string (##fixnum.- e n) #\0)
                                   (if (chez-fp-syntax) ".0" ".")))))

          ((and (##not (##fixnum.< e -2)) ; -2<=e<=-1
                (##not (##fixnum.< -1 e)))

           ; Format 5: .0DDD or .00DDD    (0.0DDD or 0.00DDD in chez-fp-syntax)

           (##string-append sign-prefix
                            (if (chez-fp-syntax) "0." ".")
                            (##make-string (##fixnum.- e) #\0)
                            d))

          (else

           ; Format 6: D.DDDeEEE
           ;
           ; This is the most general format.  We insert a period after
           ; the first digit (unless there is only one digit) and add
           ; an exponent.

           (##string-append sign-prefix
                            (##substring d 0 1)
                            (if (##fixnum.= n 1) "" ".")
                            (##substring d 1 n)
                            "e"
                            (##number->string (##fixnum.- e 1) 10))))))

(define (##flonum.number->string* x rad force-sign?)

  (define (non-neg-num->str x rad sign-prefix)
    (if (##flonum.zero? x)
      (##string-append sign-prefix (if (chez-fp-syntax) "0.0" "0."))
      (##flonum.printout x sign-prefix)))

  (cond ((##flonum.nan? x)
         (##string-copy (if (chez-fp-syntax) "+nan.0" "+nan.")))
        ((##flonum.negative? (##flonum.copysign (inexact-+1) x))
         (let ((abs-x (##flonum.copysign x (inexact-+1))))
           (cond ((##flonum.= abs-x (inexact-+inf))
                  (##string-copy (if (chez-fp-syntax) "-inf.0" "-inf.")))
                 (else
                  (non-neg-num->str abs-x rad "-")))))
        (else
         (cond ((##flonum.= x (inexact-+inf))
                (##string-copy (if (chez-fp-syntax) "+inf.0" "+inf.")))
               (force-sign?
                (non-neg-num->str x rad "+"))
               (else
                (non-neg-num->str x rad ""))))))

(define (##cpxnum.number->string* x rad force-sign?)
  (let* ((real
          (cpxnum-real x))
         (real-str
          (if (##eq? real 0) "" (##number->string* real rad force-sign?))))
    (let ((imag (cpxnum-imag x)))
      (cond ((##eq? imag 1)
             (##string-append real-str "+i"))
            ((##eq? imag -1)
             (##string-append real-str "-i"))
            (else
             (##string-append real-str
                              (##number->string* imag rad #t)
                              "i"))))))

(define (number->string n #!optional (r (absent-obj)))
  (force-vars (n r)
    (let ((rad (if (##eq? r (absent-obj)) 10 r)))

      (define (range-error)
        (##trap-check-range 'number->string n r))

      (define (type-error1)
        (##trap-check-number 'number->string n r))

      (define (type-error2)
        (##trap-check-exact-int 'number->string n r))

      (if (exact-int? rad)
        (if (or (##eq? rad 2)
                (##eq? rad 8)
                (##eq? rad 10)
                (##eq? rad 16))
          (let ((result (##number->string* n rad #f)))
            (if (##null? result)
              (type-error1)
              result))
          (range-error))
        (type-error2)))))

(define (##string->number s rad)

  (define (make-real e n r p)
    (let ((x (##* n (##expt r p))))
      (if (##eq? e 'e) x (##exact->inexact x))))

  (define (make-nan)
    (inexact-+nan))

  (define (make-inf)
    (inexact-+inf))

  (define (make-rec a b)
    (##make-rectangular a b))

  (define (make-pol a b)
    (##make-polar a b))

  (define (ex e x)
    (if (##eq? e 'i) (##exact->inexact x) x))

  (define (setsign +/- x)
    (if (##eq? +/- '-)
      (if (##flonum? x) (##flonum.copysign x (inexact--1)) (##negate x))
      x))

  (define (end s i x)
    (if (##eq? i (##string-length s)) x #f))

  (define (radix-prefix s i)
    (if (##fixnum.< (##fixnum.+ i 1) (##string-length s))
      (if (##char=? (##string-ref s i) #\#)
        (let ((c (##string-ref s (##fixnum.+ i 1))))
          (cond ((or (##char=? c #\b) (##char=? c #\B))  2)
                ((or (##char=? c #\o) (##char=? c #\O))  8)
                ((or (##char=? c #\d) (##char=? c #\D)) 10)
                ((or (##char=? c #\x) (##char=? c #\X)) 16)
                (else                                   #f)))
        #f)
      #f))

  (define (exactness-prefix s i)
    (if (##fixnum.< (##fixnum.+ i 1) (##string-length s))
      (if (##char=? (##string-ref s i) #\#)
        (let ((c (##string-ref s (##fixnum.+ i 1))))
          (cond ((or (##char=? c #\i) (##char=? c #\I)) 'i)
                ((or (##char=? c #\e) (##char=? c #\E)) 'e)
                (else                                   #f)))
        #f)
      #f))

  (define (sign s i)
    (if (##fixnum.< i (##string-length s))
      (let ((c (##string-ref s i)))
        (cond ((##char=? c #\+) '+)
              ((##char=? c #\-) '-)
              (else             #f)))
      #f))

  (define (imaginary s i)
    (if (##fixnum.< i (##string-length s))
      (and (let ((c (##string-ref s i)))
             (or (##char=? c #\i) (##char=? c #\I)))
           (or (##fixnum.= (##fixnum.+ i 1) (##string-length s))
               (let ((c (##string-ref s (##fixnum.+ i 1))))
                 (##not (or (##char=? c #\n) (##char=? c #\N))))))
      #f))

  (define (nan s i)
    (if (##fixnum.< (##fixnum.+ i (if (chez-fp-syntax) 4 3))
                    (##string-length s))
      (and (let ((c (##string-ref s i)))
             (or (##char=? c #\n) (##char=? c #\N)))
           (let ((c (##string-ref s (##fixnum.+ i 1))))
             (or (##char=? c #\a) (##char=? c #\A)))
           (let ((c (##string-ref s (##fixnum.+ i 2))))
             (or (##char=? c #\n) (##char=? c #\N)))
           (##char=? (##string-ref s (##fixnum.+ i 3)) #\.)
           (if (chez-fp-syntax)
             (##char=? (##string-ref s (##fixnum.+ i 4)) #\0)
             #t))
      #f))

  (define (inf s i)
    (if (##fixnum.< (##fixnum.+ i (if (chez-fp-syntax) 4 3))
                    (##string-length s))
      (and (let ((c (##string-ref s i)))
             (or (##char=? c #\i) (##char=? c #\I)))
           (let ((c (##string-ref s (##fixnum.+ i 1))))
             (or (##char=? c #\n) (##char=? c #\N)))
           (let ((c (##string-ref s (##fixnum.+ i 2))))
             (or (##char=? c #\f) (##char=? c #\F)))
           (##char=? (##string-ref s (##fixnum.+ i 3)) #\.)
           (if (chez-fp-syntax)
             (##char=? (##string-ref s (##fixnum.+ i 4)) #\0)
             #t))
      #f))

  (define (polar s i)
    (if (##fixnum.< i (##string-length s))
      (##char=? (##string-ref s i) #\@)
      #f))

  (define (ratio s i)
    (if (##fixnum.< i (##string-length s))
      (##char=? (##string-ref s i) #\/)
      #f))

  (define (exponent s i)
    (if (##fixnum.< i (##string-length s))
      (let ((c (##string-ref s i)))
        (cond ((or (##char=? c #\e) (##char=? c #\E)) 'e)
              ((or (##char=? c #\s) (##char=? c #\S)) 's)
              ((or (##char=? c #\f) (##char=? c #\F)) 'f)
              ((or (##char=? c #\d) (##char=? c #\D)) 'd)
              ((or (##char=? c #\l) (##char=? c #\L)) 'l)
              (else                                   #f)))
      #f))

  (define (digit c r)
    (let ((d (cond ((##not (or (##char<? c #\0) (##char<? #\9 c)))
                    (##fixnum.- (##fixnum.<-char c) 48))
                   ((##not (or (##char<? c #\a) (##char<? #\z c)))
                    (##fixnum.- (##fixnum.<-char c) 87))
                   ((##not (or (##char<? c #\A) (##char<? #\Z c)))
                    (##fixnum.- (##fixnum.<-char c) 55))
                   (else
                    #f))))
      (if (and d (##fixnum.< d r)) d #f)))

  (define (prefix s i r cont)
    (let ((e1 (exactness-prefix s i)))
      (if e1
        (let ((r1 (radix-prefix s (##fixnum.+ i 2))))
          (if r1
            (cont s (##fixnum.+ i 4) r1 e1)
            (cont s (##fixnum.+ i 2) r e1)))
        (let ((r2 (radix-prefix s i)))
          (if r2
            (let ((e2 (exactness-prefix s (##fixnum.+ i 2))))
              (if e2
                (cont s (##fixnum.+ i 4) r2 e2)
                (cont s (##fixnum.+ i 2) r2 #f)))
            (cont s i r #f))))))

  (define (num s i r)
    (prefix s i r complex))

  (define (complex s i r e)
    (let ((+/- (sign s i)))
      (ucomplex s (if +/- (##fixnum.+ i 1) i) r e +/-)))

  (define (ucomplex s i r e +/-)
    (if (and +/- (imaginary s i))
      (end s (##fixnum.+ i 1)
        (make-rec (ex e 0) (ex e (if (##eq? +/- '-) -1 1))))
      (ureal s i r e +/- #f
        (lambda (s i r e +/- dummy x)
          (let ((y (setsign +/- x)))
            (cond ((and +/- (imaginary s i))
                   (end s (##fixnum.+ i 1) (make-rec (ex e 0) y)))
                  ((polar s i)
                   (let ((+/-2 (sign s (##fixnum.+ i 1))))
                     (ureal s (##fixnum.+ i (if +/-2 2 1)) r e +/-2 y
                       (lambda (s i r e +/-2 y z)
                         (end s i
                           (make-pol y (setsign +/-2 z)))))))
                  (else
                   (let ((+/-2 (sign s i)))
                     (if +/-2
                       (if (imaginary s (##fixnum.+ i 1))
                         (end s (##fixnum.+ i 2)
                           (make-rec y (ex e (if (##eq? +/-2 '-) -1 1))))
                         (ureal s (##fixnum.+ i 1) r e +/-2 y
                           (lambda (s i r e +/-2 y z)
                             (and (imaginary s i)
                                  (end s (##fixnum.+ i 1)
                                    (make-rec y (setsign +/-2 z)))))))
                       (end s i y))))))))))

  (define (ureal s i r e +/- x cont)
    (if (and (##eq? +/- '+) (nan s i))
      (cont s (##fixnum.+ i (if (chez-fp-syntax) 5 4)) r e +/- x (make-nan))
      (if (and +/- (inf s i))
        (cont s (##fixnum.+ i (if (chez-fp-syntax) 5 4)) r e +/- x (make-inf))
        (uinteger s i r e +/- x cont (##eq? r 10)
          (lambda (s i r e +/- x cont ex? n p)
            (if p ; decimal point or exponent?
              (cont s i r e +/- x (make-real e n r p))
              (if (ratio s i)
                (uinteger s (##fixnum.+ i 1) r e +/- x cont #f
                  (lambda (s i r e +/- x cont ex2? n2 p2)
                    (let ((y (##/ n n2)))
                      (cont s i r e +/- x (ex (or e (if (and ex? ex2?) #f 'i)) y)))))
                (cont s i r e +/- x (ex (or e (if ex? #f 'i)) n)))))))))
  
  (define (uinteger s i r a1 a2 a3 a4 decimal? cont)
    (let loop1 ((i i) (state 0) (n 0) (p #f))

      (define (suffix)
        (if (##eq? state 0)
          #f
          (let ((mark (exponent s i)))
            (if (and mark decimal?)
              (let ((+/- (sign s (##fixnum.+ i 1))) (p (or p 0)))
                (let loop2 ((i (##fixnum.+ i (if +/- 2 1))) (j #f))
                  (if (and (##fixnum.< i (##string-length s))
                           (digit (##string-ref s i) 10))
                    (loop2 (##fixnum.+ i 1)
                           (##+ (##* (or j 0) 10)
                                (digit (##string-ref s i) 10)))
                    (and j (cont s i r a1 a2 a3 a4 #f n
                             (##+ p (if (##eq? +/- '-) (##negate j) j)))))))
              (cont s i r a1 a2 a3 a4 (##not (or (##eq? state 2) p)) n p)))))

      (if (##fixnum.< i (##string-length s))
        (let ((c (##string-ref s i)))
          (if (and (##char=? c #\.) decimal? (##not p))
            (loop1 (##fixnum.+ i 1) state n 0)
            (if (and (##char=? c #\#) (##fixnum.< 0 state))
              (loop1 (##fixnum.+ i 1) 2 (##* n r) (and p (##fixnum.- p 1)))
              (if (##fixnum.< state 2)
                (let ((d (digit c r)))
                  (if d
                    (loop1 (##fixnum.+ i 1)
                           1
                           (##+ (##* n r) d)
                           (and p (##fixnum.- p 1)))
                    (suffix)))
                (suffix)))))
        (suffix))))

  (num s 0 rad))

(define (string->number s #!optional (r (absent-obj)))
  (force-vars (s r)
    (check-string s (string->number s r)
      (let ((rad (if (##eq? r (absent-obj)) 10 r)))

        (define (range-error)
          (##trap-check-range 'string->number s r))

        (define (type-error)
          (##trap-check-exact-int 'string->number s r))

        (if (exact-int? rad)
          (if (or (##eq? rad 2)
                  (##eq? rad 8)
                  (##eq? rad 10)
                  (##eq? rad 16))
            (##string->number s rad)
            (range-error))
          (type-error))))))

;------------------------------------------------------------------------------
