package biss.jde;

import biss.FileLib;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.PrintStream;
import java.util.BitSet;
import java.util.Vector;

/**
 * Parser to analyze CUBrowser level information (types, imports, fields
 * methods) of a CompileUnit (Java source)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
class ParseException
  extends Exception
{
	String Msg;

public ParseException ( String msg ) {
	Msg = msg;
}

public ParseException ( String msg, JavaScanner scan ) {
	String found;
	if ( scan.Tok == JavaScanner.CHAR || scan.Tok > 0 )
		found = "'" + scan.character() + "'";
	else if ( scan.Tok == JavaScanner.STRING )
		found = "\"" + scan.string() + "\"";
	else
		found = scan.keyword( scan.Tok);

	Msg = msg + ", found " + found;
}

public String toString() {
	return Msg;
}
}

class Parser
{
	PrintStream Out = System.out;
	ByteArrayOutputStream OutBuffer;
	String FileName;
	byte Src[];
	JavaScanner Scan;
	int Level = 0;
	String CurNt;
// current nonterminal
	int CurLn;
// current line of nonterminal
	CompileUnit CU;
	TypeDecl TDecl;
	int IStart;
	int LStart;
	static BitSet CorI_Modifier;
	static BitSet MorD_Modifier;

static {
	CorI_Modifier = new BitSet( JavaScanner.NKeys);
	CorI_Modifier.set( -JavaScanner.PUBLIC );
	CorI_Modifier.set( -JavaScanner.FINAL );
	CorI_Modifier.set( -JavaScanner.ABSTRACT );
	CorI_Modifier.set( -JavaScanner.SYNCHRONIZED );

	MorD_Modifier = new BitSet ( JavaScanner.NKeys);
	MorD_Modifier.set( -JavaScanner.PRIVATE);
	MorD_Modifier.set( -JavaScanner.PROTECTED);
	MorD_Modifier.set( -JavaScanner.PUBLIC);
	MorD_Modifier.set( -JavaScanner.STATIC);
	MorD_Modifier.set( -JavaScanner.FINAL);
	MorD_Modifier.set( -JavaScanner.ABSTRACT);
	MorD_Modifier.set( -JavaScanner.NATIVE);
	MorD_Modifier.set( -JavaScanner.SYNCHRONIZED);
	MorD_Modifier.set( -JavaScanner.TRANSIENT);
	MorD_Modifier.set( -JavaScanner.VOLATILE);
}

// to enable subclassing
protected Parser () {}

public Parser ( String src ) {
	int len = src.length();

	Src = new byte[len];
	src.getBytes( 0, len, Src, 0);
	FileName = "<text>";
	Scan = new JavaScanner( Src);
}

public Parser ( String fname, byte src[] ) {
	FileName = fname;
	Src = src;
	Scan = new JavaScanner( Src);
}

int array_dim () throws ParseException {
	// logEnter( "array_spec");
	int dim = 0;

	while ( isMatchValChar( '[') ){
		matchValChar( ']');
		dim++;
	}

	// logExit( String.valueOf(dim));
	return dim;
}

void class_decl ( BitSet mods ) throws ParseException {
	CurNt = "class decl"; CurLn = Scan.Line;
	// logEnter( "class_decl");

	String id;
	String base;
	Vector interfaces;

	// match( Scan.CLASS );
	id = matchString( "class name expected");
	ClassDecl cls = new ClassDecl( mods, id);
	cls.Comment = Scan.lastComment();
	cls.Line = LStart;
	TDecl = cls;
	CU.addTypeDecl( TDecl);

	if ( isMatch( Scan.EXTENDS ) ){
		base = matchString( "base class name expected");
		cls.setSuper( base);
	}

	if ( isMatch( Scan.IMPLEMENTS) ){
		interfaces = id_list();
		TDecl.setInterfaces( interfaces);
	}

	matchValChar( '{' );
	while ( !isMatchValChar( '}') )
		field_decl();

	// logExit( id);
}

//////////////////////////////////////////////////////////// nonteminal funcs
void compile_unit () throws ParseException {
	// logEnter( "compile_unit");

	CU = new CompileUnit( FileName);

	if ( isMatch( Scan.PACKAGE ) )
		package_spec();

	while ( isMatch( Scan.IMPORT ) )
		import_spec();

	while ( Scan.Tok != Scan.UNKNOWN )
		type_decl();

	// logExit( "finished");
}

void data_decl ( BitSet mods, String id, String type )
	throws ParseException {
	CurNt = "data"; CurLn = Scan.Line;
	// logEnter( "data_decl");

	String  array;
	String  initExpr;
	Data    data = new Data( mods, type, id);
	data.Line = LStart;
	data.Comment = Scan.lastComment();
	do {
		if ( Scan.Tok == '[' ) {
			int dim = array_dim();
			array = "[]";
			for ( int l=1; l<dim; l++ ) array += "[]";
			data.setArrayDim( array);
		}

		if ( Scan.Tok == '=' ){
			Scan.skipBlanks();
			Scan.setStartMark();
			Scan.skipExpr();         // skips to ';'
			initExpr = Scan.string();
			data.setInitExpr( initExpr);

			Scan.nextToken();
		}

		TDecl.addData( data);

		if ( isMatchValChar( ',') ){
			id = matchString( "data id expected");
			data = new Data( mods, type, id);
			continue;
		}
		else
			break;
	} while ( true );

	isMatchValChar( ';');  // ';' might be required otherwise (init-expr)

	// logExit( type + ' ' + id + array + " = " + initExpr);
}

void field_decl () throws ParseException {
	CurNt = "field decl"; CurLn = Scan.Line;
	// logEnter( "field_decl");

	String type, id;
	BitSet mods = new BitSet( Scan.NKeys);

	LStart = Scan.Line;

	for ( int t = Scan.Tok; isMatchSet( MorD_Modifier); t = Scan.Tok )
		mods.set( -t);  // Token ids are negative

	IStart = Scan.I0;
	if ( Scan.Tok == '{' ) {
		static_init();
	}
	else {
		type = res_type();
		if ( Scan.Tok == '(' ){           // ctor (if type == clsName)
			method_decl( mods, type, "");
		}
		else {
			id = matchString( "method or data id expected");

			if ( Scan.Tok == '(' )
				method_decl( mods, id, type);
			else
				data_decl( mods, id, type);
		}
	}

	while ( isMatchValChar( ';'));

	// logExit( "");
}

public ByteArrayOutputStream getOutBuffer () {
	return OutBuffer;
}

Vector id_list () throws ParseException {
	// logEnter( "id_list");

	Vector list = new Vector( 3);
	String s;

	do {
		s = matchString( "id [, id..] expected");
		list.addElement( s);
	} while ( isMatchValChar( ',') );

	// logExit( "");
	return list;
}

void import_spec () throws ParseException {
	CurNt = "import"; CurLn = Scan.Line;
	// logEnter( "import_spec");

	String importSpec;
	// match ( Scan.IMPORT);
	importSpec = matchString( "import name expected");
	CU.addImport( new Import(importSpec));
	while ( isMatchValChar( ';'));

	// logExit( importSpec);
}

void interface_decl ( BitSet mods ) throws ParseException {
	CurNt = "interface"; CurLn = Scan.Line;
	// logEnter( "interface_decl");

	String id;
	Vector interfaces;


	// match( Scan.INTERFACE );
	id = matchString( "interface name expected");
	InterfaceDecl ifc =new InterfaceDecl(mods,id);
	ifc.Comment = Scan.lastComment();
	ifc.Line = LStart;
	TDecl = ifc;

	if ( isMatch( Scan.EXTENDS) ) {
		interfaces = id_list();
		TDecl.setInterfaces( interfaces);
	}

	matchValChar( '{' );
	while ( !isMatchValChar( '}') )
		field_decl();

	CU.addTypeDecl( TDecl);
	// logExit( id);
}

boolean isMatch ( int tokenType ) {
	if ( Scan.Tok == tokenType ){
		Scan.nextToken();
		return true;
	}
	else
		return false;
}

boolean isMatchSet ( BitSet set ) {
	if ( (Scan.Tok < 0) && (set.get( -Scan.Tok)) ) {
		Scan.nextToken();
		return true;
	}

	return false;
}

boolean isMatchValChar ( char c ) {
	if ( Scan.Tok == c ){
		Scan.nextToken();
		return true;
	}
	else
		return false;
}

void logEnter ( String msg ) {
	logPrefix();
	Level++;
	for ( int i=0; i<Level; i++ )
		System.out.print( "  ");

	System.out.println( msg);
}

void logExit ( String msg ) {
	int i;
	logPrefix();

	for ( i=0; i<Level; i++ )
		System.out.print( "  ");

	for ( ;i<8;i++ )
		System.out.print( "--");

	System.out.println( ' ' + msg );
	Level--;
}

void logPrefix(){
	int i;
	String prfx = FileName +':' + Scan.Line;
	System.out.print( prfx);

	for ( i=prfx.length(); i < 20; i++ )
		System.out.print( ' ');
}

void match ( int tokenType ) throws ParseException {
	if ( Scan.Tok == tokenType )
		Scan.nextToken();
	else 
		throw new ParseException( Scan.keyword( tokenType)
		                          + " expected", Scan);
}

void matchBlock () throws ParseException {
	if ( Scan.Tok == '{' ){
		Scan.skipBlock( '{', '}');
		Scan.nextToken();
	}
	else
		throw new ParseException( "{..} expected", Scan);
}

String matchString ( String failMsg ) throws ParseException {
	if (Scan.Tok == Scan.STRING ){
		String s = Scan.string();
		Scan.nextToken();
		return s;
	}
	else
		throw new ParseException( failMsg, Scan);
}

void matchValChar ( char c ) throws ParseException {
	if ( Scan.Tok == (int)c )
		Scan.nextToken();
	else
		throw new ParseException( "'" + c + "' expected", Scan);
}

void method_decl ( BitSet mods, String name, String resType )
	throws ParseException {
	CurNt = "method"; CurLn = Scan.Line;
	// logEnter( "method_decl");

	Vector exceptions;
	Method mth = new Method( mods, resType, name);
	mth.Line = LStart;
	mth.Comment = Scan.lastComment();
	parm_list( mth );

	if ( isMatch( Scan.THROWS) ){
		exceptions = id_list();
		mth.setExceptions( exceptions);
	}

	if ( Scan.Tok == '{' ) {
		Scan.skipBlock( '{', '}');
		mth.Body = Scan.stringFrom( IStart);
		Scan.nextToken();
	}
	else if ( Scan.Tok ==  ';' ) {
		mth.Body = Scan.stringFrom( IStart);
		Scan.nextToken();
	}
	else
		throw new ParseException( "{..} or ';' expected", Scan);

	TDecl.addMethod( mth);

	// logExit( resType + ' ' + id);
}

int nextToken () {
	int r = Scan.nextToken();
	System.out.print( "[" + Scan.Line + ',' + Scan.I + "] : " + Scan.Tok + " : ");
	System.out.println( Scan.tokenString());
	return r;
}

void package_spec () throws ParseException {
	CurNt = "package"; CurLn = Scan.Line;
	// logEnter( "package_spec");

	String packageId;
	// match ( Scan.PACKAGE);
	packageId = matchString( "package name expected");
	CU.setPackage( packageId);

	while ( isMatchValChar( ';'));

	// logExit( packageId);
}

void parm_list ( Method mth ) throws ParseException {
	// logEnter( "parm_list");

	String pType, pName, pArr;
	int dim, l1, l2;
	char sig[];
	int i=0;
	char norm[];
	int j=0;

	matchValChar( '(');

	if ( Scan.Tok == ')' ){
		mth.setParms( "()");
		mth.setSignature( "()");
		Scan.nextToken();
	}
	else {
		sig = new char[100];  i = 0;
		norm = new char[200]; j = 0;

		sig[i++] = '(';
		norm[j++] = '(';
		do {
			i += Scan.copyTo( sig, i);
			j += Scan.copyTo( norm, j);
			match( Scan.STRING);
			dim = 0;
			while ( Scan.Tok == '[' || Scan.Tok == ']' ){
				Scan.nextToken(); dim++;
			}
			l1 = Scan.I0; l2 = Scan.I;   
			match( Scan.STRING);

			while ( Scan.Tok == '[' || Scan.Tok == ']' ){
				if ( dim > 0 ) dim=0;
				Scan.nextToken(); dim++;
			}
			for ( ; dim > 0; dim -= 2 ){
				norm[j++] = '['; norm[j++] = ']';
				sig[i++] = '[';  sig[i++] = ']';
			}

			for ( norm[j++]= ' '; l1 < l2; l1++ )
				norm[j++] = (char)Scan.C[l1];

			if ( Scan.Tok == ',' ) {
				norm[j++] = ',';
				sig[i++] = ',';
				Scan.nextToken();
			}
			else if ( Scan.Tok == ')' ) {
				norm[j++] = ')';
				sig[i++] = ')';
				Scan.nextToken();
				break;
			}
			else 
				throw new ParseException( "')' or ',' expected", Scan);

		} while ( true );
		mth.setParms( new String( norm, 0, j));
		mth.setSignature( new String( sig, 0, i));
	}


	// logExit( new String( norm, 0, j) );
}

public CompileUnit parseCompileUnit () throws ParseException {
	CU = null;

	Scan.nextToken();
	compile_unit();

	if ( CU != null )
		CU.IsExpanded = true;
	return CU;
}

void reportException ( ParseException x ) {
	Out.print( FileName + ':' + Scan.Line + ": ");
	Out.print( x.toString() + " while parsing " + CurNt );
	Out.print( " at line " + CurLn + "\n");
	Out.println( Scan.currentLine());
	Out.println();
}

String res_type () throws ParseException {
	// logEnter( "res_type");
	String s;
	int dim;

	s = matchString( "type name expected");
	if ( Scan.Tok == '[' ){
		dim =  array_dim();
		for ( int l=0; l<dim; l++ )
			s += "[]";
	}

	// logExit( s);
	return s;
}

void reset () {
	Scan.reset();
	CU = null;
	TDecl = null;
	IStart = 0;
}

void setOutputStream ( PrintStream out ) {
	Out = out;
}

void static_init () throws ParseException {
	CurNt = "static init"; CurLn = Scan.Line;
	// logEnter( "static_init");
	String body;
	Scan.skipBlock( '{', '}');
	body = Scan.stringFrom( IStart);
	Method mth = Method.staticInit( body);
	mth.Comment = Scan.lastComment();
	mth.Line = LStart;
	TDecl.addMethod( mth);
	Scan.nextToken();

	// logExit( "");
}

public void storeOutput () {
	OutBuffer = new ByteArrayOutputStream( 300);
	Out = new PrintStream( OutBuffer);
}

void type_decl () throws ParseException {
	CurNt = "type decl"; CurLn = Scan.Line;
	// logEnter( "type_decl");
	BitSet mods = new BitSet( Scan.NKeys);
	LStart = Scan.Line;
	for ( int t = Scan.Tok; isMatchSet( CorI_Modifier); t = Scan.Tok )
		mods.set( -t);  // JavaScanner token ids are negative

	if ( isMatch( Scan.CLASS ) )
		class_decl ( mods);
	else if ( isMatch( Scan.INTERFACE ) )
		interface_decl( mods);
	else
		throw new ParseException( "'class' or 'interface' expected", Scan);

	while ( isMatchValChar( ';'));

	// logExit( "");
}
}
