package biss.jde;

import biss.ObserverSocket;
import biss.VectorLib;
import biss.awt.Awt;
import biss.awt.Control;
import biss.awt.HotKey;
import biss.awt.HotKeyProcessor;
import biss.awt.Key;
import biss.awt.Layouter;
import biss.awt.Menu;
import java.awt.Color;
import java.awt.Event;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.PrintStream;
import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

/**
 * specialized widget class to display/modify widget representations from
 * within the Designer
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
class DesignPane
  extends Control
  implements Observer, HotKeyProcessor
{
	int LastAction;
	Vector Comps;
	DesignObject Selection;
	int CPA = -1;
	Graphics Gr;
	ObserverSocket OsModify = new ObserverSocket( this);
	Point DragOffs = new Point( 0, 0);
	int Grid = 3;
	Vector Layouts = new Vector( 3);
	boolean XOrMode = false;
	int CurLo;
	final static int RESHAPE = 1;
	final static int SELECT = 2;
	final static int CLEAR = 3;
	final static int SET_UNIT = 4;
	final static int SET_POS = 5;
	final static int SET_LAYOUT = 6;
	final static int REMOVE = 7;
	final static int ADD = 8;
	Vector MultiSels;
	final static int TOGGLE = 9;
	Point LastGridPos = new Point( -1, -1);
	Rectangle RefRect;
	final static int REF_POS = 10;
	final static int REF_SWITCH = 11;
	int SOF = 3;

DesignPane () {
	setFont( Awt.StaticFont);
	setForeground( Color.black);
	setBackground( Color.lightGray);
	buildMenu();

	registerObserver();
}

public void addLayout () {
	Layouts.addElement( new Vector( 10));
	setLayout( Layouts.size()-1);
}

public void addObject ( Object obj ) {
	Selection = (DesignObject) obj;
	Comps.addElement( Selection );
	redraw( Gr);
}

public void addObject ( String clsName, String fieldName, 
                 String text, DesignObject refObj ) {
	Selection = (DesignObject)Selection.clone();
	Selection.FldName = fieldName;
	Selection.setClass( clsName);
	Selection.setInitParams( text);
	Selection.updateHandles( Width, Height);
	Comps.addElement( Selection );
	redraw( Gr);
}

public void addObject ( String clsName, String fieldName, 
                 String text, boolean duOffs ) {
	Selection = new DesignObject( clsName, fieldName, text, Width, Height);
	Comps.addElement( Selection );
	setSelectionUnit( duOffs);
	redraw( Gr);
}

int adjust( int pos, int min, int max) {
	if ( pos < min) return min;
	if ( pos > max) return max;
	return pos;
}

boolean alignBottom() {
	if ( (MultiSels == null) || (RefRect == null) )
		return false;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.y = RefRect.y + SOF - dob.Rect.height;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
	}
	redraw( Gr);
	return true;
}

boolean alignLeft() {
	if ( (MultiSels == null) || (RefRect == null) )
		return false;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.x = RefRect.x + SOF;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
	}
	redraw( Gr);
	return true;
}

boolean alignRight() {
	if ( (MultiSels == null) || (RefRect == null) )
		return false;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.x = RefRect.x - dob.Rect.width + SOF;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
	}
	redraw( Gr);
	return true;
}

boolean alignTop() {
	if ( (MultiSels == null) || (RefRect == null) )
		return false;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.y = RefRect.y + SOF;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
	}
	redraw( Gr);
	return true;
}

Vector allObjects () {
	Vector va = new Vector();
	for ( Enumeration le = Layouts.elements(); le.hasMoreElements();){
		Vector v = (Vector) le.nextElement();
		VectorLib.addAllElements( va, v);
	}
	return va;
}

boolean assignHeight() {
	if ( (MultiSels == null) || (Selection == null) )
		return false;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.height = Selection.Rect.height;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
	}
	redraw( Gr);
	return true;
}

boolean assignWidth() {
	if ( (MultiSels == null) || (Selection == null) )
		return false;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.width = Selection.Rect.width;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
	}
	redraw( Gr);
	return true;
}

void buildMenu() {
	Menu m = new Menu();
	m.addItem( "~Mark");
	m.addItem( "~Clear");
	m.addItem( "GroupSlider");
	m.addSubItem( "GroupSlider", "~Show");
	m.addSubItem( "GroupSlider", "~Hide");
	m.addItem( "Align");
	m.addSubItem( "Align", "~Left");
	m.addSubItem( "Align", "~Right");
	m.addSubItem( "Align", "~Top");
	m.addSubItem( "Align", "~Bottom");
	m.addItem( "Assign");
	m.addSubItem( "Assign", "Width");
	m.addSubItem( "Assign", "Height");
	setMenu( m);
}

void checkInners( DesignObject os) {
	for ( Enumeration e = Comps.elements(); e.hasMoreElements();) {
		DesignObject osc = (DesignObject)e.nextElement();
		if ( os == osc)
			continue;
		if ( (os.Rect.width > osc.Rect.width) && (os.Rect.height > osc.Rect.height) ) {
			if ( os.Rect.intersects( osc.Rect)){
				if ( os.Rect.intersection( osc.Rect).equals( osc.Rect) )
					os.HasInners = true;
			}
		}
	}
}

void checkIntersects() {
	for ( Enumeration e = Comps.elements(); e.hasMoreElements();) {
		DesignObject osc = (DesignObject)e.nextElement();
		osc.HasInners = false;
		checkInners( osc);
	}
}

public void cleanUp () {
	Layouts.removeAllElements();
	addLayout();
}

public void clearSelections () {
	Selection = null;
	MultiSels = null;
	redraw( Gr);
	LastAction = CLEAR;
	OsModify.notifyObservers( this);
}

public int countLayouts () {
	return Layouts.size();
}

boolean dragging() {
	return ( (DragOffs.x != 0) && (DragOffs.y != 0) );
}

void drawRefLines( int x, int y) {
	if ( RefRect != null) {
		Gr.setColor( Color.green);
		Gr.drawLine( 0, y, Width, y);
		Gr.drawLine( x, 0, x, Height);
		Gr.fillRect( RefRect.x, RefRect.y, RefRect.width, RefRect.height);
	}
}

boolean equiDistX( int pels) {
	if ( (MultiSels == null) || (Selection == null) )
		return false;
	int x = Selection.Rect.x + Selection.Rect.width + pels;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.x = x;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
		x += dob.Rect.width + pels;
	}
	redraw( Gr);
	return true;
}

boolean equiDistY( int pels) {
	if ( (MultiSels == null) || (Selection == null) )
		return false;
	int y = Selection.Rect.y + Selection.Rect.height + pels;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		dob.Rect.y = y;
		dob.updateHandles( Width, Height);
		dob.updateLayoutSpec( Width, Height);
		y += dob.Rect.height + pels;
	}
	redraw( Gr);
	return true;
}

void getControlPoint( int x, int y) {

	CPA        = -1;
	DragOffs.x = 0;
	DragOffs.y = 0;

	if ( (RefRect != null) && RefRect.inside( x, y)) {
		setCursor( Frame.MOVE_CURSOR);
		CPA = 12;
		return;
	}
	if ( MultiSels != null){
		setCursor( Frame.MOVE_CURSOR);
		CPA = 11;
		DragOffs.x = x / Grid * Grid;
		DragOffs.y = y / Grid * Grid;
		return;
	}
	if ( Selection == null){
		setCursor( Frame.DEFAULT_CURSOR);
		return;
	}

	CPA = Selection.detectHandlePos( x, y);

	switch( CPA) {
	case 0:
		setCursor( Frame.W_RESIZE_CURSOR);
		break;
	case 1:
		setCursor( Frame.NW_RESIZE_CURSOR);
		break;
	case 2:
		setCursor( Frame.N_RESIZE_CURSOR);
		break;
	case 3:
		setCursor( Frame.NE_RESIZE_CURSOR);
		break;
	case 4:
		setCursor( Frame.SW_RESIZE_CURSOR);
		break;
	case 5:
		setCursor( Frame.S_RESIZE_CURSOR);
		break;
	case 6:
		setCursor( Frame.SE_RESIZE_CURSOR);
		break;
	case 7:
		setCursor( Frame.E_RESIZE_CURSOR);
		break;
	case 8:
		DragOffs.x = (x - Selection.Rect.x) / Grid * Grid;
		DragOffs.y = (y - Selection.Rect.y) / Grid * Grid;
		setCursor( Frame.MOVE_CURSOR);
		break;
	case 9:
	case 10:
		setCursor( Frame.MOVE_CURSOR);
		break;
	default:
		setCursor( Frame.DEFAULT_CURSOR);
	}
}

public boolean includesObject ( String fieldName) {
	if ( Comps == null)
		return false;
	for ( int i=0; i<Comps.size(); i++) {
		DesignObject deo = (DesignObject)Comps.elementAt( i);
		if ( deo.FldName.equals( fieldName))
			return true;
	}
	return false;
}

public boolean isDefaultObject( DesignObject os) {
	Vector v = (Vector)Layouts.firstElement();
	return v.contains( os);
}

public void markObject( DesignObject os ) {
	if ( os != null) {
		LastAction = TOGGLE;
		OsModify.notifyObservers( os);
		redraw( Gr);
	}
}

public void mouse1Down( Event evt) {

	if ( (CPA != 8) && (CPA != 11) && (CPA != -1)) //already on control point
		return;

	DesignObject ls = null;
	int      s = Comps.size();
	boolean cd = evt.controlDown();

	for ( int i=0; i<s; i++){
		DesignObject os = (DesignObject)Comps.elementAt( i);
		if ( os.Rect.inside( evt.x, evt.y)){
			if ( cd) {
				if (! os.HasInners ){
					markObject( os);
					return;
				}
			}
			else if ( os != Selection){
				selectObject( os);
				return;
			}
			else
				ls = os;
		}
	}

	if ( ls == null)
		clearSelections();
}

public void mouse2Down( Event evt) {
	blank( Gr);
	XOrMode = true;
	Gr.setXORMode( getBackground() );
	redraw( Gr);
}

public void mouse2Drag( Event evt) {

	if (CPA == -1) //detect handle operation
		getControlPoint( evt.x, evt.y);

	int x = (evt.x / Grid) * Grid;
	int y = (evt.y / Grid) * Grid;

	if ( (x == LastGridPos.x) && (y == LastGridPos.y))
		return;

	if ( CPA == 12)
		moveRefPoint( x, y);
	else if ( CPA == 11)
		moveMultiSels( x, y);
	else if ( Selection != null)
		reshapeSelection( x, y);

	LastGridPos.x = x;
	LastGridPos.y = y;
}

public void mouse2Up( Event evt) {
	XOrMode = false;
	Gr.setPaintMode();
	redraw( Gr);

	CPA = -1;
}

public void mouseMove( Event evt) {
	getControlPoint( evt.x, evt.y);
}

void moveMultiSels( int x, int y) {
	int dx = x - DragOffs.x;
	int dy = y - DragOffs.y;
	for (Enumeration e = MultiSels.elements(); e.hasMoreElements(); ) {
		DesignObject dob = (DesignObject)e.nextElement();
		Rectangle rob = dob.Rect;
		dob.drawMinimal( Gr, false, true);
		rob.move( rob.x+dx, rob.y+dy);
		dob.updateHandles( Width, Height);
		dob.drawMinimal( Gr, false, true);
		dob.updateLayoutSpec( Width, Height);
	}
	DragOffs.x += dx;
	DragOffs.y += dy;

	LastAction = RESHAPE;
	OsModify.notifyObservers( this);
}

void moveRefPoint( int x, int y) {
	//blank
	drawRefLines( RefRect.x + SOF, RefRect.y + SOF);
	//update
	RefRect.move( adjust( x - SOF, -SOF, Width - SOF),      
	              adjust( y - SOF, -SOF, Height -SOF) );
	//redraw
	drawRefLines( RefRect.x + SOF, RefRect.y + SOF);

	LastAction = REF_POS;
	OsModify.notifyObservers( this);
}

public void paint( Graphics g){
	//first valid resident graphics can be obtained here
	if ( Gr == null)
		Gr = getGraphics();
	super.paint( g);
}

protected void posChanged() {
	if ( Gr != null) {
		Gr.dispose();
		Gr = getGraphics();
	}
	if ( Comps == null)
		return;
	for ( int i=0; i<Comps.size(); i++){
		DesignObject os = (DesignObject) Comps.elementAt( i);
		os.updateRect( Width, Height);
	}
	redraw( Gr);
	LastAction = RESHAPE;
	OsModify.notifyObservers( this);
}

void printLayoutOn ( PrintStream s, String loFldName) {
	for ( int i=0; i<Layouts.size(); i++) {
		Vector v = (Vector)Layouts.elementAt( i);
		if ( v.size() == 0)
			continue;
		if ( i>0 )
			s.println( "\n\t" + loFldName + ".newLayout();" );
		for ( Enumeration se = v.elements(); se.hasMoreElements();){
			DesignObject os = (DesignObject)se.nextElement();
			if ( i>0 && isDefaultObject( os))
				s.println( '\t' + loFldName + ".addLikeDefault( " + os.FldName + ");" );
			else {
				s.print( '\t' + loFldName + ".add( ");
				os.printLayoutOn( s);
				s.println( ");" );
			}
		}
	}
}

public void processKey( Object sel) {
	if ( sel.equals( "clear"))
		clearSelections();
}

public void redraw (Graphics g) {
	if ( Gr == null)
		return;

	blank( Gr);

	if ( Comps == null)
		return;  

	if ( ! XOrMode)
		checkIntersects();

	int s = Comps.size();
	int ms = (MultiSels == null) ? 0 : MultiSels.size();

	for ( int i=0; i<s; i++){
		DesignObject os = (DesignObject)Comps.elementAt( i);
		boolean mSel = ms > 0 ? MultiSels.contains( os) : false;
		if ( XOrMode) os.drawMinimal( Gr, os == Selection, mSel);
		else          os.drawExtended( Gr, os == Selection, mSel);
	}

	//redraw selection to top ratio handles
	if ( (Selection != null) && ( !XOrMode))
		Selection.drawExtended( Gr, true, false);

	if ( RefRect != null)
		drawRefLines( RefRect.x + SOF, RefRect.y + SOF);
}

String refPosString() {
	if ( RefRect != null)
		return ( "R: " + (RefRect.x + SOF) + "," + (RefRect.y + SOF) );
	return null;
}

public void registerObserver () {
	OsCommand.addObserver( this);
	addHotKey( Key.Escape, HotKey.Lone, this, "clear" );
}

public void removeObject ( DesignObject os) {
	Comps.removeElement( os);
	if ( os == Selection)
		Selection = null;
	redraw( Gr);
	LastAction = REMOVE;
	OsModify.notifyObservers( this);
}

public void removeSelection () {
	if ( Selection != null)
		removeObject( Selection);
}

void reshapeSelection( int x, int y) {
	int xc, mw = 15;
	Rectangle r = Selection.Rect;

	Selection.drawMinimal( Gr, true, false);

	switch( CPA) {
	case 0:
		xc = r.x + r.width;
		r.x = adjust( x, 0, xc-mw);
		r.width = xc - r.x;
		break;
	case 1:	  
		xc = r.x + r.width;
		r.x = adjust( x, 0, xc-mw);
		r.width = xc - r.x;
		xc = r.y + r.height;
		r.y = adjust( y, 0, xc-mw);
		r.height = xc - r.y;
		break;
	case 2:
		xc = r.y + r.height;
		r.y = adjust( y, 0, xc-mw);
		r.height = xc - r.y;
		break;
	case 3:
		xc = r.y + r.height;
		r.y = adjust( y, 0, xc-mw);
		r.height = xc - r.y;
		r.width = Math.max( Math.min( Width, x) - r.x, mw);
		break;
	case 4:
		xc = r.x + r.width;
		r.x = adjust( x, 0, xc-mw);
		r.width = xc - r.x;
		r.height = Math.max( Math.min( Height, y) - r.y, mw);
		break;
	case 5:
		r.height = Math.max( Math.min( Height, y) - r.y, mw);
		break;
	case 6:
		r.width = Math.max( Math.min( Width, x) - r.x, mw);
		r.height = Math.max( Math.min( Height, y) - r.y, mw);
		break;
	case 7:
		r.width = Math.max( Math.min( Width, x) - r.x, mw);
		break;
	case 8:
		r.move( adjust( x-DragOffs.x, 0, Width-r.width),
		        adjust( y-DragOffs.y, 0, Height-r.height) );
		/*
		  r.move( Math.min( Math.max(x - DragOffs.x, 0),
		  Width - r.width),
		  Math.min( Math.max(y - DragOffs.y, 0),
		  Height - r.height) );
		 */
		break;
	case 9:
		Selection.RatioOrigin.x = adjust( x, 0, Width);
		Selection.RatioOrigin.y = adjust( y, 0, Height);
		break;
	case 10:
		Selection.RatioCorner.x = adjust( x, 0, Width);
		Selection.RatioCorner.y = adjust( y, 0, Height);
		break;
	}

	Selection.updateHandles( Width, Height);
	Selection.drawMinimal( Gr, true, false);

	Selection.updateLayoutSpec( Width, Height);
	LastAction = RESHAPE;
	OsModify.notifyObservers( this);
}

public void selectObject( DesignObject os ) {
	Selection = os;
	redraw( Gr);
	LastAction = SELECT;
	OsModify.notifyObservers( this);
}

public void setLayout ( int id) {
	CurLo = id;
	Comps = (Vector)Layouts.elementAt( id);
	for ( int i=0; i<Comps.size(); i++){
		DesignObject os = (DesignObject)Comps.elementAt( i);
		os.updateRect( Width, Height);
	}
	Selection = null;
	MultiSels = null;
	redraw( Gr);
	LastAction = SET_LAYOUT;
	OsModify.notifyObservers( this);
}

public void setLayouts ( Vector v) {
	Layouts = v;
	setLayout( 0);
}

void setSelectionClass (String cls) {
	if ( (Selection != null) && (cls != null)){
		Selection.setClass( cls);
		Selection.drawExtended( Gr, true, false);
	}
}

void setSelectionField (String fld) {
	if ( (Selection != null) && (fld != null))
		Selection.FldName = fld;
}

void setSelectionSplits (boolean left, boolean right, boolean top, boolean bottom) {
	if (Selection != null){
		int vs = left ? Layouter.LEFT : 0;
		if ( right) vs |= Layouter.RIGHT;
		if ( top)   vs |= Layouter.TOP;
		if ( bottom)vs |= Layouter.BOTTOM;
		Selection.LoSpec.VarSides = vs;
	}
}

void setSelectionText (String txt) {
	if ( Selection != null) {
		Selection.setInitParams( txt);
		Selection.drawExtended( Gr, true, false);
	}
}

void setSelectionUnit (boolean dlg) {
	if ( Selection != null) {
		Selection.LoSpec.DUOffset = dlg;
		Selection.updateLayoutSpec( Width, Height);
		LastAction = SET_UNIT;
		OsModify.notifyObservers( this);
	}
}

void setUnits (boolean dlg) {
	for ( int i=0; i<Comps.size(); i++) {
		DesignObject deo = (DesignObject)Comps.elementAt( i);
		deo.LoSpec.DUOffset = dlg;
		deo.updateLayoutSpec( Width, Height);
	}
	LastAction = SET_UNIT;
	OsModify.notifyObservers( this);
}

void showRefLines( boolean state) {
	if ( state ) {
		if (RefRect == null){
			RefRect = new Rectangle( Width/2 - SOF, Height/2 - SOF, 7, 7);
			redraw( Gr);
		}
	}
	else {
		RefRect = null;
		redraw( Gr);
	}
	LastAction = REF_SWITCH;
	OsModify.notifyObservers( this);
}

void update () {
	redraw( Gr);
}

public void update ( Observable obs, Object arg ) {
	if ( obs == OsCommand) {
		if ( "Mark".equals( arg))
			markObject( Selection);
		else if ( "Clear".equals( arg))
			clearSelections();
		else if ( "Show".equals( arg))
			showRefLines( true);
		else if ( "Hide".equals( arg))
			showRefLines( false);
		else if ( "Left".equals( arg))
			alignLeft();
		else if ( "Right".equals( arg))
			alignRight();
		else if ( "Top".equals( arg))
			alignTop();
		else if ( "Bottom".equals( arg))
			alignBottom();
		else if ( "Width".equals( arg))
			assignWidth();
		else if ( "Height".equals( arg))
			assignHeight();
	}
	else
		super.update( obs, arg);
}

void updateSelectionPos () {
	if ( Selection != null) {
		Selection.updateRect( Width, Height);
		redraw( Gr);
		LastAction = SET_POS;
		OsModify.notifyObservers( this);
	}
}

void updateSelectionRatio ( int pos, boolean corner, boolean yPos) {
	// update Layout ratios and PatioPoints from
	// Rectangle and ratio pos (ppm)
	if ( Selection != null) {
		DesignObjLayout lo = Selection.LoSpec;

		if ( yPos) {
			if ( corner) lo.Y1 = pos;
			else         lo.Y0 = pos;
		}
		else { 
			if ( corner) lo.X1 = pos;
			else         lo.X0 = pos;
		}
		lo.ratioToPelPoints( Width, Height, Selection.RatioOrigin, Selection.RatioCorner);
		Selection.updateHandles( Width, Height);

		redraw( Gr);

		Selection.updateLayoutSpec( Width, Height);
		LastAction = SET_POS;
		OsModify.notifyObservers( this);
	}
}

void updateSelectionRect () {
	if ( Selection != null) {
		Selection.updateRect( Width, Height);
		redraw( Gr);
		LastAction = SET_POS;
		OsModify.notifyObservers( this);
	}
}

protected int xGridVal( int xPos) {
	return (xPos / Grid) * Grid;
}

protected int yGridVal( int yPos) {
	return (yPos / Grid) * Grid;
}
}
