package biss.jde;

import biss.CLPrintStream;
import biss.DummyCLPrintStream;
import biss.FileLib;
import biss.ObserverSocket;
import biss.VectorLib;
import biss.VectorSorter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

interface CUObserver
{
public boolean expandCU ( CompileUnit cu );

public boolean needsExpandedCU ( CompileUnit cu );

}

/**
 * model class describing the contents of a Java source file
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class CompileUnit
  extends Observable
  implements TypeDeclContainer
{
	String DirName;
	String FileName;
	Vector Imports = new Vector( 5);
	String MainTypeName;
	String Package;
	File SrcFile;
	Vector TypeDecls = new Vector( 5);
	boolean IsExpanded = false;
	long Released;
	int ExpandObservers = 0;
	int LastChange = 0;
	final public static int CHG_PCKG = 1;
	final public static int CHG_TYPE = 2;
	final public static int CHG_IMPT = 4;
	final public static int CHG_DATA = 8;
	final public static int CHG_MTHD = 16;
	final public static int CHG_FILE = 32;
	final public static int CHG_NONE = 0;

public CompileUnit ( String fname ) {
	setFileName( fname);
	MainTypeName = typeNameFromFile( SrcFile);
}

public boolean addImport ( Import imp ) {
	VectorLib.sortIn( Imports, imp, Import.Sorter);
	LastChange &= CHG_IMPT;
	return true;
}

public void addObserver ( Observer obj ) {
	super.addObserver( obj);  
	if ( obj instanceof CUObserver ) {
		CUObserver cuo = (CUObserver) obj;
		if ( cuo.needsExpandedCU( this) ) {
			Released = 0;
			ExpandObservers++;
			if ( !IsExpanded )
				cuo.expandCU( this);
			else
				notifyObservers( null);
		}
	}
}

public boolean addTypeDecl ( TypeDecl td ) {
	td.CU = this;

	VectorLib.sortIn( TypeDecls, td, TypeDecl.Sorter);
	LastChange &= CHG_TYPE;
	return true;
}

boolean backup () {
	return FileLib.backup( SrcFile);
}

void delImports ( Vector list ) {
	for ( Enumeration e=list.elements(); e.hasMoreElements(); ) {
		Import imp = (Import) e.nextElement();
		Imports.removeElement( imp);
	}
	LastChange &= CHG_IMPT;
}

void delTypeDecls ( Vector list ) {
	for ( Enumeration e=list.elements(); e.hasMoreElements(); ) {
		TypeDecl t = (TypeDecl) e.nextElement();
		TypeDecls.removeElement( t);
	}
	LastChange &= CHG_TYPE;
}

public void deleteObserver ( Observer obj ) {
	super.deleteObserver( obj);
	if ( obj instanceof CUObserver ) {
		if ( ((CUObserver)obj).needsExpandedCU( this) && --ExpandObservers == 0 ) {
			Released = System.currentTimeMillis();
		}
	}  
}

public synchronized void expandFrom ( CompileUnit cu ) {
	mergeTypes( cu.TypeDecls, true);
	Imports = cu.Imports;
	IsExpanded = true;
	notifyObservers( this);
}

public String fileInfoString ( boolean fixedLength ) {
	return FileLib.fileInfoString( SrcFile, fixedLength );
}

public void format () {
	Enumeration e;
	MethodFormatter mf = new MethodFormatter();

	for ( e = TypeDecls.elements(); e.hasMoreElements(); ){
		((TypeDecl)e.nextElement()).formatAllMethods( mf);
	}
}

public String getDirName() {
	return DirName;
}

public String getFileName () {
	return FileName;
}

public String getPackage () {
	return Package;
}

public boolean hasLastChange ( int testVal ) {
	return ((LastChange & testVal) != 0);
}

public boolean isExpanded () {
	return IsExpanded;
}

TypeDecl mainType () {
	return typeWithName( MainTypeName);
}

public String mainTypeTemplate() {
	return "public class " + MainTypeName + "\n    extends \n    implements";
}

boolean mergeImports ( Vector newImports ) {
	for ( Enumeration e=newImports.elements(); e.hasMoreElements(); ){
		Import imp = (Import) e.nextElement();
		if ( !Imports.contains( imp) )
			addImport( imp);
	}
	LastChange &= CHG_IMPT;
	return true;
}

boolean mergeTypes ( Vector newTypes, boolean full ) {
	for ( Enumeration e=newTypes.elements(); e.hasMoreElements(); ){
		TypeDecl dNew = (TypeDecl) e.nextElement();
		TypeDecl dOld = (TypeDecl) VectorLib.firstEqualElement( TypeDecls, dNew);

		if ( dOld != null )
			dOld.merge( dNew, full);
		else {
			dNew.CU = this;
			TypeDecls.addElement( dNew);
		}
	}
	LastChange &= CHG_TYPE;
	return true;
}

public void notifyObservers ( Object arg ) {
	setChanged();
	super.notifyObservers( arg);
	clearChanged();
}

String packageDefinition () {
	if ( Package != null )
		return ("package " + Package);
	else
		return "package ";
}

public void printOn ( PrintStream s ) {
	Enumeration e;

	if ( Package != null ) {
		s.print( "package ");
		s.print( Package);
		s.println( ";\n");
	}

	for ( e = Imports.elements() ; e.hasMoreElements() ; ) {
		((Import)e.nextElement()).printOn( s);
	}

	for ( e = TypeDecls.elements(); e.hasMoreElements(); ){
		s.println();
		((TypeDecl)e.nextElement()).printOn( s, true, true, true);
	}
}

String qualifiedClsName ( String cls ) {
	return (Package != null) ? Package + '.' + cls : cls;
}

void refreshLines () {
	PrintStream s = new DummyCLPrintStream(); 
	printOn( s);
	s.close();
}

boolean renameType ( String oldName, String newName ) {
	TypeDecl t = typeWithName( oldName);
	if ( t == null )
		return false;

	if ( oldName.equals( MainTypeName) ) {
		File newSrc = new File( SrcFile.getParent(),  newName + ".java");
		if ( SrcFile.renameTo( newSrc) ){
			t.Id = newName;
			VectorLib.quickSort( TypeDecls, TypeDecl.Sorter);

			MainTypeName = newName;
			SrcFile = newSrc;
			FileName = newSrc.getAbsolutePath();
		}
		else
			return false;
	}
	else {
		t.Id = newName;
		VectorLib.quickSort( TypeDecls, TypeDecl.Sorter);
	}
	return true;
}

public boolean save () {
	return save( false);
}

public boolean save ( boolean trackLines ) {
	try {
		FileOutputStream fo = new FileOutputStream( FileName);
		ByteArrayOutputStream o = new ByteArrayOutputStream( 40000);
		PrintStream s = trackLines ? new CLPrintStream(o) : new PrintStream(o); 

		printOn( s);

		o.writeTo( fo);
		fo.close();
		o.close();
		s.close();

		LastChange &= CHG_FILE;
		return true;
	}
	catch ( IOException x ) {
		System.out.println( x.toString());
	}
	return false;
}

public void searchMainType () {
	if ( TypeDecls.size() == 1 ){
		MainTypeName = ((TypeDecl)TypeDecls.firstElement()).Id;
		return;
	}

	for ( Enumeration e=TypeDecls.elements(); e.hasMoreElements(); ){
		TypeDecl t = (TypeDecl)e.nextElement();
		if ( t.hasMainMethod() ){
			MainTypeName = t.Id;
			return;
		}
	}

	String mType = typeNameFromFile( SrcFile);
	TypeDecl tBig = (TypeDecl) TypeDecls.firstElement();

	for ( Enumeration e=TypeDecls.elements(); e.hasMoreElements(); ){
		TypeDecl t = (TypeDecl)e.nextElement();
		if ( t.Id.equals( mType) ) {
			MainTypeName = mType;
			return;
		}
		if ( t.Methods.size() > tBig.Methods.size() )
			tBig = t;
	}
	MainTypeName = tBig.Id;
}

public void setFileName ( String fname ) {
	SrcFile = new File( fname);
	FileName = SrcFile.getAbsolutePath();
	DirName = SrcFile.getParent();

	if ( DirName == null )
		DirName = ".";
	else
		DirName = DirName.intern();
}

public void setImports ( Vector newImports ) {
	Imports = newImports;
}

public void setLastChange ( int newVal ) {
	LastChange = newVal;
}

public void setPackage ( String packageName ) {
	if ( packageName != null && packageName.length() > 0 )
		Package = packageName.intern();
	else
		Package = null;

	LastChange &= CHG_PCKG;
}

public synchronized void shrink () {
	for ( Enumeration e = TypeDecls.elements(); e.hasMoreElements(); ) {
		TypeDecl t = (TypeDecl)e.nextElement();
		t.shrink();
	}
	Imports = new Vector(5);
	IsExpanded = false;
	notifyObservers( this);
}

public String toString () {
	return MainTypeName;
}

static String typeNameFromFile ( File file ) {
	String tName = file.getName();
	int i;
	if ( (i = tName.lastIndexOf( '.')) > 0 )
		tName = tName.substring( 0, i);
	return tName;
}

TypeDecl typeOfLine ( int line ) {
	TypeDecl  t = null;
	int       dist = 100000;

	for ( Enumeration e=TypeDecls.elements(); e.hasMoreElements(); ){
		TypeDecl tt = (TypeDecl) e.nextElement();
		int d = line - tt.Line;
		if ( d > 0 && d < dist ) {
			dist = d;
			t = tt;
		}
	}

	return t;
}

TypeDecl typeWithName ( String name ) {
	for ( Enumeration e=TypeDecls.elements(); e.hasMoreElements(); ) {
		TypeDecl t = (TypeDecl)e.nextElement();
		if ( name.equals( t.Id) )
			return t;
	}
	return null;
}
}

class CompileUnitSorter
  implements VectorSorter
{
	boolean UseQualified = true;

public CompileUnitSorter ( boolean useQualifiedTypes ) {
	UseQualified = useQualifiedTypes;
}

public int compare ( Object a, Object b ) {
	CompileUnit ca = (CompileUnit) a;
	CompileUnit cb = (CompileUnit) b;

	String ka = ca.MainTypeName;
	String kb = cb.MainTypeName;

	if ( UseQualified ) {
		ka = ca.qualifiedClsName( ka);
		kb = cb.qualifiedClsName( kb);
	}

	return ka.compareTo( kb);
}
}
