/* --------------------------------------------------------------------*/
/*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \   /  '                               */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome. Send them to                                          */
/*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
/*-------------------------------------------------------------------- */
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/writer.c                */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Tue Dec 17 09:44:20 1991                          */
/*    Last change :  Thu Feb 12 06:28:28 1998 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Object (that have to be non recursives) printing.                */
/*=====================================================================*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <bigloo1.9c.h>

/*---------------------------------------------------------------------*/
/*    Les recuperations externes                                       */
/*---------------------------------------------------------------------*/
extern obj_t c_constant_string_to_string();
extern obj_t write_object( obj_t, obj_t );
extern obj_t write_ucs2( obj_t, obj_t );
extern obj_t display_ucs2string( obj_t, obj_t );
extern char *real_to_string( double );

/*---------------------------------------------------------------------*/
/*    Les noms des caracateres                                         */
/*---------------------------------------------------------------------*/
static char *char_name[] = {
   "","","","","","","","",
   "",  "tab", "newline", "", "", "return", "", "",
   "", "","","","","","","",
   "", "", "","","", "", "", "",
   "space", "!", "\"","#","$","%","&","'",
   "(", ")", "*", "+", ",", "-", ".", "/",
   "0", "1", "2", "3", "4", "5", "6", "7",
   "8", "9", ":", ";", "<", "=", ">", "?",
   "@", "A", "B", "C", "D", "E", "F", "G",
   "H", "I", "J", "K", "L", "M", "N", "O",
   "P", "Q", "R", "S", "T", "U", "V", "W",
   "X", "Y", "Z", "[", "\\", "]", "^", "_",
   "`", "a", "b", "c", "d", "e", "f", "g",
   "h", "i", "j", "k", "l", "m", "n", "o",
   "p", "q", "r", "s", "t", "u", "v", "w",
   "x", "y", "z", "{", "|", "}", "~", ""
};


/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strputc ...                                                      */
/*---------------------------------------------------------------------*/
obj_t
strputc( char c, obj_t p )
{
   long offset;
   
   if( END_OF_STRING_PORTP( p ) )
      strport_grow( p );

   offset = OUTPUT_STRING_PORT( p ).offset;

   OUTPUT_STRING_PORT( p ).buffer[ offset ] = c;
   OUTPUT_STRING_PORT( p ).offset = offset + 1;

   return p;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    lstrputs ...                                                     */
/*---------------------------------------------------------------------*/
obj_t
lstrputs( char *s, obj_t p, long len )
{
   long offset;
   
   offset = OUTPUT_STRING_PORT( p ).offset;

   while((OUTPUT_STRING_PORT( p ).offset+len) > OUTPUT_STRING_PORT( p ).size)
      strport_grow( p );

   memcpy( &(OUTPUT_STRING_PORT(p).buffer[offset] ), s, len);

   OUTPUT_STRING_PORT( p ).offset = offset + len;

   return p;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strputs ...                                                      */
/*---------------------------------------------------------------------*/
obj_t
strputs( char *s, obj_t p )
{
   return lstrputs( s, p, strlen( s ) );
}
   
/*---------------------------------------------------------------------*/
/*    We catch the `escape_char_found' variable from Clib/cstring.c    */
/*---------------------------------------------------------------------*/
extern int escape_char_found;

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_string ...                                               */
/*---------------------------------------------------------------------*/
obj_t
display_string( obj_t o, obj_t port )
{
   if( OUTPUT_STRING_PORTP( port ) )
      lstrputs( BSTRING_TO_STRING( o ),
	        port,
	        STRING_LENGTH( o ) );
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;
      long  len  = STRING_LENGTH( o );
      char *aux  = &STRING_REF( o, 0 );
      
      fwrite( aux, 1, len, fout );
   }
   
   return o;
}

/*---------------------------------------------------------------------*/
/*    write_string ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
write_string( obj_t string, obj_t port )
{
   char *aux = BSTRING_TO_STRING( string );
   long len  = STRING_LENGTH( string );
   
   if( OUTPUT_STRING_PORTP( port ) )
   {
      if( escape_char_found )
         strputc( '#', port );

      strputc( '"', port );
      lstrputs( aux, port,len );
      strputc( '"', port );
   }
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;

      if( escape_char_found )
         fputc( '#', fout );
   
      fputc( '"', fout );
      fwrite( aux, 1, len, fout );
      fputc( '"', fout );
      return string;
   }

   return string;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_fixnum ...                                               */
/*---------------------------------------------------------------------*/
obj_t
display_fixnum( obj_t o, obj_t port )
{
   if( OUTPUT_STRING_PORTP( port ) )
   {
      char new[ 100 ];
      
      sprintf( new, "%ld", CINT( o ) );
      strputs( new, port );
   }
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;
      
      fprintf( fout, "%ld", CINT( o ) );
   }
   
   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_flonum ...                                               */
/*    -------------------------------------------------------------    */
/*    Many thanks to Raj Manandhar <raj@droid.msfc.nasa.gov> for       */
/*    providing this code.                                             */
/*---------------------------------------------------------------------*/
obj_t
display_flonum( obj_t o, obj_t port )
{
   char *new = real_to_string( REAL( o ).real );

   if( OUTPUT_STRING_PORTP( port ) )
      strputs( new, port );
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;

      fputs( new, fout );
   }
   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_char ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
display_char( obj_t o, obj_t port )
{
   if( OUTPUT_STRING_PORTP( port ) )
      strputc( CCHAR( o ), port );
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;
         
      fputc( CCHAR( o ), fout );
   }
      
   return o;
}

/*---------------------------------------------------------------------*/
/*    write_char ...                                                   */
/*---------------------------------------------------------------------*/
obj_t
write_char( obj_t c, obj_t port )
{
   int  cc = CCHAR( c );
   
   if( OUTPUT_STRING_PORTP( port ) )
   {
      if( (cc > 0) && (cc < 128) && (char_name[ cc ][ 0 ]) )
      {
      
         lstrputs( "#\\", port, 2 );
         strputs( char_name[ cc ], port );
      }
      else
      {
         char aux[ 10 ];
         
         sprintf( aux, "#a%03d", (unsigned char)(cc) );
         strputs( aux, port );
      }
   }
   else
   {
      FILE *f = OUTPUT_PORT( port ).file;
   
      if( (cc > 0) && (cc < 128) && (char_name[ cc ][ 0 ]) )
         fprintf( f, "#\\%s", char_name[ CCHAR( c ) ] );
      else
         fprintf( f, "#a%03d", (unsigned char)(cc) );
   }
   return c;
}

/*---------------------------------------------------------------------*/
/*    ill_char_rep ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
ill_char_rep( unsigned char c )
{
   char aux[ 10 ];

   sprintf( aux, "#a%03d", c );

   return c_constant_string_to_string( aux );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    write_object ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
write_object( obj_t o, obj_t port ) 
{
   if( INTEGERP( o ) )
      return display_fixnum( o, port );
   
   if( CHARP( o ) )
      return display_char( o, port );

   if( UCS2P( o ) )
      return write_ucs2( o, port );

#if defined( TAG_STRING )
   if( STRINGP( o ) )
      return display_string( o, port );
#endif  
      
#if defined( TAG_REAL )
   if( REALP( o ) )
      return display_flonum( o, port );   
#endif

   switch( (long)o )
   {
    case (long)BNIL :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "()", port, 2 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
               fputs( "()", fout );
            }

            return o;

    case (long)BUNSPEC :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#unspecified", port, 12 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
	       fputs( "#unspecified", fout );
            }
         
            return o;
    
    case (long)BFALSE :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#f", port, 2 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;

               fputs( "#f", fout );
            }

            return o;
    
    case (long)BTRUE :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#t", port, 2 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;

               fputs( "#t", fout );
            }

            return o;

    case (long)BEOF :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#eof-object", port, 14 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
	       fputs( "#eof-object", fout);
            }
         
            return o;

      case (long)BOPTIONAL :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#!optional", port, 11 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file; 
               
	       fputs( "#!optional", fout);
            }
         
            return o;

      case (long)BREST :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#!rest", port, 7 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
	       fputs( "#!rest", fout);
            }
         
            return o;

      case (long)BKEY :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#!key", port, 6 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
	       fputs( "#!key", fout);
            }
         
            return o;

	 
    default :
            if( CNSTP( o ) )
            {
               if( OUTPUT_STRING_PORTP( port ) )
               {
                  char aux[ 7 ];
               
                  sprintf( aux, "#<%04x>", (int)CCNST( o ) );
                  lstrputs( aux, port, 7 );
               
               }
               else
               {
                  FILE *fout = OUTPUT_PORT( port ).file;
               
                  fprintf( fout, "#<%04x>", (int)CCNST( o ) );
               }
               
               return o;
            }
                
            if( !POINTERP( o ) )
            {
               if( OUTPUT_STRING_PORTP( port ) )
               {
                  char aux[ 16 ];
               
                  sprintf( aux, "#<???:%08x>", (unsigned long)o );
                  strputs( aux, port );
               }
               else
               {
                  FILE *fout = OUTPUT_PORT( port ).file;
                  
                  fprintf( fout, "#<???:%08x>", (unsigned long)o );
               }
               
               return o;
            }
	    else 
	    switch( TYPE( o ) )
	    {
#if( !defined( TAG_STRING ) )
               case STRING_TYPE :
	             return display_string( o, port );
#endif

	       case UCS2_STRING_TYPE:
		     return display_ucs2string( o, port );
		     
	       case SYMBOL_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
		        strputs( SYMBOL( o ).name, port );
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fputs( SYMBOL( o ).name, fout );
                     }
                     
		     return o;

	       case KEYWORD_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
		     {
		        strputs( KEYWORD( o ).name, port );
			lstrputs( ":", port, 1 );
		     }
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;

			fprintf( fout, "%s:", KEYWORD( o ).name );
                     }
                     
		     return o;

#if( !defined( TAG_REAL ) )
	       case REAL_TYPE :
	             return display_flonum( o, port );
#endif
                        
	       case PROCEDURE_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 100 ];
                  
                        sprintf( new, "#<procedure:%x.%d>",
                                 (unsigned long) o,
                                 (long)PROCEDURE( o ).arity );
                        strputs( new, port );
                     }
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fprintf( fout, "#<procedure:%x.%d>", (unsigned long)o,
				 (long)PROCEDURE( o ).arity );
                     }
                     
                     return o;
        
	       case OUTPUT_PORT_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 100 ];
                        
			sprintf( new, "#<output_port:%s>",
				 OUTPUT_PORT( o ).name ); 
                        strputs( new, port );
                     }
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                     
                        fprintf( fout, "#<output_port:%s>",
                                 OUTPUT_PORT( o ).name );
                     }
                        
                     return o;
                  
	       case OUTPUT_STRING_PORT_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
		        lstrputs( "#<output_string_port>", port, 21 );
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fputs( "#<output_string_port>", fout );
                     }
                        
                     return o;
                  
	       case INPUT_PORT_TYPE : 
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 500 ];
                        
			sprintf( new, "#<input_port:%s.%d>",
				 INPUT_PORT( o ).name,
				 (long)INPUT_PORT( o ).bufsiz );
                        strputs( new, port );
                     } 
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fprintf( fout, "#<input_port:%s.%d>",
                                 INPUT_PORT( o ).name,
                                 (long)INPUT_PORT( o ).bufsiz );

                     }
                     
		     return o;
      
	       case BINARY_PORT_TYPE : 
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 500 ];
                        
			sprintf( new, "#<binary_input_port:%s.%s>",
				 BINARY_PORT( o ).name,
                                 BINARY_PORT_INP( o ) ? "in" : "out" );
                        strputs( new, port );
                     } 
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fprintf( fout, "#<binary_input_port:%s.%s>",
                                 BINARY_PORT( o ).name,
                                 BINARY_PORT_INP( o ) ? "in" : "out" );
                     }
                                
		     return o;

	       case ELONG_TYPE:
		     if( OUTPUT_STRING_PORTP( port ) )
		     {
			char new[ 100 ];
			
			sprintf( new, "#e%ld", BELONG_TO_LONG( o ) );
			strputs( new, port );
		     }
		     else
		     {
			FILE *fout = OUTPUT_PORT( port ).file;
			
			fprintf( fout, "#e%ld", BELONG_TO_LONG( o ) );
		     }
		     
		     return o;

	       case LLONG_TYPE:
		     if( OUTPUT_STRING_PORTP( port ) )
		     {
			char new[ 100 ];
			
			sprintf( new, "#l%ld", BLLONG_TO_LLONG( o ) );
			strputs( new, port );
		     }
		     else
		     {
			FILE *fout = OUTPUT_PORT( port ).file;
			
			fprintf( fout, "#l%ld", BLLONG_TO_LLONG( o ) );
		     }
		     
		     return o;
                  
	       case FOREIGN_TYPE :
	          if( OUTPUT_STRING_PORTP( port ) )
		  {
		     char new[ 500 ];
		     
		     lstrputs( "#<foreign:", port, 10 );
		     write_object( FOREIGN_ID( o ), port );

		     sprintf( new, ":%x>", FOREIGN_TO_COBJ( o ) );
		     strputs( new, port );
	 	  }
		  else
		  {
		     FILE *fout = OUTPUT_PORT( port ).file;

		     fputs( "#<foreign:", fout );
		     write_object( FOREIGN_ID( o ), port );
		     fprintf( fout, ":%x>", FOREIGN_TO_COBJ( o ) );
		  }
		  return o;

	       case PROCESS_TYPE:
	          if( OUTPUT_STRING_PORTP( port ) )
		  {
		     char new[ 500 ];

		     sprintf( new, "#<process:%d>", PROCESS_PID( o ) );
		     strputs( new, port );
	 	  }
		  else
		  {
		     FILE *fout = OUTPUT_PORT( port ).file;

		     fprintf( fout, "#<process:%d>", PROCESS_PID( o ) );
		  }
		  return o;

	       default :
                  if( OUTPUT_STRING_PORTP( port ) )
                  {
                     char aux[ 20 ];
                  
                     sprintf( aux,
			      "#<???:%d:%08x>",
			      TYPE( o ),
			      (unsigned long)o );
                     strputs( aux, port );
                  }                            
                  else
                  {
                     FILE *fout = OUTPUT_PORT( port ).file;

                     fprintf( fout,
			      "#<???:%d:%08x>",
			      TYPE( o ),
			      (unsigned long)o );
                  }
                        
		  return o;
            }
   }
}


