// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: pkgelement.cc,v 1.11 1998/04/29 00:52:23 jgg Exp $
/* ######################################################################

   Package Element - A single tag/value pair that describes a portion of 
                     the package.
   Element Id - Identity of a Package Element. (Tag portion)

   A static STL map is used to store the pkgElementId::ID records.
   
   ##################################################################### */
									/*}}}*/
#include <pkglib/pkgelement.h>
#include <stdio.h>
#include <ctype.h>

// Storage for statics
typedef map<string,pkgElementId::Id *,less<string> > IdMap_t;
IdMap_t pkgElementId::IdMap;

// ElementId::ElementId - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* This constructor is used when an existing ID is desired. */
pkgElementId::pkgElementId(string Name) : ID(0)
{
   IdMap_t::iterator I = IdMap.find(Name);
   if (I == IdMap.end())
      return;
   ID = (*I).second;
}
/* This is a faster version that advoids the string memory allocation in 
   the inner loop of the loaders */
pkgElementId::pkgElementId(const char *Name) : ID(0)
{
   // Null ID
   if (Name == 0)
      return;
   
   IdMap_t::iterator I = IdMap.find(Name);
   if (I == IdMap.end())
      return;
   ID = (*I).second;
}
									/*}}}*/
// ElementId::ElementId - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* This constructor is used to generate new Element ID's. It is passed 
   a function to generate new instances of a pkgElement. If an existing 
   instance is found then Newer is ignored. */
pkgElementId::pkgElementId(string Name,NewFunc Newer)
{
   // Locate an already inserted item or insert a new one
   Id *&Itm = IdMap[Name];
   
   // Already exists
   if (Itm != 0)
   {
      ID = Itm;
      return;
   }
   
   // Generate the new item
   Itm = new Id(Name,Newer);
   ID = Itm;
}
									/*}}}*/
// PkgElement::~pkgElement - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgElement::~pkgElement()
{
}
									/*}}}*/
// ElmStringVal::pkgElmStringVal - Simple element			/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgElmStringVal::pkgElmStringVal(pkgElementId ID) : pkgElement(ID)
{
}
									/*}}}*/
// ElmVersion::pkgElmVersion - Version string				/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgElmVersion::pkgElmVersion(pkgElementId ID) : pkgElement(ID)
{
}
									/*}}}*/
// ElmPackage::pkElmPackage - Package element				/*{{{*/
// ---------------------------------------------------------------------
/* The package element is the name of the package */
pkgElmPackage::pkgElmPackage() : pkgElmStringVal(pkgElementId("Package"))
{
}
									/*}}}*/
// ElmIntegerVal::Value - Converts the string to an integer w/error	/*{{{*/
// ---------------------------------------------------------------------
/* The integer is signed, this is okay for the current package format 
   but we may want to adjust it later. */
bool pkgElmIntegerVal::Value(string Val)
{
   char *End = 0;
   signed long Res = strtol(Val.c_str(),&End,0);

   // Strip any ending white space
   for(;End != 0 && *End != 0 && isspace(*End) == 1;End++);
   
   if (End == 0 || *End == 0)
   {
      IValue = Res;
      return true;
   }   
   return false;
}
									/*}}}*/
// ElmIntegerVal::Value - Converts the string to an integer w/error	/*{{{*/
// ---------------------------------------------------------------------
/* The integer is signed, this is okay for the current package format 
   but we may want to adjust it later. */
string pkgElmIntegerVal::Value() const
{
   char S[40];
   sprintf(S,"%li",IValue);
   return S;
   
}
									/*}}}*/
// ElmIntegerVal::pkgElmIntegerVal - Integer Element			/*{{{*/
// ---------------------------------------------------------------------
/* */
pkgElmIntegerVal::pkgElmIntegerVal(pkgElementId ID) : pkgElement(ID), 
                  IValue(0)
{
}
									/*}}}*/
// ElmMD5Sum::pkgElmMD5Sum - Constructor				/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgElmMD5Sum::pkgElmMD5Sum(pkgElementId ID) : pkgElement(ID)
{
}
									/*}}}*/
// ElmMultiLine::pkgElmMultiLine - Constructor				/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgElmMultiLine::pkgElmMultiLine(pkgElementId ID) : pkgElement(ID)
{
}
									/*}}}*/
// ElmPkgList::pkgElmVersion						/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class*/
pkgElmPkgList::pkgElmPkgList(pkgElementId ID) : pkgElement(ID)
{
}
									/*}}}*/
// ElmPkgList::operator == - Comparison operator			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool pkgElmPkgList::operator ==(const pkgElement &) const
{
   return false;
}
									/*}}}*/
// ElmPkgList::Value - Convert the list into a string			/*{{{*/
// ---------------------------------------------------------------------
/* */
string pkgElmPkgList::Value() const
{
   string Res;
   Res.reserve(100);
   for (const Item *I = List.begin(); I != List.end(); I++)
   {
      Res += I->Package;

      if (string(I->Version).empty() == true)
      {
	 if (I + 1 == List.end())
	    continue;
	 
	 if ((I->Operator & pkgOP_OR) == pkgOP_OR)
	    Res += "| ";
	 else
	    Res += ", ";
      }      
      else
      {
	 switch (I->Operator & 0x0F)
	 {
	    case pkgOP_EQUALS:
	    Res += " (= ";
	    break;

	    case pkgOP_LESSEQ:
	    Res += " (<= ";
	    break;
	    
	    case pkgOP_GREATEREQ:
	    Res += " (>= ";
	    break;
	    
	    case pkgOP_LESS:
	    Res += " (<< ";
	    break;

	    case pkgOP_GREATER:
	    Res += " (>> ";
	    break;

	    default:
	    Res += " ( ";
	    break;
	 }

	 Res += I->Version;

	 if (I + 1 == List.end())
	 {
	    Res += ")";
	    continue;
	 }
	 	 
	 if ((I->Operator & pkgOP_OR) == pkgOP_OR)
	    Res += ") | ";
	 else
	    Res += "), ";
      }      
   }   

   return Res;
}
									/*}}}*/
// ElmPkgList::Value - Convert a string into the list			/*{{{*/
// ---------------------------------------------------------------------
/* The format is Package [(op ver)] |,...
   
   We parse it by first locating the package name then considering a bracketed
   version expression and then finally the delimiter.
   
   The | syntax is done with a special flag on the package before the or, 
   which marks it as being ored with the next package.
 */
bool pkgElmPkgList::Value(string Val)
{
   const char *I = Val.begin();
   
   while (I != Val.end())
   {
      const char *Start = I;
      Item New;
      
      // Strip off leading space
      for (;Start != Val.end() && isspace(*Start) != 0; Start++);
      I = Start;
      
      // Parse off the package name
      for (;I != Val.end() && isspace(*I) == 0 && *I != '(' && *I != ')' &&
	   *I != ',' && *I != '|'; I++);

      // Malformed, no '('
      if (I != Val.end() && *I == ')')
	 return false;     
      
      if (I == Start)
	 return false;
      
      // Stash the package name
      New.Package.assign(Start,I - Start);
      
      // Skip white space
      for (;I != Val.end() && isspace(*I) != 0 ; I++);
      if (I == Val.end())
      {
	 List.push_back(New);
	 continue;
      }
      
      
      // Parse a version
      if (*I == '(')
      {
	 // Skip the '('
	 for (I++;I != Val.end() && isspace(*I) != 0 ; I++);
	 if (I + 3 >= Val.end())
	    return false;

	 // Determine the operator
	 switch (*I)
	 {
	    case '<':
	    I++;
	    if (*I == '=')
	    {
	       I++;
	       New.Operator = pkgOP_LESSEQ;
	       break;
	    }
	    
	    if (*I == '<')
	    {
	       I++;
	       New.Operator = pkgOP_LESS;
	       break;
	    }
	    
	    // < is the same as <= and << is really Cs < for some reason
	    New.Operator = pkgOP_LESSEQ;
	    break;
	    
	    case '>':
	    I++;
	    if (*I == '=')
	    {
	       I++;
	       New.Operator = pkgOP_GREATEREQ;
	       break;
	    }
	    
	    if (*I == '>')
	    {
	       I++;
	       New.Operator = pkgOP_GREATER;
	       break;
	    }
	    
	    // > is the same as >= and >> is really Cs > for some reason
	    New.Operator = pkgOP_GREATEREQ;
	    break;
	    
	    case '=':
	    New.Operator = pkgOP_EQUALS;
	    I++;
	    break;

	    // HACK around bad package definitions
	    default:
	    New.Operator = pkgOP_EQUALS;
	    break;
//	    return false;
	 }

	 // Skip whitespace
	 for (;I != Val.end() && isspace(*I) != 0; I++);
	 Start = I;
	 for (;I != Val.end() && *I != ')'; I++);
	 if (I == Val.end() || Start == I)
	    return false;     

	 New.Version = string(Start,I-Start);
	 I++;
      }
      
      // Skip whitespace
      for (;I != Val.end() && isspace(*I) != 0; I++);
      if (I == Val.end() || *I == ',')
      {
	 List.push_back(New);
	 
	 if (I == Val.end())
	    continue;
	 
	 for (I++; I != Val.end() && isspace(*I) != 0; I++);
	 continue;
      }
      
      if (*I == '|')
      {
	 New.Operator |= pkgOP_OR;
	 List.push_back(New);
	 for (I++; I != Val.end() && isspace(*I) != 0; I++);
	 continue;
      }      

      return false;
   }

   return true;
}
									/*}}}*/
// ElmPkgList::SetLines - Handle multi line depends			/*{{{*/
// ---------------------------------------------------------------------
/* Apparently dpkg allows multi-line depends, it's somewhat weird, but 
   we allow it as well. */
bool pkgElmPkgList::SetLines(const vector<string> &Lines)
{
   for (vector<string>::const_iterator I = Lines.begin();
	I < Lines.end(); I++)
      if (Value(*I) == false)
	 return false;
   
   return true;
}
									/*}}}*/

// pkgElmRegister - Registers all built in elements			/*{{{*/
// ---------------------------------------------------------------------
/* This primes the pkgElementID's table with a list of all known elements. 
   It is not required that an element exist in this list to be loaded. If 
   the loader detects an unknown element it will construct a class that is
   capable of storing it on it's own. For text files this will always be a
   multi-line element. The loader also adds an entry to the id table
   so future references will also use a multiline element. */
void pkgElmRegister()
{
   pkgElementId("Package",pkgElmPackage::Newer);
   pkgElementId("Priority",pkgElmStringVal::Newer);
   pkgElementId("Section",pkgElmStringVal::Newer);
   pkgElementId("Size",pkgElmIntegerVal::Newer);
   pkgElementId("Maintainer",pkgElmStringVal::Newer);
   pkgElementId("Architecture",pkgElmStringVal::Newer);
   pkgElementId("Source",pkgElmStringVal::Newer);
   pkgElementId("Version",pkgElmVersion::Newer);

   pkgElementId("Depends",pkgElmPkgList::Newer);
   pkgElementId("Recommends",pkgElmPkgList::Newer);
   pkgElementId("Suggests",pkgElmPkgList::Newer);
   pkgElementId("Provides",pkgElmPkgList::Newer);
   pkgElementId("Conflicts",pkgElmPkgList::Newer);
   pkgElementId("Replaces",pkgElmPkgList::Newer);
   pkgElementId("Essential",pkgElmPkgList::Newer);   
   pkgElementId("Pre-Depends",pkgElmPkgList::Newer);   
   
   pkgElementId("Filename",pkgElmStringVal::Newer);
   pkgElementId("MSDOS-Filename",pkgElmStringVal::Newer);
   pkgElementId("Installed-Size",pkgElmIntegerVal::Newer);
   pkgElementId("MD5sum",pkgElmMD5Sum::Newer);
   pkgElementId("Description",pkgElmMultiLine::Newer);

   // Unstandard I guess:
   pkgElementId("comment",pkgElmStringVal::Newer);
}
									/*}}}*/
