/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */
#ifndef ICEWM_H
#define ICEWM_H

//#pragma interface "icewm.h"

#include "config.h"

#define ICON_SMALL 16   // small: 16x16
#define ICON_LARGE 32   // large: 32x32

#define SCROLLBAR_SIZE 16

#ifdef __EMX__ 
#define PATHSEP ';'
#define SLASH '\\'
#define ISSLASH(c) ((c) == '/' || (c) == '\\')
#else
#define PATHSEP ':'
#define SLASH '/'
#define ISSLASH(c) ((c) == '/')
#endif

extern "C" {
#include <assert.h>
#include <stdio.h>
#include <string.h>
#ifdef NEED_STRINGS_H
#include <strings.h>
#endif
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#ifdef NEED_TIME_H
#include <time.h>
#endif
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#ifdef NEED_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <limits.h>
#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include <ctype.h>
#include <fcntl.h>

#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include <X11/xpm.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#ifdef SM
#include <X11/SM/SMlib.h>
#endif
#ifndef NO_MWM_HINTS
#include "MwmUtil.h"
#endif

#include "WinMgr.h"
    
#ifdef __EMX__
char* __XOS2RedirRoot(const char*);
#define REDIR_ROOT(path) __XOS2RedirRoot(path)
#else
#define REDIR_ROOT(path) (path)
#endif
}

#ifndef O_TEXT
#define O_TEXT 0
#endif
#ifndef O_BINARY
#define O_BINARY 0
#endif

#ifdef NEED_BOOL
enum bool_t { false = 0, true = 1 };
typedef int bool;
#endif

#include "default.h"

#ifdef CONFIG_GUIEVENTS
#include "guievent.h"
#endif

#ifndef NO_CONFIGURE_MENUS
extern char *menuFile;
extern char *programFile;
extern char *toolbarFile;
#endif
#ifndef NO_WINDOW_OPTIONS
extern char *winOptFile;
#endif

extern char **progNames;
extern char **progCmds;
extern char ***progArgs;
extern int progCount;

#define MAXWORKSPACES 64
#define INVALID_WORKSPACE 0xFFFFFFFF

extern CARD32 workspaceCount;
extern char *workspaceNames[MAXWORKSPACES];

#define DO_BORDER(x) 0 // (x)      // 0 looks much better
#define KEY_MODMASK(x) ((x) & (app->KeyMask))
#define BUTTON_MODMASK(x) ((x) & (app->ButtonKeyMask))

#define ISMASK(w,e,n) (((w) & ~(n)) == (e))
#define HASMASK(w,e,n) ((((w) & ~(n)) & (e)) == (e))

#define ISLOWER(c) ((c) >= 'a' && (c) <= 'z')
#define TOUPPER(c) (ISLOWER(c) ? (c) - 'a' + 'A' : (c))

void die(int exitcode, const char *msg, ...);
void warn(const char *msg, ...);

class YApplication;
class YTimer;
class YWindow;
class YDesktop;
#ifdef CONFIG_TOOLTIP
class YToolTip;
#endif
class YFocusWindow;
class YPopupWindow;
class YMenu;
class YButton;
class YWMApp;
class YWindowManager;
class YFrameWindow;
class YFrameControl;
class YFrameTitle;
class YFrameButton;
class YFrameClient;
#ifdef CONFIG_TASKBAR
class TaskBar;
class TaskBarApp;
#endif
class MiniIcon;

#ifdef CONFIG_WINLIST
class YListItem;
class YListBox;
class YScrollBar;
class WindowListItem;
class WindowListBox;
#endif
class MailBoxStatus;
class CPUStatus;

class TimerListener;

#ifdef DEBUG
extern bool debug;
extern bool debug_z;

void msg(const char *msg, ...);
void logEvent(XEvent xev);
void dumpZorder(const char *oper, YFrameWindow *w, YFrameWindow *a = 0);

#define DBG if (debug)
#define MSG(x) DBG msg x
#else

#define DBG if (0)
#define MSG(x)
#endif

extern YApplication *app;
extern YWMApp * wmapp;
extern YWindowManager *manager;
extern YDesktop *desktop;
#ifdef CONFIG_TASKBAR
extern TaskBar *taskBar;
#endif

enum WMCommand {
    cmdNone = 0,
    cmdSubmenu,
    cmdSeparator,
    cmdRestore,
    cmdMove,
    cmdSize,
    cmdMinimize, cmdHide,
    cmdMaximize, cmdMaximizeVert, cmdMaximizeHorz,
    cmdRollup,
    cmdLower,    cmdRaise,
    cmdClose,    cmdKill,
    cmdSysMenu,
    cmdActivate,
    cmdActivateWindow,
    cmdRefresh,
    cmdCloseSession,
    cmdExit,
    cmdExec,
    cmdRun,
    cmdRestart,
    cmdOccupyAll,
    cmdOccupyAllOrCurrent,
    cmdOccupyWorkspace,
    cmdOccupyOnlyWorkspace,
    cmdMoveToWorkspace,
    cmdActivateWorkspace,
    cmdTermProcess,
    cmdKillProcess,

    cmdCascade,
    cmdTileVertical,
    cmdTileHorizontal,
    cmdShow,
    cmdSetLayer,
    cmdOpen
#ifdef GNOME
    ,cmdGnome
#endif
#ifdef CONFIG_WINLIST
    , cmdWindowList
#endif
};

enum PhaseType {
    phaseStartup,
    phaseShutdown,
    phaseRunning,
    phaseRestart
};

struct WindowOption {
    char *name;
    char *icon;
    unsigned long functions, function_mask;
    unsigned long decors, decor_mask;
    unsigned long options, option_mask;
    CARD32 workspace;
    CARD32 layer;
};

class WindowOptions {
public:
    WindowOptions();
    ~WindowOptions();

    WindowOption *getWindowOption(char *name, bool create, bool remove = false);
    void setWinOption(char *class_instance, char *opt, char *arg);

    static void combineOptions(WindowOption &cm, WindowOption &n);
private:
    WindowOption *winOptions;
    int winOptionsCount;
};

typedef int FrameState;

class YColor {
public:
    YColor(unsigned short red, unsigned short green, unsigned short blue);
    YColor(const char *clr);

    void alloc();
    unsigned long pixel() { return color.pixel; }

    YColor *darker();
    YColor *brighter();
private:
    XColor color;
    YColor *fDarker;
    YColor *fBrighter;
};

class YFont {
public:
    YFont(const char *name);
    ~YFont();

    void alloc();


#ifndef I18N
    XFontStruct *getFontStruct() { return font; }

    int height() { return font->max_bounds.ascent + font->max_bounds.descent; }
    int descent() { return font->max_bounds.descent; }
    int ascent() { return font->max_bounds.ascent; }

    int textWidth(const char *str) {
        return XTextWidth(font, str, strlen(str));
    }
    int textWidth(const char *str, int len) {
        return XTextWidth(font, str, len);
    }
private:
    XFontStruct *font;
#else
    XFontSet getFontStruct() { return font; }

    int height() { return fontascent + fontdescent; }
    int descent() { return fontdescent; }
    int ascent() { return fontascent; }

    int textWidth(char *str) {
        return XmbTextEscapement(font, str, strlen(str));
    }
    int textWidth(char *str, int len) {
        return XmbTextEscapement(font, str, len);
    }
private:
    XFontSet font;
    int fontascent, fontdescent;
#endif
};

class YPixmap {
public:
    YPixmap(const char *fileName);
    YPixmap(const char *fileName, int w, int h);
    YPixmap(Pixmap pixmap, Pixmap mask, int w, int h);
    ~YPixmap();

    Pixmap pixmap() { return fPixmap; }
    Pixmap mask() { return fMask; }
    unsigned int width() { return fWidth; }
    unsigned int height() { return fHeight; }
private:
    Pixmap fPixmap;
    Pixmap fMask;
    unsigned int fWidth, fHeight;
    bool fOwned;
};

class YIcon {
public:
    YIcon(const char *fileName);
    YIcon(int count, int reclen, CARD32 *elem);
    ~YIcon();

    YPixmap *large();
    YPixmap *small();

    char *iconName() { return fPath; }
    YIcon *next() { return fNext; }
private:
    char *fPath;
    YPixmap *fSmall;
    YPixmap *fLarge;
    YIcon *fNext;
    int loadedS;
    int loadedL;

    int findIcon(char **fullPath, int size);
    YPixmap *loadIcon(int size);
};

extern YIcon *defaultAppIcon;

class CommandListener {
public:
    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers) = 0;
};

class PopDownListener {
public:
    virtual void handlePopDown(YPopupWindow *popup) = 0;
};

class TimerListener {
public:
    virtual bool handleTimer(YTimer *timer) = 0;
};

class YButtonListener {
public:
    virtual void ButtonClick(YButton *button, unsigned int modifiers) = 0;
};

class Graphics {
public:
    Graphics(YWindow *window);
    Graphics(YPixmap *pixmap);
    virtual ~Graphics();

    void copyArea(int x, int y, int width, int height, int dx, int dy) {
        XCopyArea(display, drawable, drawable, gc,
                  x, y, width, height, dx, dy);
    }
    void copyPixmap(YPixmap *pixmap, int x, int y, int width, int height, int dx, int dy) {
        XCopyArea(display, pixmap->pixmap(), drawable, gc,
                  x, y, width, height, dx, dy);
    }
    void drawPoint(int x, int y) {
        XDrawPoint(display, drawable, gc, x, y);
    }
    void drawLine(int x1, int y1, int x2, int y2) {
        XDrawLine(display, drawable, gc, x1, y1, x2, y2);
    }
    void drawRect(int x, int y, int width, int height) {
        XDrawRectangle(display, drawable, gc, x, y, width, height);
    }
    void drawChars(char *data, int offset, int len, int x, int y) {
#ifndef I18N
        XDrawString(display, drawable, gc, x, y, data + offset, len);
#else
        XmbDrawString(display, drawable, font->getFontStruct(), gc, x, y, data + offset, len);
#endif
    }
    void drawCharUnderline(int x, int y, char *str, int charPos) {
#ifndef I18N
        int left = XTextWidth(font->getFontStruct(), str, charPos);
        int right = XTextWidth(font->getFontStruct(), str, charPos + 1) - 1;
#else
        int left = XmbTextEscapement(font->getFontStruct(), str, charPos);
        int right = XmbTextEscapement(font->getFontStruct(), str, charPos + 1) - 1;
#endif

        drawLine(x + left, y + 2, x + right, y + 2);
    }

    void drawPixmap(YPixmap *pix, int x, int y);
    void drawClippedPixmap(Pixmap pix, Pixmap clip,
                           int x, int y, int w, int h, int toX, int toY);
    void fillRect(int x, int y, int width, int height) {
        XFillRectangle(display, drawable, gc,
                       x, y, width, height);
    }
    void fillPolygon(XPoint *points, int n, int shape, int mode) {
        XFillPolygon(display, drawable, gc, points, n, shape, mode);
    }
    void setColor(YColor *aColor) {
        color = aColor;
        XSetForeground(display, gc, color->pixel());
    }
    void setFont(YFont *aFont) {
        font = aFont;
#ifndef I18N
        XSetFont(display, gc, font->getFontStruct()->fid);
#endif
    }

    void draw3DRect(int x, int y, int w, int h, bool raised);
    void drawBorderW(int x, int y, int w, int h, bool raised);
    void drawBorderM(int x, int y, int w, int h, bool raised);
    void drawBorderG(int x, int y, int w, int h, bool raised);
    void drawCenteredPixmap(int x, int y, int w, int h, YPixmap *pixmap);
    void drawOutline(int l, int t, int r, int b, int iw, int ih);
    void repHorz(YPixmap *pixmap, int x, int y, int w);
    void repVert(YPixmap *pixmap, int x, int y, int h);

    void drawArrow(int direction, int style, int x, int y, int size);

    YColor *getColor() { return color; }
    YFont *getFont() { return font; }
    GC handle()  { return gc; }
private:
    Display *display;
    Drawable drawable;
    GC gc;

    YColor *color;
    YFont *font;
};

class YWindow {
public:
    YWindow(YWindow *aParent = 0, Window win = 0);
    virtual ~YWindow();

    void setStyle(unsigned long aStyle);

    void show();
    void hide();
    virtual void raise();
    virtual void lower();

    void repaint();
    void repaintFocus();

    void reparent(YWindow *parent, int x, int y);

    void setFocus();

    void setGeometry(int x, int y, unsigned int width, unsigned int height);
    void setSize(unsigned int width, unsigned int height);
    void setPosition(int x, int y);
    virtual void configure(int x, int y, unsigned int width, unsigned int height);

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void paintFocus(Graphics &g, int x, int y, unsigned int w, unsigned int h);

    virtual void handleEvent(const XEvent &event);

    virtual void handleExpose(const XExposeEvent &expose);
    virtual void handleGraphicsExpose(const XGraphicsExposeEvent &graphicsExpose);
    virtual void handleConfigure(const XConfigureEvent &configure);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleProperty(const XPropertyEvent &property);
    virtual void handleColormap(const XColormapEvent &colormap);
    virtual void handleFocus(const XFocusChangeEvent &focus);
    virtual void handleClientMessage(const XClientMessageEvent &message);
#if 0
    virtual void handleVisibility(const XVisibilityEvent &visibility);
    virtual void handleCreateWindow(const XCreateWindowEvent &createWindow);
#endif
    virtual void handleMap(const XMapEvent &map);
    virtual void handleUnmap(const XUnmapEvent &unmap);
    virtual void handleDestroyWindow(const XDestroyWindowEvent &destroyWindow);
    virtual void handleConfigureRequest(const XConfigureRequestEvent &configureRequest);
    virtual void handleMapRequest(const XMapRequestEvent &mapRequest);
#ifdef SHAPE
    virtual void handleShapeNotify(const XShapeEvent &shape);
#endif

    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);

    virtual void handleClose();
    
    void setPointer(Cursor pointer);
    void setGrabPointer(Cursor pointer);
    void grabKey(int key, unsigned int modifiers);
    void grabButton(int button, unsigned int modifiers);

    void captureEvents();
    void releaseEvents();

    Window handle();
    YWindow *parent() { return fParentWindow; }

    Graphics &getGraphics();

    int x() { return fX; }
    int y() { return fY; }
    unsigned int width() { return fWidth; }
    unsigned int height() { return fHeight; }

    int visible() { return (flags & wfVisible) ? 1 : 0; }
    int created() { return (flags & wfCreated) ? 1 : 0; }
    int adopted() { return (flags & wfAdopted) ? 1 : 0; }
    int destroyed() { return (flags & wfDestroyed) ? 1 : 0; }
    int unmapped() { return (flags & wfUnmapped) ? 1 : 0; }

    virtual void donePopup(YPopupWindow * /*command*/);

    typedef enum {
        wsOverrideRedirect = 1 << 0,
        wsSaveUnder        = 1 << 1,
        wsManager          = 1 << 2
    } WindowStyle;

    virtual bool isFocusTraversable();
    bool isFocused();
    bool isEnabled() { return fEnabled; }
    void setEnabled(bool enable);
    void nextFocus();
    void prevFocus();
    void requestFocus();
    void setFocus(YWindow *window);
    virtual void gotFocus();
    virtual void lostFocus();

    bool isToplevel() { return fToplevel; }
    void setToplevel(bool toplevel) { fToplevel = toplevel; }

    void installAccelerator(unsigned int key, unsigned int mod, YWindow *win);
    void removeAccelerator(unsigned int key, unsigned int mod, YWindow *win);

    void setToolTip(const char *tip);

    void mapToGlobal(int &x, int &y);
    void mapToLocal(int &x, int &y);

    void setWinGravity(int gravity);

    void setDND(bool enabled);

    void XdndStatus(bool acceptDrop, Atom dropAction);
    virtual void handleXdnd(const XClientMessageEvent &message);

    virtual void handleDNDEnter();
    virtual void handleDNDLeave();

private:
    typedef enum {
        wfVisible   = 1 << 0,
        wfCreated   = 1 << 1,
        wfAdopted   = 1 << 2,
        wfDestroyed = 1 << 3,
        wfUnmapped  = 1 << 4
    } WindowFlags;

    void create();
    void destroy();

    void insertWindow();
    void removeWindow();
    
    YWindow *fParentWindow;
    YWindow *fNextWindow;
    YWindow *fPrevWindow;
    YWindow *fFirstWindow;
    YWindow *fLastWindow;
    YWindow *fFocusedWindow;

    Window fHandle;
    unsigned long flags;
    unsigned long fStyle;
    int fX, fY;
    unsigned int fWidth, fHeight;
    Cursor fPointer;
    int unmapCount;
    Graphics *fGraphics;
    long fEventMask;

    bool fEnabled;
    bool fToplevel;

    typedef struct YAccelerator {
        unsigned int key;
        unsigned int mod;
        YWindow *win;
        YAccelerator *next;
    } YAccelerator;

    YAccelerator *accel;
#ifdef CONFIG_TOOLTIP
    YToolTip *fToolTip;
#endif

    static XButtonEvent fClickEvent;
    static YWindow *fClickWindow;
    static Time fClickTime;
    static int fClickCount;
    static int fClickDrag;
    static unsigned int fClickButton;
    static unsigned int fClickButtonDown;
    static YTimer *fToolTipTimer;

    bool fDND;
    Window XdndDragSource;
    Window XdndDropTarget;
};

class YDesktop: public YWindow {
public:
    YDesktop(YWindow *aParent = 0, Window win = 0);
    virtual ~YDesktop();
    
    virtual void resetColormap(bool active);
    virtual void resetInputFocus();
};

#ifdef CONFIG_TOOLTIP
class YToolTip: public YWindow, public TimerListener {
public:
    YToolTip(YWindow *aParent = 0, Window win = 0);
    virtual ~YToolTip();
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void setText(const char *tip);
    virtual bool handleTimer(YTimer *t);
    void locate(YWindow *w, const XCrossingEvent &crossing);

private:
    void display();

    char *fText;

    static YColor *toolTipBg;
    static YColor *toolTipFg;
    static YFont *toolTipFont;
};
#endif

class MoveSizeStatus: public YWindow {
public:
    MoveSizeStatus(YWindow *aParent, Window win = 0);
    virtual ~MoveSizeStatus();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void setStatus(YFrameWindow *frame, int x, int y, int width, int height);
    void setStatus(YFrameWindow *frame);
    void begin(YFrameWindow *frame);
    void end() { hide(); }
private:
    int fX, fY, fW, fH;

    static YColor *statusFg;
    static YColor *statusBg;
    static YFont *statusFont;
};

class YPopupWindow: public YWindow {
public:
    YPopupWindow(YWindow *aParent);
    virtual ~YPopupWindow();

    virtual void sizePopup();
    
    bool popup(YWindow *forWindow,
               CommandListener *commandWindow,
               PopDownListener *popDown,
               int x, int y,
               int x_delta, int y_delta,
               unsigned int flags);
    bool popup(YWindow *forWindow,
               CommandListener *commandWindow,
               PopDownListener *popDown,
               unsigned int flags);
    void popdown();

    virtual void updatePopup();
    void finishPopup(WMCommand command, void *context, unsigned int modifiers);
    void cancelPopup();

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);

    virtual void activatePopup();
    virtual void deactivatePopup();

    unsigned int popupFlags() { return fFlags; }

    CommandListener *commandListener() { return fCommandListener; }
    YPopupWindow *prevPopup() { return fPrevPopup; }
    void setPrevPopup(YPopupWindow *prevPopup) { fPrevPopup = prevPopup; }

    enum {
        pfButtonDown        = 1 << 0,
        pfCanFlipVertical   = 1 << 1,
        pfCanFlipHorizontal = 1 << 2,
        pfFlipVertical      = 1 << 3,
        pfFlipHorizontal    = 1 << 4,
        pfNoPointerChange   = 1 << 5,
        pfPopupMenu         = 1 << 6
    } PopupFlags;

private:
    unsigned int fFlags;
    YWindow *fForWindow;
    CommandListener *fCommandListener;
    PopDownListener *fPopDownListener;
    YPopupWindow *fPrevPopup;
};

class YMenu: public YPopupWindow {
public:
    class YMenuItem {
    public:
        YMenuItem(const char *name, int hotCharPos, const char *param, WMCommand command, YMenu *submenu, void *context = 0);
        YMenuItem() { fName = 0; fHotCharPos = -1; fParam = 0; fCommand = cmdSeparator; fContext = 0; fEnabled = 0; fSubmenu = 0; }
        ~YMenuItem();
        char *name() { return fName; }
        char *param() { return fParam; }
        WMCommand command() { return fCommand; }
        void *context() { return fContext; }
        YMenu *submenu() { return fSubmenu; }
        int hotChar() { return (fName && fHotCharPos >= 0) ? fName[fHotCharPos] : -1; }
        int hotCharPos() { return fHotCharPos; }

        YPixmap *getPixmap() { return fPixmap; }
        void setPixmap(YPixmap *pixmap);
        void setChecked(bool c);
        int isChecked() { return fChecked; }
        int isEnabled() { return fEnabled; }
        void setEnabled(bool e) { fEnabled = e; }
    private:
        char *fName;
        char *fParam;
        WMCommand fCommand;
        void *fContext;
        int fHotCharPos;
        YMenu *fSubmenu;

        YPixmap *fPixmap;
        bool fChecked;
        bool fEnabled;
    };

    YMenu(YWindow *parent = 0);
    virtual ~YMenu();

    virtual void sizePopup();
    virtual void activatePopup();
    virtual void deactivatePopup();
    virtual void donePopup(YPopupWindow *popup);
    
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    
    YMenuItem *add(YMenuItem *item);
    YMenuItem *addItem(const char *name, int hotCharPos, const char *param, WMCommand command, void *context = 0);
    YMenuItem *addSubmenu(const char *name, int hotCharPos, YMenu *submenu);
    YMenuItem *addSeparator();
    void removeAll();
    YMenuItem *findItem(WMCommand command);
    YMenuItem *findSubmenu(YMenu *sub);

    void enableCommand(WMCommand cmd); // cmdNone == All
    void disableCommand(WMCommand cmd); // cmdNone == All

    int itemCount() { return fItemCount; }
    YMenuItem *item(int n) { return fItems[n]; }

    bool isShared() { return fShared; }
    void setShared(bool shared) { fShared = shared; }

private:
    int fItemCount;
    YMenuItem **fItems;
    int selectedItem;
    int paintedItem;
    int paramPos;
    int namePos;
    YPopupWindow *fPopup;
    YPopupWindow *fPopupActive;
    bool fShared;

    int getItemHeight(int itemNo, int &h, int &top, int &bottom, int &pad);
    void getItemWidth(int i, int &iw, int &nw, int &pw);
    void getOffsets(int &left, int &top, int &right, int &bottom);
    void getArea(int &x, int &y, int &w, int &h);

    void drawSeparator(Graphics &g, int x, int y, int w);
    
    void paintItem(Graphics &g, int i, int &l, int &t, int &r, int paint);
    void paintItems();
    int findItemPos(int item, int &x, int &y);
    int findItem(int x, int y);
    int findActiveItem(int cur, int direction);
    void focusItem(int item, int submenu, int byMouse);
    int activateItem(int no, int byMouse, unsigned int modifiers);
    int onCascadeButton(int selectedItem, int x, int y);
};

class ObjectContainer {
public:
    // !!! eliminate 'int cmd', make context an object
    virtual void addObject(char *name, int cmd, YIcon *icon, void *context) = 0;
    virtual void addSeparator() = 0;
    virtual void addContainer(char *name, YIcon *icon, ObjectContainer *container) = 0;
};

class ObjectMenu: public YMenu, public ObjectContainer {
public:
    ObjectMenu(YWindow *parent = 0);
    virtual ~ObjectMenu();

    virtual void addObject(char *name, int cmd, YIcon *icon, void *context);
    virtual void addSeparator();
    virtual void addContainer(char *name, YIcon *icon, ObjectContainer *container);
};

class ObjectBar: public YWindow, public ObjectContainer, public YButtonListener {
public:
    ObjectBar(YWindow *parent);
    virtual ~ObjectBar();

    virtual void addObject(char *name, int cmd, YIcon *icon, void *context);
    virtual void addSeparator();
    virtual void addContainer(char *name, YIcon *icon, ObjectContainer *container);

    virtual void ButtonClick(YButton *button, unsigned int modifiers);

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void addButton(char *name, YIcon *icon, YButton *button);
};

class WindowListMenu: public YMenu {
public:
    WindowListMenu(YWindow *parent = 0);
    virtual void updatePopup();
};

class BrowseMenu: public ObjectMenu {
public:
    BrowseMenu(const char *path, YWindow *parent = 0);
    virtual ~BrowseMenu();
    virtual void updatePopup();
private:
    char *fPath;
    time_t fModTime;
};

class GnomeMenu: public ObjectMenu {
public:
    GnomeMenu(YWindow *parent, const char *path);
    virtual ~GnomeMenu();
    virtual void updatePopup();
private:
    char *fPath;
    time_t fModTime;
};

class SwitchWindow: public YPopupWindow {
public:
    SwitchWindow(YWindow *parent = 0);
    virtual ~SwitchWindow();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    YFrameWindow *nextWindow(YFrameWindow *from, bool zdown, bool next);
    void begin(bool zdown);

    virtual void activatePopup();
    virtual void deactivatePopup();
    
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);

    void destroyedFrame(YFrameWindow *frame);
private:
    YFrameWindow *fActiveWindow;

    static YColor *switchFg;
    static YColor *switchBg;
    static YFont *switchFont;
};

class YWindowManager: public YDesktop {
public:
    YWindowManager(YWindow *parent, Window win = 0);
    virtual ~YWindowManager();

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual bool handleKey(const XKeyEvent &key);

    virtual void handleConfigureRequest(const XConfigureRequestEvent &configureRequest);
    virtual void handleMapRequest(const XMapRequestEvent &mapRequest);
    virtual void handleUnmap(const XUnmapEvent &unmap);
    virtual void handleDestroyWindow(const XDestroyWindowEvent &destroyWindow);
    virtual void handleClientMessage(const XClientMessageEvent &message);
    virtual void handleProperty(const XPropertyEvent &property);

    void manageClients();
    void unmanageClients();

    YFrameClient *manageWindow(Window win);
    YFrameWindow *mapWindow(Window win, int newClient = 1);
    YFrameClient *unmapWindow(Window win);
    void unmanageWindow(Window win, bool map);
    void destroyedWindow(Window win);
    YFrameClient *manageClient(Window win);
    YFrameClient *findClient(Window win);
    
    void unmanageFrame(YFrameWindow *frame);

    YFrameWindow *getFrame(Window win);
    YFrameWindow *createFrame(YFrameClient *client);

    YFrameClient *getClient(Window win);
    YFrameClient *createClient(Window win);

    void setFocus(YFrameWindow *window, bool canWarp = false);
    void loseFocus(YFrameWindow *window);
    void loseFocus(YFrameWindow *window,
                   YFrameWindow *next,
                   YFrameWindow *prev);
    YFrameWindow *focus() { return fFocus; }
    void activate(YFrameWindow *frame, bool canWarp = false);

    void installColormap(Colormap cmap);
    void setColormapWindow(YFrameWindow *frame);
    YFrameWindow *colormapWindow() { return fColormapWindow; }

    int minX(CARD32 layer);
    int minY(CARD32 layer);
    int maxX(CARD32 layer);
    int maxY(CARD32 layer);
    int maxWidth(CARD32 layer) { return maxX(layer) - minX(layer); }
    int maxHeight(CARD32 layer) { return maxY(layer) - minY(layer); }

    void placeWindow(YFrameWindow *frame, int x, int y, int newClient = 1);

    YFrameWindow *top(CARD32 layer) { return fTop[layer]; }
    void setTop(CARD32 layer, YFrameWindow *top);
    YFrameWindow *bottom(CARD32 layer) { return fBottom[layer]; }
    void setBottom(CARD32 layer, YFrameWindow *bottom) { fBottom[layer] = bottom; }

    YFrameWindow *topLayer();
    YFrameWindow *bottomLayer();

    void restackWindows(YFrameWindow *win, int oper = 0);
    void focusTopWindow();
    void relocateWindows(int dx, int dy);

    YMenu *createWindowMenu(YMenu *menu, CARD32 workspace);
    int windowCount(CARD32 workspace);
    void popupWindowListMenu(int x, int y);

    CARD32 activeWorkspace() { return fActiveWorkspace; }
    void activateWorkspace(CARD32 workspace);
    CARD32 workspaceCount() { return ::workspaceCount; }
    char *workspaceName(CARD32 workspace) { return ::workspaceNames[workspace]; }

    void popupStartMenu();
    void popupWindowListMenu();

    void putWorkArea();
    void setWinWorkspace(CARD32 workspace);
    void updateWorkArea();

    void getIconPosition(YFrameWindow *frame, int *iconX, int *iconY);

    void wmCloseSession();

    virtual void resetColormap(bool active);
    virtual void resetInputFocus();
private:
    YFrameWindow *fFocus;
    YFrameWindow *fTop[WinLayerCount];
    YFrameWindow *fBottom[WinLayerCount];
    CARD32 fActiveWorkspace;
    YFrameWindow *fColormapWindow;
    int fMinX, fMinY, fMaxX, fMaxY;
    bool metaDown;
    unsigned short buttonsPressed;
};

class YFrameControl: public YWindow {
public:
    YFrameControl(YWindow *parent, YFrameWindow *frame, Window win = 0): YWindow((YWindow *)parent, win) { fFrame = frame; }
    virtual ~YFrameControl() { }

    YFrameWindow *frame() { return fFrame; }
    virtual void setFrame(YFrameWindow *newFrame);
private:
    YFrameWindow *fFrame;
};

class YFrameClient: public YFrameControl {
public:
    YFrameClient(YWindow *parent, YFrameWindow *frame, Window win = 0);
    virtual ~YFrameClient();

    virtual void handleProperty(const XPropertyEvent &property);
    virtual void handleColormap(const XColormapEvent &colormap);
    virtual void handleUnmap(const XUnmapEvent &unmap);
    virtual void handleDestroyWindow(const XDestroyWindowEvent &destroyWindow);
    virtual void handleClientMessage(const XClientMessageEvent &message);
#ifdef SHAPE
    virtual void handleShapeNotify(const XShapeEvent &shape);
#endif

    Window clientWindow() { return fClient; }

    unsigned int border() { return fBorder; }
    void setBorder(unsigned int border) { fBorder = border; }
    void setFrame(YFrameWindow *newFrame);

    enum {
        wpDeleteWindow = 1 << 0,
        wpTakeFocus    = 1 << 1
    } WindowProtocols;

    void sendMessage(Atom msg, Time timeStamp = CurrentTime);

    enum {
        csKeepX = 1,
        csKeepY = 2,
        csRound = 4
    };
    
    void constrainSize(int &w, int &h, CARD32 layer, int flags = 0);
    void gravityOffsets (int &xp, int &yp);

    Colormap colormap() { return fColormap; }
    void setColormap(Colormap cmap);

    FrameState getFrameState();
    void setFrameState(FrameState state);

    void getWMHints();
    XWMHints *hints() { return fHints; }

    void getSizeHints();
    XSizeHints *sizeHints() { return fSizeHints; }

    unsigned long protocols() { return fProtocols; }
    void getProtocols();

    void getTransient();
    Window ownerWindow() { return fTransientFor; }

    void getClassHint();
    XClassHint *classHint() { return fClassHint; }

    void getNameHint();
    void getIconNameHint();
    void setWindowTitle(unsigned char *aWindowTitle);
    void setIconTitle(unsigned char *aIconTitle);
#ifdef I18N
    void setWindowTitle(XTextProperty *aWindowTitle);
    void setIconTitle(XTextProperty *aIconTitle);
#endif
    unsigned char *windowTitle() { return fWindowTitle; }
    unsigned char *iconTitle() { return fIconTitle; }

    bool getWinIcons(Atom *type, int *count, CARD32 **elem);

    void setWinWorkspaceHint(CARD32 workspace);
    bool getWinWorkspaceHint(CARD32 *workspace);

    void setWinLayerHint(CARD32 layer);
    bool getWinLayerHint(CARD32 *layer);

    void setWinStateHint(CARD32 mask, CARD32 state);
    bool getWinStateHint(CARD32 *mask, CARD32 *state);

#ifndef NO_MWM_HINTS
    MwmHints *mwmHints() { return fMwmHints; }
    void getMwmHints();
    CARD32 mwmFunctions();
    CARD32 mwmDecors();
#endif

#ifndef NO_KWM_HINTS
    bool getKwmIcon(int *count, Pixmap **pixmap);
#endif

#ifdef SHAPE
    int shaped() { return fShaped; }
    void queryShape();
#endif

private:
    int fProtocols;
    int haveButtonGrab;
    Window fClient;
    unsigned int fBorder;
    XSizeHints *fSizeHints;
    XClassHint *fClassHint;
    XWMHints *fHints;
    Colormap fColormap;
    int fShaped;

    unsigned char *fWindowTitle;
    unsigned char *fIconTitle;

    MwmHints *fMwmHints;

    Window fTransientFor;
    Pixmap *kwmIcons;
};

class YClientContainer: public YFrameControl {
public:
    YClientContainer(YWindow *parent, YFrameWindow *frame);
    //: YFrameControl(parent, frame) { isActive = 0; movingWindow = 0; }
    virtual ~YClientContainer();

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleConfigureRequest(const XConfigureRequestEvent &configureRequest);
    virtual void handleMapRequest(const XMapRequestEvent &mapRequest);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    void grabButtons();
    void releaseButtons();
    int haveGrab() { return haveButtonGrab; }
private:
    int haveButtonGrab;
};

class YFrameTitleBar: public YFrameControl {
public:
    YFrameTitleBar(YWindow *parent, YFrameWindow *frame): YFrameControl(parent, frame) { movingWindow = 0; }
    virtual ~YFrameTitleBar() { }

    void activate();
    void deactivate();
    
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);

    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);

private:
    int buttonDownX, buttonDownY;
    int movingWindow;
};

class YFrameButton: public YFrameControl {
public:
    YFrameButton(YWindow *parent, YFrameWindow *frame, WMCommand command, WMCommand command2 = cmdNone);
    virtual ~YFrameButton();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void activate();
    void deactivate();

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    void setCommand(WMCommand command, WMCommand command2 = cmdNone);

    virtual void donePopup(YPopupWindow *popup);
    void popupMenu();
    void popdownMenu();
        
private:
    WMCommand fCommand;
    WMCommand fCommand2;

    int isDown;
    int isHilited;
    int isActive;
    int wasPopupActive;
    YPopupWindow *fPopup;
};

class YFocusWindow: public YWindow {
public:
    YFocusWindow(YWindow *parent, Window win = 0): YWindow(parent, win) { }
    virtual ~YFocusWindow() { }
};

class YFrameWindow: public YFocusWindow, public CommandListener, public TimerListener, public PopDownListener {
public:
    YFrameWindow(YWindow *parent, YFrameClient *client);
    virtual ~YFrameWindow();

    void manage(YFrameClient *client);
    void unmanage();
    void sendConfigure();

    void createPointerWindows();
    void grabKeys();

    void focus(bool canWarp = false);
    void activate(bool canWarp = false);
    void deactivate();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);
    virtual void handleMotion(const XMotionEvent &motion);
    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleConfigure(const XConfigureEvent &configure);

    virtual bool handleTimer(YTimer *t);

    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers);
    
    void wmRestore();
    void wmMinimize();
    void wmMaximize();
    void wmMaximizeVert();
    void wmMaximizeHorz();
    void wmRollup();
    void wmHide();
    void wmShow();
    void wmLower();
    void doLower();
    void wmRaise();
    void doRaise();
    void wmClose();
    void wmKill();
    void wmNextWindow();
    void wmPrevWindow();
    void wmLastWindow();
    void wmMove();
    void wmSize();
    void wmOccupyAll();
    void wmOccupyAllOrCurrent();
    void wmOccupyWorkspace(CARD32 workspace);
    void wmOccupyOnlyWorkspace(CARD32 workspace);
    void wmMoveToWorkspace(CARD32 workspace);
    void wmSetLayer(CARD32 layer);

    void DoMaximize(CARD32 flags);

    void loseFocus();
    void setFocus(bool canWarp = false);
    bool focused() { return fFocused; }
    void focusOnMap();

    YFrameClient *client() { return fClient; }
    YFrameTitleBar *titlebar() { return fTitleBar; }
    YClientContainer *container() { return fClientContainer; }

    void startMoveSize(int doMove, int byMouse,
                       int sideX, int sideY,
                       int mouseXroot, int mouseYroot);
    void endMoveSize();
    void moveWindow(int newX, int newY);
    void manualPlace();

    void drawOutline(int x, int y, int w, int h);
    int handleMoveKeys(const XKeyEvent &xev, int &newX, int &newY);
    int handleResizeKeys(const XKeyEvent &key,
                         int &newX, int &newY, int &newWidth, int &newHeight,
                         int incX, int incY);
    void handleMoveMouse(const XMotionEvent &motion, int &newX, int &newY);
    void handleResizeMouse(const XMotionEvent &motion,
                           int &newX, int &newY, int &newWidth, int &newHeight);

    void outlineMove();
    void outlineResize();
    
    void constrainPositionByModifier(int &x, int &y, const XMotionEvent &motion);
    void constrainMouseToWorkspace(int &x, int &y);

    void getDefaultOptions();

    bool canSize(bool boriz = true, bool vert = true);
    bool canMove();
    bool canClose();
    bool canMaximize();
    bool canMinimize();
    bool canRollup();
    bool canHide();
    bool canLower();
    bool canRaise();

    void insertFrame();
    void removeFrame();
    void setAbove(YFrameWindow *aboveFrame); // 0 = at the bottom
    void setBelow(YFrameWindow *belowFrame); // 0 = at the top
    YFrameWindow *next() { return fNextFrame; }
    YFrameWindow *prev() { return fPrevFrame; }
    void setNext(YFrameWindow *next) { fNextFrame = next; }
    void setPrev(YFrameWindow *prev) { fPrevFrame = prev; }

    typedef enum {
        fwfVisible    = 1 << 0, // visible windows only
        fwfCycle      = 1 << 1, // cycle when bottom(top) reached
        fwfBackward   = 1 << 2, // go up in zorder (default=down)
        fwfNext       = 1 << 3, // start from next window
        fwfFocusable  = 1 << 4, // only focusable windows
        fwfWorkspace  = 1 << 5, // current workspace only
        fwfSame       = 1 << 6, // return same if no match and same matches
        fwfLayers     = 1 << 7, // find windows in other layers
        fwfSwitchable = 1 << 8, // window can be Alt+Tabbed
        fwfMinimized  = 1 << 9, // minimized/visible windows
        fwfUnminimized = 1 << 10 // normal/rolledup only
    } FindWindowFlags;

    YFrameWindow *findWindow(int flag);

    YFrameButton *menuButton() { return fMenuButton; }
    YFrameButton *closeButton() { return fCloseButton; }
    YFrameButton *minimizeButton() { return fMinimizeButton; }
    YFrameButton *maximizeButton() { return fMaximizeButton; }
    void updateMenu();

    virtual void raise();
    virtual void lower();

    void popupSystemMenu();
    void popupSystemMenu(int x, int y,
                         int x_delta, int y_delta,
                         unsigned int flags,
                         YWindow *forWindow = 0);
    virtual void handlePopDown(YPopupWindow *popup);

    virtual void configure(int x, int y, unsigned int width, unsigned int height);
    
    void configureClient(const XConfigureRequestEvent &configureRequest);

#ifdef SHAPE
    void setShape();
#endif

    enum {
        ffMove          = (1 << 0),
        ffResize        = (1 << 1),
        ffClose         = (1 << 2),
        ffMinimize      = (1 << 3),
        ffMaximize      = (1 << 4),
        ffHide          = (1 << 5),
        ffRollup        = (1 << 6)
    } YFrameFunctions;

    enum {
        fdTitleBar      = (1 << 0),
        fdSysMenu       = (1 << 1),
        fdBorder        = (1 << 2),
        fdResize        = (1 << 3),
        fdClose         = (1 << 4),
        fdMinimize      = (1 << 5),
        fdMaximize      = (1 << 6),
        fdHide          = (1 << 7)
    } YFrameDecors;

    enum YFrameOptions {
        foAllWorkspaces = (1 << 1),
        foIgnoreTaskBar = (1 << 2),
        foIgnoreWinList = (1 << 3),
        foFullKeys      = (1 << 4),
        foIgnoreQSwitch = (1 << 5),
        foNoFocusOnAppRaise = (1 << 6),
        foIgnoreNoFocusHint = (1 << 7)
    };

    unsigned long frameFunctions() { return fFrameFunctions; }
    unsigned long frameDecors() { return fFrameDecors; }
    unsigned long frameOptions() { return fFrameOptions; }
    void getFrameHints();
#ifndef NO_WINDOW_OPTIONS
    void getWindowOptions(WindowOption &opt, bool remove); /// !!! fix kludges
    void getWindowOptions(WindowOptions *list, WindowOption &opt, bool remove);
#endif

    YMenu *windowMenu();

    unsigned int borderX() {
        return
            (frameDecors() & fdBorder) ?
            ((frameDecors() & fdResize) ? wsBorderX : wsDlgBorderX) : 0;
    }
    unsigned int borderY() {
        return
            (frameDecors() & fdBorder) ?
            ((frameDecors() & fdResize) ? wsBorderX : wsDlgBorderX) : 0;
    }
    unsigned int titleY() {
        return (frameDecors() & fdTitleBar) ? wsTitleBar : 0;
    }
    
    void layoutTitleBar();
    void layoutButtons();
    void layoutResizeIndicators();
    void layoutClient();

    //void workspaceShow();
    //void workspaceHide();

    void addToMenu(YMenu *menu);

    YFrameWindow *nextLayer();
    YFrameWindow *prevLayer();
#ifdef CONFIG_WINLIST
    WindowListItem *winListItem() { return fWinListItem; }
    void setWinListItem(WindowListItem *i) { fWinListItem = i; }
#endif

    void addAsTransient();
    void removeAsTransient();
    void addTransients();
    void removeTransients();

    void setTransient(YFrameWindow *transient) { fTransient = transient; }
    void setNextTransient(YFrameWindow *nextTransient) { fNextTransient = nextTransient; }
    void setOwner(YFrameWindow *owner) { fOwner = owner; }
    YFrameWindow *transient() { return fTransient; }
    YFrameWindow *nextTransient() { return fNextTransient; }
    YFrameWindow *owner() { return fOwner; }

    YIcon *getClientIcon() { return fFrameIcon; }
    YIcon *clientIcon();

    void getNormalSize(int *x, int *y, int *w, int *h);
    void updateNormalSize();

    void updateTitle();
    void updateIconTitle();
    void updateIcon();
    void updateState();
    void updateWorkspace();
    void updateLayout();

    void updateMwmHints();
    void updateProperties();
#ifdef CONFIG_TASKBAR
    void updateTaskBar();
#endif

    CARD32 getWorkspace() { return fWinWorkspace; }
    void setWorkspace(CARD32 workspace);
    CARD32 getLayer() { return fWinLayer; }
    void setLayer(CARD32 layer);
    CARD32 getState() { return fWinState; }
    void setState(CARD32 mask, CARD32 state);

    bool isMaximized() { return (getState() &
                                 (WinStateMaximizedHoriz |
                                  WinStateMaximizedVert)) ? true : false; }
    bool isMaximizedVert() { return (getState() & WinStateMaximizedVert) ? true : false; }
    bool isMaximizedHoriz() { return (getState() & WinStateMaximizedHoriz) ? true : false; }
    bool isMinimized() { return (getState() & WinStateMinimized) ? true : false; }
    bool isHidden() { return (getState() & WinStateHidden) ? true : false; }
    bool isRollup() { return (getState() & WinStateRollup) ? true : false; }
    bool isSticky() { return (getState() & WinStateAllWorkspaces) ? true : false; }
    bool isHidWorkspace() { return (getState() & WinStateHidWorkspace) ? true : false; }
    bool isHidTransient() { return (getState() & WinStateHidTransient) ? true : false; }

    bool isIconic() { return isMinimized() && minimizeToDesktop && fMiniIcon; }

    MiniIcon *getMiniIcon() { return fMiniIcon; }

    void setSticky(bool sticky);

    int visibleOn(CARD32 workspace) {
        return (isSticky() || getWorkspace() == workspace);
    }
    int visibleNow() { return !isHidWorkspace(); }

    bool isModal();
    bool hasModal();
    bool isFocusable();

private:
    /*typedef enum {
        fsMinimized       = 1 << 0,
        fsMaximized       = 1 << 1,
        fsRolledup        = 1 << 2,
        fsHidden          = 1 << 3,
        fsWorkspaceHidden = 1 << 4
    } FrameStateFlags;*/

    bool fFocused;
    unsigned long fFrameFunctions;
    unsigned long fFrameDecors;
    unsigned long fFrameOptions;

    int normalX, normalY;
    unsigned int normalWidth, normalHeight;
    int iconX, iconY;

    YFrameClient *fClient;
    YClientContainer *fClientContainer;
    YFrameTitleBar *fTitleBar;
    YFrameButton *fCloseButton;
    YFrameButton *fMenuButton;
    YFrameButton *fMaximizeButton;
    YFrameButton *fMinimizeButton;

    YPopupWindow *fPopupActive;

    //CARD32 fStyle;
    //FrameState fSavedFrameState;

    int buttonDownX, buttonDownY;
    int grabX, grabY;
    int movingWindow, sizingWindow;
    int sizeByMouse;
    int origX, origY, origW, origH;

    YFrameWindow *fNextFrame; // window below this one
    YFrameWindow *fPrevFrame; // window above this one

    Window topSide, leftSide, rightSide, bottomSide;
    Window topLeftCorner, topRightCorner, bottomLeftCorner, bottomRightCorner;
    int indicatorsVisible;

#ifdef CONFIG_TASKBAR
    TaskBarApp *fTaskBarApp;
#endif
    MiniIcon *fMiniIcon;
#ifdef CONFIG_WINLIST
    WindowListItem *fWinListItem;
#endif
    YIcon *fFrameIcon;

    YFrameWindow *fOwner;
    YFrameWindow *fTransient;
    YFrameWindow *fNextTransient;

    static YTimer *fAutoRaiseTimer;
    static YTimer *fDelayFocusTimer;

    CARD32 fWinWorkspace;
    CARD32 fWinLayer;
    CARD32 fWinState;
    CARD32 fWinStateMask;
};

class YButton: public YWindow {
public:
    YButton(YWindow *parent, WMCommand command, void *context);
    YButton(YWindow *parent, YPopupWindow *popup, CommandListener *command);
    virtual ~YButton();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void paintFocus(Graphics &g, int x, int y, unsigned int w, unsigned int h);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    void setCommand(WMCommand command, void *context);
    void setPopup(YPopupWindow *popup, CommandListener *command);
    void setPixmap(YPixmap *pixmap);
    void setText(const char *str, int hot = -1);
    void setPressed(int pressed);

    virtual bool isFocusTraversable();

    virtual void donePopup(YPopupWindow *popup);

    void setListener(YButtonListener *listener) { fListener = listener; }

    void popupMenu();

    void *getContext() { return fContext; }
    WMCommand getCommand() { return fCommand; }

private:
    WMCommand fCommand;
    YPopupWindow *fPopup;
    YPixmap *fPixmap;
    char *fText;
    void *fContext;
    int fPressed;
    int fHotCharPos;
    int hotKey;

    int selected;
    int isActive;
    int wasPopupActive;

    YButtonListener *fListener;
    CommandListener *fCommandListener;

    void popup();
    void popdown();

    static YColor *normalButtonBg;
    static YColor *normalButtonFg;
    
    static YColor *activeButtonBg;
    static YColor *activeButtonFg;
    
    static YFont *normalButtonFont;
    static YFont *activeButtonFont;
};

class MiniIcon: public YFrameControl {
public:
    MiniIcon(YWindow *aParent, YFrameWindow *frame, Window win = 0);
    virtual ~MiniIcon();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);

private:
    int selected;
};

#ifdef CONFIG_CLOCK
class YClock: public YWindow, public TimerListener {
public:
    YClock(YWindow *aParent = 0, Window win = 0);
    virtual ~YClock();

    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual bool handleTimer(YTimer *t);

    YPixmap *getPixmap(char ch);
    int calcWidth(char *s, int count);
private:
    int selected;
    YTimer *clockTimer;

    static YColor *clockBg;
    static YColor *clockFg;
    static YFont *clockFont;
};
#endif

#ifdef CONFIG_TASKBAR
class TaskBarApp: public YWindow, public TimerListener {
public:
    TaskBarApp(YFrameWindow *frame, YWindow *aParent, Window win = 0);
    virtual ~TaskBarApp();

    virtual bool isFocusTraversable();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleDNDEnter();
    virtual void handleDNDLeave();
    virtual bool handleTimer(YTimer *t);

    YFrameWindow *frame() { return fFrame; }

    void setShown(bool show);
    bool getShown() { return fShown; }
    
    TaskBarApp *getNext() { return fNext; }
    TaskBarApp *getPrev() { return fPrev; }
    void setNext(TaskBarApp *next) { fNext = next; }
    void setPrev(TaskBarApp *prev) { fPrev = prev; }

private:
    YFrameWindow *fFrame;
    TaskBarApp *fPrev, *fNext;
    bool fShown;
    int selected;
    static YTimer *fRaiseTimer;
};

class WorkspaceButton: public YButton, public TimerListener, public YButtonListener {
public:
    WorkspaceButton(CARD32 workspace, YWindow *parent, WMCommand command, void *context);

    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);

    virtual void handleDNDEnter();
    virtual void handleDNDLeave();
    virtual bool handleTimer(YTimer *t);

    virtual void ButtonClick(YButton *button, unsigned int modifiers);
private:
    static YTimer *fRaiseTimer;
    CARD32 fWorkspace;
};

class TaskBar: public YWindow, public TimerListener {
public:
    TaskBar(YWindow *aParent, Window win = 0);
    virtual ~TaskBar();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);

    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual bool handleTimer(YTimer *t);

    void insert(TaskBarApp *tapp);
    void remove(TaskBarApp *tapp);

    TaskBarApp *addApp(YFrameWindow *frame);
    void removeApp(YFrameWindow *frame);

    void updateLocation();
    void relayout() { fNeedRelayout = true; }
    void relayoutNow();

    YClock *clock() { return fClock; }
    WorkspaceButton *workspaceButton(CARD32 n) { return fWorkspaceButton[n]; }

    void popupStartMenu();
    void popupWindowListMenu();
    TaskBarApp *getFirst() { return fFirst; }
    TaskBarApp *getLast() { return fLast; }
private:
    TaskBarApp *fFirst, *fLast;
    int fCount;
    YClock *fClock;
    MailBoxStatus *fMailBoxStatus;
    CPUStatus *fCPUStatus;
    ObjectBar *fObjectBar;
    YButton *fWinList;
    YButton *fApplications;
    WorkspaceButton **fWorkspaceButton;
    int leftX, rightX;
    bool fNeedRelayout;
    bool fIsHidden;
    YTimer *fAutoHideTimer;

    friend class WindowList;
    friend class WindowListBox;
};
#endif

#ifdef CONFIG_WINLIST
class YScrollListener {
public:
    virtual void scroll(YScrollBar *scroll, int delta) = 0;
    virtual void move(YScrollBar *scroll, int pos) = 0;
};

class YListItem {
public:
    YListItem();
    virtual ~YListItem();
    
    YListItem *getNext();
    YListItem *getPrev();
    void setNext(YListItem *next);
    void setPrev(YListItem *prev);
    
    int getSelected();
    void setSelected(int aSelected);

    virtual int getOffset();

    virtual unsigned char *getText();
    virtual YIcon *getIcon();
private:
    YListItem *fPrevItem, *fNextItem;
    int fSelected;
};

class WindowListItem: public YListItem {
public:
    WindowListItem(YFrameWindow *frame);
    virtual ~WindowListItem();

    virtual int getOffset();
    
    virtual unsigned char *getText();
    virtual YIcon *getIcon();
    YFrameWindow *frame();
private:
    YFrameWindow *fFrame;
};

class YListBox: public YWindow, public YScrollListener {
public:
    YListBox(YScrollBar *vert, YWindow *aParent, Window win = 0);
    virtual ~YListBox();

    int addItem(YListItem *item);
    int addAfter(YListItem *prev, YListItem *item);
    void removeItem(YListItem *item);
    YListItem *findItem(int mouseY, int &no);
    int itemCount();
    YListItem *item(int no);

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    //virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    //virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);
    virtual void handleMotion(const XMotionEvent &motion);

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void scroll(YScrollBar *sb, int delta);
    virtual void move(YScrollBar *sb, int pos);

    void selectItem(YListItem *i);
    int getLineHeight();
    void ensureVisibility(int item);
    void focusVisible();

    virtual bool isFocusTraversable();

    YListItem *getFirst() { return fFirst; }
    YListItem *getLast() { return fLast; }

    virtual void activateItem(YListItem *item);

    bool hasSelection();

private:
    YScrollBar *fVerticalScroll;
    int fOffsetY;
    YListItem *fFirst, *fLast;
    int fFocusedItem;
    int fSelectStart, fSelectEnd;
    int fSelecting;

    void setSelection();
    void updateSelection(int apply);
};

class WindowListBox: public YListBox, public CommandListener {
public:
    WindowListBox(YScrollBar *vert, YWindow *aParent, Window win = 0);
    virtual ~WindowListBox();

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    
    virtual void activateItem(YListItem *item);
    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers);
};

class YScrollBar: public YWindow {
public:
    enum Orientation {
        Vertical, Horizontal
    };

    YScrollBar(YWindow *aParent);
    YScrollBar(Orientation anOrientation, YWindow *aParent);
    YScrollBar(Orientation anOrientation,
               int aValue, int aVisibleAmount, int aMin, int aMax,
               YWindow *aParent);
    virtual ~YScrollBar();

    Orientation getOrientation() { return fOrientation; }
    int getMaximum() { return fMaximum; }
    int getMinimum() { return fMinimum; }
    int getVisibleAmount() { return fVisibleAmount; }
    int getUnitIncrement() { return fUnitIncrement; }
    int getBlockIncrement() { return fBlockIncrement; }
    int getValue() { return fValue; }

    void setOrientation(Orientation anOrientation);
    void setMaximum(int aMaximum);
    void setMinimum(int aMinimum);
    void setVisibleAmount(int aVisibleAmount);
    void setUnitIncrement(int anUnitIncrement);
    void setBlockIncrement(int aBlockIncrement);
    void setValue(int aValue);
    void setValues(int aValue, int aVisibleAmount, int aMin, int aMax);

    bool handleScrollKeys(const XKeyEvent &key);
    bool handleScrollMouse(const XButtonEvent &button);
    
private:
    Orientation fOrientation;
    int fMaximum;
    int fMinimum;
    int fValue;
    int fVisibleAmount;
    int fUnitIncrement;
    int fBlockIncrement;
public:
    void scroll(int delta);
    void move(int pos);

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    void setListener(YScrollListener *notify) { fListener = notify; }
private:
    enum {
        goUp, goDown, goPageUp, goPageDown, goPosition, goNone
    } fScrollTo;

    int fGrabDelta;
    YScrollListener *fListener;
};

class WindowList: public YFrameClient {
public:
    WindowList(YWindow *aParent, Window win = 0);
    virtual ~WindowList();

    void handleFocus(const XFocusChangeEvent &focus);
    virtual void handleClose();

    virtual void configure(int x, int y, unsigned int width, unsigned int height);
    void relayout();

    WindowListItem *addWindowListApp(YFrameWindow *frame);
    void removeWindowListApp(WindowListItem *item);

    void showFocused();

    WindowListBox *getList() { return list; }

private:
    WindowListBox *list;
    YScrollBar *scrollVert;
};
#endif

#ifdef CONFIG_MAILBOX
class MailBoxStatus: public YWindow, public TimerListener {
public:
    enum MailBoxState {
        mbxNoMail,
        mbxHasMail,
        mbxHasUnreadMail,
        mbxHasNewMail
    };
    
    MailBoxStatus(const char *mailBox, const char *mailCommand, YWindow *aParent = 0);
    virtual ~MailBoxStatus();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);

    void checkMail();
    void newMailArrived();
    void countMessages();
    
    virtual bool handleTimer(YTimer *t);
private:
    const char *fMailBox;
    const char *fMailCommand;
    time_t fLastCheckTime;
    unsigned long fLastSize;
    MailBoxState fState;
    YTimer *fMailboxCheckTimer;
};
#endif

class CPUStatus: public YWindow, public TimerListener {
public:
    CPUStatus(YWindow *aParent = 0);
    virtual ~CPUStatus();
    
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual bool handleTimer(YTimer *t);
    void updateStatus();
    void getStatus(int *cpuval);
private:
#define CPU_SAMPLES 20
    int cpu[CPU_SAMPLES][4];
    long last_cpu[4];
    YColor *color[4];
    YTimer *fUpdateTimer;
};

class CtrlAltDelete: public YWindow, public YButtonListener {
public:
    CtrlAltDelete(YWindow *parent);
    virtual ~CtrlAltDelete();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void ButtonClick(YButton *button, unsigned int modifiers);

    void activate();
    void deactivate();
private:
    YButton *lockButton;
    YButton *logoutButton;
    YButton *cancelButton;
};

class YTimer {
public:
    YTimer();
    ~YTimer();

    void setListener(TimerListener *listener) { fListener = listener; }
    TimerListener *getListener() { return fListener; }
    
    void setInterval(long ms) { fInterval = ms; }
    long getInterval() { return fInterval; }

    void startTimer();
    void stopTimer();
    bool isRunning() { return fRunning; }

private:
    TimerListener *fListener;
    long fInterval;
    bool fRunning;
    YTimer *fPrev;
    YTimer *fNext;

    struct timeval timeout;

    friend class YApplication;
};

class YApplication {
public:
    YApplication(int *argc, char ***argv, const char *displayName = 0);
    virtual ~YApplication();

    int mainLoop();
    void exitLoop(int exitCode);
    void exit(int exitCode);
    
    //YRootWindow *root() { return fRoot; }
    Display *display() { return fDisplay; }

    void saveEventTime(XEvent &xev);
    Time getEventTime() { return lastEventTime; }

    int grabEvents(YWindow *win, Cursor ptr, unsigned int eventMask, int grabMouse = 1, int grabKeyboard = 1, int grabTree = 0);
    int releaseEvents();
    void handleGrabEvent(YWindow *win, XEvent &xev);

    void captureGrabEvents(YWindow *win);
    void releaseGrabEvents(YWindow *win);

    void dispatchEvent(YWindow *win, XEvent &e);

    YPopupWindow *popup() { return fPopup; }
    bool popup(YWindow *forWindow, YPopupWindow *popup);
    void popdown(YPopupWindow *popdown);

    YWindow *grabWindow() { return fGrabWindow; }

    virtual void handleSignal(int sig);
    virtual void handleIdle();

    void blockSignal(int sig);
    void resetSignals();
    //void unblockSignal(int sig);

    void initModifiers();

    unsigned int AltMask;
    unsigned int MetaMask;
    unsigned int NumLockMask;
    unsigned int ScrollLockMask;

    unsigned int KeyMask;
    unsigned int ButtonMask;
    unsigned int ButtonKeyMask;
private:
    Display *fDisplay;
    Time lastEventTime;
    YPopupWindow *fPopup;

    int fGrabTree;
    YWindow *fXGrabWindow;
    int fGrabMouse;
    YWindow *fGrabWindow;

    YTimer *fFirstTimer, *fLastTimer;

    int fLoopLevel;
    int fExitLoop;
    int fExitCode;
    int fExitApp;

    sigset_t oldSignalMask;
    struct sigaction oldSignalCHLD;


    friend class YTimer;
    
    void registerTimer(YTimer *t);
    void unregisterTimer(YTimer *t);
    void getTimeout(struct timeval *timeout);
    void handleTimeouts();
};

class YWMApp: public YApplication, public CommandListener {
public:
    YWMApp(int *argc, char ***argv, const char *displayName = 0);
    ~YWMApp();
#ifdef CONFIG_GUIEVENTS
    void signalGuiEvent(GUIEvent ge);
#endif

    virtual void handleSignal(int sig);
    virtual void handleIdle();
    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers);

private:
    YWindowManager *fWindowManager;

};

extern Colormap defaultColormap;

extern GC blackGC;
extern GC whiteGC;
extern GC outlineGC;
extern GC clipPixmapGC;

extern YColor *black;
extern YColor *white;

extern YColor *dialogBg;

extern YColor *titleButtonBg;
extern YColor *titleButtonFg;

#ifdef CONFIG_WINLIST
extern YColor *scrollBarBg;
extern YColor *scrollBarArrow;
extern YColor *scrollBarSlider;

extern YColor *listBoxBg;
extern YColor *listBoxFg;
extern YColor *listBoxSelBg;
extern YColor *listBoxSelFg;
#endif

extern YColor *menuBg;
extern YColor *menuItemFg;
extern YColor *activeMenuItemBg;
extern YColor *activeMenuItemFg;
extern YColor *disabledMenuItemFg;

extern YColor *activeBorderBg;
extern YColor *inactiveBorderBg;

extern YColor *activeTitleBarBg;
extern YColor *activeTitleBarFg;

extern YColor *inactiveTitleBarBg;
extern YColor *inactiveTitleBarFg;

extern YColor *normalMinimizedWindowBg;
extern YColor *normalMinimizedWindowFg;

extern YColor *activeMinimizedWindowBg;
extern YColor *activeMinimizedWindowFg;

#ifdef CONFIG_TASKBAR
extern YColor *taskBarBg;

extern YColor *normalTaskBarAppFg;
extern YColor *normalTaskBarAppBg;

extern YColor *activeTaskBarAppFg;
extern YColor *activeTaskBarAppBg;

extern YColor *minimizedTaskBarAppFg;
extern YColor *minimizedTaskBarAppBg;
#endif

#ifdef CONFIG_LOOK_PIXMAP
extern YPixmap *frameTL[2][2];
extern YPixmap *frameT[2][2];
extern YPixmap *frameTR[2][2];
extern YPixmap *frameL[2][2];
extern YPixmap *frameR[2][2];
extern YPixmap *frameBL[2][2];
extern YPixmap *frameB[2][2];
extern YPixmap *frameBR[2][2];

extern YPixmap *titleL[2];
extern YPixmap *titleT[2];
extern YPixmap *titleM[2];
extern YPixmap *titleB[2];
extern YPixmap *titleR[2];

extern YPixmap *menuButton[2];
#endif

extern YPixmap *closePixmap[2];
extern YPixmap *minimizePixmap[2];
extern YPixmap *maximizePixmap[2];
extern YPixmap *restorePixmap[2];

extern YPixmap *startPixmap;
extern YPixmap *windowsPixmap;
#ifdef CONFIG_MAILBOX
extern YPixmap *mailPixmap;
extern YPixmap *unreadMailPixmap;
extern YPixmap *newMailPixmap;
#endif

#ifdef CONFIG_CLOCK
extern YPixmap *PixNum[10];
extern YPixmap *PixSpace;
extern YPixmap *PixColon;
extern YPixmap *PixSlash;
extern YPixmap *PixA;
extern YPixmap *PixP;
extern YPixmap *PixM;
extern YPixmap *PixDot;
#endif

/* small pixmaps (16x16) */
extern YPixmap pixWinRestore;
extern YPixmap pixWinMove;
extern YPixmap pixWinSize;
extern YPixmap pixWinMinimize;
extern YPixmap pixWinMaximize;
extern YPixmap pixWinRaise;
extern YPixmap pixWinLower;
extern YPixmap pixWinRollup;
extern YPixmap pixWinHide;
extern YPixmap pixWinMoveToWorkspace;
extern YPixmap pixWinOccupy;
extern YPixmap pixWinClose;
extern YPixmap pixWinList;

extern YFont *titleFont;
extern YFont *menuFont;

#ifdef CONFIG_TASKBAR
extern YFont *normalTaskBarFont;
extern YFont *activeTaskBarFont;
#endif
#ifdef CONFIG_WINLIST
extern YFont *minimizedWindowFont;
extern YFont *windowListFont;
#endif

extern Cursor leftPointer;
extern Cursor rightPointer;

extern Cursor movePointer;

extern Cursor sizeRightPointer;
extern Cursor sizeTopRightPointer;
extern Cursor sizeTopPointer;
extern Cursor sizeTopLeftPointer;
extern Cursor sizeLeftPointer;
extern Cursor sizeBottomLeftPointer;
extern Cursor sizeBottomPointer;
extern Cursor sizeBottomRightPointer;

#ifdef SHAPE
extern int shapesSupported;
extern int shapeEventBase, shapeErrorBase;
#endif

extern XContext windowContext;
extern XContext frameContext;
extern XContext clientContext;

extern YMenu *windowMenu;
extern YMenu *occupyMenu;
extern YMenu *layerMenu;
extern YMenu *moveMenu;
extern YMenu *windowListMenu;
extern YMenu *windowListPopup;
extern YMenu *maximizeMenu;
extern ObjectMenu *rootMenu;
extern MoveSizeStatus *statusMoveSize;
extern SwitchWindow *switchWindow;
#ifdef CONFIG_WINLIST
extern WindowList *windowList;
#endif
extern CtrlAltDelete *ctrlAltDelete;

extern Atom _XA_WM_PROTOCOLS;
extern Atom _XA_WM_TAKE_FOCUS;
extern Atom _XA_WM_DELETE_WINDOW;
extern Atom _XA_WM_STATE;
extern Atom _XA_WM_CHANGE_STATE;
extern Atom _XATOM_MWM_HINTS;
extern Atom _XA_WM_COLORMAP_WINDOWS;

extern Atom _XA_WIN_PROTOCOLS;
extern Atom _XA_WIN_WORKSPACE;
extern Atom _XA_WIN_WORKSPACE_COUNT;
extern Atom _XA_WIN_WORKSPACE_NAMES;
extern Atom __XA_WIN_WORKSPACE_NAMES;
extern Atom _XA_WIN_WORKAREA;
extern Atom _XA_WIN_LAYER;
extern Atom _XA_WIN_ICONS;
extern Atom _XA_WIN_HINTS;
extern Atom _XA_WIN_STATE;

/* KDE specific */
extern Atom _XA_KWM_WIN_ICON;

/* Xdnd */
extern Atom XA_XdndAware;
extern Atom XA_XdndEnter;
extern Atom XA_XdndLeave;
extern Atom XA_XdndPosition;
extern Atom XA_XdndStatus;
extern Atom XA_XdndDrop;
extern Atom XA_XdndFinished;

extern Atom XA_IcewmWinOptHint;

extern const char *clrBlack;
extern const char *clrWhite;

extern int configurationLoaded;
extern PhaseType phase;

extern WindowOptions *defOptions;
extern WindowOptions *hintOptions;

void getOptions(int &argc, char **argv);
void loadConfiguration(const char *fileName);
void loadMenus(const char *fileName, ObjectContainer *container);
#ifndef NO_WINDOW_OPTIONS
void loadWinOptions(const char *optFile);
#endif
int addProgram(const char *name, const char *command, char **args = 0);
void addWorkspace(const char *name);
int findPath(const char *path, int mode, const char *name, char **fullname);
char *findProgFile(const char *name);
char *getArgument(char *dest, int maxLen, char *p);

YIcon *getIcon(const char *name);
void freeIcons();

void freeConfig();
void freePrograms();

void runProgram(const char *str, char **args);
void runCommand(const char *prog);
void runRestart(char *str, char **args);

typedef struct {
    const char **root;
    const char *rdir;
    const char **sub;
}  pathelem;

extern pathelem icon_paths[10];

void joinPath(char *path, pathelem *pe, const char *base, const char *name);
void verifyPaths(pathelem *search, const char *base);

void findThemes(const char *path, YMenu *container);

int is_reg(const char *path);

void *MALLOC(size_t len);
void *REALLOC(void *p, size_t new_len);
void FREE(void *p);

#endif
