/*
  Copyright: GPL. Author: Joost witteveen 
                 (joostje@debian.org, joosteto@esperanto.nu)
*/
/* #define _POSIX_C_SOURCE 199309L whatever that may mean...*/ 
/* #define _BSD_SOURCE             I use strdup, S_IFDIR, etc */ 
/* #define _GNU_SOURCE */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <dlfcn.h>
#include <unistd.h> 
#include <dirent.h>
#include <errno.h>

#include "vsearch.h"
#include "libtricks.h"
#include "libfake.h"

/* 
   where are those shared libraries? 
   If I knew of a configure/libtool way to find that out, I'd use it. Or
   any other way other than the method I'm using below. Does anybody know
   how I can get that location? (BTW, symply linking a programme, and running
   `ldd' on it isn't the option, as Digital Unix doesn't have an ldd)
*/


/* 
   Note that LIBCPATH isn't actually used on Linux, as RTLD_NEXT
   is defined and we use that to get the `next_*' functions

   Linux:
*/
#ifdef __alpha__
#define LIBCPATH "/lib/libc.so.6.1"
#else
#define LIBCPATH "/lib/libc.so.6"
#endif

/* OSF1 :*/
/*#define LIBCPATH "/usr/shlib/libc.so"*/


struct next_wrap_st{
  void **doit;
  char *name;
};

void *get_libc(){
 
#ifndef RTLD_NEXT
 void *lib=0;
 if(!lib){ 
   lib= dlopen(LIBCPATH,RTLD_LAZY);
 }
 return lib;
#else
  return RTLD_NEXT;
#endif
}
void load_library_symbols(void);

#include "wrapped.h"
#include "wraptmpf.h"
#include "wrapdef.h"
#include "wrapstruct.h"


void load_library_symbols(void){
  /* this function loads all original functions from the C library.
     I ran into problems when let each function individually
     loaded it's original counterpart, as RTLD_NEXT seems to have
     a different meaning in files with different names than libtricks.c
     (I.E, dlsym(RTLD_NEXT, ...) called in vsearch.c returned funtions
     defined in libtricks */
  /* The calling of this function itself is somewhat tricky:
     the awk script wrapawk generates several .h files. In wraptmpf.h
     there are temporary definitions for tmp_*, that do the call
     to this function. The other generated .h files do even more tricky
     things :) */

  static int done=0;
  int i;
  
  if(!done){
    for(i=0; next_wrap[i].doit; i++)
      *(next_wrap[i].doit)=dlsym(get_libc(), next_wrap[i].name);
  }
}

const char *env_var_set(const char *env){
  const char *s;
  
  s=getenv(env);
  
  if(s && *s)
    return s;
  else
    return NULL;
}

key_t get_ipc_key(){
  const char *s;
  if((s=env_var_set(FAKEROOTKEY_ENV)))
    return atoi(s);
  else
    return 0;
}


/* this is for communication with the faked */

int init_get_msg(){
  /* a msgget call generates a fstat() call. As fstat() is wrapped,
     that call will in turn call semaphore_up(). So, before 
     the semaphores are setup, we should make sure we already have
     the msg_get and msg_set id.
     This is why semaphore_up() calls this function.*/
  static int done=0;
  key_t key;

  if(!done){
    done=1;
    key=get_ipc_key();
    
    if(key){
      // if(msg_snd==-1){
      msg_snd=msgget(get_ipc_key(),IPC_CREAT|0600);
      //}
      //if(msg_get==-1){
      msg_get=msgget(get_ipc_key()+1,IPC_CREAT|0600);
      //}
    }
    else{
      msg_get=-1;
      msg_snd=-1;
    }
  }
  return msg_snd;
}

int fake_root_env(){
  return init_get_msg()!=-1;
}

/* a few functions that probe environment variables once, and remember
   the results
*/

static uid_t faked_uid(){
  static int inited=0;
  static int uid;
  const char *s; 

  if(!inited){
    if((s=env_var_set(FAKEROOTUID_ENV)))
      uid=atoi(s);
    else
      uid=0;
    inited=1;
  }
  return uid;    
}
static uid_t faked_gid(){
  static int inited=0;
  static int gid;
  const char *s;

  if(!inited){
    if((s=env_var_set(FAKEROOTGID_ENV)))
      gid=atoi(s);
    else
      gid=0;
    inited=1;
  }
  return gid;    
}

static int dont_try_chown(){
  static int inited=0;
  static int donttry;

  if(!inited){
    donttry=(env_var_set(FAKEROOTDONTTRYCHOWN_ENV)!=NULL);
    inited=1;
  }
  return donttry;
}


static int do_show_symlinks(){
  static int inited=0;
  static int doshow;

  if(!inited){
    doshow=(env_var_set(VPATHSYMLINK)!=NULL);
    inited=1;
  }
  return doshow;
}
/* The wrapped functions */


int wrapped_open(int opts, const char *pathname, int flags, mode_t mode,
		 int *vpath_errno){
  char buf[PATH_MAX];
  const char *s;
  mode_t need;
  struct stat st;

  if(opts & DONTVPATH){
    s=pathname;
    return next_open(s,flags,mode);    
  }
  else{
    need=(flags & 3) +1;
    if(flags & O_EXCL)
      need|=NEED_EXCL;
    
    if(vpath_findpath(buf,pathname,need, &st, vpath_errno)){
      s=buf;
    }else
      s=pathname;
    if (*vpath_errno == -1)
      return next_open(s,flags,mode);
    else
      return -1;
  }
}

int open(const char *pathname, int flags, ...){
  va_list ap;
  mode_t mode;
  int vpath_errno;
  int r;
  va_start(ap, flags);
  mode=va_arg(ap, mode_t);
  r=wrapped_open(0,pathname,flags,mode, &vpath_errno);
  if(vpath_errno != -1){
    errno=vpath_errno;
    return -1;
  }else
    return r;
}

mode_t char_mode(const char *mode){
  /* return an int describing the permission of the
     mode described in the string mode. */
  mode_t m;

  if (mode[1]=='+')
    m=NEED_READ|NEED_WRITE;
  else {
    switch (mode[0]){
    case 'r': m=NEED_READ; break;
    case 'w': m=NEED_WRITE; break;
    case 'a': m=NEED_WRITE; break;
    default:m=0;
    }
  }
  return m;
}

FILE *fopen(const char *pathname, const char *mode){
  char buf[PATH_MAX];
  int vpath_errno=0;
  int need;
  struct stat st;
  const char *s;

  need=char_mode(mode);
  if(vpath_findpath(buf,pathname,need, &st, &vpath_errno)){
    s=buf;
  } else {
    s=pathname;
  }
  if(vpath_errno != -1){
    errno=vpath_errno;
    return NULL;
  }else
    return next_fopen(pathname,mode);
  
}

FILE *freopen(const char *pathname, const char *mode, FILE *stream){
  char buf[PATH_MAX];
  int vpath_errno=0;
  int need;
  struct stat st;
  const char *s;

  need=char_mode(mode);
  if(vpath_findpath(buf,pathname,need, &st, &vpath_errno)){
    s=buf;
  } else {
    s=pathname;
  }
  if(vpath_errno != -1){
    errno=vpath_errno;
    return NULL;
  }else
    return next_freopen(pathname,mode,stream);
  
}


int readlink(const char *path, READLINK_BUF_TYPE *buf, READLINK_BUFSIZE_TYPE bufsize){
  char b[PATH_MAX];
  struct stat st;
  int vpath_errno;

  if(vpath_findpath(b,path,NEED_STAT, &st, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    if(strlen(b)<bufsize){
      strcpy(buf, b);
      return strlen(buf);
    }
    else{
      strncpy(buf, b, bufsize);
      return bufsize;
    }
  } else
  return next_readlink(path, buf, bufsize);
}

int wrapped_lxstat(int opts, ARG_IFSTATVER(int ver) 
		   const char *file_name, struct stat *st,
		   int *vpath_errno){

  int r=0;
  char b[PATH_MAX];
  short makelink=0;
  
  if(opts & DONTVPATH){
    r=next___lxstat(ARG_IFSTATVER(ver) file_name, st);
  } else {
    if(vpath_findpath(b,file_name, NEED_STAT, st, vpath_errno)){
      if(*vpath_errno!=-1)
	return -1;
      makelink=do_show_symlinks();
    } else
      r=next___lxstat(ARG_IFSTATVER(ver) file_name, st);
  }
  if(r)
    return r;

  if(!(opts & DONTFAKE))
    send_get_stat(st);

  if(makelink)
    st->st_mode=(st->st_mode|S_IFLNK)&(~S_IFDIR);

  return 0; 
}

int
#ifdef _STAT_VER
 __lxstat
#else
  lstat
#endif
       (ARG_IFSTATVER(int ver) const char *file_name, 
	     struct stat *statbuf){

  int vpath_errno;
  int r;
  r=wrapped_lxstat(0, ARG_IFSTATVER(ver) file_name,statbuf, &vpath_errno);
  if(vpath_errno!=-1){
    errno=vpath_errno;
    return -1;
  }
  return r;
}


int wrapped_xstat(int opts, ARG_IFSTATVER(int ver)
		  const char *file_name, struct stat *st,
		  int *vpath_errno){
  int r=0;
  char b[PATH_MAX];
  short makelink=0;

  if(opts & DONTVPATH){
    r=next___xstat(ARG_IFSTATVER(ver) file_name,st);
  } else {
    if(vpath_findpath(b,file_name, NEED_STAT, st, vpath_errno)){
      if(*vpath_errno!=-1){
	return -1;
      }
      makelink=do_show_symlinks();
    } else
      r=next___xstat(ARG_IFSTATVER(ver) file_name,st);
  }
  if(r)
    return r;

  if(!(opts & DONTFAKE))
    send_get_stat(st);

  if(makelink)
    st->st_mode=(st->st_mode|S_IFLNK)&(~S_IFDIR);

  return 0; 
}


int
#ifdef _STAT_VER
 __xstat
#else
 stat
#endif
        (ARG_IFSTATVER(int ver)  const char *file_name, struct stat *st){
  int vpath_errno;
  int r;

  r=wrapped_xstat(0, ARG_IFSTATVER(ver) file_name,st, &vpath_errno);
  if(vpath_errno!=-1){
    errno=vpath_errno;
    return -1;
  }
  return r;
}

int wrapped_fxstat(int opts, ARG_IFSTATVER(int ver)
		   int fd, struct stat *st){

  int r;

  r=next___fxstat(ARG_IFSTATVER(ver) fd,st);
  if(r)
    return r;

  if(opts & DONTFAKE)
    send_get_stat(st);

  return 0; 
}

int 
#ifdef _STAT_VER
  __fxstat
#else
  fstat
#endif 
      (ARG_IFSTATVER(int ver) int fd, struct stat *st){

  return wrapped_fxstat(0, ARG_IFSTATVER(ver) fd,st);
}

/*************************************************************/
/*
  Wrapped functions general info:
  
  In general, the structure is as follows:
    - If one of the arguments to the function is a path,
      first pass that path to vpath_findpath() to check if
      it matches a component of the VPATH environment variable.
      If it does, use the path supplied by vpath_findpath().
    - Then, if the function does things that (possibly) fail by
      other users than root, allow for `fake' root privileges.
      Do this by obtaining the inode the function works on, and then
      informing faked (the deamon that remembers all `fake' file
      permissions e.d.) about the intentions of the user.
      Faked maintains a list of inodes and their respective
      `fake' ownerships/filemodes.
    - Or, if the function requests information that we should
      fake, again get the inode of the file, and ask faked about the
      ownership/filemode data it maintains for that inode.

*/
/*************************************************************/



/*  
    opendir, closedir, readdir.
    
    The vpath_opendir called by opendir() creates it's own DIR
    data. If directory `name' isn't subject to VPATH, the DIR
    returned by VPATH is just a pointer to the real DIR structure.
    If directory `name' is mangled by VPATH, then vpath_opendir()
    will read the complete `directory contents' (may be spread out
    over several directories), and put it in the DIR structure.
    
    The vpath_readdir() and vpath_opendir() functions will check
    whether the DIR argument points to a `real' DIR pointer, or to
    the VPATH created data, and act appropriately.
 */

DIR *opendir(const char *name){
  int vpath_errno;
  DIR *r;
  r= (DIR *) vpath_opendir(name, &vpath_errno);
  if(vpath_errno !=-1){
    errno=vpath_errno;
    return NULL;
  }
  return r;
}

int closedir(DIR *dir){

  return vpath_closedir(dir);
}

struct dirent *readdir(DIR *dir){
  
  return vpath_readdir(dir);
}

/* chown, lchown, fchown, chmod, fchmod, mknod functions
   
   quite general. See the `Wrapped functions general info:' above
   for more info.
 */

int chown(const char *path, uid_t owner, gid_t group){
  struct stat st;
  int r=0;
  char b[PATH_MAX];
  int vpath_errno;

  /* unfortunately, the old linux kernels (2.1.81 and older) didn't
     have a lchown syscall, and therefore chown() did what POSIX
     lchown() should do. This changed, but for compatibility reasons,
     on i386 chown still does for now (libc2.0 at least). So, on
     alpha we need to get the inode with xstat, while on i386
     we use lxstat */

  if(vpath_findpath(b, path, NEED_CHFL, &st, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    path=b;
  }
  else if (fake_root_env())
#ifdef __alpha__  /* see above */
    r=wrapped_xstat(DONTFAKE|DONTVPATH, 
		     ARG_IFSTATVER(_STAT_VER) 
		     path, &st, NULL);
#else
    r=wrapped_lxstat(DONTFAKE|DONTVPATH, 
		     ARG_IFSTATVER(_STAT_VER) 
		     path, &st, NULL);
#endif
  if (!fake_root_env())
    return next_chown(path,owner,group);
  else {
    if(r)
      return r;
    st.st_uid=owner;
    st.st_gid=group;
    send_stat(&st,chown_func);
    if(!dont_try_chown())
      r=next_chown(path,owner,group);
    
    if(r&&(errno==EPERM))
      r=0;
  }

  return r;
}


int lchown(const char *path, uid_t owner, gid_t group){
  struct stat st;
  int r=0;
  char b[PATH_MAX];
  int vpath_errno;

  if(vpath_findpath(b, path, NEED_CHFL, &st, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    path=b;
  }
  else if (fake_root_env())
    r=wrapped_lxstat(DONTFAKE|DONTVPATH, 
		     ARG_IFSTATVER(_STAT_VER) 
		     path, &st, NULL);
  if (!fake_root_env())
    return next_chown(path,owner,group);
  else {
    if(r)
      return r;
    st.st_uid=owner;
    st.st_gid=group;
    send_stat(&st,chown_func);
    if(!dont_try_chown())
      r=next_lchown(path,owner,group);
    
    if(r&&(errno==EPERM))
      r=0;
  }

  return r;
}


int fchown(int fd, uid_t owner, gid_t group){
  struct stat st;
  int r;

  if(!fake_root_env())
    return next_fchown(fd,owner,group);
  else {
    r=wrapped_fxstat(DONTFAKE, ARG_IFSTATVER(_STAT_VER) fd, &st);
    if(r)
      return r;

    st.st_uid=owner;
    st.st_gid=group;
    send_stat(&st, chown_func);
    
    if(!dont_try_chown())
      r=next_fchown(fd,owner,group);
    
    if(r&&(errno==EPERM))
      r=0;
    
    return r;
  }
}

int chmod(const char *path, mode_t mode){
  struct stat st;
  int r;
  char b[PATH_MAX];
  int vpath_errno;

  /*chmod(sym-link) works on the file pointed to, use stat, not lstat: */
  /*as vpath_findpath() does lstat, we shouldn't use the struct stat it*/
  /* gives us */
  if(vpath_findpath(b, path, NEED_CHFL, &st, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    path=b;
  }
  if (!fake_root_env())
    return next_chmod(path, mode);
  else {
    r=wrapped_xstat(DONTFAKE, ARG_IFSTATVER(_STAT_VER) path, &st, &vpath_errno);
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    
    if(r)
      return r;
    
    st.st_mode=(mode&ALLPERMS)|(st.st_mode&~ALLPERMS);
    
    send_stat(&st, chmod_func);
    
    /* if a file is unwritable, then root can still write to it
       (no matter who owns the file). If we are fakeroot, the only
       way to fake this is to always make the file writable, readable
       etc for the real user (who started fakeroot). Also holds for
       the exec bit of directories
    */
    mode |= 0600;
    if(S_ISDIR(st.st_mode))
      mode |= 0100;
    
    r=next_chmod(path, mode);
    if(r&&(errno==EPERM))
      r=0;
    return r;
  }
}

int fchmod(int fd, mode_t mode){
  int r;
  struct stat st;


  if(!fake_root_env())
    return next_fchmod(fd, mode);
  else {
    r=wrapped_fxstat(DONTFAKE, ARG_IFSTATVER(_STAT_VER) fd, &st);
    
    if(r)
      return(r);
    
    st.st_mode=(mode&ALLPERMS)|(st.st_mode&~ALLPERMS);
    send_stat(&st,chmod_func);  
    
    /* see chmod() for comment */
    mode |= 0600;
    if(S_ISDIR(st.st_mode))
      mode |= 0100;
    
    r=next_fchmod(fd, mode);
    if(r&&(errno==EPERM))
      r=0;
    return r;
  }
}

int __xmknod(ARG_IFSTATVER(int ver) const char *pathname, 
             mode_t mode, dev_t *dev){
  struct stat st;
  int old_mask=umask(022);
  int fd,r;
  char b[PATH_MAX];
  int vpath_errno;

  umask(old_mask);
  
  if(vpath_findpath(b, pathname, NEED_WRITE, &st, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    pathname=b;
  }
  if (!fake_root_env())
    return next___xmknod(ARG_IFSTATVER(ver) pathname, mode, dev);
  else {
    /*Don't bother to mknod the file, that probably doesn't work.
      just create it as normal file, and leave the premissions
      to the fakemode.*/
    
    fd=wrapped_open(DONTFAKE, pathname, O_WRONLY|O_CREAT|O_TRUNC, 
		    0644, &vpath_errno);
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    if(fd>=0){
      r=0;
      close(fd);
    }else
      return -1;
    
    r=wrapped_lxstat(DONTFAKE, ARG_IFSTATVER(_STAT_VER) pathname, 
		     &st, &vpath_errno);
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    
    st.st_mode= mode & ~old_mask;
    st.st_dev= *dev;
    
    send_stat(&st,mknod_func);
    
    return r;
  }
}

/* See the `Wrapped functions general info:' above for info on the
   general structure of these wrapped functions
*/

int mkdir(const char *path, mode_t mode){
  struct stat st;
  int r;
  char b[PATH_MAX];
  int vpath_errno;

  if(vpath_findpath(b, path, NEED_WRITE, &st, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    path=b;
  }

  /* we need to tell the fake deamon the real mode. In order
     to communicate with faked we need an struct stat, so we now
     do a stat of the new directory (just for the inode/dev) */

  if (fake_root_env()){

    r=next_mkdir(path, mode|0700); 
                    /* mode|0700: see `remove functions' above */
    if(r)
      return r;
    r=wrapped_xstat(DONTFAKE, ARG_IFSTATVER(_STAT_VER) path, 
		    &st, &vpath_errno);
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    
    if(r)
      return -1;
    
    st.st_mode=(mode&ALLPERMS)|(st.st_mode&~ALLPERMS)|S_IFDIR;
    send_stat(&st, chmod_func);
  } else {
    r=next_mkdir(path, mode);
    if(r)
      return r;
  }
  return r;
}


/* 
   The remove funtions: unlink, rmdir, rename.
   These functions can all remove inodes from the system.
   I need to inform faked about the removal of these inodes because
   of the following:
    # rm -f file
    # touch file 
    # chown admin file
    # rm file
    # touch file
   In the above example, assuming that for both touch-es, the same
   inode is generated, faked will still report the owner of `file'
   as `admin', unless it's informed about the removal of the inode.
   
   See the `Wrapped functions general info:' above for info on the
   general structure of these wrapped functions
*/

int unlink(const char *pathname){
  int r,s;
  struct stat st;
  char b[PATH_MAX];
  int vpath_errno;

  if((s=vpath_findpath(b, pathname, NEED_UNLI, &st, &vpath_errno))){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    pathname=b;
  } else if (fake_root_env())
    s=wrapped_lxstat(DONTFAKE|DONTVPATH, 
		     ARG_IFSTATVER(_STAT_VER) pathname, &st, NULL);
  
  r=next_unlink(pathname);  

  if(fake_root_env()&&(r==0)&&(s==0))
    send_stat(&st, unlink_func);
  
  return r;
}

/*
  See the `remove funtions:' comments above for more info on
  these remove function wrappers.
*/
int rmdir(const char *pathname){
  int r,s;
  struct stat st;
  char b[PATH_MAX];
  int vpath_errno;

  if((s=vpath_findpath(b, pathname, NEED_UNLI, &st, &vpath_errno))){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    pathname=b;
  }
  else if (fake_root_env())
    s=wrapped_lxstat(DONTFAKE|DONTVPATH, 
		     ARG_IFSTATVER(_STAT_VER) 
		     pathname, &st, NULL);

  r=next_rmdir(pathname);  

  if(fake_root_env()&&(r==0)&&(s==0))
    send_stat(&st,unlink_func);  

  return r;
}

/*
  See the `remove funtions:' comments above for more info on
  these remove function wrappers.
*/
int remove(const char *pathname){
  int r, s;
  struct stat st;
  char b[PATH_MAX];
  int vpath_errno;

  if((s=vpath_findpath(b, pathname, NEED_UNLI, &st, &vpath_errno))){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    pathname=b;
  } else if (fake_root_env())
    s=wrapped_lxstat(DONTFAKE|DONTVPATH, 
		     ARG_IFSTATVER(_STAT_VER) 
		     pathname, &st, NULL);

  
  r=next_remove(pathname);  
  if(fake_root_env()&&(r==0)&&(s==0))
    send_stat(&st,unlink_func);  
  
  return r;
}

/*
  if the second argument to the rename() call points to an
  existing file, then that file will be removed. So, we have
  to treat this function as one of the general `remove functions'.

  See the `remove funtions:' comments above for more info on
  these remove function wrappers.
*/

int rename(const char *oldpath, const char *newpath){
  int r,s;
  /* actually, I could do with just one struct stat here, and do the
     stat(oldpath) before the stat(newpath). But exchanging the two
     will give very irreproducable, and hard to find bugs, so I'll play
     safe here, and I'll declare two struct stats */
  struct stat st_old;
  struct stat st_new;     
  char b_old[PATH_MAX];
  char b_new[PATH_MAX];
  int vpath_errno;

  /* If newpath points to an existing file, that file will be 
     unlinked.   Make sure we tell the faked daemon about this! */

  if(vpath_findpath(b_old, oldpath, NEED_CHFL, &st_old, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    oldpath=b_old;
  }
  if((s=vpath_findpath(b_new, newpath, NEED_UNLI, &st_new, &vpath_errno))){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    newpath=b_new;
  }
  else if (fake_root_env())
    /* we need the st_new struct in order to inform faked about the
       (possible) unlink of the file */
    s=wrapped_lxstat(DONTFAKE|DONTVPATH, 
		     ARG_IFSTATVER(_STAT_VER) 
		     newpath, &st_new, NULL);
  
  r=next_rename(oldpath, newpath);

  if(fake_root_env()&&(r==0)&&(s==0))
    send_stat(&st_new,unlink_func);

  return r;
}

/* at least in GNU libc 2.0 we don't need to wrap getwd or
   get_current_working_dir_name, as those functions use getcwd 
*/
char *getcwd(char *buf, size_t size){
  return vpath_getcwd(buf,size);
}

/* Another GNU libc 2.0 -ism: all exec* functions eventually use __execve,
   so just wrap that one */
int __execve(const char *path, char *const argv[],
	     char *const envp[]){
  struct stat st;
  char b[PATH_MAX];
  int vpath_errno;

  if(vpath_findpath(b, path, NEED_EXEC, &st, &vpath_errno)){
    if(vpath_errno!=-1){
      errno=vpath_errno;
      return -1;
    }
    path=b;
  }
  return next___execve(path,argv,envp);
}

int execve(const char *path, char *const argv[],
	     char *const envp[]){
  return __execve(path,argv,envp);
}

uid_t getuid(void){
  if(fake_root_env())
    return faked_uid();
  else 
    return next_getuid();
}

uid_t geteuid(void){
  if(fake_root_env())
    return faked_uid();
  else 
    return next_geteuid();
}

uid_t getgid(void){
  if(fake_root_env())
    return faked_gid();
  else 
    return next_getgid();
}

uid_t getegid(void){
  if(fake_root_env())
    return faked_gid();
  else
    return next_geteuid();
}

