/******************************************************************************
*   WMSound	WindowMaker Sound Server	                              *
*   Copyright (C) 1998  Anthony Quinn <southgat@frontiernet.net               *
*                       William Moore <billy@mud.mind.net>		      *
*									      *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.					      *
*									      *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
*******************************************************************************/
#include "wmsound.h"

proplist_t	WMSound;		        // Options PropList
SPrefrences	sPrefrences;			// Options Structure
const char *	ProgName;			// Program Name
const char *	LocalDomain;                    // Local Domain
Display		*dpy;				// X Windows Display
Window		leader;				// Main Application Window
Atom		_XA_WINDOWMAKER_WM_FUNCTION; 	// WindowMaker Funciton Id
Atom		DeleteWindow;			// Window Delete Function Id
long		SoundIndex[WMSound_MaxSound];   // Index of Sound Events
long		TimeStamp;			// Time file was last changed

int sLoadConfig();

int main (int argc, char **argv ) {
  int error;
	
  ProgName = PACKAGE;
	
  // Setting LocalDomain, as a const global because we
  // will be refering to it pretty often, so we save the
  // overhead of calling defaultsPathForDomain( ) every
  // time we want it;
  LocalDomain = defaultsPathForDomain(CFG_DOMAIN,YES);
  if (sLoadConfig( ) < 0) {
    sBugf("could not load WindowMaker configuration file '%s'",LocalDomain);
    exit(1);
  }
	
  if ( sInitSoundIndex( ) ) {
    sBugf("Could not initialize Sound Index");
    exit(1);
  }
	
  if ( sInitDisplay(argc,argv) >= 0) {
    sDispatchStartup( );
  } 
  else {
    sBugf("Could not initialize Display");
    exit(1);
  }
	
  for( ; ; ) {
    XEvent event;
    XNextEvent(dpy, &event);
    switch(event.type) {
      case ClientMessage:
      {
        if (event.xclient.data.l[0]==DeleteWindow) {
	  XCloseDisplay(dpy);
	  exit(1);
        }
	if (event.xclient.message_type == _XA_WINDOWMAKER_WM_FUNCTION) {
	  long	tmp;
	  // Before we play a sound, lets go ahead and see if our
	  // Domain has changed. If so, we'll load it up, and change
	  // options so new sounds take.
					
	  if((tmp=getTimeStamp(LocalDomain)) > TimeStamp) {
	    proplist_t *tmplist;
						
            tmplist = PLGetProplistWithPath(LocalDomain);
	    if(!PLIsEqual(tmplist,WMSound)) {
	      PLMergeDictionaries(WMSound,tmplist);
	      sUpdateOptions( );
	    }
	    PLRelease(tmplist);
	    TimeStamp = getTimeStamp(LocalDomain);
	  } 
	  else if (!tmp) { // Hrm, Our Domain went somewhere?!, lets resave it
	    PLSave(WMSound,YES);
	    TimeStamp = getTimeStamp(LocalDomain);
	  }
	    if((error=sPlayEvent(event.xclient.data.l[0]))>0) {
	      sBugf(ErrorMsg[error]);
	    }
	  XSync(dpy,YES);
        } 
      } /* end case ClientMessage */
    } /* end switch */
  } /* end for loop */
  return 0;
}

int sLoadConfig(void) {	
  proplist_t config;		// Proplist that will load file
  const char *GlobalDomain;	// Global Domain

  GlobalDomain = defaultsPathForDomain(CFG_DOMAIN,NO);	
	
  // Initialize ALL WMSound Options to sane values
  // and sets WMSoundAttributes' filename to the
  // Local Domain
  sReadDefaults(&WMSound,LocalDomain);
	
  // This nasty looking if statment tries to first of all
  // load the Local Domain.  If that does not exist, then
  // it looks for the Global Domain.  If that does not exist
  // then it's still ok, because WMSound has sane defaults
  // for all its options.  If Global Domain is loaded, then
  // It will merge the loaded options into WMSoundAttributes,
  // then save it to Local Domain.  In the case that Global
  // Domain does not exist, then the default options are saved
  // to the local domain.
  if((config=PLGetProplistWithPath(LocalDomain)) == NULL) { 
    if((config=PLGetProplistWithPath(GlobalDomain)) != NULL) {
      PLMergeDictionaries(WMSound,config);
    } 
    else {
      printf("Could not find '%s'\n",LocalDomain);
    }
  } 
  else {
    PLMergeDictionaries(WMSound,config);
  }
  PLSave(WMSound,YES);

  // By the time we get here, we know that the Local Domain file
  // exists, so for optimization, we'll just go with that instaed
  // of extracting it from the proplist, or calling the domain 
  // function
  TimeStamp=getTimeStamp(LocalDomain);

  // Since WMSound actually works with sPrefrences struct,
  // we need to update this in order for changes to take.
  sUpdateOptions( );
  return 0;
}

int sInitDisplay(int argc, char **argv) {
  XClassHint	ClassHints;
  XWMHints	*WMHints;
  XSizeHints	SHints;
  unsigned long	black,white;
  Atom		Protos[2];

  if((dpy=XOpenDisplay("")) == NULL){
    return -1;
  }
	
  black = BlackPixel(dpy,DefaultScreen(dpy));
  white = WhitePixel(dpy,DefaultScreen(dpy));
  leader = XCreateSimpleWindow(dpy,DefaultRootWindow(dpy),10,10,150,50,black,white,0);
  _XA_WINDOWMAKER_WM_FUNCTION = XInternAtom(dpy,"_WINDOWMAKER_WM_FUNCTION",False);
  DeleteWindow = XInternAtom(dpy,"WM_DELETE_WINDOW",False);

  ClassHints.res_name = "wmsoundserver";
  ClassHints.res_class = "WMSoundServer";
	
  Protos[0] = _XA_WINDOWMAKER_WM_FUNCTION;
  Protos[1] = DeleteWindow;
	
  WMHints = XAllocWMHints( );
  WMHints->window_group = leader;
  WMHints->initial_state = WithdrawnState;
  WMHints->flags = WindowGroupHint|StateHint;
	
  SHints.min_width = 64;
  SHints.min_height = 64;
  SHints.max_width = 64;
  SHints.max_height = 64;
  SHints.flags = PMinSize|PMaxSize;
	
  XSetClassHint(dpy,leader,&ClassHints);
  XSetWMHints(dpy,leader,WMHints);
  XSetWMNormalHints(dpy,leader, &SHints);
  XSetCommand(dpy,leader,argv,argc);
  XSetWMProtocols(dpy,leader,Protos,1);
  XSelectInput(dpy,leader,ExposureMask|ButtonPressMask|KeyPressMask);
  XClearWindow(dpy,leader);
  XMapWindow(dpy,leader);
  return 0;
}


int sInitSoundIndex ( ) {
  int i;
	
  for(i=0;i<WMSound_MaxSound;i++) {
    SoundIndex[i] = 1000 + i;
  }
  return 0;
}

void sDispatchStartup ( ) {
  XEvent SoundEvent;
	
  SoundEvent.xclient.type = ClientMessage;
  SoundEvent.xclient.message_type = _XA_WINDOWMAKER_WM_FUNCTION; 
  SoundEvent.xclient.format = 32;
  SoundEvent.xclient.display = dpy;
  SoundEvent.xclient.window = leader;
  SoundEvent.xclient.data.l[0] = SoundIndex[WMSound_Startup];
  XSendEvent(dpy,leader,False,NoEventMask,&SoundEvent);
  XFlush(dpy);
}

int sPlayEvent (long sidx) {
  int i;
  char *sound_file = NULL;
  /* Find the sound file to play according to sidx */
  for(i=0;i<WMSound_MaxSound;i++) {
    if(SoundIndex[i] == sidx) {
      if(!strcmp("None",sPrefrences.Sound[i]))
        return 0;
      sound_file = find_file_in_list(sPrefrences.SoundPath,sPrefrences.Sound[i]);
      break;
    }
  }

  if(!sound_file) {
    sBugf("Could not find sound '%s'",sPrefrences.Sound[i]);
    return 0;
  }
  return (sPlayFile(sound_file, sPrefrences.SoundDevice));
}
