/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for PCM (OSS 3.XX compatible)
 */

#include "driver.h"
#include "pcm.h"

#if 0
#define PCM_DEBUG_BUFFERS
#endif

static void pcm_fill_with_neutral( gus_pcm_channel_t *pchn );
static void pcm_compute_blocks( gus_pcm_t *pcm, gus_pcm_channel_t *pchn );

/*
 *
 */

static void gus_pcm_init1( gus_card_t *card, gus_pcm_t *pcm )
{
  pcm -> flags = PCM_LFLG_NONE;
  pcm -> card = card;
  SLEEP_PREPARE( pcm, playback )
  SLEEP_PREPARE( pcm, record )
}

void gus_pcm_init( gus_card_t *card )
{
  gus_pcm_init1( card, &card -> pcm[ 0 ] );
  gus_pcm_init1( card, &card -> pcm[ 1 ] );
}
 
static unsigned short pcm_file_flags( unsigned short f_flags )
{
  switch ( f_flags & O_ACCMODE ) {
    case O_WRONLY:	return PCM_LFLG_PLAY;
    case O_RDONLY:	return PCM_LFLG_RECORD;
    default:		return PCM_LFLG_BOTH;
  }
}

/*
 *  user to dma
 */

static int pcm_user_to_dma( gus_pcm_t *pcm, char *buf, int count )
{
  unsigned long flags;
  gus_pcm_channel_t *pchn;
  int result, tmp, realtime;

  if ( count <= 0 || !buf ) return 0;
  if ( VERIFY_AREA( VERIFY_READ, buf, count ) ) return -EACCES;

  pchn = &pcm -> chn[ PCM_PLAYBACK ];
#if 0
  PRINTK( "pcm_user_to_dma: buf=0x%x, count=0x%x,%i (%s)\n", (int)buf, count, count, pchn -> flags & PCM_FLG_NONBLK ? "non blocked" : "blocked" );
#endif
  count &= ~( (pchn -> voices * (pchn -> mode & PCM_MODE_16 ? 2 : 1)) - 1 );
  pchn -> flags &= ~PCM_FLG_ABORT;
  if ( pchn -> flags & PCM_FLG_MMAP ) return -EIO;	/* go to hell... */
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( pcm, pchn );
  if ( pchn -> flags & PCM_FLG_NEUTRAL )
    pcm_fill_with_neutral( pchn );

  result = 0;

  realtime = 0;
  if ( !(pchn -> flags & PCM_FLG_REALTM) &&
       !(pcm -> flags & PCM_LFLG_REALTIME) )
    realtime = 1;

  /*
   * OK. This is patch for .au files which begin with .snd header...
   * This is a little bit hard - it's application work to do conversions...
   */
  if ( (pchn -> mode & PCM_MODE_ULAW) && !pchn -> processed_bytes && !pchn -> used && count > 31 )
    {
      unsigned char buffer[ 8 ];
      
      MEMCPY_FROMFS( buffer, buf, 8 );
      if ( !memcmp( buffer, ".snd\0\0\0\x20", 8 ) )
        {
          buf += 32;
          count -= 32;
          result += 32;
        }
    }

  while ( count > 0 )
    {
      while ( pchn -> used >= pchn -> blocks )
        {
          if ( pchn -> flags & PCM_FLG_NONBLK ) return result;
          if ( !(pchn -> flags & PCM_FLG_ENABLE) ) return result;
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEP( pcm, playback, 10 * HZ );
          pchn -> flags &= ~PCM_FLG_SLEEP;
          if ( TABORT( pcm, playback ) )
            {
              pchn -> flags |= PCM_FLG_ABORT;
              return result;
            }
          if ( pchn -> used >= pchn -> blocks && TIMEOUT( pcm, playback ) )
            {
              PRINTK( "pcm_user_to_dma: timeout, new block discarded\n" );
              return -EIO;
            }
        }
        
      /* ok.. interresant part.. we must find first free & not locked block.. */
        
      tmp = 0;
      CLI( &flags );
      while ( PCM_ISLOCK( pchn, pchn -> head ) )
        {
          pchn -> discarded++;
          pchn -> used++;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
          if ( pchn -> used >= pchn -> blocks )
            {
              tmp++;
              break;
            }
        }
      STI( &flags );
      if ( tmp ) continue;		/* go to sleep loop */

      /* ok.. now we have right block - fill it */
      
      tmp = pchn -> block_size - pchn -> sizes[ pchn -> head ];
      if ( tmp > count ) tmp = count;	/* correction */

#if 0
      printk( "to dma: size = %i\n", tmp );
#endif
      if ( VERIFY_AREA( VERIFY_READ, buf, tmp ) )
        return -EACCES;
      pchn -> hw_dma( pcm, pchn, buf, tmp );

      count -= tmp;
      result += tmp;
      buf += tmp;

      pchn -> sizes[ pchn -> head ] += tmp;
      if ( pchn -> sizes[ pchn -> head ] == pchn -> block_size )
        {
#if 0
          printk( "block filled = %i, flags = 0x%x\n", pchn -> head, card -> pcm.flags  );
#endif
          pchn -> used++;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
          if ( (pchn -> flags & PCM_FLG_ENABLE) && pchn -> used > realtime )
            pchn -> hw_init( pcm );
        }
#if 0
      PRINTK( "pcm_user_to_dma: end count=0x%x\n", count );
#endif
    }
#if 0
  PRINTK( "pcm_user_to_dma: end\n" );
#endif
  return result;
}

/*
 *  dma to user
 */

static int pcm_dma_to_user( gus_pcm_t *pcm, char *buf, int count )
{
  unsigned long flags;
  gus_pcm_channel_t *pchn;
  int result, tmp;

  if ( count <= 0 ) return 0;
  if ( VERIFY_AREA( VERIFY_WRITE, buf, count ) ) return -EACCES;
  
  pchn = &pcm -> chn[ PCM_RECORD ];
#if 0
  PRINTK( "pcm_user_from_dma: buf=0x%x, count=0x%x (%s)\n", (int)buf, count, pchn -> flags & PCM_FLG_NONBLK ? "non blocked" : "blocked" );
#endif
  count &= ~( (pchn -> voices * (pchn -> mode & PCM_MODE_16 ? 2 : 1)) - 1 );
  pchn -> flags &= ~PCM_FLG_ABORT;
  if ( pchn -> flags & PCM_FLG_MMAP ) return -EIO;
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( pcm, pchn );
  if ( pchn -> flags & PCM_FLG_NEUTRAL )
    pcm_fill_with_neutral( pchn );

  if ( pchn -> flags & PCM_FLG_ENABLE )
    pchn -> hw_init( pcm );
  result = 0;
  
  while ( count > 0 )
    {
      while ( !pchn -> used )
        {
          if ( pchn -> flags & PCM_FLG_NONBLK ) return result;
          if ( !(pchn -> flags & PCM_FLG_ENABLE) ) return result;
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEP( pcm, record, 10 * HZ );
          pchn -> flags &= ~PCM_FLG_SLEEP;
          if ( TABORT( pcm, record ) )
            {
              pchn -> flags |= PCM_FLG_ABORT;
              return -EINTR;
            }
          if ( !pchn -> used && TIMEOUT( pcm, record ) )
            {
              PRINTK( "pcm_dma_to_user: data timeout\n" );
              return -EIO;
            }
        }
      
      tmp = 0;
      CLI( &flags );
      while ( PCM_ISLOCK( pchn, pchn -> tail ) )
        {
          pchn -> discarded++;
          pchn -> used--;
          pchn -> tail++;
          pchn -> tail %= pchn -> blocks;
          if ( !pchn -> used )
            {
              tmp++;
              break;
            }
        }
      STI( &flags );
      if ( tmp ) continue;		/* go to sleep loop */
        
      tmp = count <= pchn -> sizes[ pchn -> tail ] ?
		      	       count : pchn -> sizes[ pchn -> tail ];
      if ( VERIFY_AREA( VERIFY_WRITE, buf, tmp ) )
        return -EACCES;
      pchn -> hw_dma( pcm, pchn, buf, tmp );

      buf += tmp;
      count -= tmp;
      result += tmp;

      pchn -> sizes[ pchn -> tail ] -= tmp; 
      if ( !pchn -> sizes[ pchn -> tail ] )
        {
          pchn -> used--;
          pchn -> tail++;
          pchn -> tail %= pchn -> blocks;
        }
    }

  return result;
}

/*
 *  synchronize playback
 */

static void pcm_sync_playback( gus_pcm_t *pcm )
{
  gus_pcm_channel_t *pchn;

  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  if ( pchn -> flags & PCM_FLG_ABORT ) return;
  pchn -> flags |= PCM_FLG_SYNC;
  if ( pchn -> used < pchn -> blocks && pchn -> sizes[ pchn -> head ] > 0 )
    {
      pchn -> hw_dma_neutral( pcm, pchn );
      pchn -> used++;
      pchn -> head++;
      pchn -> head %= pchn -> blocks;
    }
  if ( !pchn -> used ) goto __end;
  pchn -> hw_sync( pcm );
  pcm_fill_with_neutral( pchn );
  pchn -> head = pchn -> tail = pchn -> used = 0;
  PCM_LOCKZERO( pchn );
  __end:
  pchn -> flags &= ~PCM_FLG_SYNC;
}

static void pcm_sync_record( gus_pcm_t *pcm )
{
  gus_pcm_channel_t *pchn;

  pchn = &pcm -> chn[ PCM_RECORD ];
  pchn -> hw_sync( pcm );
  pcm_fill_with_neutral( pchn );
  pchn -> head = pchn -> tail = pchn -> used = 0;
  PCM_LOCKZERO( pchn );
}

/*
 *  other things
 */

void gus_pcm_set_dma_struct( gus_card_t *card, int dma_number, gus_pcm_channel_t *pchn )
{
  int i;
  struct GUS_STRU_DMA *pdma;

  pdma = card -> dmas[ dma_number ];
  pchn -> _dma = pdma;
  pchn -> dma = pdma -> dma;
  pchn -> size = pdma -> usize;
  pchn -> buffer = pdma -> buf;
  for ( i = 0; i < PCM_MAX_BLOCKS; i++ )
    pchn -> sizes[ i ] = 0;
  pchn -> flags |= PCM_FLG_DMAOK;
}

static void pcm_fill_with_neutral( gus_pcm_channel_t *pchn )
{
  MEMSET( pchn -> buffer, pchn -> neutral_byte, pchn -> used_size );
  pchn -> flags &= ~PCM_FLG_NEUTRAL;
}

static void pcm_compute_blocks( gus_pcm_t *pcm, gus_pcm_channel_t *pchn )
{
  int record;
  unsigned int block, blocks, bps;
  unsigned long flags;
#if 0
  unsigned int max;
#endif

  record = pchn == &pcm -> chn[ PCM_RECORD ];
  if ( !record )
    pcm_sync_playback( pcm );
   else
    pcm_sync_record( pcm );
  CLI( &flags );
  if ( pchn -> requested_block_size )
    {
      block = pchn -> requested_block_size;
#if 0
      if ( block > pchn -> size / pchn -> requested_blocks )
        block = pchn -> size / pchn -> requested_blocks;
#endif
      block &= ~3;	/* align to 4 */
      if ( pcm -> info_flags & GUS_PCM_INFO_PLAYBACK_BATCH )	/* GF1? */
        block &= ~31;
      blocks = pchn -> size / block;
      if ( blocks > pchn -> requested_blocks )
        blocks = pchn -> requested_blocks;
    }
   else
    {
      bps = pchn -> rate * pchn -> voices;
      if ( pchn -> mode & PCM_MODE_16 ) bps <<= 1;
      if ( pchn -> mode & PCM_MODE_ADPCM ) bps >>= 2;
      block = pchn -> size;
      while ( block > bps ) block >>= 1;
      if ( block == pchn -> size ) block >>= 1;
      if ( record )
        block /= 4;			/* small fragment when recording */
      block &= ~3;			/* align to 4 */
      blocks = pchn -> size / block;
    }
#if 0
   max = pchn -> hw_max_dma_block( card );
   if ( max < block ) block = max;
#endif
   if ( blocks > PCM_MAX_BLOCKS ) blocks = PCM_MAX_BLOCKS;
   pchn -> used_size = blocks * block;
   pchn -> blocks = blocks;
   pchn -> block_size = block;
   pchn -> flags |= PCM_FLG_NEUTRAL;
   pchn -> flags &= ~(PCM_FLG_BUFFERS|PCM_FLG_REALTM);
   STI( &flags );
#ifdef PCM_DEBUG_BUFFERS
   PRINTK( "used_size = %i, blocks = %i, blocks_size = %i, mmap = %s\n", pchn -> used_size, pchn -> blocks, pchn -> block_size, pchn -> flags & PCM_FLG_MMAP ? "yes" : "no" );
#endif
}

static int pcm_set_subdivision( gus_pcm_t *pcm, unsigned short flags, unsigned int subdivision )
{
  int idx;
  gus_pcm_channel_t *pchn;

  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &pcm -> chn[ idx - 1 ];
        if ( pchn -> requested_subdivision != subdivision )
          {
            if ( idx == 1 )
              pcm_sync_playback( pcm );
             else
              pcm_sync_record( pcm );
            pchn -> requested_subdivision = subdivision;
            pchn -> flags |= PCM_FLG_NEUTRAL | PCM_FLG_BUFFERS;
          }
      }  
  return 0;
}

/*
 * SET FRAGMENT notes:
 *   Real Video 5.0b1 - fragment = 0x04 (rate & format setup before)
 *   Libmmoss 1.1 - fragment = 0x20009 (rate & format setup before)
 *   MTV - fragment = 0x7f000a (rate & format setup after!!)
 *   RAT - fragment = 0x7fff0007 (rate & format setup after!!)
 */

static int pcm_set_fragment( gus_pcm_t *pcm, unsigned short flags, unsigned int fragment )
{
  int idx, size, bytes, count, min_fragment;
  gus_pcm_channel_t *pchn;

  bytes = fragment & 0xffff;
  if ( !bytes ) {
    count = PCM_MAX_BLOCKS;
    bytes = 10;
  }
  min_fragment = 4;
  if ( flags & PCM_LFLG_PLAY ) 
    if ( pcm -> chn[ PCM_PLAYBACK ].min_fragment > min_fragment )
      min_fragment = pcm -> chn[ PCM_PLAYBACK ].min_fragment;
  if ( flags & PCM_LFLG_RECORD ) 
    if ( pcm -> chn[ PCM_RECORD ].min_fragment > min_fragment )
      min_fragment = pcm -> chn[ PCM_RECORD ].min_fragment;
  if ( bytes < min_fragment ) bytes = min_fragment;
  if ( fragment == 0x7fff0007 ) bytes = 9;	/* bad hack for RAT */
  if ( bytes > 17 ) bytes = 17;
  size = 1 << bytes;

  count = fragment >> 16;
  if ( !count ) {
    count = PCM_MAX_BLOCKS;
    size = 512;
  }
  if ( count < 2 ) count = 2;
  if ( count > PCM_MAX_BLOCKS ) count = PCM_MAX_BLOCKS;

#ifdef PCM_DEBUG_BUFFERS
  printk( "set fragment: fragment = 0x%x, size = %i, count = %i\n", fragment, size, count );
#endif

  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &pcm -> chn[ idx - 1 ];
        if ( pchn -> requested_block_size != size ||
             pchn -> requested_blocks != count )
          {
            if ( idx == 1 )
              pcm_sync_playback( pcm );
             else
              pcm_sync_record( pcm );
            pchn -> requested_block_size = size;
            pchn -> requested_blocks = count;
            pchn -> flags |= PCM_FLG_NEUTRAL | PCM_FLG_BUFFERS;
          }
      }  
  return ( count << 16 ) | bytes;
}


static unsigned int pcm_get_block_size( gus_pcm_t *pcm, unsigned short flags )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( ( flags & PCM_LFLG_PLAY ) && ( pcm -> chn[ PCM_PLAYBACK ].flags & PCM_FLG_BUFFERS ) )
    pcm_compute_blocks( pcm, &pcm -> chn[ PCM_PLAYBACK ] );
  if ( ( flags & PCM_LFLG_RECORD ) && ( pcm -> chn[ PCM_RECORD ].flags & PCM_FLG_BUFFERS ) )
    pcm_compute_blocks( pcm, &pcm -> chn[ PCM_RECORD ] );
  return pcm -> chn[ ( flags - 1 ) & 1 ].block_size;
}

static void pcm_set_mode( gus_pcm_t *pcm, unsigned short flags, unsigned int mode )
{
  int idx;
  gus_pcm_channel_t *pchn;

#ifdef PCM_DEBUG_BUFFERS
  printk( "mode = 0x%x\n", mode );
#endif
  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &pcm -> chn[ idx - 1 ];
        if ( pchn -> mode != mode )
          {
            if ( idx == 1 )
              pcm_sync_playback( pcm );
             else 
              pcm_sync_record( pcm );
            pchn -> neutral_byte = ( mode & PCM_MODE_16 ) ? 0x00 : 0x80;
            pchn -> mode = mode;
            pchn -> flags |= PCM_FLG_NEUTRAL | PCM_FLG_BUFFERS;
          }
      }
}

static unsigned int pcm_get_mode( gus_pcm_t *pcm, unsigned short flags )
{
  return pcm -> chn[ ( flags - 1 ) & 1 ].mode;
}

static void pcm_set_format( gus_pcm_t *pcm, unsigned short flags, unsigned int format )
{
  int idx;
  gus_pcm_channel_t *pchn;

#ifdef PCM_DEBUG_BUFFERS
  printk( "format = 0x%x\n", format );
#endif
  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &pcm -> chn[ idx - 1 ];
        pchn -> format = format;
      }
}

static unsigned int pcm_get_format( gus_pcm_t *pcm, unsigned short flags )
{
  return pcm -> chn[ ( flags - 1 ) & 1 ].format;
}

static void pcm_set_rate( gus_pcm_t *pcm, unsigned short flags, unsigned int rate )
{
  int idx;
  gus_pcm_channel_t *pchn;

#ifdef PCM_DEBUG_BUFFERS
  printk( "rate = %i\n", rate );
#endif
  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &pcm -> chn[ idx - 1 ];
        if ( pchn -> rate != rate )
          {
            if ( idx == 1 )
              pcm_sync_playback( pcm );
             else
              pcm_sync_record( pcm );
            pchn -> rate = rate;
            pchn -> flags |= PCM_FLG_BUFFERS; 
          }
    }
}

static unsigned int pcm_get_rate( gus_pcm_t *pcm, unsigned short flags )
{
  return pcm -> chn[ ( flags - 1 ) & 1 ].rate;
}

static int set_format( gus_pcm_t *pcm, unsigned short flags, int format )
{
  unsigned int new_mode;
  unsigned int new_format;

  if ( format != AFMT_QUERY )
    {
#if 0
      printk( "format = 0x%x\n", format );
#endif
      if ( !format )
        {
          new_format = pcm_get_format( pcm, flags );
          new_mode = pcm_get_mode( pcm, flags );
        }
       else
        {
          new_format = 0;
          if ( flags & PCM_LFLG_PLAY ) {
            new_format = format & pcm -> chn[ PCM_PLAYBACK ].formats;
            if ( !new_format )		/* not supported format */
              new_format = AFMT_U8;	/* always supported */
          }
          if ( flags & PCM_LFLG_RECORD ) {
            if ( new_format )
              new_format &= pcm -> chn[ PCM_RECORD ].formats;
             else
              new_format = format & pcm -> chn[ PCM_RECORD ].formats;
            if ( !new_format )		/* not supported format */
              new_format = AFMT_U8;	/* always supported */
          }
          new_mode = pcm_get_mode( pcm, flags ) & ~PCM_MODE_TYPE;
          new_mode |= PCM_MODE_VALID;
          if ( new_format & ( AFMT_MU_LAW | AFMT_A_LAW | AFMT_U8 | 
                              AFMT_U16_LE | AFMT_U16_BE ) )
            new_mode |= PCM_MODE_U;
          if ( new_format & ( AFMT_S16_LE | AFMT_S16_BE | AFMT_U16_LE | 
                              AFMT_U16_BE | AFMT_IMA_ADPCM ) )
            new_mode |= PCM_MODE_16;
          if ( new_format & AFMT_MU_LAW )
            new_mode |= PCM_MODE_ULAW;
          if ( new_format & AFMT_A_LAW )
            new_mode |= PCM_MODE_ALAW;
          if ( new_format & AFMT_IMA_ADPCM )
            new_mode |= PCM_MODE_ADPCM;
        }
      if ( new_mode != pcm_get_mode( pcm, flags ) )
        {
          pcm_set_format( pcm, flags, new_format );
          pcm_set_mode( pcm, flags, new_mode );
        }
    }

  return pcm_get_format( pcm, flags );
}

static int set_rate( gus_pcm_t *pcm, unsigned short flags, unsigned int rate )
{
  unsigned int max_rate;

  if ( rate > 0 ) 
    {
      max_rate = 48000;
      if ( flags & PCM_LFLG_PLAY ) {
        if ( max_rate > pcm -> chn[ PCM_PLAYBACK ].max_rate )
          max_rate = pcm -> chn[ PCM_PLAYBACK ].max_rate;
      }
      if ( flags & PCM_LFLG_RECORD ) {
        if ( max_rate > pcm -> chn[ PCM_RECORD ].max_rate )
          max_rate = pcm -> chn[ PCM_RECORD ].max_rate;
      }
      if ( rate > max_rate ) rate = max_rate;
      pcm_set_rate( pcm, flags, rate );
    }
   else
    rate = pcm_get_rate( pcm, flags );

  return rate;
}

static int pcm_set_channels( gus_pcm_t *pcm, unsigned short flags, int channels )
{
  gus_pcm_channel_t *pchn;

#if 0
  printk( "set channels = %i\n", channels );
#endif
  if ( channels >= 0 )
    {
      if ( channels < 1 || channels > 32 ) return -EINVAL;
      if ( flags & PCM_LFLG_PLAY ) {
        pchn = &pcm -> chn[ PCM_PLAYBACK ];
        if ( channels > pchn -> max_voices ) return -EINVAL;
        pcm_sync_playback( pcm );
        pchn -> voices = channels;
        pchn -> flags |= PCM_FLG_BUFFERS;
      }
      if ( flags & PCM_LFLG_RECORD ) {
        pchn = &pcm -> chn[ PCM_RECORD ];
        if ( channels > pchn -> max_voices ) return -EINVAL;
        pcm_sync_playback( pcm );
        pchn -> voices = channels;
        pchn -> flags |= PCM_FLG_BUFFERS;
      }
    }
   else
    {
      if ( flags & PCM_LFLG_PLAY )
        channels = pcm -> chn[ PCM_PLAYBACK ].voices;
      if ( flags & PCM_LFLG_RECORD )
        channels = pcm -> chn[ PCM_RECORD ].voices;
    }
  return channels;
}

static void pcm_reset( gus_pcm_t *pcm, unsigned short flags )
{
  int idx;
  gus_pcm_channel_t *pchn;

#ifdef PCM_DEBUG_BUFFERS
  printk( "pcm_reset!!!\n" );
#endif
  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &pcm -> chn[ idx - 1 ];
        if ( idx == 1 )
          pcm_sync_playback( pcm );
         else
          pcm_sync_record( pcm );
#if 0
        pchn -> requested_block_size =
        pchn -> requested_blocks =
        pchn -> requested_subdivision = 0;
        pchn -> processed_bytes = 0;
        pchn -> interrupts = 0;
        pchn -> flags |= PCM_FLG_BUFFERS;
#endif
        pchn -> flags &= ~PCM_FLG_ABORT;
      }
}

static int pcm_get_trigger( gus_pcm_t *pcm, unsigned short flags )
{
  unsigned long iflags;
  int result;

  result = 0;
  CLI( &iflags );
  if ( ( pcm -> chn[ PCM_PLAYBACK ].flags & PCM_FLG_TRIGGER ) && ( flags & PCM_LFLG_PLAY ) ) result |= PCM_ENABLE_OUTPUT;
  if ( ( pcm -> chn[ PCM_RECORD ].flags & PCM_FLG_TRIGGER ) && ( flags & PCM_LFLG_RECORD ) ) result |= PCM_ENABLE_INPUT;
  STI( &iflags );
#if 0
  printk( "get trigger = 0x%x\n", result );
#endif
  return result;
}

static int pcm_set_trigger( gus_pcm_t *pcm, unsigned short flags, int trigger )
{
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

#if 0
  printk( "set trigger = 0x%x\n", trigger );
#endif
  card = pcm -> card;
  if ( flags & PCM_LFLG_PLAY )
    {
      pchn = &pcm -> chn[ PCM_PLAYBACK ];
      if ( trigger & PCM_ENABLE_OUTPUT )
        {
          pchn -> flags |= PCM_FLG_ENABLE;
          if ( !(pchn -> flags & PCM_FLG_TRIGGER) )
            {
              if ( (pchn -> flags & PCM_FLG_MMAP) || pchn -> used > 0 )
                {
                  if ( pchn -> flags & PCM_FLG_BUFFERS )
                    pcm_compute_blocks( pcm, pchn );
                  pchn -> hw_init( pcm );
                }
            }
        }
       else
        {
          pchn -> flags &= ~PCM_FLG_ENABLE;
          if ( pchn -> flags & PCM_FLG_TRIGGER )
            pchn -> hw_done( pcm );
        }
    }
  if ( flags & PCM_LFLG_RECORD )
    {
      pchn = &pcm -> chn[ PCM_RECORD ];
      if ( trigger & PCM_ENABLE_INPUT )
        {
          pchn -> flags |= PCM_FLG_ENABLE;
          if ( !(pchn -> flags & PCM_FLG_TRIGGER) )
            {
              if ( pchn -> flags & PCM_FLG_BUFFERS )
                pcm_compute_blocks( pcm, pchn );
              pchn -> hw_init( pcm );
            }
        }
       else
        {
          pchn -> flags &= ~PCM_FLG_ENABLE;
          if ( pchn -> flags & PCM_FLG_TRIGGER )
            pchn -> hw_done( pcm );
        }
    }
  return pcm_get_trigger( pcm, flags );
}

#ifdef GUSCFG_OSS

int gus_pcm_trigger_sequencer( unsigned int devmask )
{
  int i;
  gus_card_t *card;
  gus_pcm_t *pcm;
  
  for ( i = 0; i < gus_cards_count * 2; i++ ) {
    if ( ( devmask & (1 << i) ) == 0 ) continue; 
    card = gus_cards[ i >> 1 ];
    pcm = &card -> pcm[ i & 1 ];
    if ( !(pcm -> flags & PCM_LFLG_PLAY) ) continue;
    pcm_set_trigger( pcm, PCM_LFLG_PLAY, PCM_ENABLE_OUTPUT );
  }
  return 0;
}

#endif

static int pcm_get_space( gus_pcm_t *pcm, unsigned short flags, int record, audio_buf_info *arg )
{
  unsigned long iflags;
  gus_pcm_channel_t *pchn;
  audio_buf_info info;

  if ( !( flags & ( 1 << record ) ) ) return -EIO;	/* bad file mode */
  if ( VERIFY_AREA( VERIFY_WRITE, arg, sizeof( info ) ) ) return -EIO;
  pchn = &pcm -> chn[ record ];
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( pcm, pchn );
  CLI( &iflags );
  info.fragments = record ? pchn -> used : pchn -> blocks - pchn -> used;
  info.fragstotal = pchn -> blocks;
  info.fragsize = pchn -> block_size;
  info.bytes = info.fragments * info.fragsize;
  if ( !record && pchn -> sizes[ pchn -> head ] )
    {
      /* info.fragments--; */
      info.bytes -= pchn -> sizes[ pchn -> head ];
    }
  STI( &iflags );
#ifdef PCM_DEBUG_BUFFERS
  printk( "ospace: frags = %i, total = %i, size = %i, bytes = %i\n", info.fragments, info.fragstotal, info.fragsize, info.bytes );
#endif
  pchn -> flags |= PCM_FLG_ENABLE;
  MEMCPY_TOFS( arg, &info, sizeof( info ) );
  return 0;
}

static int pcm_get_ptr( gus_pcm_t *pcm, unsigned short flags, int record, count_info *arg )
{
  unsigned long iflags;
  gus_pcm_channel_t *pchn;
  count_info info;

  if ( !( flags & ( 1 << record ) ) ) return -EIO;	/* bad file mode */
  if ( VERIFY_AREA( VERIFY_WRITE, arg, sizeof( info ) ) ) return -EIO;
  pchn = &pcm -> chn[ record ];
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( pcm, pchn );
  CLI( &iflags );
  pchn -> flags |= PCM_FLG_REALTM;
  info.bytes = pchn -> processed_bytes;
  info.ptr = pchn -> hw_pointer( pcm );
  info.bytes += info.ptr % pchn -> block_size;
  info.blocks = pchn -> interrupts;
  if ( pchn -> flags & PCM_FLG_MMAP ) {
    pchn -> interrupts = 0;
  }
#if 0
  printk( "ptr: bytes = %i, ptr = %i, blocks = %i\n", info.bytes, info.ptr, info.blocks );
#endif
  STI( &iflags );
  MEMCPY_TOFS( arg, &info, sizeof( info ) );
  return 0;
}

static int pcm_get_mapbuf( gus_pcm_t *pcm, unsigned short flags, int record, buffmem_desc *arg )
{
  unsigned long iflags;
  gus_pcm_channel_t *pchn;
  buffmem_desc info;

  if ( !( flags & ( 1 << record ) ) ) return -EIO;	/* bad file mode */
  if ( VERIFY_AREA( VERIFY_WRITE, arg, sizeof( info ) ) ) return -EIO;
  pchn = &pcm -> chn[ record ];
  CLI( &iflags );
  info.buffer = (unsigned *)pchn -> buffer;
  info.size = pchn -> size;
  STI( &iflags );
  MEMCPY_TOFS( arg, &info, sizeof( info ) );
  return 0;
}

int gus_pcm_info_flags( gus_pcm_t *pcm, struct GUS_PCM_INFO *info, int control )
{
  struct GUS_PCM_INFO ginfo;
  gus_pcm_channel_t *pchn_p, *pchn_r;

  if ( control ) {
    gus_card_t *card;
    int val;
    
    if ( VERIFY_AREA( VERIFY_READ, info, sizeof( ginfo ) ) ) return -EIO;
    MEMCPY_FROMFS( &ginfo, info, sizeof( ginfo ) );
    val = ginfo.device >> 16;
    if ( val >= gus_cards_count ) return -ENODEV;
    card = gus_cards[ val ];
    val = ginfo.device & 0xffff;
    if ( val >= GUS_PCM_DEVICES ) return -ENODEV;
    pcm = &card -> pcm[ val ];
    if ( !(pcm -> flags & PCM_LFLG_USED) ) return -ENODEV;
  }
  if ( VERIFY_AREA( VERIFY_WRITE, info, sizeof( ginfo ) ) ) return -EIO;
  MEMSET( &ginfo, 0, sizeof( ginfo ) );
  pchn_p = &pcm -> chn[ PCM_PLAYBACK ];
  pchn_r = &pcm -> chn[ PCM_RECORD ];
  ginfo.flags = pcm -> info_flags;
  strcpy( ginfo.name, pcm -> info_name );
  ginfo.max_rate_play = pchn_p -> max_rate;
  ginfo.max_rate_record = pchn_r -> max_rate;
  ginfo.formats_play = pchn_p -> formats;
  ginfo.formats_record = pchn_r -> formats;
  ginfo.max_channels_play = pchn_p -> max_voices;
  ginfo.max_channels_record = pchn_r -> max_voices;
  ginfo.dma_size_play = pchn_p -> size;
  ginfo.dma_size_record = pchn_r -> size;
  MEMCPY_TOFS( info, &ginfo, sizeof( ginfo ) );
  return 0;
}

static int get_osscaps( gus_pcm_t *pcm, int flags )
{
  unsigned int result;
  
  result = DSP_CAP_REALTIME | DSP_CAP_TRIGGER | DSP_CAP_MMAP;
  if ( flags & PCM_LFLG_PLAY ) {
    if ( pcm -> info_flags & GUS_PCM_INFO_PLAYBACK_BATCH )
      result |= DSP_CAP_BATCH;
  } else {
    if ( pcm -> info_flags & GUS_PCM_INFO_RECORD_BATCH )
      result |= DSP_CAP_BATCH;
  }
  if ( pcm -> info_flags & GUS_PCM_INFO_DUPLEX )
    result |= DSP_CAP_DUPLEX;
  return result;
}

static int gus_open_pcm_card( gus_pcm_t *pcm, short minor, struct file *file )
{
  unsigned short flags;
  int res;

  flags = pcm_file_flags( file -> f_flags );
#if 0
  PRINTK( "gus_open_pcm (1) - card = 0x%lx, minor = %i, flags = %i, pcm = %i\n", (long)card, minor, flags, card -> pcm.flags );
#endif
  if ( flags & pcm -> flags ) return -EBUSY;	/* channel(s) already used */
#if 0
  PRINTK( "gus_open_pcm - minor = %i, flags = %i, pcm = %i\n", minor, flags, card -> pcm.flags );
#endif
  if ( flags & PCM_LFLG_PLAY )
    if ( !(pcm -> info_flags & GUS_PCM_INFO_PLAYBACK) ) return -ENODEV;
  if ( flags & PCM_LFLG_RECORD )
    if ( !(pcm -> info_flags & GUS_PCM_INFO_RECORD) ) return -ENODEV;
  if ( flags == PCM_LFLG_BOTH )
    if ( !(pcm -> info_flags & GUS_PCM_INFO_DUPLEX) ) return -ENODEV;
  pcm -> mask = PCM_LFLG_BOTH;
  pcm_set_format( pcm, flags, 0 );
  pcm_set_mode( pcm, flags, 0 );
  if ( minor < GUS_MINOR_GBEGIN ) {
    switch ( minor & GUS_MINOR_OSS_MASK ) {
      case GUS_MINOR_AUDIO:
        set_format( pcm, flags, AFMT_MU_LAW );
        break;
      case GUS_MINOR_PCM_8:
        set_format( pcm, flags, AFMT_U8 );
        break; 
      case GUS_MINOR_PCM_16:
      case GUS_MINOR_PCM1:
        set_format( pcm, flags, AFMT_U16_LE );
        break;
      default:
        PRINTK( "pcm: bad minor value\n" );
        return -EINVAL;
    }
  } else {
    set_format( pcm, flags, AFMT_U16_LE );
  }
  pcm_set_rate( pcm, flags, PCM_DEFAULT_RATE );
  pcm_set_channels( pcm, flags, 1 );
  if ( flags & PCM_LFLG_PLAY )
    {
      if ( ( res = pcm -> chn[ PCM_PLAYBACK ].hw_open( pcm ) ) < 0 )
        return res;
      pcm -> chn[ PCM_PLAYBACK ].flags |= PCM_FLG_ENABLE;
    }
  if ( flags & PCM_LFLG_RECORD )
    {
      if ( ( res = pcm -> chn[ PCM_RECORD ].hw_open( pcm ) ) < 0 )
        {
          if ( flags & PCM_LFLG_PLAY )
            {
              pcm -> chn[ PCM_PLAYBACK ].hw_close( pcm );
              pcm -> chn[ PCM_PLAYBACK ].flags &= ~PCM_FLG_ENABLE;
            }
          return res;
        }
      pcm -> chn[ PCM_RECORD ].flags |= PCM_FLG_ENABLE;
    }
  pcm -> flags |= flags;

#if 0
  printk( "pcm open - done...\n" );
#endif

  return 0;
}

static void gus_close_pcm_card( gus_pcm_t *pcm, struct file *file )
{
  unsigned short flags;
  
  if ( !pcm ) return;
  flags = pcm_file_flags( file -> f_flags ) & pcm -> flags;
#if 0
  PRINTK( "gus_release_pcm - flags = %i\n", flags );
#endif
  if ( flags & PCM_LFLG_PLAY )
    {
      pcm_sync_playback( pcm );		/* synchronize playback */
      pcm -> chn[ PCM_PLAYBACK ].hw_close( pcm );
      pcm -> flags &= ~PCM_LFLG_PLAY;
      MEMSET( &pcm -> chn[ PCM_PLAYBACK ], 0, sizeof( gus_pcm_channel_t ) - ( PCM_HW_FUNCTIONS * sizeof( long ) ) - PCM_ZERO_RESERVED );
    }
  if ( flags & PCM_LFLG_RECORD )
    {
      pcm_sync_record( pcm );		/* and record */
      pcm -> chn[ PCM_RECORD ].hw_close( pcm );
      pcm -> flags &= ~PCM_LFLG_RECORD;
      MEMSET( &pcm -> chn[ PCM_RECORD ], 0, sizeof( gus_pcm_channel_t ) - ( PCM_HW_FUNCTIONS * sizeof( long ) ) - PCM_ZERO_RESERVED );
    }
#if 0
  printk( "release pcm: done\n" );
#endif
}

int gus_open_pcm( unsigned short minor, struct file *file )
{
  short dev;
  int idx, res;
  gus_card_t *card;
  gus_pcm_t *pcm;

  dev = ( minor >> 4 ) & 7;
  if ( dev >= gus_cards_count ) return -ENODEV;
  idx = 0;
  if ( minor >= GUS_MINOR_GBEGIN ) {
    if ( ( minor & GUS_MINOR_GDEVMASK ) == GUS_MINOR_GPCM1 ) idx = 1;
  } else {
    if ( ( minor & GUS_MINOR_OSS_MASK ) == GUS_MINOR_PCM1 ) idx = 1;
  }
  card = gus_cards[ dev ];
  file -> private_data = pcm = &card -> pcm[ idx ];
  if ( ( res = gus_open_pcm_card( pcm, minor, file ) ) < 0 )
    {
      file -> private_data = NULL;
      return res;
    }
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_pcm( struct file *file )
{
  gus_close_pcm_card( (gus_pcm_t *)file -> private_data, file );
  MOD_DEC_USE_COUNT;
}

int gus_ioctl_pcm( struct file *file, unsigned int cmd, unsigned long arg )
{
  gus_pcm_t *pcm;
  gus_card_t *card;
  unsigned short flags;

  pcm = (gus_pcm_t *)file -> private_data;
  card = pcm -> card;
  flags = pcm_file_flags( file -> f_flags ) & pcm -> mask;
  if ( ( ( cmd >> 8 ) & 0xff ) == 'M' )	/* mixer ioctl - for OSS (grrr) compatibility */
    return gus_ioctl_mixer( card, file, cmd, arg );
  if ( ( ( cmd >> 8 ) & 0xff ) != 'P' ) return -EIO;
#if 0
  printk( "cmd = 0x%x, arg = 0x%x, flags = 0x%x\n", cmd, IOCTL_IN( arg ), flags );
#endif
  switch ( cmd ) {
    case SOUND_PCM_CARDS:
      return IOCTL_OUT( arg, gus_cards_count );
    case SOUND_PCM_WRITE_RATE:
      return IOCTL_OUT( arg, set_rate( pcm, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_READ_RATE:
      return IOCTL_OUT( arg, pcm_get_rate( pcm, flags ) );
    case SNDCTL_DSP_STEREO:
      return IOCTL_OUT( arg, pcm_set_channels( pcm, flags, IOCTL_IN( arg ) > 0 ? 2 : 1 ) - 1 );
    case SOUND_PCM_WRITE_CHANNELS:
      return IOCTL_OUT( arg, pcm_set_channels( pcm, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_READ_CHANNELS:
      return IOCTL_OUT( arg, pcm_set_channels( pcm, flags, -1 ) );
    case SOUND_PCM_GETFMTS:
      return IOCTL_OUT( arg, pcm -> chn[ flags & PCM_LFLG_PLAY ? PCM_PLAYBACK : PCM_RECORD ].formats );
    case SOUND_PCM_SETFMT:
      return IOCTL_OUT( arg, set_format( pcm, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_GETCAPS:
      return IOCTL_OUT( arg, get_osscaps( pcm, flags ) );
    case SOUND_PCM_READ_BITS:
      return IOCTL_OUT( arg, pcm_get_format( pcm, flags ) );
    case SNDCTL_DSP_GETBLKSIZE:
      return IOCTL_OUT( arg, pcm_get_block_size( pcm, flags ) );
    case SOUND_PCM_SUBDIVIDE:
      return IOCTL_OUT( arg, pcm_set_subdivision( pcm, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_SETFRAGMENT:
      return IOCTL_OUT( arg, pcm_set_fragment( pcm, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_SYNC:
    case SOUND_PCM_POST:	/* wrong implementation */
      if ( flags & PCM_LFLG_PLAY ) pcm_sync_playback( pcm );
      if ( flags & PCM_LFLG_RECORD ) pcm_sync_record( pcm );
      return 0;
    case SOUND_PCM_RESET:
      pcm_reset( pcm, flags );
      return 0;
    case SOUND_PCM_WRITE_FILTER:
    case SOUND_PCM_READ_FILTER:
      return -EINVAL;
    case SOUND_PCM_NONBLOCK:
      if ( flags & PCM_LFLG_PLAY ) pcm -> chn[ PCM_PLAYBACK ].flags |= PCM_FLG_NONBLK;
      if ( flags & PCM_LFLG_RECORD ) pcm -> chn[ PCM_RECORD ].flags |= PCM_FLG_NONBLK;
      return 0;
    case SOUND_PCM_GETTRIGGER:
      return IOCTL_OUT( arg, pcm_get_trigger( pcm, flags ) );
    case SOUND_PCM_SETTRIGGER:
      return IOCTL_OUT( arg, pcm_set_trigger( pcm, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_GETOSPACE:
    case SOUND_PCM_GETISPACE:
      return pcm_get_space( pcm, flags, cmd == SOUND_PCM_GETISPACE, (audio_buf_info *)arg );
    case SOUND_PCM_GETIPTR:
    case SOUND_PCM_GETOPTR:
      return pcm_get_ptr( pcm, flags, cmd == SOUND_PCM_GETIPTR, (count_info *)arg );
    case SOUND_PCM_MAPINBUF:
    case SOUND_PCM_MAPOUTBUF:
      return pcm_get_mapbuf( pcm, flags, cmd == SOUND_PCM_MAPINBUF, (buffmem_desc *)arg );
    case SNDCTL_DSP_SETDUPLEX:
      if ( get_osscaps( pcm, flags ) & DSP_CAP_DUPLEX ) return 0;
      return -EIO;
    case SOUND_PCM_SETSYNCRO:
      /* stop DMA now.. */
      return 0;
    case SOUND_PCM_GUSINFO:
      return gus_pcm_info_flags( pcm, (struct GUS_PCM_INFO *)arg, 0 );
    case SOUND_PCM_GUSMASK:
      {
        int val;
      
        if ( pcm_file_flags( file -> f_flags ) != PCM_LFLG_BOTH )
          return -EIO;
        val = IOCTL_IN( arg );
        val &= PCM_LFLG_BOTH;
        if ( !val ) return -EINVAL;
        pcm -> mask = val;
        return 0;
      }
    default:
      PRINTK( "pcm: unknown command = 0x%x\n", cmd ); 
  }
  return -EIO;
}

int gus_read_pcm( struct file *file, char *buf, int count )
{
  gus_pcm_t *pcm;

  pcm = (gus_pcm_t *)file -> private_data;
  if ( !pcm ) return -EIO;
  if ( !( pcm_file_flags( file -> f_flags ) & PCM_LFLG_RECORD ) ||
       !( pcm -> flags & PCM_LFLG_RECORD ) ) return -EIO;
  return pcm_dma_to_user( pcm, buf, count );
}

int gus_write_pcm( struct file *file, char *buf, int count )
{
  gus_pcm_t *pcm;

  pcm = (gus_pcm_t *)file -> private_data;
  if ( !pcm ) return -EIO;
  if ( !( pcm_file_flags( file -> f_flags ) & PCM_LFLG_PLAY ) ||
       !( pcm -> flags & PCM_LFLG_PLAY ) ) return -EIO;
  return pcm_user_to_dma( pcm, buf, count );
}

#ifdef GUS_POLL
unsigned int gus_poll_pcm( struct file *file, poll_table *wait )
{
  gus_pcm_t *pcm;
  unsigned long flags;
  unsigned int mask;
  gus_pcm_channel_t *record, *playback;

  pcm = (gus_pcm_t *)file -> private_data;
  if ( !pcm ) return 0;

  record = &pcm -> chn[ PCM_RECORD ];
  playback = &pcm -> chn[ PCM_PLAYBACK ];
  CLI( &flags );
  record -> flags |= PCM_FLG_SLEEP;
  SLEEP_POLL( pcm, record, wait );
  playback -> flags |= PCM_FLG_SLEEP;
  SLEEP_POLL( pcm, playback, wait );
  STI( &flags );

  mask = 0;
  if ( record -> flags & PCM_FLG_MMAP )
    {
      if ( record -> interrupts ) mask |= POLLIN | POLLRDNORM;
    }
   else
    {
      if ( record -> used || (record -> flags & PCM_FLG_BUFFERS) ) mask |= POLLIN | POLLRDNORM;
    }
  if ( playback -> flags & PCM_FLG_MMAP )
    {
      if ( playback -> interrupts ) mask |= POLLOUT | POLLWRNORM;
    }
   else
    {
      if ( playback -> used < playback -> blocks || (playback -> flags & PCM_FLG_BUFFERS) ) mask |= POLLOUT | POLLWRNORM;
    }

  return mask;
}
#else
int gus_select_pcm( struct file *file, int sel_type, select_table *wait )
{
  int ok;
  gus_pcm_t *pcm;
  unsigned long flags;
  gus_pcm_channel_t *pchn;

  pcm = (gus_pcm_t *)file -> private_data;
  if ( !pcm ) return -EIO;
  switch ( sel_type ) {
    case SEL_IN:
      pchn = &pcm -> chn[ PCM_RECORD ];
      CLI( &flags );
      if ( pchn -> flags & PCM_FLG_MMAP )
        ok = !pchn -> interrupts; 
       else
        ok = !pchn -> used && !(pchn -> flags & PCM_FLG_BUFFERS);
      if ( ok )
        {
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEPS( pcm, record, wait );
          STI( &flags );
          return 0;
        }
      pchn -> flags &= ~PCM_FLG_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      pchn = &pcm -> chn[ PCM_PLAYBACK ];
      CLI( &flags );
      if ( pchn -> flags & PCM_FLG_MMAP )
        ok = !pchn -> interrupts; 
       else
        ok = pchn -> used >= pchn -> blocks && !(pchn -> flags & PCM_FLG_BUFFERS);
      if ( ok )
        {
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEPS( pcm, playback, wait );
          STI( &flags );
          return 0;
        }
      pchn -> flags &= ~PCM_FLG_SLEEP;
      STI( &flags );
      return 1;
    case SEL_EX:
      break;
  }
  return 0;
}
#endif

#ifdef LINUX_2_1_X
int gus_mmap_pcm( struct file *file, struct vm_area_struct *vma )
#else
int gus_mmap_pcm( struct inode *inode, struct file *file, struct vm_area_struct *vma )
#endif
{
  gus_pcm_t *pcm;
  unsigned short flags;
  gus_pcm_channel_t *pchn;
  unsigned long size;

  pcm = (gus_pcm_t *)file -> private_data;
  flags = pcm_file_flags( file -> f_flags );
  if ( ( vma -> vm_flags & ( VM_READ | VM_WRITE ) ) == ( VM_READ | VM_WRITE ) )
    return -EINVAL;
  if ( vma -> vm_flags & VM_READ )
    {
      if ( !(flags & PCM_LFLG_RECORD) ) return -EINVAL;
      pchn = &pcm -> chn[ PCM_RECORD ];
    }
   else
  if ( vma -> vm_flags & VM_WRITE )
    {
      if ( !(flags & PCM_LFLG_PLAY) ) return -EINVAL;
      pchn = &pcm -> chn[ PCM_PLAYBACK ];
    }
   else
    return -EINVAL;
  if ( vma -> vm_offset != 0 ) return -EINVAL;
  size = vma -> vm_end - vma -> vm_start;
  if ( size != pchn -> size )
    printk( "pcm: wrong mmap() size 0x%x, should be 0x%x\n", (unsigned int)size, pchn -> size );
#if 0
  printk( "remap: to=0x%x from=0x%x size=0x%x\n", (int)vma -> vm_start, (int)virt_to_bus( pchn -> buffer ), (int)size );
#endif
  if ( remap_page_range( vma -> vm_start,
                         virt_to_bus( pchn -> buffer ), 
                         size,
                         vma -> vm_page_prot ) )
    return -EAGAIN;
#if LinuxVersionCode( 2, 1, 45 ) > LINUX_VERSION_CODE
  vma -> vm_inode = inode;
#if LinuxVersionCode( 2, 1, 43 ) > LINUX_VERSION_CODE
  inode -> i_count++;
#else
  atomic_inc( &inode -> i_count );
#endif
#else
  vma -> vm_dentry = dget( file -> f_dentry );
#endif
  pchn -> _dma -> mmaped = 1;
  pchn -> flags |= PCM_FLG_MMAP;
  pcm_fill_with_neutral( pchn );
  return 0;
}

void gus_pcm_info( gus_card_t *card, gus_info_buffer_t *buffer )
{
  int i;
  gus_pcm_t *pcm;

  for ( i = 0; i < 2; i++ ) {
    pcm = &card -> pcm[ i ];
    gus_iprintf( buffer, "PCM [%s]:\n", pcm -> info_name );
    gus_iprintf( buffer, "  realtime buffering : %s\n", pcm -> flags & PCM_LFLG_REALTIME ? "on" : "off" );
    gus_iprintf( buffer, "  playback underflow : %i\n", pcm -> chn[ PCM_PLAYBACK ].discarded );
    gus_iprintf( buffer, "  record overflow    : %i\n", pcm -> chn[ PCM_RECORD ].discarded );
  }
}
