/*
 * Copyright (c) 1994-97 by Jaroslav Kysela (Perex soft)
 */

#ifndef __GUS_H
#define __GUS_H

#if defined( LINUX ) || defined( __LINUX__ )
#include <asm/byteorder.h>
#include <linux/ioctl.h>
#endif

#define GUS_CARDS		8	/* number of supported GUS cards */

/*
 *  section for direct support of GF1 synthesizer - /dev/gus
 */

#define GUS_SYNTH_VERSION	0x0710
	/* current version of protocol - 7.10; main mask is 0xff00 */
	/* changes in low 8 bites -> protocols are backwards compatible */
	/* high 8. bites -> protocols are totaly uncompatible */

/* some typedef's */

typedef struct GUS_STRU_RESET gus_reset_t;
typedef struct GUS_STRU_INFO gus_info_t;
typedef struct GUS_STRU_ULTRACLICK gus_ultraclick_t;
typedef struct GUS_STRU_WAVE gus_wave_t;
typedef struct GUS_STRU_LAYER gus_layer_t;
typedef struct GUS_STRU_INSTRUMENT gus_instrument_t;
typedef struct GUS_STRU_INSTRUMENT_NAME gus_instrument_name_t;
typedef struct GUS_STRU_MEMORY_BLOCK gus_memory_block_t;
typedef struct GUS_STRU_MEMORY_LIST gus_memory_list_t;
typedef struct GUS_STRU_MEMORY_DUMP gus_memory_dump_t;
typedef struct GUS_STRU_EFFECT gus_effect_t;
typedef struct GUS_STRU_MIDI_DEVICES gus_midi_devices_t;
typedef struct GUS_STRU_MIDI_DEVICE_INFO gus_midi_device_info_t;
typedef struct GUS_STRU_MIDI_THRU gus_midi_thru_t;

/* commands over ioctl function - control */
 
#define GUS_IOCTL_CARDS		_IOR ( 'g', 0x00, int )
			/* return number of installed cards */
#define GUS_IOCTL_VERSION	_IOR ( 'g', 0x01, int )
			/* return version of protocol (current GUS_SYNTH_VERSION) */
#define GUS_IOCTL_RESET0	_IOR ( 'g', 0x02, struct GUS_STRU_RESET )
			/* reset synth engine only - not GF1 */
#define GUS_IOCTL_RESET		_IOWR( 'g', 0x02, struct GUS_STRU_RESET )
			/* first arg is number of active voices 14-32 */
#define GUS_IOCTL_INFO		_IOR ( 'g', 0x03, struct GUS_STRU_INFO )
			/* arg is ptr to GUS_STRU_INFO */

/* commands over ioctl function - setup */

#define GUS_IOCTL_GET_ULTRACLICK _IOR ( 'g', 0x20, struct GUS_STRU_ULTRACLICK )
#define GUS_IOCTL_SET_ULTRACLICK _IOW ( 'g', 0x20, struct GUS_STRU_ULTRACLICK )
#define GUS_IOCTL_GET_MIDI_EMUL	_IOR ( 'g', 0x21, int )
#define GUS_IOCTL_SET_MIDI_EMUL	_IOW ( 'g', 0x21, int )
#define GUS_IOCTL_EFFECT_RESET	_IO  ( 'g', 0x22 )
#define GUS_IOCTL_EFFECT_SETUP	_IOW ( 'g', 0x22, struct GUS_STRU_EFFECT )

/* commands over ioctl function - timer */

#define GUS_IOCTL_TIMER_START	_IO  ( 'g', 0x40 )
			/* arg is count in one us */
#define GUS_IOCTL_TIMER_STOP	_IO  ( 'g', 0x41 )
			/* stop timer */
#define GUS_IOCTL_TIMER_CONTINUE _IO ( 'g', 0x42 )
#define GUS_IOCTL_TIMER_BASE	_IOW ( 'g', 0x43, int )
			/* set time base */
#define GUS_IOCTL_TIMER_TEMPO	_IOW ( 'g', 0x44, int )
			/* set timer tempo */
#define GUS_IOCTL_TIMER_MASTER	_IO  ( 'g', 0x45 )
			/* mark this card as master */
#define GUS_IOCTL_TIMER_SLAVE	_IOW ( 'g', 0x46, int )
			/* mark this card as slave to specified card */

/* commands over ioctl function - memory */

#define GUS_IOCTL_MEMORY_RESET	_IOW  ( 'g', 0x60, int )
			/* reset GUS's memory manager */
#define GUS_IOCTL_MEMORY_TEST	_IOW ( 'g', 0x61, struct GUS_STRU_INSTRUMENT )
			/* arg is ptr to GUS_STRU_INSTRUMENT - test if instrument can be allocated */
#define GUS_IOCTL_MEMORY_ALLOC	_IOW ( 'g', 0x62, struct GUS_STRU_INSTRUMENT )
			/* arg is ptr to GUS_STRU_INSTRUMENT - allocate instrument */
#define GUS_IOCTL_MEMORY_FREE	_IOW ( 'g', 0x63, struct GUS_STRU_INSTRUMENT )
			/* arg is ptr to GUS_STRU_INSTRUMENT - free instrument */
#define GUS_IOCTL_MEMORY_PACK	_IO  ( 'g', 0x64 )
			/* pack waves in GUS's memory */
#define GUS_IOCTL_MEMORY_BALLOC	_IOW ( 'g', 0x65, struct GUS_STRU_MEMORY_BLOCK )
			/* allocate GUS's memory block */
#define GUS_IOCTL_MEMORY_BFREE	_IOW ( 'g', 0x66, struct GUS_STRU_MEMORY_BLOCK )
			/* free GUS's memory block */
#define GUS_IOCTL_MEMORY_GET_NAME _IOR ( 'g', 0x67, struct GUS_STRU_INSTRUMENT_NAME )
			/* arg is ptr to GUS_STRU_INSTRUMENT_NAME */
#define GUS_IOCTL_MEMORY_LIST	_IOWR( 'g', 0x67, struct GUS_STRU_MEMORY_LIST )

#define GUS_IOCTL_MEMORY_DUMP	_IOW ( 'g', 0x70, struct GUS_STRU_MEMORY_DUMP )
			/* arg is ptr to GUS_STRU_MEMORY_DUMP */

/* commands over ioctl function - command queue */

#define GUS_IOCTL_WQUEUE_SET_SIZE _IOW ( 'g', 0x80, int )
			/* set write queue size in items (records) */
#define GUS_IOCTL_WQUEUE_GET_SIZE _IOR ( 'g', 0x80, int )
			/* get write queue size in items (records) */
#define GUS_IOCTL_WQUEUE_FREE	_IOR ( 'g', 0x81, int )
			/* get free write queue items (records) */
#define GUS_IOCTL_WQUEUE_THRESHOLD _IOW ( 'g', 0x82, int )
			/* set min wait free queue items to wakeup */

#define GUS_IOCTL_RQUEUE_SET_SIZE _IOW ( 'g', 0x88, int )
			/* set read queue size in items (records) */
#define GUS_IOCTL_RQUEUE_GET_SIZE _IOR ( 'g', 0x88, int )
			/* get read queue size in items (records) */
#define GUS_IOCTL_RQUEUE_USED	_IOR ('g', 0x89, int )
			/* get used read queue items (records) */

#define GUS_IOCTL_FLUSH		_IO  ( 'g', 0x90 )
			/* flush command queue */
#define GUS_IOCTL_ABORT		_IO  ( 'g', 0x91 )
			/* abort command queue and stop gf1 */
#define GUS_IOCTL_QABORT	_IO  ( 'g', 0x92 )
			/* abort only command queue to command STOP */
#define GUS_IOCTL_STOP		_IO  ( 'g', 0x93 )
			/* stop playing */
#define GUS_IOCTL_CONTINUE	_IO  ( 'g', 0x94 )
			/* continue playing */

/* MIDI emulation */

#define GUS_MIDI_EMUL_GM	0x0000
#define GUS_MIDI_EMUL_GS	0x0001
#define GUS_MIDI_EMUL_MT32	0x0002
#define GUS_MIDI_EMUL_AUTO	0x00ff

/*
 *  commands in read/write queues
 *
 *  format of queue (one record have 8 bytes):
 *
 *    0. byte		command		see GUS_CMD_XXXX constants
 *    1. byte		voice or channel #
 *    2.-7. byte	arguments 
 */

/*
 *  first level - simple voice commands for MOD players
 */

/* voice stop controllers */
#define GUS_STOP_NOW		0x00
#define GUS_STOP_LOOP		0x01
#define GUS_STOP_ENVELOPE	0x02
/* frequency controllers */
#define GUS_FREQ_HZ		0x00000000
#define GUS_FREQ_NOTE		0x40000000
#define GUS_FREQ_PITCHBEND	0x80000000
#define GUS_FREQ_OSSNOTE	0xc0000000
#define GUS_FREQ_MASK		0xc0000000
#define GUS_FREQ_DATA		(unsigned int)(~GUS_FREQ_MASK)
/* volume controllers */
#define GUS_VOLUME_LINEAR	0x0000
#define GUS_VOLUME_MIDI		0x4000
#define GUS_VOLUME_EXP		0x8000
#define GUS_VOLUME_MASK		0xc000
#define GUS_VOLUME_DATA		(unsigned short)(~GUS_VOLUME_MASK)
/* pan controllers */
#define GUS_PAN_LINEAR		0x0000
#define GUS_PAN_MASK		0xc000
#define GUS_PAN_DATA		(unsigned short)(~GUS_PAN_MASK)
/* lfo commands */
#define GUS_LFO_SETUP		0x00
#define GUS_LFO_FREQ		0x01
#define GUS_LFO_DEPTH		0x02
#define GUS_LFO_ENABLE		0x03
#define GUS_LFO_DISABLE		0x04
#define GUS_LFO_SHUTDOWN	0x05
/* private1 subcommands */
#define GUS_PRIV1_IW_EFFECT	0x00

#define GUS_CMD_VOICE_PROGRAM	0x00	/* change voice program */
#define GUS_CMD_VOICE_START	0x01	/* voice start */
#define GUS_CMD_VOICE_STOP	0x02	/* voice stop */
#define GUS_CMD_VOICE_CONTROL	0x03	/* voice control */
#define GUS_CMD_VOICE_FREQ	0x04	/* voice frequency */
#define GUS_CMD_VOICE_VOLUME	0x05	/* voice volume */
#define GUS_CMD_VOICE_SLOOP	0x06	/* voice loop start in bytes * 16 */
#define GUS_CMD_VOICE_ELOOP	0x07	/* voice loop end in bytes * 16 */
#define GUS_CMD_VOICE_RAMP	0x08	/* voice ramp range & control */
#define GUS_CMD_VOICE_POS	0x09	/* voice position in bytes * 16 (lowest 4 bits - fraction) */
#define GUS_CMD_VOICE_PAN	0x0a	/* voice pan (balance) */
#define GUS_CMD_VOICE_LFO	0x0b	/* voice lfo setup */
#define GUS_CMD_VOICE_PRIVATE1	0x20	/* voice private1 command */

/*
 *  second level - channel commands
 */

/* midi controllers - by MMA */
#define GUS_MCTL_MSB_BANK		0x00	/* implemented */
#define GUS_MCTL_MSB_MODWHEEL           0x01	/* implemented */
#define GUS_MCTL_MSB_BREATH             0x02
#define GUS_MCTL_MSB_FOOT               0x04
#define GUS_MCTL_MSB_PORTNAMENTO_TIME   0x05
#define GUS_MCTL_MSB_DATA_ENTRY		0x06	/* implemented */
#define GUS_MCTL_MSB_MAIN_VOLUME        0x07	/* implemented */
#define GUS_MCTL_MSB_BALANCE            0x08
#define GUS_MCTL_MSB_PAN                0x0a	/* implemented */
#define GUS_MCTL_MSB_EXPRESSION         0x0b	/* implemented */
#define GUS_MCTL_MSB_EFFECT1		0x0c
#define GUS_MCTL_MSB_EFFECT2		0x0d
#define GUS_MCTL_MSB_GENERAL_PURPOSE1   0x10
#define GUS_MCTL_MSB_GENERAL_PURPOSE2   0x11
#define GUS_MCTL_MSB_GENERAL_PURPOSE3   0x12
#define GUS_MCTL_MSB_GENERAL_PURPOSE4   0x13
#define GUS_MCTL_LSB_BANK		0x20	/* implemented */
#define GUS_MCTL_LSB_MODWHEEL           0x21	/* ignored */
#define GUS_MCTL_LSB_BREATH             0x22
#define GUS_MCTL_LSB_FOOT               0x24
#define GUS_MCTL_LSB_PORTNAMENTO_TIME   0x25
#define GUS_MCTL_LSB_DATA_ENTRY		0x26	/* ignored */
#define GUS_MCTL_LSB_MAIN_VOLUME        0x27	/* ignored */
#define GUS_MCTL_LSB_BALANCE            0x28
#define GUS_MCTL_LSB_PAN                0x2a	/* ignored */
#define GUS_MCTL_LSB_EXPRESSION         0x2b	/* ignored */
#define GUS_MCTL_LSB_EFFECT1		0x2c
#define GUS_MCTL_LSB_EFFECT2		0x2d
#define GUS_MCTL_LSB_GENERAL_PURPOSE1   0x30
#define GUS_MCTL_LSB_GENERAL_PURPOSE2   0x31
#define GUS_MCTL_LSB_GENERAL_PURPOSE3   0x32
#define GUS_MCTL_LSB_GENERAL_PURPOSE4   0x33
#define GUS_MCTL_SUSTAIN                0x40	/* implemented */
#define GUS_MCTL_PORNAMENTO             0x41
#define GUS_MCTL_SOSTENUTO              0x42
#define GUS_MCTL_SOFT_PEDAL             0x43
#define GUS_MCTL_LEGATO_FOOTSWITCH	0x44
#define GUS_MCTL_HOLD2                  0x45
#define GUS_MCTL_SC1_SOUND_VARIATION	0x46
#define GUS_MCTL_SC2_TIMBRE		0x47
#define GUS_MCTL_SC3_RELEASE_TIME	0x48
#define GUS_MCTL_SC4_ATTACK_TIME	0x49
#define GUS_MCTL_SC5_BRIGHTNESS		0x4a
#define GUS_MCTL_SC6			0x4b
#define GUS_MCTL_SC7			0x4c
#define GUS_MCTL_SC8			0x4d
#define GUS_MCTL_SC9			0x4e
#define GUS_MCTL_SC10			0x4f
#define GUS_MCTL_GENERAL_PURPOSE5       0x50
#define GUS_MCTL_GENERAL_PURPOSE6       0x51
#define GUS_MCTL_GENERAL_PURPOSE7       0x52
#define GUS_MCTL_GENERAL_PURPOSE8       0x53
#define GUS_MCTL_PORNAMENTO_CONTROL	0x54
#define GUS_MCTL_E1_REVERB_DEPTH	0x5b	/* implemented - InterWave only */
#define GUS_MCTL_E2_TREMOLO_DEPTH	0x5c
#define GUS_MCTL_E3_CHORUS_DEPTH	0x5d	/* implemented - InterWave only */
#define GUS_MCTL_E4_DETUNE_DEPTH	0x5e
#define GUS_MCTL_E5_PHASER_DEPTH	0x5f
#define GUS_MCTL_DATA_INCREMENT         0x60	/* implemented */
#define GUS_MCTL_DATA_DECREMENT         0x61	/* implemented */
#define GUS_MCTL_NONREG_PARM_NUM_LSB    0x62	/* implemented */
#define GUS_MCTL_NONREG_PARM_NUM_MSB    0x63	/* implemented */
#define GUS_MCTL_REGIST_PARM_NUM_LSB    0x64	/* implemented */
#define GUS_MCTL_REGIST_PARM_NUM_MSB    0x65	/* implemented */
#define GUS_MCTL_ALL_SOUNDS_OFF		0x78	/* implemented */
#define GUS_MCTL_RESET_CONTROLLERS	0x79	/* implemented */
#define GUS_MCTL_LOCAL_CONTROL_SWITCH	0x7a
#define GUS_MCTL_ALL_NOTES_OFF		0x7b	/* implemented */
#define GUS_MCTL_OMNI_OFF		0x7c	/* same as all notes off */
#define GUS_MCTL_OMNI_ON		0x7d	/* same as all notes off */
#define GUS_MCTL_MONO1			0x7e	/* same as all notes off */
#define GUS_MCTL_MONO2			0x7f	/* same as all notes off */
/* extended controllers (only for GUS driver) */
#define GUS_MCTL_TREMOLO		0x80
#define GUS_MCTL_VIBRATO		0x81
#define GUS_MCTL_LEGATO			0x82

#define GUS_CMD_CHN_PROGRAM	0x40	/* change program */
#define GUS_CMD_CHN_NOTE_ON	0x41	/* note on */
#define GUS_CMD_CHN_NOTE_OFF	0x42	/* note off */
#define GUS_CMD_CHN_PITCHBEND	0x43	/* pitch bend change */
#define GUS_CMD_CHN_BALANCE	0x45	/* change balance */
#define GUS_CMD_CHN_CONTROL	0x46	/* channel control */
#define GUS_CMD_CHN_NOTE_PRESSURE 0x47	/* note pressure */
#define GUS_CMD_CHN_PRESSURE	0x48	/* channel pressure */

/* 0x80-0xff - second byte is extended argument, not voice */

#define GUS_CMD_WAIT		0x80	/* wait x us (uint) 1-(uint max) */
#define GUS_CMD_STOP		0x81	/* see GUS_IOCTL_ABORT ioctl */
#define GUS_CMD_TEMPO		0x82	/* set tempo */

/* 0xf0-0xff - input commands */

#define GUS_CMD_ECHO		0xf0	/* this will be send to file input - read() function */
#define GUS_CMD_VOICES_CHANGE	0xf1	/* active voices is changed */

/* defines for GUS_IOCTL_MEMORY_RESET */

#define GUS_DOWNLOAD_MODE_NORMAL 0x0000
#define GUS_DOWNLOAD_MODE_TEST	 0x0001

/* structures */

struct GUS_STRU_RESET {
  unsigned int voices;	  	/* active voices */
  unsigned int channel_voices;	/* bitmap of voices used for channel commands (dynamicaly allocated) */
};

#define GUS_CARD_VERSION_CLASSIC	0x0024		/* revision 2.4 */
#define GUS_CARD_VERSION_CLASSIC1	0x0034		/* revision 3.4? */
#define GUS_CARD_VERSION_CLASSIC_ICS	0x0037		/* revision 3.7 (ICS mixer) */
#define GUS_CARD_VERSION_EXTREME	0x0050		/* GUS Extreme */
#define GUS_CARD_VERSION_ACE		0x0090		/* GUS ACE */
#define GUS_CARD_VERSION_MAX		0x00a0		/* GUS MAX - revision 10 */
#define GUS_CARD_VERSION_MAX1		0x00a1		/* GUS MAX - revision 11 */
#define GUS_CARD_VERSION_PNP		0x0100		/* GUS Plug & Play */
#define GUS_CARD_VERSION_DYNASONIC	0x0101		/* DynaSonic 3-D */

#define GUS_STRU_INFO_F_DB16		0x00000001	/* 16-bit daughter board present */
#define GUS_STRU_INFO_F_PCM		0x00000004	/* GF1 PCM during SYNTH enabled */
#define GUS_STRU_INFO_F_ENHANCED	0x00000008	/* InterWave - enhanced mode */
#define GUS_STRU_INFO_F_DAEMON		0x00000010	/* instrument daemon is present */

struct GUS_STRU_INFO {
  unsigned char id[ 8 ];	  /* id of this card (warning! maybe unterminated!!!) */ 

  unsigned int flags;		  /* some info flags - see to GUS_STRU_INFO_F_XXXX */
  unsigned int version;		  /* see to GUS_CARD_VERSION_XXXX constants */

  unsigned short port;
  unsigned short irq;
  unsigned short dma1;		  /* DMA1 - GF1 download & codec record */
  unsigned short dma2;		  /* DMA2 - GF1 record & codec playback */

  unsigned int mixing_freq;	  /* mixing frequency in Hz */

  unsigned int memory_size;	  /* in bytes */
  unsigned int memory_free;	  /* in bytes */
  unsigned int memory_block_8; 	  /* largest free 8-bit block in memory */
  unsigned int memory_block_16;	  /* largest free 16-bit block in memory */
  struct {
    unsigned int size;		  /* size of this bank in bytes */
    unsigned int address;	  /* start address of this bank */
  } memory_banks[ 4 ];
  unsigned int memory_rom_size;	  /* in bytes */
  unsigned int memory_rom_present; /* bitmask */
  unsigned int memory_rom_banks;  /* count of rom banks (all have same size) */
};

struct GUS_STRU_ULTRACLICK {
  unsigned int smooth_pan: 1,     /* 1 = enable, 0 = disable */
  	       full_range_pan: 1, /* 1 = enable, 0 = disable */
  	       volume_ramp: 8;	  /* 0 = disable, 1-255 */
};

/* bits for format variable in GUS_STRU_WAVE */

#define GUS_WAVE_16BIT		0x01	/* 16-bit wave */
#define GUS_WAVE_UNSIGNED	0x02	/* unsigned wave */
#define GUS_WAVE_INVERT		0x02	/* same as unsigned wave */
#define GUS_WAVE_BACKWARD	0x04	/* forward mode */
#define GUS_WAVE_LOOP		0x08	/* loop mode */
#define GUS_WAVE_BIDIR		0x10	/* bidirectional mode */
#define GUS_WAVE_ULAW		0x20	/* uLaw compressed wave */
#define GUS_WAVE_RAM		0x40	/* wave is _preloaded_ in RAM (it is used for ROM simulation) */
#define GUS_WAVE_ROM		0x80	/* wave is in ROM */

#define GUS_WAVE_PATCH_ENVELOPE	0x01	/* envelopes on */
#define GUS_WAVE_PATCH_SUSTAIN	0x02	/* sustain mode */

struct GUS_STRU_WAVE {
  unsigned int share_id1;		/* share id (part 1) - 0 = no sharing enabled */
  unsigned int share_id2;		/* share id (part 2) - 0 = no sharing enabled */

  unsigned char mode;			/* see to GUS_INSTR_XXXX constants */
  unsigned char format;			/* see to GUS_WAVE_XXXX constants */
  unsigned int size;			/* size of waveform in bytes */
  unsigned int start;			/* start offset in bytes * 16 (lowest 4 bits - fraction) */
  unsigned int loop_start;		/* bits loop start offset in bytes * 16 (lowest 4 bits - fraction) */
  unsigned int loop_end;		/* loop start offset in bytes * 16 (lowest 4 bits - fraction) */
  unsigned short loop_repeat;		/* loop repeat - 0 = forever */
  struct {
    unsigned int memory;		/* begin of waveform in GUS's memory */
    unsigned char *ptr;			/* pointer to waveform in system memory */
  } begin;

  union {
    struct {
      unsigned char flags;
      unsigned int sample_rate;
      unsigned int low_frequency;	/* low frequency range for this waveform */
      unsigned int high_frequency;	/* high frequency range for this waveform */
      unsigned int root_frequency;	/* root frequency for this waveform */
      signed short tune;
      unsigned char balance;
      unsigned char envelope_rate[ 6 ];
      unsigned char envelope_offset[ 6 ];
      unsigned char tremolo_sweep;
      unsigned char tremolo_rate;
      unsigned char tremolo_depth;
      unsigned char vibrato_sweep;
      unsigned char vibrato_rate;
      unsigned char vibrato_depth;
      unsigned short scale_frequency;
      unsigned short scale_factor;	/* 0-2048 or 0-2 */
    } patch;
    struct {
      unsigned int sample_ratio;	/* sample ratio (44100 * 1024 / rate) */
      unsigned char attenuation;	/* 0 - 127 (no corresponding midi controller) */
      unsigned char low_note;		/* lower frequency range for this waveform */
      unsigned char high_note;		/* highter frequency range for this waveform */
    } iw;
  } data;
  
  struct GUS_STRU_WAVE *next;
};

#define GUS_STRU_IW_LFO_SHAPE_TRIANGLE		0
#define GUS_STRU_IW_LFO_SHAPE_POSTRIANGLE	1

struct GUS_STRU_IW_LFO {
  unsigned short freq;		/* (0-2047) 0.01Hz - 21.5Hz */
  signed short depth;		/* volume +- (0-255) 0.48675dB/step */
  signed short sweep;		/* 0 - 950 deciseconds */
  unsigned char shape;		/* see to GUS_STRU_IW_LFO_SHAPE_XXXX */
  unsigned char delay;		/* 0 - 255 deciseconds */
};

/* struct GUS_STRU_IW_ENV - flags */

#define GUS_STRU_IW_ENV_F_RETRIGGER	0x0001	/* flags - retrigger */

#define GUS_STRU_IW_ENV_M_ONE_SHOT	0x0001	/* mode - one shot */
#define GUS_STRU_IW_ENV_M_SUSTAIN	0x0002	/* mode - sustain */
#define GUS_STRU_IW_ENV_M_NO_SUSTAIN	0x0003	/* mode - no sustain */

#define GUS_STRU_IW_ENV_I_VELOCITY	0x0001	/* index - velocity */
#define GUS_STRU_IW_ENV_I_FREQUENCY	0x0002	/* index - frequency */

struct GUS_STRU_IW_ENV_RECORD {
  unsigned short nattack;
  unsigned short nrelease;
  unsigned short sustain_offset;
  unsigned short sustain_rate;
  unsigned short release_rate;
  unsigned char hirange;
  struct {
    unsigned short offset;
    unsigned short rate;
  } *points;			/* count of points = nattack + nrelease */
  struct GUS_STRU_IW_ENV_RECORD *next;
};

struct GUS_STRU_IW_ENV {
  unsigned char flags: 8,
  		mode: 4,
  		index: 4;
  struct GUS_STRU_IW_ENV_RECORD *record;
};

/* strcut GUS_STRU_LAYER - data.iw.flags */

#define GUS_LAYER_IW_F_RETRIGGER	0x0001	/* retrigger */

#define GUS_LAYER_IW_V_VEL_TIME		0x0000	/* velocity mode = time */
#define GUS_LAYER_IW_V_VEL_RATE		0x0001	/* velocity mode = rate */

#define GUS_LAYER_IW_E_KUP		0x0000	/* layer event - key up */
#define GUS_LAYER_IW_E_KDOWN		0x0001	/* layer event - key down */
#define GUS_LAYER_IW_E_RETRIG		0x0002	/* layer event - retrigger */
#define GUS_LAYER_IW_E_LEGATO		0x0003	/* layer event - legato */

struct GUS_STRU_LAYER {
  unsigned char mode;			/* see to GUS_INSTR_XXXX constants */

  union {
    struct {
      unsigned short flags: 8,		/* see to GUS_LAYER_IW_F_XXXX */
      		     velocity_mode: 4,	/* see to GUS_LAYER_IW_V_XXXX */
      		     layer_event: 4;	/* see to GUS_LAYER_IW_E_XXXX */
      unsigned char low_range;		/* range for layer based */
      unsigned char high_range;		/* on either velocity or frequency */
      unsigned char pan;		/* pan offset from CC1 (0 left - 127 right) */
      unsigned char pan_freq_scale;	/* position based on frequency (0-127) */
      struct GUS_STRU_IW_LFO tremolo;	/* tremolo effect */
      struct GUS_STRU_IW_LFO vibrato;	/* vibrato effect */
      unsigned char attenuation;	/* 0-127 (no corresponding midi controller) */
      unsigned short freq_scale;	/* 0-2048, 1024 is equal to semitone scaling */
      unsigned char freq_center;	/* center for keyboard frequency scaling */
      struct GUS_STRU_IW_ENV penv;	/* pitch envelope */
      struct GUS_STRU_IW_ENV venv;	/* volume envelope */
    } iw;
  } data;

  struct GUS_STRU_WAVE *wave;
  struct GUS_STRU_LAYER *next;
};

/* struct GUS_STRU_INSTRUMENTS - mode */

#define GUS_INSTR_SIMPLE	0x00	/* simple format - for MOD players */
#define GUS_INSTR_PATCH		0x01	/* old GF1 patch format */
#define GUS_INSTR_INTERWAVE	0x02	/* new InterWave format */
#define GUS_INSTR_COUNT		3

/* struct GUS_STRU_INSTRUMENTS - flags */

#define GUS_INSTR_F_NORMAL	0x0000	/* normal mode */
#define GUS_INSTR_F_NOT_FOUND	0x0001	/* instrument cann't be loaded */
#define GUS_INSTR_F_ALIAS	0x0002	/* alias */
#define GUS_INSTR_F_NOT_LOADED	0x00ff	/* instrument not loaded (not found) */

#define GUS_INSTR_E_NONE	0x0000	/* exclusion mode - none */
#define GUS_INSTR_E_SINGLE	0x0001	/* exclude single - single note from this instrument */
#define GUS_INSTR_E_MULTIPLE	0x0002	/* exclude multiple - stop only same note from this instrument */

#define GUS_INSTR_L_NONE	0x0000	/* not layered */
#define GUS_INSTR_L_ON		0x0001	/* layered */
#define GUS_INSTR_L_VELOCITY	0x0002	/* layered by velocity */
#define GUS_INSTR_L_FREQUENCY	0x0003	/* layered by frequency */

/* struct GUS_STRU_INSTRUMENTS - data.iw.effect[12] */

#define GUS_INSTR_IE_NONE	0
#define GUS_INSTR_IE_REVERB	1
#define GUS_INSTR_IE_CHORUS	2
#define GUS_INSTR_IE_ECHO	3

struct GUS_STRU_INSTRUMENT {
  union {
    struct {
#ifdef __LITTLE_ENDIAN
      unsigned short prog;
      unsigned short bank;
#else
      unsigned short bank;
      unsigned short prog;
#endif
    } midi;
    unsigned int instrument;		/* instrument number */
  } number;

  char *name;				/* name of this instrument or NULL */

  unsigned int mode: 8,			/* see to GUS_INSTR_XXXX */
   	       flags: 8,		/* see to GUS_INSTR_F_XXXX */
               exclusion: 4,		/* see to GUS_INSTR_E_XXXX */
               layer: 4;		/* see to GUS_INSTR_L_XXXX */
  unsigned short exclusion_group;	/* 0 - none, 1-65535 */

  union {
    struct {
      unsigned char effect1 : 4,	/* use global effect if available */
      		    effect2 : 4;	/* use global effect if available */
      unsigned char effect1_depth;	/* 0-127 */
      unsigned char effect2_depth;	/* 0-127 */
    } iw;
    struct {
      unsigned char effect1 : 4,	/* use global effect if available */
      		    effect2 : 4;	/* use global effect if available */
      unsigned char effect1_depth;	/* 0-127 */
      unsigned char effect2_depth;	/* 0-127 */
    } patch;
  } data;

  union {
    struct GUS_STRU_LAYER *layer;	/* first layer */
    struct {
#ifdef __LITTLE_ENDIAN
      unsigned short prog;
      unsigned short bank;
#else
      unsigned short bank;
      unsigned short prog;
#endif
    } midi_alias;
    unsigned int alias;			/* pointer to instrument */
  } info;

  gus_instrument_t *left;		/* used by driver for search speedup */
  gus_instrument_t *right;		/* used by driver for search speedup */

  gus_instrument_t *next;		/* next instrument */
};

struct GUS_STRU_INSTRUMENT_NAME {
  union {
    struct {
#ifdef __LITTLE_ENDIAN
      unsigned short prog;
      unsigned short bank;
#else
      unsigned short bank;
      unsigned short prog;
#endif
    } midi;
    unsigned int instrument;		/* instrument number */
  } number;
  char name[ 64 ];			/* name of this instrument */
  unsigned int mode: 8,			/* see to GUS_INSTR_XXXX */
  	       flags: 8;		/* see to GUS_INSTR_F_XXXX */
};

struct GUS_STRU_MEMORY_LIST {
  unsigned int count;			/* count of instruments */
  union {
    unsigned int *instruments;
    struct {
#ifdef __LITTLE_ENDIAN
      unsigned short prog;
      unsigned short bank;
#else
      unsigned short bank;
      unsigned short prog;
#endif
    } *midi;
  } info;
};

#define GUS_MEMORY_DUMP_RAM	0x0001
#define GUS_MEMORY_DUMP_ROM	0x0002
#define GUS_MEMORY_DUMP_WRITE	0x0004	/* if RAM - write block to RAM */
#define GUS_MEMORY_DUMP_INVERT	0x0008
#define GUS_MEMORY_DUMP_UNSIGNED 0x0008
#define GUS_MEMORY_DUMP_16BIT	0x0010	/* 16-bit wave */

struct GUS_STRU_MEMORY_DUMP {
  unsigned short flags;		  /* see to GUS_MEMORY_DUMP_XXXX constants */
  unsigned int address;		  /* linear address to GUS's memory */
  unsigned int size;		  /* size of transferred block */
  unsigned char *dump;		  /* pointer to destonation memory block */
};

#define GUS_MEMORY_BLOCK_LOCKED	0x0001	/* lock memory (don't free this by close()) */
#define GUS_MEMORY_BLOCK_16BIT	0x0002	/* for old GF1 chip (256k boundaries) */
#define GUS_MEMORY_BLOCK_ALREADY 0x0004	/* block is already allocated */

struct GUS_STRU_MEMORY_BLOCK {
  unsigned short flags;	          /* see to GUS_MEMORY_BLOCK_XXXX constants */
  unsigned int address;		  /* linear address in GUS's memory - if alloc - filled by driver */
  unsigned int size;		  /* size of this block */
  unsigned char id[ 256 ];	  /* identification (for lock) */
};

/* this structure is for InterWave only */

#define GUS_EFFECT_CHIP_INTERWAVE	0x0001

#define GUS_EFFECT_F_IW_USED		0x0001	/* is this voice used? */
#define GUS_EFFECT_F_IW_ALTERNATE	0x0002	/* enable alternate path for this effect processor */

struct GUS_STRU_EFFECT_INTERWAVE {
  unsigned short flags;			/* see to GUS_EFFECT_F_IW_???? */
  unsigned short number;		/* effect number */

  unsigned int buffer_addr; 		/* used by driver */
  unsigned int buffer_size; 		/* write buffer size */
  unsigned int buffer_pos;  		/* initial write position in buffer */
  unsigned short volume;		/* volume - 0-4095 */
  unsigned short effect_volume; 	/* effect volume - 0-4095 */
  unsigned short l_offset;  		/* left offset - 0-4095 */
  unsigned short r_offset;  		/* right offset - 0-4095 */

  unsigned short vibrato_control;	/* control bits + freq */
  unsigned short vibrato_depth;		/* 0-8191 */
  unsigned short tremolo_control;	/* control bits + freq */
  unsigned short tremolo_depth;		/* 0-8191 */

  unsigned char write_mask;		/* write mask for accumulators */
};

struct GUS_STRU_EFFECT {
  unsigned short chip_type;	  	/* see to GUS_EFFECT_CHIP_???? */
  union {
    struct {
      unsigned char voice_output[ 8 ];	/* voice accumulator bitmap */
      /* eight effects processors should be enough */
      struct GUS_STRU_EFFECT_INTERWAVE voices[ 8 ];
    } interwave;
  } chip;
};

/*
 *  Section for direct support of GF1 synthesizer - /dev/gusctl?
 */

#define GUS_SYNTH_CTL_VERSION	GUS_SYNTH_VERSION
	
/* commands over ioctl function - control */

#define GUS_IOCTL_CTL_CARDS		_IOR ( 'g', 0x00, int )
			/* get number of cards */
#define GUS_IOCTL_CTL_VERSION		_IOR ( 'g', 0x01, int )
			/* get version */
#define GUS_IOCTL_CTL_INFO		_IOR ( 'g', 0x01, struct GUS_STRU_INFO )
			/* get various information */
#define GUS_IOCTL_CTL_PCM_INFO		_IOWR( 'g', 0x01, struct GUS_PCM_INFO )
			/* get info about PCM, device member must be filled!! */

/*
 *
 *  MIDI interface
 *
 */

#define GUS_MIDID_UART		0x00
#define GUS_MIDID_SYNTH		0x01
#define GUS_MIDID_LAST		GUS_MIDID_SYNTH
#define GUS_MIDID_COMMON	0xff

#define GUS_MCMD_NOTE_OFF               0x80
#define GUS_MCMD_NOTE_ON                0x90
#define GUS_MCMD_NOTE_PRESSURE          0xa0
#define GUS_MCMD_CONTROL                0xb0
#define GUS_MCMD_PGM_CHANGE             0xc0
#define GUS_MCMD_CHANNEL_PRESSURE       0xd0
#define GUS_MCMD_BENDER                 0xe0

#define GUS_MCMD_COMMON_SYSEX           0xf0
#define GUS_MCMD_COMMON_MTC_QUARTER	0xf1
#define GUS_MCMD_COMMON_SONG_POS	0xf2
#define GUS_MCMD_COMMON_SONG_SELECT	0xf3
#define GUS_MCMD_COMMON_TUNE_REQUEST	0xf6
#define GUS_MCMD_COMMON_SYSEX_END       0xf7
#define GUS_MCMD_COMMON_CLOCK		0xf8	/* ignored - not passed to MIDI input queue */
#define GUS_MCMD_COMMON_START		0xfa
#define GUS_MCMD_COMMON_CONTINUE	0xfb
#define GUS_MCMD_COMMON_STOP		0xfc
#define GUS_MCMD_COMMON_SENSING		0xfe	/* ignored - not passed to MIDI input queue */
#define GUS_MCMD_COMMON_RESET		0xff	/* ignored - not passed to MIDI input queue */

#define GUS_MIDI_CARDS			_IO  ( 'g', 0x00 )
#define GUS_MIDI_DEVICE_OPEN		_IOW ( 'g', 0x01, struct GUS_STRU_MIDI_OPEN )
#define GUS_MIDI_DEVICE_CLOSE		_IO  ( 'g', 0x01 )
#define GUS_MIDI_TIMER_BASE		_IOW ( 'g', 0x02, int )
#define GUS_MIDI_TIMER_TEMPO		_IOW ( 'g', 0x03, int )
#define GUS_MIDI_TIMER_START		_IO  ( 'g', 0x04 )
#define GUS_MIDI_TIMER_STOP		_IO  ( 'g', 0x05 )
#define GUS_MIDI_TIMER_CONTINUE		_IO  ( 'g', 0x06 )
#define GUS_MIDI_THRESHOLD		_IOW ( 'g', 0x07, int )
#define GUS_MIDI_FLUSH			_IO  ( 'g', 0x08 )
#define GUS_MIDI_REALTIME		_IOW ( 'g', 0x08, struct GUS_STRU_MIDI_REALTIME )
#define GUS_MIDI_ABORT			_IO  ( 'g', 0x09 )
#define GUS_MIDI_SET_THRU		_IOW ( 'g', 0x0a, struct GUS_STRU_MIDI_THRU )
#define GUS_MIDI_GET_THRU		_IOR ( 'g', 0x0a, struct GUS_STRU_MIDI_THRU )

#define GUS_MIDI_PRELOAD_BANK		_IOW ( 'g', 0x10, struct GUS_STRU_MIDI_PRELOAD_BANK )
#define GUS_MIDI_PRELOAD_INSTRUMENTS	_IOW ( 'g', 0x11, struct GUS_STRU_MIDI_PRELOAD_INSTRUMENTS )
#define GUS_MIDI_GET_EMULATION		_IOR ( 'g', 0x12, struct GUS_STRU_MIDI_EMULATION )
#define GUS_MIDI_SET_EMULATION		_IOW ( 'g', 0x12, struct GUS_STRU_MIDI_EMULATION )

#define GUS_MIDI_MEMORY_RESET		_IOW ( 'g', 0x13, struct GUS_STRU_MIDI_MEMORY_RESET )
#define GUS_MIDI_MEMORY_TEST		_IOW ( 'g', 0x14, struct GUS_STRU_MIDI_INSTRUMENT )
#define GUS_MIDI_MEMORY_ALLOC		_IOW ( 'g', 0x15, struct GUS_STRU_MIDI_INSTRUMENT )
#define GUS_MIDI_MEMORY_FREE		_IOW ( 'g', 0x16, struct GUS_STRU_MIDI_INSTRUMENT )
#define GUS_MIDI_MEMORY_PACK		_IOW ( 'g', 0x17, struct GUS_STRU_MIDI_MEMORY_PACK )
#define GUS_MIDI_MEMORY_BALLOC		_IOW ( 'g', 0x18, struct GUS_STRU_MIDI_MEMORY_BLOCK )
#define GUS_MIDI_MEMORY_BFREE		_IOW ( 'g', 0x19, struct GUS_STRU_MIDI_MEMORY_BLOCK )
#define GUS_MIDI_MEMORY_GET_NAME	_IOR ( 'g', 0x1a, struct GUS_STRU_MIDI_INSTRUMENT_NAME )
#define GUS_MIDI_MEMORY_DUMP		_IOW ( 'g', 0x1b, struct GUS_STRU_MIDI_MEMORY_DUMP )
#define GUS_MIDI_MEMORY_LIST		_IOWR( 'g', 0x1c, struct GUS_STRU_MIDI_MEMORY_LIST )

#define GUS_MIDI_CARD_INFO		_IOR ( 'g', 0x1d, struct GUS_STRU_MIDI_CARD_INFO )

#define GUS_MIDI_EFFECT_RESET		_IOW ( 'g', 0x1e, struct GUS_STRU_MIDI_EFFECT_RESET )
#define GUS_MIDI_EFFECT_SETUP		_IOW ( 'g', 0x1f, struct GUS_STRU_MIDI_EFFECT )

#define GUS_MIDI_DEVICES		_IOR ( 'g', 0x20, struct GUS_STRU_MIDI_DEVICES )
#define GUS_MIDI_DEVICE_INFO		_IOWR( 'g', 0x21, struct GUS_STRU_MIDI_DEVICE_INFO )

/* common commands */

#define GUS_MCMD_WAIT		0x00	/* wait n-ticks (unsigned int) */
#define GUS_MCMD_TEMPO		0x01	/* set tempo (unsigned short) */
#define GUS_MCMD_TBASE		0x02	/* set time base (unsigned short) */
#define GUS_MCMD_ECHO		0x80	/* echo command */

#define GUS_MIDI_OPEN_MODE_READ		0x01
#define GUS_MIDI_OPEN_MODE_WRITE	0x02

struct GUS_STRU_MIDI_OPEN {
  unsigned short devices_used;
  struct {
    unsigned char device;
    unsigned char mode; 
  } devices[ GUS_MIDID_LAST * GUS_CARDS ];
};

struct GUS_STRU_MIDI_THRU {
  unsigned char device;			/* device which will generate MIDI THRU */
  struct {
    unsigned char device;		/* card + device - 0xff - not used */
    unsigned char channel;		/* destonation channel */
    unsigned char velocity;		/* destonation velocity - 0xff - no change */
  } routing[ 16 ][ GUS_MIDID_LAST + 1 ]; /* 16 midi channels and X midi devices */
};

struct GUS_STRU_MIDI_PRELOAD_BANK {
  unsigned char device;
  char name[ 64 ];
};

struct GUS_STRU_MIDI_PRELOAD_INSTRUMENTS {
  unsigned char device;
  unsigned short instruments_used;
  unsigned int instruments[ 64 ];
};

struct GUS_STRU_MIDI_EMULATION {
  unsigned char device;
  unsigned short emulation;
};

struct GUS_STRU_MIDI_MEMORY_RESET {
  unsigned char device;
};

struct GUS_STRU_MIDI_INSTRUMENT {
  unsigned char device;
  gus_instrument_t *instrument;
};

struct GUS_STRU_MIDI_MEMORY_PACK {
  unsigned char device;
};

struct GUS_STRU_MIDI_MEMORY_BLOCK {
  unsigned char device;
  struct GUS_STRU_MEMORY_BLOCK *block;
};

struct GUS_STRU_MIDI_INSTRUMENT_NAME {
  unsigned char device;
  struct GUS_STRU_INSTRUMENT_NAME *name;
};

struct GUS_STRU_MIDI_MEMORY_DUMP {
  unsigned char device;
  struct GUS_STRU_MEMORY_DUMP *dump;
};

struct GUS_STRU_MIDI_MEMORY_LIST {
  unsigned char device;
  struct GUS_STRU_MEMORY_LIST *list;
};

struct GUS_STRU_MIDI_CARD_INFO {
  unsigned char device;
  gus_info_t *info;
};

struct GUS_STRU_MIDI_EFFECT_RESET {
  unsigned char device;
};

struct GUS_STRU_MIDI_EFFECT {
  unsigned char device;
  struct GUS_STRU_EFFECT *effect;
};

struct GUS_STRU_MIDI_DEVICES {
  int count;
  unsigned char devices[ 256 ];
  int reserved[ 31 ];
};

#define GUS_MIDI_CAP_INPUT	0x00000001	/* input possible */
#define GUS_MIDI_CAP_OUTPUT	0x00000002	/* output possible */
#define GUS_MIDI_CAP_SYNTH	0x00000004	/* emulated MIDI */
#define GUS_MIDI_CAP_MEMORY	0x00000008	/* memory routines can be used */

struct GUS_STRU_MIDI_DEVICE_INFO {
  unsigned char device;
  unsigned char name[ 64 ];
  unsigned int cap;
  unsigned int reserved[ 31 ];
};

struct GUS_STRU_MIDI_REALTIME {
  unsigned char *buffer;
  unsigned int count;
};

/*
 *  MIDI daemon
 */

#define GUS_DAEMON_MSG_REQUEST	0x0000	/* request to daemon */
#define GUS_DAEMON_MSG_REPLY	0x0001	/* reply from daemon */

#define GUS_DAEMON_MSG_END_OF_REQUEST 0x0000	/* end of request */
#define GUS_DAEMON_MSG_PRELOAD	0x0001	/* preload specified bank */
#define GUS_DAEMON_MSG_DOWNLOAD	0x0002	/* instrument download request */
#define GUS_DAEMON_MSG_EMUL	0x0003	/* emulation change */
#define GUS_DAEMON_MSG_BFREE	0x0004	/* block free */
#define GUS_DAEMON_MSG_IWEFF	0x0005	/* interwave effects */

struct GUS_STRU_DAEMON_MESSAGE {
  unsigned short direction;
  unsigned short command;
  union {
    unsigned short emul;		/* emulation */
    unsigned char bank_name[ 64 ]; 
    unsigned int instruments[ 64 ];	/* instrument numbers */
    int iw_effect[ 2 ];			/* interwave effect numbers */
    gus_instrument_t *instrument;
    struct GUS_STRU_EFFECT *effect;	/* effect */
  } info; 
};

#define GUS_IOCTL_DMN_INFO		_IOR	( 'g', 0x00, struct GUS_STRU_INFO )
#define GUS_IOCTL_DMN_INSTRUMENT	_IOWR	( 'g', 0x01, int )
#define GUS_IOCTL_DMN_FREE		_IOR	( 'g', 0x02, int )
#define GUS_IOCTL_DMN_MEMORY_DUMP	_IOW	( 'g', 0x03, struct GUS_STRU_MEMORY_DUMP )
#define GUS_IOCTL_DMN_MEMORY_BALLOC	_IOW	( 'g', 0x04, struct GUS_STRU_MEMORY_BLOCK )
#define GUS_IOCTL_DMN_MEMORY_BFREE	_IOW	( 'g', 0x05, struct GUS_STRU_MEMORY_BLOCK )
#define GUS_IOCTL_DMN_LOCK		_IO	( 'g', 0x06 )
#define GUS_IOCTL_DMN_UNLOCK		_IO	( 'g', 0x07 )
#define GUS_IOCTL_DMN_GET_EMULATION	_IOR	( 'g', 0x08, int )

/*
 *  Compatibility section for VoxWare 3.XX.
 *  VoxWare 3.XX - Copyright (c) by Hannu Savolainen.
 *  Code for gus module driver isn't copied from VoxWare driver - 
 *  It's copyrighted by me (Jaroslav Kysela).
 */

#ifndef SOUND_VERSION
#include <linux/soundcard.h>
#include <linux/ultrasound.h>
#if SOUND_VERSION < 300
#error "Bad version of VoxWare header files... You must use VoxWare version 3.00+."
#endif
#endif

#if 0
#include <sys/voxware_soundcard.h>
#endif

/*
 *  some mixer extensions
 */

#define GUS_MIXER_S_NONE	0
#define GUS_MIXER_S_IW		1

#define GUS_MIXER_S_IW_SERIAL_NONE			0
#define GUS_MIXER_S_IW_SERIAL_DSP_TO_RECORD		1
#define GUS_MIXER_S_IW_SERIAL_DSP_TO_PLAYBACK		2
#define GUS_MIXER_S_IW_SERIAL_RECORD_TO_PLAYBACK	3
#define GUS_MIXER_S_IW_SERIAL_DSP_TO_EXTOUT		4
#define GUS_MIXER_S_IW_SERIAL_RECORD_TO_EXTOUT		5
#define GUS_MIXER_S_IW_SERIAL_EXTIN_TO_PLAYBACK_1	6
#define GUS_MIXER_S_IW_SERIAL_EXTIN_TO_PLAYBACK_2	7

struct GUS_MIXER_SPECIAL {
  unsigned int what;
  union {
    unsigned char bytes[ 32 ];
    unsigned short words[ 16 ];
    unsigned int dwords[ 8 ];
    struct {
      unsigned char serial;
    } interwave;
  } data;
};

#define SOUND_MIXER_CARDS		_IOR ( 'M', 111, int )
#define SOUND_MIXER_GUS_VERSION		_IOR ( 'M', 112, int )
#define SOUND_MIXER_MUTE_FLAG		0x40000000	/* maybe wrong */
#define SOUND_MIXER_READ_DEVS_LMUTE	_IOR ( 'M', 113, int )
#define SOUND_MIXER_WRITE_DEVS_LMUTE	_IOW ( 'M', 113, int )
#define SOUND_MIXER_READ_DEVS_RMUTE	_IOR ( 'M', 114, int )
#define SOUND_MIXER_WRITE_DEVS_RMUTE	_IOW ( 'M', 114, int )
#define SOUND_MIXER_SPECIAL_READ	_IOR ( 'M', 115, struct GUS_MIXER_SPECIAL )
#define SOUND_MIXER_SPECIAL_WRITE	_IOW ( 'M', 115, struct GUS_MIXER_SPECIAL )

/*
 *  some PCM extensions
 */

#define GUS_PCM_DEVICES			2
 
#define SOUND_PCM_CARDS			_IOR ( 'P', 240, int )
#define SOUND_PCM_GUSINFO		_IOR ( 'P', 240, struct GUS_PCM_INFO )
#define SOUND_PCM_GUSMASK		_IOW ( 'P', 240, int )

#define GUS_PCM_INFO_CODEC		0x00000001
#define GUS_PCM_INFO_DSP		GUS_PCM_INFO_CODEC
#define GUS_PCM_INFO_MMAP		0x00000002
#define GUS_PCM_INFO_PLAYBACK		0x00000100
#define GUS_PCM_INFO_RECORD		0x00000200
#define GUS_PCM_INFO_DUPLEX		0x00000400
#define GUS_PCM_INFO_DUPLEX_LIMIT	0x00000800
#define GUS_PCM_INFO_PLAYBACK_BATCH	0x00001000
#define GUS_PCM_INFO_RECORD_BATCH	0x00002000

struct GUS_PCM_INFO {
  unsigned int device;			/* device # */
  unsigned int flags;			/* see to GUS_PCM_INFO_XXXX */
  unsigned int formats_play;		/* supported formats */
  unsigned int formats_record;		/* supported formats */
  unsigned int max_rate_play;		/* maximal rate (in Hz) */
  unsigned int max_rate_record;		/* maximal rate (in Hz) */
  unsigned int max_channels_play;	/* maximal channels # */ 
  unsigned int max_channels_record;	/* maximal channels # */
  unsigned int dma_size_play;		/* playback DMA size */
  unsigned int dma_size_record;		/* record DMA size */
  unsigned char name[ 64 ];		/* name of this device */
  unsigned int reserved[ 23 ];		/* reserved for future... */
};

#define GUS_PCM_MASK_PLAYBACK		0x0001
#define GUS_PCM_MASK_RECORD		0x0002
#define GUS_PCM_MASK_DUPLEX		(GUS_PCM_MASK_PLAYBACK|GUS_PCM_MASK_RECORD)
#define GUS_PCM_MASK_BOTH		GUS_PCM_MASK_DUPLEX

/*
 *
 */
 
#endif /* __GUS_H */
