%{

/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>

#include "icfg.h"

#if 0
#define YYERROR_VERBOSE
#endif

#define DEF_FLAGS_IW_ROM_PROG		0x00000001
#define DEF_FLAGS_IW_ROM_BANK		0x00000002
#define DEF_FLAGS_IW_FILE_PROG		0x00000004
#define DEF_FLAGS_IW_FILE_BANK		0x00000008

#undef yyerror
#define yyerror(args...)		{gus_icfg_parser_abort();gus_icfg_error( 0, ## args );}

#if 0
#define DEBUG_PARSER
#endif
#ifdef DEBUG_PARSER
#define dprintf(args...)	printf(## args )
#else
#define dprintf(args...)	/* nothing */
#endif

	/* insgus_lexer.c */

int yylex( void );

extern int line_count;
extern FILE *yyin;
	
	/* local functions */

void gus_icfg_parser_abort( void );

static void condition_go( int true );
static void condition_end( void );

static int file_exist( char *file );
static int patch_exist( char *patch );
static int rom_exist( int bank, char *name );

static void source_init( void );
static void source( char *source, char *description );
static void source_option( char *source, char *option );
static void preload_source_init( void );
static void preload_source( char *source, char *description );
static void preload_source_option( char *source, char *option );
static void external_init( void );
static void external( char *source, char *description );
static void external_option( char *source, char *option );

static void melodic_start( int number, char *name );
static void melodic_stop( void );
static void melodic_item( int bank, int prog, int voices, char *name );

static void drums_start( int number, char *name );
static void drums_stop( void );
static void drums_item( int bank, int prog, int voices, char *name );

static void iw_ffff_rom( int number, char *id, int bank, int file );
static void iw_ffff_file( int number, char *fff, char *dat );
static void iw_ffff_start( int number, int preload );
static void iw_ffff_stop( void );
static void iw_ffff_item( int bank, int prog );
static void iw_ffff_range( int bank, int min_prog, int max_prog );
static void iw_ffff_option( char *option );

static void patches_start( char *dir, int preload );
static void patches_stop( void );
static void patches_item( int bank, int prog, char *file );
static void patches_item_options( int bank, int prog, char *file, char *options );
static void patches_option( char *option );
static void patches_preload_item( int bank, int prog );
static void patches_preload_range( int bank, int min_prog, int max_prog );

static void alias( int src_bank, int src_prog, int dst_bank, int dst_prog );
static void alias_range( int src_bank, int src_min_prog, int src_max_prog, int dst_bank, int dst_prog );

static void preload_start( char *name );
static void preload_stop( void );

%}

%start lines

%union {
    int i_value;
    char *s_value;
  };

%token <i_value> L_INTEGER L_TRUE L_FALSE L_BOOL
%token <s_value> L_STRING L_ETC_DIR L_LIB_DIR L_INS_DIR
%token <s_value> L_VARIABLE_ASSIGN L_LOCAL_VARIABLE_ASSIGN

%token L_VARNONE
%token L_LE L_LEQ L_GT L_GTE L_EQ L_NEQ L_OR L_AND
%token L_ASSIGN L_UNSET

%token L_IF L_IFE L_ELSE

%token L_MESSAGE L_ABORT L_INCLUDE
%token L_FILE_EXIST L_PATCH_EXIST L_ROM_EXIST

%token L_POLYPHONY

%token L_SOURCE_INIT L_SOURCE L_SOURCE_OPTION
%token L_PRELOAD_SOURCE_INIT L_PRELOAD_SOURCE L_PRELOAD_SOURCE_OPTION
%token L_EXTERNAL_INIT L_EXTERNAL L_EXTERNAL_OPTION

%token L_EXIT

%token L_IW_FFFF_ROM L_IW_FFFF_FILE L_IW_FFFF L_PATCHES

%token L_ITEM L_ITEMGM L_ITEMGS L_ITEMMT L_RANGE L_RANGEGM L_RANGEGS L_RANGEMT
%token L_ALIAS L_ALIASGM L_ALIASGS L_ALIASMT
%token L_ALIAS_RANGE L_ALIAS_RANGEGM L_ALIAS_RANGEGS L_ALIAS_RANGEMT

%token L_MELODIC L_DRUMS

%token L_PRELOAD L_OPTION

%type <i_value> bool integer
%type <s_value> string

%left L_LE L_NEQ L_GT L_GTQ
%left L_OR L_AND
%left '-' '+'
%left '*' '/'
%left '&' '|'
%left NEG
%left L_RSHIFT L_LSHIFT

%%

lines	: line
	| lines line
	| keyword
	| lines keyword
	;

	/*
         *  main
	 */

line	: L_POLYPHONY '(' integer ')' ';' { ; }
	| L_SOURCE_INIT '(' ')' ';' { if ( gus_icfg_config -> condition_true ) source_init(); }
	| L_SOURCE '(' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) source( $3, $5 ); else { free( $3 ); free( $5 ); } }
	| L_SOURCE_OPTION '(' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) source_option( $3, $5 ); else { free( $3 ); free( $5 ); } }
	| L_PRELOAD_SOURCE_INIT '(' ')' ';' { if ( gus_icfg_config -> condition_true ) preload_source_init(); }
	| L_PRELOAD_SOURCE '(' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) preload_source( $3, $5 ); else { free( $3 ); free( $5 ); } }
	| L_PRELOAD_SOURCE_OPTION '(' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) preload_source_option( $3, $5 ); else { free( $3 ); free( $5 ); } }
	| L_EXTERNAL_INIT '(' ')' ';' { if ( gus_icfg_config -> condition_true ) external_init(); }
	| L_EXTERNAL '(' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) external( $3, $5 ); else { free( $3 ); free( $5 ); } }
	| L_EXTERNAL_OPTION '(' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) external_option( $3, $5 ); else { free( $3 ); free( $5 ); } }
	| L_MELODIC '(' integer ',' string ')' '{' { if ( gus_icfg_config -> condition_true ) melodic_start( $3, $5 ); else free( $5 ); } melodic { if ( gus_icfg_config -> condition_true ) melodic_stop(); } '}'
	| L_DRUMS '(' integer ',' string ')' '{' { if ( gus_icfg_config -> condition_true ) drums_start( $3, $5 ); else free( $5 ); } drums { if ( gus_icfg_config -> condition_true ) drums_stop(); } '}'
	| L_IW_FFFF_ROM '(' integer ',' string ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true ) iw_ffff_rom( $3, $5, $7, $9 ); else free( $5 ); }
	| L_IW_FFFF_FILE '(' integer ',' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) iw_ffff_file( $3, $5, $7 ); else { free( $5 ); free( $7 ); } }
	| L_IW_FFFF '(' integer ')' '{' { if ( gus_icfg_config -> condition_true ) iw_ffff_start( $3, 0  ); } iw_ffff { if ( gus_icfg_config -> condition_true ) iw_ffff_stop(); } '}'
	| L_PATCHES '(' string ')' '{' { if ( gus_icfg_config -> condition_true ) patches_start( $3, 0 ); else free( $3 ); } patches { if ( gus_icfg_config -> condition_true ) patches_stop(); } '}'
	| L_PRELOAD '(' string ')' '{' { if ( gus_icfg_config -> condition_true ) preload_start( $3 ); else free( $3 ); } preload { if ( gus_icfg_config -> condition_true ) preload_stop(); } '}'
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' lines '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' lines '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' lines '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in main section" ); YYABORT; }
	| ';'			{ ; }
	;

melodic	: melodic1
	| melodic melodic1
	| keyword
	| melodic keyword
	;

melodic1
	: L_ITEM '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) melodic_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_ITEMGM '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) melodic_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_ITEMGS '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) melodic_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_ITEMMT '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) melodic_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' melodic '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' melodic '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' melodic '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in melodic() {} section" ); YYABORT; }
	| ';'			{ ; }
	;

drums	: drums1
	| drums drums1
	| keyword
	| drums keyword
	;

drums1	: L_ITEM '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) drums_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_ITEMGM '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) drums_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_ITEMGS '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) drums_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_ITEMMT '(' integer ',' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) drums_item( $3, $5, $7, $9 ); else free( $9 ); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' drums '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' drums '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' drums '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in drums() {} section" ); YYABORT; }
	| ';'			{ ; }
	;

iw_ffff	: iw_ffff1
	| iw_ffff iw_ffff1
	| alias
	| iw_ffff alias
	| keyword
	| iw_ffff keyword
	;

iw_ffff1
	: L_ITEM '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true ) iw_ffff_item( $3, $5 ); }
	| L_ITEMGM '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) iw_ffff_item( $3, $5 ); }
	| L_ITEMGS '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) iw_ffff_item( $3, $5 ); }
	| L_ITEMMT '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) iw_ffff_item( $3, $5 ); }
	| L_RANGE '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true ) iw_ffff_range( $3, $5, $7 ); }
	| L_RANGEGM '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) iw_ffff_range( $3, $5, $7 ); }
	| L_RANGEGS '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) iw_ffff_range( $3, $5, $7 ); }
	| L_RANGEMT '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) iw_ffff_range( $3, $5, $7 ); }
	| L_OPTION '(' string ')' ';' { if ( gus_icfg_config -> condition_true ) iw_ffff_option( $3 ); else free( $3 ); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' iw_ffff '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' iw_ffff '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' iw_ffff '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in iw_ffff() {} section" ); YYABORT; }
	| ';'			{ ; }
	;

patches	: patches1
	| patches patches1
	| keyword
	| alias
	| patches alias
	| patches keyword
	;

patches1
	: L_ITEM '(' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) patches_item( $3, $5, $7 ); else free( $7 ); }
	| L_ITEM '(' integer ',' integer ',' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true ) patches_item_options( $3, $5, $7, $9 ); else { free( $7 ); free( $9 ); } }
	| L_ITEMGM '(' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) patches_item( $3, $5, $7 ); else free( $7 ); }
	| L_ITEMGM '(' integer ',' integer ',' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) patches_item_options( $3, $5, $7, $9 ); else { free( $7 ); free( $9 ); } }
	| L_ITEMGS '(' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) patches_item( $3, $5, $7 ); else free( $7 ); }
	| L_ITEMGS '(' integer ',' integer ',' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) patches_item_options( $3, $5, $7, $9 ); else { free( $7 ); free( $9 ); } }
	| L_ITEMMT '(' integer ',' integer ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) patches_item( $3, $5, $7 ); else free( $7 ); }
	| L_ITEMMT '(' integer ',' integer ',' string ',' string ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) patches_item_options( $3, $5, $7, $9 ); else { free( $7 ); free( $9 ); } }

	| L_ITEM '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true ) patches_preload_item( $3, $5 ); }
	| L_ITEMGM '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) patches_preload_item( $3, $5 ); }
	| L_ITEMGS '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) patches_preload_item( $3, $5 ); }
	| L_ITEMMT '(' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) patches_preload_item( $3, $5 ); }
	| L_RANGE '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true ) patches_preload_range( $3, $5, $7 ); }
	| L_RANGEGM '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) patches_preload_range( $3, $5, $7 ); }
	| L_RANGEGS '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) patches_preload_range( $3, $5, $7 ); }
	| L_RANGEMT '(' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) patches_preload_range( $3, $5, $7 ); }

	| L_OPTION '(' string ')' ';' { if ( gus_icfg_config -> condition_true ) patches_option( $3 ); else free( $3 ); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' patches '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' patches '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' patches '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in patches() {} section" ); YYABORT; }
	| ';'			{ ; }
	;

preload	: preload1
	| preload preload1
	| keyword
	| preload keyword
	;

preload1
	: L_IW_FFFF '(' integer ')' '{' { if ( gus_icfg_config -> condition_true ) iw_ffff_start( $3, 1 ); } iw_ffff { if ( gus_icfg_config -> condition_true ) iw_ffff_stop(); } '}'
	| L_PATCHES '(' string ')' '{' { if ( gus_icfg_config -> condition_true ) patches_start( $3, 1 ); else free( $3 ); } patches { if ( gus_icfg_config -> condition_true ) patches_stop(); } '}'
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' preload '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' preload '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' preload '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in preload() {} section" ); YYABORT; }
	| ';'			{ ; }
	;

	/*
	 * aliases
	 */

alias	: L_ALIAS '(' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true ) alias( $3, $5, $7, $9 ); }
	| L_ALIASGM '(' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) alias( $3, $5, $7, $9 ); }
	| L_ALIASGS '(' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) alias( $3, $5, $7, $9 ); }
	| L_ALIASMT '(' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) alias( $3, $5, $7, $9 ); }
	| L_ALIAS_RANGE '(' integer ',' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true ) alias_range( $3, $5, $7, $9, $11 ); }
	| L_ALIAS_RANGEGM '(' integer ',' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM ) alias_range( $3, $5, $7, $9, $11 ); }
	| L_ALIAS_RANGEGS '(' integer ',' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS ) alias_range( $3, $5, $7, $9, $11 ); }
	| L_ALIAS_RANGEMT '(' integer ',' integer ',' integer ',' integer ',' integer ')' ';' { if ( gus_icfg_config -> condition_true && gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32 ) alias_range( $3, $5, $7, $9, $11 ); }
	;

	/*
	 * keywords
         */

keyword	: L_VARIABLE_ASSIGN string ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_lexer_variable_set_string( $1, $2, GUS_ICFG_FVAR_GLOBAL ); free( $1 ); free( $2 ); }
	| L_LOCAL_VARIABLE_ASSIGN string ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_lexer_variable_set_string( $1, $2, GUS_ICFG_FVAR_NONE ); free( $1 ); free( $2 ); }
	| L_VARIABLE_ASSIGN integer ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_lexer_variable_set_integer( $1, $2, GUS_ICFG_FVAR_GLOBAL ); free( $1 ); }
	| L_LOCAL_VARIABLE_ASSIGN integer ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_lexer_variable_set_integer( $1, $2, GUS_ICFG_FVAR_NONE ); free( $1 ); }
	| L_VARIABLE_ASSIGN bool ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_lexer_variable_set_bool( $1, $2, GUS_ICFG_FVAR_GLOBAL ); free( $1 ); }
	| L_LOCAL_VARIABLE_ASSIGN bool ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_lexer_variable_set_bool( $1, $2, GUS_ICFG_FVAR_NONE ); free( $1 ); }
	| L_VARIABLE_ASSIGN L_VARNONE ';' { ; }
	| L_LOCAL_VARIABLE_ASSIGN L_VARNONE ';' { ; }
	| L_EXIT '(' integer ')' { if ( gus_icfg_config -> condition_true ) gus_icfg_config -> exit_value = $3; }
	| L_UNSET '(' { gus_icfg_config -> unset_flag = 1; } ')' { gus_icfg_config -> unset_flag = 0; } ';'
	| L_INCLUDE '(' string ')' ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_lexer_include( $3 ); else free( $3 ); }
	| L_MESSAGE '(' string ')' ';' { if ( gus_icfg_config -> condition_true ) gus_icfg_error( 1, $3 ); free( $3 ); }
	| L_ABORT '(' ')' ';'	{ if ( gus_icfg_config -> condition_true ) { yyerror( "User abort..." ); YYABORT; } }
	;

	/*
         *  types
	 */

bool	: L_BOOL		{ $$ = $1; }
	| L_VARNONE		{ $$ = 0; }
	| L_TRUE		{ $$ = 1; }
	| L_FALSE		{ $$ = 0; }
	| L_FILE_EXIST '(' string ')' { $$ = file_exist( $3 ) != 0 ? 1 : 0; }
	| L_PATCH_EXIST '(' string ')' { $$ = patch_exist( $3 ) != 0 ? 1 : 0; }
	| L_ROM_EXIST '(' integer ',' string ')' { $$ = rom_exist( $3, $5 ) != 0 ? 1 : 0; }
	| integer L_GT integer	{ $$ = $1 > $3; }
	| integer L_GTQ integer { $$ = $1 >= $3; }
	| integer L_LE integer	{ $$ = $1 < $3; }
	| integer L_LEQ integer	{ $$ = $1 <= $3; }
	| integer L_EQ integer	{ $$ = $1 == $3; }
	| integer L_NEQ integer	{ $$ = $1 != $3; }
	| string L_EQ string	{ $$ = strcmp( $1, $3 ) == 0; free( $1 ); free( $3 ); }
	| string L_NEQ string	{ $$ = strcmp( $1, $3 ) != 0; free( $1 ); free( $3 ); }
	| bool L_EQ bool	{ $$ = $1 == $3; }
	| bool L_NEQ bool	{ $$ = $1 != $3; }
	| bool L_OR bool	{ $$ = $1 || $3; }
	| bool L_AND bool	{ $$ = $1 && $3; }
	| '!' bool %prec NEG	{ $$ = $2 != 0 ? 0 : 1; }
	| '(' bool ')'		{ $$ = $2; }
	;

integer : L_INTEGER		{ $$ = $1; }
	| L_VARNONE		{ $$ = 0; }
	| integer '+' integer	{ $$ = $1 + $3; }
	| integer '-' integer	{ $$ = $1 - $3; }
	| integer '*' integer	{ $$ = $1 * $3; }
	| integer '/' integer	{ $$ = $1 / $3; }
	| integer L_RSHIFT integer { $$ = $1 >> $3; }
	| integer L_LSHIFT integer { $$ = $1 << $3; }
	| integer '&' integer	{ $$ = $1 & $3; }
	| integer '|' integer	{ $$ = $1 | $3; }
	| '-' integer %prec NEG { $$ = -$2; }
	| '(' integer ')'	{ $$ = $2; }
	;

string	: L_STRING		{ $$ = $1; }
	| L_VARNONE		{ $$ = strdup( "" ); if ( !$$ ) yyerror( "malloc error" ); }
	| string '+' string	{ char *result;
				  result = malloc( strlen( $1 ) + strlen( $3 ) + 1 );
			          if ( !result )
				    yyerror( "malloc error" );
				  strcpy( result, $1 );
				  strcat( result, $3 );
				  free( $1 ); free( $3 );
				  $$ = result;
				}
	| L_ETC_DIR '(' string ')'
				{ char *x = (char *)malloc( sizeof( GUS_PATH_ETC ) + strlen( $3 ) );
                                  strcpy( x, GUS_PATH_ETC );
                                  strcat( x, $3 );
                                  free( $3 );
                                  $$ = x;
                                }
	| L_LIB_DIR '(' string ')'
				{ char *x = (char *)malloc( sizeof( GUS_PATH_LIB ) + strlen( $3 ) );
                                  strcpy( x, GUS_PATH_LIB );
                                  strcat( x, $3 );
                                  free( $3 );
                                  $$ = x;
                                }
	| L_INS_DIR '(' string ')'
				{ char *x = (char *)malloc( sizeof( GUS_PATH_INS ) + strlen( $3 ) );
                                  strcpy( x, GUS_PATH_INS );
                                  strcat( x, $3 );
                                  free( $3 );
                                  $$ = x;
                                }
	;


%%

void gus_icfg_parser_abort( void )
{
  gus_icfg_config -> abort_flag = 1;
  gus_icfg_config -> condition_true = 0;
  gus_icfg_config -> condition_false = 0;
}

void gus_icfg_parser_init( int action )
{
#ifdef YYDEBUG
  yydebug = 1;
#endif
  gus_icfg_config -> condition_true = 1;
  gus_icfg_config -> condition_false = 0;
  gus_icfg_config -> condition_stack_ptr = 0;
  gus_icfg_config -> abort_flag = 0;
  gus_icfg_config -> source_flag =
  gus_icfg_config -> preload_source_flag =
  gus_icfg_config -> external_flag =
  gus_icfg_config -> exit_value = 0;
  gus_icfg_config -> tmp_group = NULL;
  gus_icfg_config -> tmp_preload_ptr = NULL;
  gus_icfg_config -> tmp_preload_format = NULL;
  gus_icfg_lexer_init( action );
}

void gus_icfg_parser_done( void )
{
  gus_icfg_free_group( gus_icfg_config -> tmp_group );
  gus_icfg_free_preload( gus_icfg_config -> tmp_preload_ptr );
  gus_icfg_free_preload_format( gus_icfg_config -> tmp_preload_format );
  gus_icfg_lexer_done();
}

static void condition_go( int true )
{
  if ( gus_icfg_config -> condition_stack_ptr >= GUS_ICFG_MAX_CONDITION ) {
    yyerror( "condition stack overflow" );
    return;
  }
  gus_icfg_config -> condition_stack[ gus_icfg_config -> condition_stack_ptr++ ] = gus_icfg_config -> condition_true;
  if ( gus_icfg_config -> abort_flag ) {
    gus_icfg_config -> condition_true = 0;
    gus_icfg_config -> condition_false = 0;
  } else {
    if ( gus_icfg_config -> condition_true ) {
      gus_icfg_config -> condition_true = true ? 1 : 0;
      gus_icfg_config -> condition_false = true ? 0 : 1;
    }
  }
}

static void condition_end( void )
{
  if ( !gus_icfg_config -> condition_stack_ptr ) {
    yyerror( "condition stack underflow" );
    return;
  }
  if ( gus_icfg_config -> abort_flag ) {
    gus_icfg_config -> condition_true = 0;
    gus_icfg_config -> condition_false = 0;
    --gus_icfg_config -> condition_stack_ptr;
  } else {
    gus_icfg_config -> condition_true = gus_icfg_config -> condition_stack[ --gus_icfg_config -> condition_stack_ptr ];
    gus_icfg_config -> condition_false = !gus_icfg_config -> condition_true;
  }
}

static int file_exist( char *file )
{
  struct stat st;

  if ( gus_icfg_config -> condition_true && !stat( file, &st ) )
    {
      free( file );
      return 1;
    }
  free( file );
  return 0;
}

static int patch_exist( char *patch )
{
  char *filename;

  if ( gus_icfg_config -> condition_true ) {
    if ( ( filename = gus_icfg_look_for_patch_file( patch ) ) != NULL )
      {
        free( filename );
        free( patch );
        return 1;
      }
  }
  free( patch );
  return 0;
}

static int rom_exist( int bank, char *name )
{
  gus_rom_interwave_header_t header;

  if ( gus_icfg_config -> condition_true )
    if ( gus_instr_ffff_get_rom_header( gus_icfg_config -> gus_fd, gus_icfg_config -> gus_access, bank, &header ) >= 0 )
      if ( !strncmp( header.series_name, name, 16 ) )
        {
          free( name );
          return 1;
        }
  free( name );
  return 0;
}

static void info_new( int type, char *name, char *description )
{
  struct gus_icfg_info *info, *pinfo;

  info = (struct gus_icfg_info *)calloc( 1, sizeof( struct gus_icfg_info ) );
  if ( !info ) {
    yyerror( "malloc error" );
    return;
  }
  info -> type = type;
  info -> name = name;
  info -> description = description;
  pinfo = gus_icfg_config -> infos;
  if ( !pinfo ) {
    gus_icfg_config -> infos = info;
  } else {
    while ( pinfo -> next ) pinfo = pinfo -> next;
    pinfo -> next = info;
  }
}

static void info_option( int type, char *name, char *value )
{
  int i;
  struct gus_icfg_info *info;
  char **x;

  for ( info = gus_icfg_config -> infos; info; info = info -> next ) {
    if ( info -> type != type || strcmp( info -> name, name ) ) continue;
    free( name );
    if ( info -> options ) {
      i = 1; while ( info -> options[ i++ ] );
    } else {
      i = 2;
    }
    x = (char **)realloc( info -> options, i * sizeof( char * ) );
    if ( !x ) { yyerror( "malloc error" ); return; }
    x[ i - 2 ] = value;
    x[ i - 1 ] = NULL;
    info -> options = x;
    return;
  }
  free( name );
  yyerror( "Item name '%s' not found.." );
}

static void source_init( void )
{
  gus_icfg_config -> source_flag = 1;
}

static void source( char *source, char *description )
{
  if ( !gus_icfg_config -> source_flag ) {
    yyerror( "Sources are not initialized, use source_init()!!!" );
    return;
  }
  info_new( GUS_ICFG_INFO_SOURCE, source, description );  
}

static void source_option( char *source, char *option )
{
  if ( !gus_icfg_config -> source_flag ) {
    yyerror( "Sources are not initialized, use source_init()!!!" );
    return;
  }
  info_option( GUS_ICFG_INFO_SOURCE, source, option );
}

static void preload_source_init( void )
{
  gus_icfg_config -> preload_source_flag = 1;
}

static void preload_source( char *source, char *description )
{
  if ( !gus_icfg_config -> preload_source_flag ) {
    yyerror( "Preload sources are not initialized, use presource_init()!!!" );
    return;
  }
  info_new( GUS_ICFG_INFO_PRELOAD, source, description );  
}

static void preload_source_option( char *source, char *option )
{
  if ( !gus_icfg_config -> preload_source_flag ) {
    yyerror( "Preload sources are not initialized, use presource_init()!!!" );
    return;
  }
  info_option( GUS_ICFG_INFO_PRELOAD, source, option );
}

static void external_init( void )
{
  gus_icfg_config -> external_flag = 1;
}

static void external( char *source, char *description )
{
  if ( !gus_icfg_config -> external_flag ) {
    yyerror( "External sources are not initialized, use external_init()!!!" );
    return;
  }
  info_new( GUS_ICFG_INFO_EXTERNAL, source, description );  
}

static void external_option( char *source, char *option )
{
  if ( !gus_icfg_config -> external_flag ) {
    yyerror( "External sources are not initialized, use external_init()!!!" );
    return;
  }
  info_option( GUS_ICFG_INFO_EXTERNAL, source, option );
}

static void melodic_start( int number, char *name )
{
  gus_icfg_config -> tmp_group = (struct gus_icfg_group *)calloc( 1, sizeof( struct gus_icfg_group ) );
  if ( !gus_icfg_config -> tmp_group ) { yyerror( "malloc error" ); return; }
  gus_icfg_config -> tmp_group -> type = GUS_ICFG_GROUP_MELODIC;
  gus_icfg_config -> tmp_group -> number = number;
  gus_icfg_config -> tmp_group -> name = name;
  gus_icfg_config -> tmp_drums_flag = 0;
}

static void melodic_stop( void )
{
  struct gus_icfg_group *group;

  group = gus_icfg_config -> groups;
  if ( !group ) {
    gus_icfg_config -> groups = gus_icfg_config -> tmp_group;
  } else {
    while ( group -> next ) group = group -> next;
    group -> next = gus_icfg_config -> tmp_group;
  }
  gus_icfg_config -> tmp_group = NULL;
}

static struct gus_icfg_instrument *find_instrument( int bank, int prog )
{
  struct gus_icfg_instrument *instr;

  bank <<= 16; bank |= prog & 0xffff;
  for ( instr = gus_icfg_config -> instruments; instr; instr = instr -> next )
    if ( instr -> number == bank ) return instr;
  return NULL;
}

static void add_instrument( struct gus_icfg_instrument *add )
{
  struct gus_icfg_instrument *instr;

  instr = gus_icfg_config -> instruments;
  if ( !instr ) {
    gus_icfg_config -> instruments = add;
  } else {
    while ( instr -> next ) instr = instr -> next;
    instr -> next = add; 
  }
}

static void verify_bank( int *bank )
{
  if ( *bank < 0 || *bank > 127 ) yyerror( "bank # out of range" );
}

static void verify_prog_drums( int *prog, int drums )
{
  *prog = *prog - 1;
  if ( drums ) {
    if ( *prog < 128 ) *prog = *prog + 128 + 1;
    if ( *prog < 128 || *prog > 255 ) { yyerror( "prog # out of range" ); return; }
  } else {
    if ( *prog < 0 || *prog > 127 ) { yyerror( "prog # out of range" ); return; }
  }
}

static void verify_prog( int *prog )
{
  *prog = *prog - 1;
  if ( *prog < 0 || *prog > 255 ) yyerror( "prog # out of range" );
}

static void melodic_item( int bank, int prog, int voices, char *name )
{
  int new_flag;
  struct gus_icfg_group_range *range, *prange;
  struct gus_icfg_instrument *instr;

  verify_bank( &bank );
  verify_prog_drums( &prog, gus_icfg_config -> tmp_drums_flag );
  for ( range = gus_icfg_config -> tmp_group -> first; range; range = range -> next ) {
    if ( range -> bank == bank && range -> max + 1 == prog ) {
      range -> max++;
    }
  }  
  if ( !range ) {
    range = (struct gus_icfg_group_range *)calloc( 1, sizeof( struct gus_icfg_group_range ) );
    if ( !range ) { yyerror( "malloc error" ); return; }
    range -> bank = bank;
    range -> min = range -> max = prog;
    prange = gus_icfg_config -> tmp_group -> first;
    if ( !prange ) {
      gus_icfg_config -> tmp_group -> first = range;
    } else {
      while ( prange -> next ) prange = prange -> next;
      prange -> next = range;
    }
  }  
  new_flag = 0;
  if ( (instr = find_instrument( bank, prog )) == NULL ) {
    new_flag = 1;
    instr = (struct gus_icfg_instrument *)calloc( 1, sizeof( struct gus_icfg_instrument ) );
    if ( !instr ) { yyerror( "malloc error" ); return; }
  }
  instr -> number = ( bank << 16 ) | ( prog & 0xffff );
  instr -> voices = voices;
  if ( instr -> name ) free( instr -> name );
  instr -> name = name;
  if ( new_flag )
    add_instrument( instr );
}

static void drums_start( int number, char *name )
{
  melodic_start( number, name );
  gus_icfg_config -> tmp_group -> type = GUS_ICFG_GROUP_DRUMS;
  gus_icfg_config -> tmp_drums_flag = 1;
}

static void drums_stop( void )
{
  melodic_stop();
}

static void drums_item( int bank, int prog, int voices, char *name )
{
  melodic_item( bank, prog, voices, name );
}

static void iw_ffff_rom( int number, char *id, int bank, int file )
{
  struct gus_icfg_iwfile *iw, *piw;

  iw = (struct gus_icfg_iwfile *)calloc( 1, sizeof( struct gus_icfg_iwfile ) );
  if ( !iw ) { yyerror( "malloc error" ); return; }
  iw -> flags |= GUS_ICFG_IWF_ROM;
  iw -> number = number;
  iw -> data.rom.bank = bank;
  iw -> data.rom.file = file;
  iw -> data.rom.name = id;
  piw = gus_icfg_config -> iwfiles;
  if ( !piw ) {
    gus_icfg_config -> iwfiles = iw;
  } else {
    while ( piw -> next ) piw = piw -> next;
    piw -> next = iw;
  }
}

static void iw_ffff_file( int number, char *fff, char *dat )
{
  struct gus_icfg_iwfile *iw, *piw;

  iw = (struct gus_icfg_iwfile *)calloc( 1, sizeof( struct gus_icfg_iwfile ) );
  if ( !iw ) { yyerror( "malloc error" ); return; }
  iw -> number = number;
  iw -> data.file.name = fff;
  iw -> data.file.data = dat;
  piw = gus_icfg_config -> iwfiles;
  if ( !piw ) {
    gus_icfg_config -> iwfiles = iw;
  } else {
    while ( piw -> next ) piw = piw -> next;
    piw -> next = iw;
  }
}

static void iw_ffff_start( int number, int preload )
{
  gus_icfg_config -> tmp_preload = preload;
  gus_icfg_config -> tmp_iwfile_number = number;
  if ( get_ffff_file( number ) == NULL ) {
    yyerror( "No FFFF file with this number is defined!!!" );
    return;
  }
  if ( preload ) {
    struct gus_icfg_preload_format *format;
    format = (struct gus_icfg_preload_format *)calloc( 1, sizeof( struct gus_icfg_preload_format ) );
    if ( !format ) { yyerror( "malloc error" ); return; }
    format -> type = GUS_ICFG_IT_IW_FILE;
    format -> data.iwfile.file = number;
    gus_icfg_config -> tmp_preload_format = format;
  }
}

static void iw_ffff_stop( void )
{
  if ( gus_icfg_config -> tmp_preload ) {
    struct gus_icfg_preload_format *format;
    format = gus_icfg_config -> tmp_preload_ptr -> formats;
    if ( !format ) {
      gus_icfg_config -> tmp_preload_ptr -> formats = gus_icfg_config -> tmp_preload_format;
    } else {
      while ( format -> next ) format = format -> next;
      format -> next = gus_icfg_config -> tmp_preload_format;
    }
    gus_icfg_config -> tmp_preload_format = NULL;
  }
}

static void new_download( int bank, int min, int max )
{
  struct gus_icfg_preload_download *download, *pdownload;

  download = (struct gus_icfg_preload_download *)calloc( 1, sizeof( struct
  gus_icfg_preload_download ) );
  if ( !download ) { yyerror( "malloc error" ); return; }
  download -> bank = bank;
  download -> min = min;
  download -> max = max;
  pdownload = gus_icfg_config -> tmp_preload_format -> downloads;
  if ( !pdownload ) {
    gus_icfg_config -> tmp_preload_format -> downloads = download;
  } else {
    while ( pdownload -> next ) pdownload = pdownload -> next;
    pdownload -> next = download;
  }
}

static void iw_ffff_item( int bank, int prog )
{
  int new_flag;
  struct gus_icfg_instrument *instr;

  verify_bank( &bank );
  verify_prog( &prog );
  if ( !gus_icfg_config -> tmp_preload ) {
    new_flag = 0;
    if ( (instr = find_instrument( bank, prog )) == NULL ) {
      new_flag = 1;
      instr = (struct gus_icfg_instrument *)calloc( 1, sizeof( struct gus_icfg_instrument ) );
      if ( !instr ) { yyerror( "malloc error" ); return; }
    }
    instr -> number = ( bank << 16 ) | ( prog & 0xffff );
    instr -> type = GUS_ICFG_IT_IW_FILE;
    instr -> flags = GUS_ICFG_IF_NONE;
    instr -> data.iwfile.file = gus_icfg_config -> tmp_iwfile_number;
    instr -> data.iwfile.bank = bank;
    instr -> data.iwfile.prog = prog;
    if ( new_flag )
      add_instrument( instr );
  } else {
    new_download( bank, prog, prog );
  }
}

static void iw_ffff_range( int bank, int min_prog, int max_prog )
{
  if ( !gus_icfg_config -> tmp_preload ) {
    for ( ; min_prog <= max_prog; min_prog++ )
      iw_ffff_item( bank, min_prog );
  } else {
    verify_bank( &bank );
    verify_prog( &min_prog );
    verify_prog( &max_prog );
    new_download( bank, min_prog, max_prog );
  }
}

static void iw_ffff_option( char *option )
{
  if ( !gus_icfg_config -> tmp_preload ) {
    free( option );
    yyerror( "unknown option" );
  } else {
    if ( !strcmp( option, "whole" ) )
      gus_icfg_config -> tmp_preload_format -> flags |= GUS_ICFG_PF_WHOLE;
     else
      yyerror( "unknown option" );
    free( option );
  }
}

static void patches_start( char *dir, int preload )
{
  int dirn;
  struct gus_icfg_gf1path *path, *ppath;

  gus_icfg_config -> tmp_preload = preload;
  if ( !preload ) {
    dirn = 0;
    while ( dirn < 0xffff ) {
      for ( path = gus_icfg_config -> gf1paths; path; path = path -> next )
        if ( path -> directory == dirn ) goto __next;
      break;
      __next:
      dirn++;
    }
    path = (struct gus_icfg_gf1path *)calloc( 1, sizeof( struct gus_icfg_gf1path ) );
    if ( !path ) { yyerror( "malloc error" ); return; }
    path -> directory = dirn;
    path -> path = dir;
    ppath = gus_icfg_config -> gf1paths;
    if ( !ppath ) {
      gus_icfg_config -> gf1paths = path;
    } else {
      while ( ppath -> next ) ppath = ppath -> next;
      ppath -> next = path;
    }
  } else {
    struct gus_icfg_preload_format *format;

    dirn = -1;
    for ( path = gus_icfg_config -> gf1paths; path; path = path -> next )
      if ( !strcmp( path -> path, dir ) ) {
        dirn = path -> directory;
        break;
      }
    if ( dirn < 0 ) {
      yyerror( "No patches in this path are defined!!!" );
      return;
    }
    format = (struct gus_icfg_preload_format *)calloc( 1, sizeof( struct gus_icfg_preload_format ) );
    if ( !format ) { yyerror( "malloc error" ); return; }
    format -> type = GUS_ICFG_IT_GF1_PATCH;
    gus_icfg_config -> tmp_preload_format = format;
  }
  gus_icfg_config -> tmp_gf1path_directory = dirn;
}

static void patches_stop( void )
{
  if ( gus_icfg_config -> tmp_preload ) {
    struct gus_icfg_preload_format *format;
    format = gus_icfg_config -> tmp_preload_ptr -> formats;
    if ( !format ) {
      gus_icfg_config -> tmp_preload_ptr -> formats = gus_icfg_config -> tmp_preload_format;
    } else {
      while ( format -> next ) format = format -> next;
      format -> next = gus_icfg_config -> tmp_preload_format;
    }
    gus_icfg_config -> tmp_preload_format = NULL;
  }
}

static void patches_item( int bank, int prog, char *file )
{
  patches_item_options( bank, prog, file, NULL );
}

static void patches_item_options( int bank, int prog, char *file, char *options )
{
  int new_flag, i;
  struct gus_icfg_instrument *instr;
  char *opt;

  verify_bank( &bank );
  verify_prog( &prog );
  if ( !gus_icfg_config -> tmp_preload ) {
    new_flag = 0;
    if ( (instr = find_instrument( bank, prog )) == NULL ) {
      new_flag = 1;
      instr = (struct gus_icfg_instrument *)calloc( 1, sizeof( struct gus_icfg_instrument ) );
      if ( !instr ) { yyerror( "malloc error" ); return; }
    }
    instr -> number = ( bank << 16 ) | ( prog & 0xffff );
    instr -> type = GUS_ICFG_IT_GF1_PATCH;
    instr -> flags = GUS_ICFG_IF_NONE;
    instr -> data.patch.directory = gus_icfg_config -> tmp_gf1path_directory;
    instr -> data.patch.filename = file;
    instr -> data.patch.exclusion = GUS_INSTR_E_NONE;
    instr -> data.patch.exclusion_group = 0;
    if ( options ) {
      opt = options;
      while ( *opt ) {
        while ( *opt && *opt <= ' ' ) opt++;
        if ( !*opt ) break;
        if ( !strncmp( opt, "exc-", 4 ) ) {
          sscanf( opt, "exc-%i", &i );
          if ( i < 1 || i > 65535 ) { yyerror( "exclude group out of range" ); return; }
          instr -> data.patch.exclusion = GUS_INSTR_E_SINGLE;
	  instr -> data.patch.exclusion_group = i;
        } else
        if ( !strncmp( opt, "mul-", 4 ) ) {
          sscanf( opt, "mul-%i", &i );
          if ( i < 1 || i > 65535 ) { yyerror( "exclude group out of range" ); return; }
          instr -> data.patch.exclusion = GUS_INSTR_E_MULTIPLE;
   	  instr -> data.patch.exclusion_group = i;        
        } else 
        if ( !strncmp( opt, "8bit", 4 ) ) {
          instr -> flags |= GUS_ICFG_IF_8BIT;
        } else { yyerror( "Unknown option for patch item..." ); return; }
        while ( *opt && *opt > ' ' ) opt++;
      }
      free( options );
    }  
    if ( new_flag )
      add_instrument( instr );  
  } else {
    yyerror( "wrong # of parameters for item() in preload section" );
  }
}

static void patches_option( char *option )
{
  if ( !gus_icfg_config -> tmp_preload ) {
    yyerror( "unknown option" );
    free( option );
    return;
  } else {

    if ( !strcmp( option, "8bit" ) )
      gus_icfg_config -> tmp_preload_format -> flags |= GUS_ICFG_PF_8BIT;
     else
      yyerror( "unknown option" );
    free( option );
  }
} 

static void patches_preload_item( int bank, int prog )
{
  patches_preload_range( bank, prog, prog );
}

static void patches_preload_range( int bank, int min_prog, int max_prog )
{
  if ( gus_icfg_config -> tmp_preload ) {
    verify_bank( &bank );
    verify_prog( &min_prog );
    verify_prog( &max_prog );
    new_download( bank, min_prog, max_prog );
  } else {
    yyerror( "wrong # of parameters for item()" );
  }
}

static void alias( int src_bank, int src_prog, int dst_bank, int dst_prog )
{
  int new_flag;
  struct gus_icfg_instrument *instr;

  if ( !gus_icfg_config -> tmp_preload ) {
    verify_bank( &src_bank );
    verify_prog( &src_prog );
    verify_bank( &dst_bank );
    verify_bank( &dst_prog );
    new_flag = 0;
    if ( (instr = find_instrument( src_bank, src_prog )) == NULL ) {
      new_flag = 1;
      instr = (struct gus_icfg_instrument *)calloc( 1, sizeof( struct gus_icfg_instrument ) );
      if ( !instr ) { yyerror( "malloc error" ); return; }
    }
    instr -> number = ( src_bank << 16 ) | ( src_prog & 0xffff );
    instr -> type = GUS_ICFG_IT_ALIAS;
    instr -> flags = GUS_ICFG_IF_NONE;
    instr -> data.alias.bank = dst_bank;
    instr -> data.alias.prog = dst_prog;
    if ( new_flag )
      add_instrument( instr );  
  } else {
    alias_range( src_bank, src_prog, src_prog, dst_bank, dst_prog );
  }
}

static void alias_range( int src_bank, int src_min_prog, int src_max_prog, int dst_bank, int dst_prog )
{
  struct gus_icfg_preload_alias *xalias, *palias;

  if ( !gus_icfg_config -> tmp_preload ) {
    for ( ; src_min_prog <= src_max_prog; src_min_prog++ )
      alias( dst_bank, dst_prog, src_bank, src_min_prog );
  } else {
    verify_bank( &src_bank );
    verify_prog( &src_min_prog );
    verify_prog( &src_max_prog );
    verify_bank( &dst_bank );
    verify_prog( &dst_prog );
    xalias = (struct gus_icfg_preload_alias *)calloc( 1, sizeof( struct gus_icfg_preload_alias ) );
    if ( !xalias ) { yyerror( "malloc error" ); return; }
    xalias -> bank = src_bank;
    xalias -> min = src_min_prog;
    xalias -> max = src_max_prog;
    xalias -> dest_bank = dst_bank;
    xalias -> dest_prog = dst_prog;
    palias = gus_icfg_config -> tmp_preload_format -> aliases;
    if ( !palias ) {
      gus_icfg_config -> tmp_preload_format -> aliases = xalias;
    } else {
      while ( palias -> next ) palias = palias -> next;
      palias -> next = xalias;
    }
  }
}

static void preload_start( char *name )
{
  struct gus_icfg_preload *preload;

  preload = (struct gus_icfg_preload *)calloc( 1, sizeof( struct gus_icfg_preload ) );
  if ( !preload ) { yyerror( "malloc error" ); return; }
  preload -> name = name;
  gus_icfg_config -> tmp_preload_ptr = preload;  
}

static void preload_stop( void )
{
  struct gus_icfg_preload *preload;

  preload = gus_icfg_config -> preloads;
  if ( !preload ) {
    gus_icfg_config -> preloads = gus_icfg_config -> tmp_preload_ptr;
  } else {
    while ( preload -> next ) preload = preload -> next;
    preload -> next = gus_icfg_config -> tmp_preload_ptr;
  }
  gus_icfg_config -> tmp_preload_ptr = NULL;  
}
