/*==============================================================================

  $Id: virtch2.c,v 1.6 1998/09/20 21:45:26 miod Exp $

  High-quality sample mixing routines, using a 32 bits mixing buffer,
  interpolation, and sample smoothing to improve sound quality and remove
  clicks.

==============================================================================*/

/*
	This library is free software; you can redistribute it and/or modify
	it under the terms of the GNU Library General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.
 
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Library General Public License for more details.
 
	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*

  Future Additions:
    Low-Pass filter to remove annoying staticy buzz.

*/

#include <mikmod.h>
  
#include <stddef.h>
#include <string.h>

/*
   Constant Definitions
   ====================

	MAXVOL_FACTOR (was BITSHIFT in virtch.c)
		Controls the maximum volume of the output data. All mixed data is
		divided by this number after mixing, so larger numbers result in
		quieter mixing.  Smaller numbers will increase the likliness of
		distortion on loud modules.

	REVERBERATION
		Larger numbers result in shorter reverb duration. Longer reverb
		durations can cause unwanted static and make the reverb sound more
		like a crappy echo.

	SAMPLING_SHIFT
		Specified the shift multiplier which controls by how much the mixing
		rate is multiplied while mixing.  Higher values can improve quality by
		smoothing the sound and reducing pops and clicks. Note, this is a shift
		value, so a value of 2 becomes a mixing-rate multiplier of 4, and a
		value of 3 = 8, etc.

	FRACBITS
		The number of bits per integer devoted to the fractional part of the
		number. Generally, this number should not be changed for any reason.

	!!! IMPORTANT !!! All values below MUST ALWAYS be greater than 0

*/

#define MAXVOL_FACTOR (1<<9)
#define	REVERBERATION 11000L

#define SAMPLING_SHIFT 2
#define SAMPLING_FACTOR (1UL<<SAMPLING_SHIFT)

#define	FRACBITS 28
#define FRACMASK ((1UL<<FRACBITS)-1UL)

#define TICKLSIZE 8192
#define TICKWSIZE (TICKLSIZE * 2)
#define TICKBSIZE (TICKWSIZE * 2)

#define CLICK_SHIFT_BASE 6
#define CLICK_SHIFT (CLICK_SHIFT_BASE * SAMPLING_SHIFT)
#define CLICK_BUFFER (1L << CLICK_SHIFT)

#ifndef MIN
#define MIN(a,b) (((a)<(b)) ? (a) : (b))
#endif

#ifdef HAS_64BIT_INT
typedef SLONGLONG FRACTYPE;
#else
#include <math.h>
typedef double FRACTYPE;
#endif

typedef struct VINFO {
	UBYTE    kick;              /* =1 -> sample has to be restarted */
	UBYTE    active;            /* =1 -> sample is playing */
	UWORD    flags;             /* 16/8 bits looping/one-shot */
	SWORD    handle;            /* identifies the sample */
	ULONG    start;             /* start index */
	ULONG    size;              /* samplesize */
	ULONG    reppos;            /* loop start */
	ULONG    repend;            /* loop end */
	ULONG    frq;               /* current frequency */
	int      vol;               /* current volume */
	int      pan;               /* current panning position */

	int      click;
	int      rampvol;
	SLONG    lastvalL,lastvalR;
	int      lvolsel,rvolsel;   /* Volume factor in range 0-255 */
	int      oldlvol,oldrvol;

	FRACTYPE current;           /* current index in the sample */
	FRACTYPE increment;         /* increment value */
} VINFO;

static	SWORD **Samples;
static	VINFO *vinf=NULL,*vnf;
static	long tickleft,samplesthatfit,vc_memory=0;
static	int vc_softchn;
static	FRACTYPE idxsize,idxlpos,idxlend;
static	SLONG *vc_tickbuf=NULL;
static	UWORD vc_mode;

/* Reverb control variables */

static	int RVc1, RVc2, RVc3, RVc4, RVc5, RVc6, RVc7, RVc8;
static	ULONG RVRindex;

/* For Mono or Left Channel */
static	SLONG *RVbufL1=NULL,*RVbufL2=NULL,*RVbufL3=NULL,*RVbufL4=NULL,
		      *RVbufL5=NULL,*RVbufL6=NULL,*RVbufL7=NULL,*RVbufL8=NULL;

/* For Stereo only (Right Channel) */
static	SLONG *RVbufR1=NULL,*RVbufR2=NULL,*RVbufR3=NULL,*RVbufR4=NULL,
		      *RVbufR5=NULL,*RVbufR6=NULL,*RVbufR7=NULL,*RVbufR8=NULL;

#ifndef HAS_64BIT_INT

/*========== Floating-point mixers */

FRACTYPE MixMonoNormal(SWORD* srce,SLONG* dest,FRACTYPE index,FRACTYPE increment,SLONG todo)
{
	SWORD sample=0;
	FRACTYPE i, f;

	while(todo--) {
		f=modf(index,&i);
		sample=(srce[(SLONG)i]*(1.0-f))+(srce[(SLONG)i+1]*f);
		index+=increment;

		if(vnf->rampvol) {
			*dest++ += (long)(
			  ( ( (FRACTYPE)(vnf->oldlvol*vnf->rampvol) +
			      (vnf->lvolsel*(CLICK_BUFFER-vnf->rampvol)) ) *
			    (FRACTYPE)sample ) / CLICK_BUFFER );
			vnf->rampvol--;
		} else
		  if(vnf->click) {
			*dest++ += (long)(
			  ( ( ((FRACTYPE)vnf->lvolsel*(CLICK_BUFFER-vnf->click)) *
			      (FRACTYPE)sample ) +
			    (vnf->lastvalL*vnf->click) ) / CLICK_BUFFER );
			vnf->click--;
		} else
			*dest++ +=vnf->lvolsel*sample;
	}
	vnf->lastvalL=vnf->lvolsel * sample;

	return index;
}

FRACTYPE MixStereoNormal(SWORD* srce,SLONG* dest,FRACTYPE index,FRACTYPE increment,ULONG todo)
{
	SWORD sample=0.0;
	FRACTYPE i, f;

	while(todo--) {
		f=modf(index,&i);
		sample=(srce[(SLONG)i]*(1.0-f))+(srce[(SLONG)i+1]*f);
		index += increment;

		if(vnf->rampvol) {
			*dest++ += (long)(
			  ( ( ((FRACTYPE)vnf->oldlvol*vnf->rampvol) +
			      (vnf->lvolsel*(CLICK_BUFFER-vnf->rampvol))
			    ) * (FRACTYPE)sample ) / CLICK_BUFFER );
			*dest++ += (long)(
			  ( ( ((FRACTYPE)vnf->oldrvol*vnf->rampvol) +
			      (vnf->rvolsel*(CLICK_BUFFER-vnf->rampvol))
			    ) * (FRACTYPE)sample ) / CLICK_BUFFER);
			vnf->rampvol--;
		} else
		  if(vnf->click) {
			*dest++ += (long)(
			  ( ( (FRACTYPE)(vnf->lvolsel*(CLICK_BUFFER-vnf->click)) *
			      (FRACTYPE)sample ) + (vnf->lastvalL * vnf->click) ) /
			    CLICK_BUFFER );
			*dest++ += (long)(
			  ( ( ((FRACTYPE)vnf->rvolsel*(CLICK_BUFFER-vnf->click)) *
			      (FRACTYPE)sample ) + (vnf->lastvalR * vnf->click) ) /
			    CLICK_BUFFER );
			vnf->click--;
		} else {
			*dest++ +=vnf->lvolsel*sample;
			*dest++ +=vnf->rvolsel*sample;
		}
	}
	vnf->lastvalL=vnf->lvolsel*sample;
	vnf->lastvalR=vnf->rvolsel*sample;

	return index;
}

FRACTYPE MixStereoSurround(SWORD* srce,SLONG* dest,FRACTYPE index,FRACTYPE increment,ULONG todo)
{
	SWORD sample=0;
	long whoop;
	FRACTYPE i, f;

	/*dest--;*/
	while(todo--) {
		f=modf(index,&i);
		sample=(srce[(SLONG)i]*(1.0-f))+(srce[(SLONG)i+1]*f);
		index+=increment;

		if(vnf->rampvol) {
			whoop=(long)(
			  ( ( (FRACTYPE)(vnf->oldlvol*vnf->rampvol) +
			      (vnf->lvolsel*(CLICK_BUFFER-vnf->rampvol)) ) *
			    (FRACTYPE)sample) / CLICK_BUFFER );
			*dest++ +=whoop;
			*dest++ -=whoop;
			vnf->rampvol--;
		} else
		  if(vnf->click) {
			whoop = (long)(
			  ( ( ((FRACTYPE)vnf->lvolsel*(CLICK_BUFFER-vnf->click)) *
			      (FRACTYPE)sample) +
			    (vnf->lastvalL * vnf->click) ) / CLICK_BUFFER );
			*dest++ +=whoop;
			*dest++ -=whoop;
			vnf->click--;
		} else {
			st++ +=vnf->lvolsel*sample;
			st++ -=vnf->rvolsel*sample;
		}
	}
	vnf->lastvalL=vnf->lvolsel*sample;
	vnf->lastvalR=vnf->rvolsel*sample;

	return index;
}

#else

/*========== 64 bit mixers */

FRACTYPE MixMonoNormal(SWORD* srce,SLONG* dest,FRACTYPE index,FRACTYPE increment,SLONG todo)
{
	SWORD sample=0;
	FRACTYPE i,f;

	while(todo--) {
		i=index>>FRACBITS,f=index&FRACMASK;
		sample=(SWORD)(((FRACTYPE)(srce[i]*(FRACMASK+1L-f)) +
		               ((FRACTYPE)srce[i+1]*f)) >> FRACBITS);
		index+=increment;

		if(vnf->rampvol) {
			*dest++ +=(
			  ( (FRACTYPE)(vnf->oldlvol*vnf->rampvol) +
			    (vnf->lvolsel*(CLICK_BUFFER-vnf->rampvol)) ) *
			  (FRACTYPE)sample ) >> CLICK_SHIFT;
			vnf->rampvol--;
		} else
		  if(vnf->click) {
			*dest++ +=(
			  ( ((FRACTYPE)vnf->lvolsel*(CLICK_BUFFER-vnf->click)) *
			    (FRACTYPE)sample ) + ((FRACTYPE)vnf->lastvalL*vnf->click) )
			  >> CLICK_SHIFT;
			vnf->click--;
		} else
			*dest++ +=vnf->lvolsel*sample;
	}
	vnf->lastvalL=vnf->lvolsel*sample;

	return index;
}

FRACTYPE MixStereoNormal(SWORD* srce,SLONG* dest,FRACTYPE index,FRACTYPE increment,ULONG todo)
{
	SWORD sample=0;
	FRACTYPE i,f;

	while(todo--) {
		i=index>>FRACBITS,f=index&FRACMASK;
		sample=(SWORD)((((FRACTYPE)srce[i]*(FRACMASK+1L-f)) +
		               ((FRACTYPE)srce[i+1] * f)) >> FRACBITS);
		index+=increment;

		if(vnf->rampvol) {
			*dest++ +=(
			  ( (FRACTYPE)(vnf->oldlvol*vnf->rampvol) +
			    (vnf->lvolsel*(CLICK_BUFFER-vnf->rampvol)) ) *
			  (FRACTYPE)sample ) >> CLICK_SHIFT;
			*dest++ +=(
			  ( (FRACTYPE)(vnf->oldrvol*vnf->rampvol) +
			    (vnf->rvolsel*(CLICK_BUFFER-vnf->rampvol)) ) *
			  (FRACTYPE)sample ) >> CLICK_SHIFT;
			vnf->rampvol--;
		} else
		  if(vnf->click) {
			*dest++ +=(
			  ( ((FRACTYPE)vnf->lvolsel*(CLICK_BUFFER-vnf->click)) *
			    (FRACTYPE)sample ) + ((FRACTYPE)vnf->lastvalL*vnf->click) )
			  >> CLICK_SHIFT;
			*dest++ +=(
			  ( ((FRACTYPE)vnf->rvolsel*(CLICK_BUFFER-vnf->click)) *
			    (FRACTYPE)sample ) + ((FRACTYPE)vnf->lastvalR*vnf->click) )
			  >> CLICK_SHIFT;
			vnf->click--;
		} else {
			*dest++ +=vnf->lvolsel*sample;
			*dest++ +=vnf->rvolsel*sample;
		}
	}
	vnf->lastvalL=vnf->lvolsel*sample;
	vnf->lastvalR=vnf->rvolsel*sample;

	return index;
}

FRACTYPE MixStereoSurround(SWORD* srce,SLONG* dest,FRACTYPE index,FRACTYPE increment,ULONG todo)
{
	SWORD sample=0;
	long whoop;
	FRACTYPE i,f;

	while(todo--) {
		i=index>>FRACBITS,f=index&FRACMASK;
		sample=(SWORD)((((FRACTYPE)srce[i]*(FRACMASK+1L-f)) +
		               ((FRACTYPE)srce[i+1]*f)) >> FRACBITS);
		index+=increment;

		if(vnf->rampvol) {
			whoop=(
			  ( (FRACTYPE)(vnf->oldlvol*vnf->rampvol) +
			    (vnf->lvolsel*(CLICK_BUFFER-vnf->rampvol)) ) *
			  (FRACTYPE)sample ) >> CLICK_SHIFT;
			*dest++ +=whoop;
			*dest++ -=whoop;
			vnf->rampvol--;
		} else
		  if(vnf->click) {
			whoop=(
			  ( ((FRACTYPE)vnf->lvolsel*(CLICK_BUFFER-vnf->click)) *
			    (FRACTYPE)sample ) + ((FRACTYPE)vnf->lastvalL*vnf->click) )
			  >> CLICK_SHIFT;
			*dest++ +=whoop;
			*dest++ -=whoop;
			vnf->click--;
		} else {
			*dest++ +=vnf->lvolsel*sample;
			*dest++ -=vnf->lvolsel*sample;
		}
	}
	vnf->lastvalL=vnf->lvolsel*sample;
	vnf->lastvalR=vnf->lvolsel*sample;

	return index;
}

#endif

static	void(*Mix32to16)(SWORD* dste,SLONG* srce,SLONG count);
static	void(*Mix32to8)(SBYTE* dste,SLONG* srce,SLONG count);
static	void(*MixReverb)(SLONG* srce,SLONG count);

/* Reverb macros */
#define COMPUTE_LOC(n) loc##n = RVRindex % RVc##n
#define COMPUTE_LECHO(n) RVbufL##n [loc##n ]=speedup+((ReverbPct*RVbufL##n [loc##n ])>>7)
#define COMPUTE_RECHO(n) RVbufR##n [loc##n ]=speedup+((ReverbPct*RVbufR##n [loc##n ])>>7)

void MixReverb_Normal(SLONG* srce,SLONG count)
{
	SLONG speedup;
	int ReverbPct;
	unsigned int loc1,loc2,loc3,loc4,loc5,loc6,loc7,loc8;

	ReverbPct=58+(md_reverb*4);

	COMPUTE_LOC(1); COMPUTE_LOC(2); COMPUTE_LOC(3); COMPUTE_LOC(4);
	COMPUTE_LOC(5); COMPUTE_LOC(6); COMPUTE_LOC(7); COMPUTE_LOC(8);

	while(count--) {
		/* Compute the left channel echo buffers */
		speedup = *srce >> 3;

		COMPUTE_LECHO(1); COMPUTE_LECHO(2); COMPUTE_LECHO(3); COMPUTE_LECHO(4);
		COMPUTE_LECHO(5); COMPUTE_LECHO(6); COMPUTE_LECHO(7); COMPUTE_LECHO(8);

		/* Prepare to compute actual finalized data */
		RVRindex++;

		COMPUTE_LOC(1); COMPUTE_LOC(2); COMPUTE_LOC(3); COMPUTE_LOC(4);
		COMPUTE_LOC(5); COMPUTE_LOC(6); COMPUTE_LOC(7); COMPUTE_LOC(8);

		/* left channel */
		*srce++ +=RVbufL1[loc1]-RVbufL2[loc2]+RVbufL3[loc3]-RVbufL4[loc4]+
		          RVbufL5[loc5]-RVbufL6[loc6]+RVbufL7[loc7]-RVbufL8[loc8];
	}
}

void MixReverb_Stereo(SLONG *srce, SLONG count)
{
	SLONG speedup;
	int ReverbPct;
	unsigned int loc1,loc2,loc3,loc4,loc5,loc6,loc7,loc8;

	ReverbPct=58+(md_reverb*4);

	COMPUTE_LOC(1); COMPUTE_LOC(2); COMPUTE_LOC(3); COMPUTE_LOC(4);
	COMPUTE_LOC(5); COMPUTE_LOC(6); COMPUTE_LOC(7); COMPUTE_LOC(8);

	while(count--) {
		/* Compute the left channel echo buffers */
		speedup = *srce >> 3;

		COMPUTE_LECHO(1); COMPUTE_LECHO(2); COMPUTE_LECHO(3); COMPUTE_LECHO(4);
		COMPUTE_LECHO(5); COMPUTE_LECHO(6); COMPUTE_LECHO(7); COMPUTE_LECHO(8);

		/* Compute the right channel echo buffers */
		speedup = srce[1] >> 3;

		COMPUTE_RECHO(1); COMPUTE_RECHO(2); COMPUTE_RECHO(3); COMPUTE_RECHO(4);
		COMPUTE_RECHO(5); COMPUTE_RECHO(6); COMPUTE_RECHO(7); COMPUTE_RECHO(8);

		/* Prepare to compute actual finalized data */
		RVRindex++;

		COMPUTE_LOC(1); COMPUTE_LOC(2); COMPUTE_LOC(3); COMPUTE_LOC(4);
		COMPUTE_LOC(5); COMPUTE_LOC(6); COMPUTE_LOC(7); COMPUTE_LOC(8);

		/* left channel */
		*srce++ +=RVbufL1[loc1]-RVbufL2[loc2]+RVbufL3[loc3]-RVbufL4[loc4]+ 
		          RVbufL5[loc5]-RVbufL6[loc6]+RVbufL7[loc7]-RVbufL8[loc8];

		/* right channel */
		*srce++ +=RVbufR1[loc1]-RVbufR2[loc2]+RVbufR3[loc3]-RVbufR4[loc4]+
		          RVbufR5[loc5]-RVbufR6[loc6]+RVbufR7[loc7]-RVbufR8[loc8];
	}
}

/* Mixing macros */
#define EXTRACT_SAMPLE(var,attenuation) var=*srce++/(MAXVOL_FACTOR*attenuation)
#define CHECK_SAMPLE(var,bound) var=(var>=bound)?bound-1:(var<-bound)?-bound:var

void Mix32To16_Normal(SWORD* dste,SLONG* srce,SLONG count)
{
	SLONG x1,x2,tmpx;
	int i;

	for(count/=SAMPLING_FACTOR;count;count--) {
		tmpx=0;

		for(i=SAMPLING_FACTOR/2;i;i--) {
			EXTRACT_SAMPLE(x1,1); EXTRACT_SAMPLE(x2,1);

			CHECK_SAMPLE(x1,32768); CHECK_SAMPLE(x2,32768);

			tmpx+=x1+x2;
		}
		*dste++ =tmpx/SAMPLING_FACTOR;
	}
}

void Mix32To16_Stereo(SWORD* dste,SLONG* srce,SLONG count)
{
	SLONG x1,x2,x3,x4,tmpx,tmpy;
	int i;

	for(count/=SAMPLING_FACTOR;count;count--) {
		tmpx=tmpy=0;

		for(i=SAMPLING_FACTOR/2;i;i--) {
			EXTRACT_SAMPLE(x1,1); EXTRACT_SAMPLE(x2,1);
			EXTRACT_SAMPLE(x3,1); EXTRACT_SAMPLE(x4,1);

			CHECK_SAMPLE(x1,32768); CHECK_SAMPLE(x2,32768);
			CHECK_SAMPLE(x3,32768); CHECK_SAMPLE(x4,32768);

			tmpx+=x1+x3;
			tmpy+=x2+x4;
		}
		*dste++ =tmpx/SAMPLING_FACTOR;
		*dste++ =tmpy/SAMPLING_FACTOR;
	}
}

void Mix32To8_Normal(SBYTE* dste,SLONG* srce,SLONG count)
{
	int x1,x2,tmpx;
	int i;

	for(count/=SAMPLING_FACTOR;count;count--) {
		tmpx = 0;

		for(i=SAMPLING_FACTOR/2;i;i--) {
			EXTRACT_SAMPLE(x1,256); EXTRACT_SAMPLE(x2,256);

			CHECK_SAMPLE(x1,128); CHECK_SAMPLE(x2,128);

			tmpx+=x1+x2;
		}
		*dste++ =(tmpx/SAMPLING_FACTOR)+128;
	}
}

void Mix32To8_Stereo(SBYTE* dste,SLONG* srce,SLONG count)
{
	int x1,x2,x3,x4,tmpx,tmpy;
	int i;

	for(count/=SAMPLING_FACTOR;count;count--) {
		tmpx=tmpy=0;

		for(i=SAMPLING_FACTOR/2;i;i--) {
			EXTRACT_SAMPLE(x1,256); EXTRACT_SAMPLE(x2,256);
			EXTRACT_SAMPLE(x3,256); EXTRACT_SAMPLE(x4,256);

			CHECK_SAMPLE(x1,128); CHECK_SAMPLE(x2,128);
			CHECK_SAMPLE(x3,128); CHECK_SAMPLE(x4,128);

			tmpx+=x1+x3;
			tmpy+=x2+x4;
		}
		*dste++ =(tmpx/SAMPLING_FACTOR)+128;        
		*dste++ =(tmpy/SAMPLING_FACTOR)+128;        
	}
}

ULONG samples2bytes(ULONG samples)
{
	if(vc_mode & DMODE_16BITS) samples <<= 1;
	if(vc_mode & DMODE_STEREO) samples <<= 1;
	return samples;
}

ULONG bytes2samples(ULONG bytes)
{
	if(vc_mode & DMODE_16BITS) bytes >>= 1;
	if(vc_mode & DMODE_STEREO) bytes >>= 1;
	return bytes;
}

void AddChannel(SLONG* ptr,SLONG todo)
{
	FRACTYPE end;
	SLONG done;
	SWORD *s;

	if(!(s=Samples[vnf->handle])) {
		vnf->current = vnf->active  = 0;
		vnf->lastvalL = vnf->lastvalR = 0;
		return;
	}

	/* update the 'current' index so the sample loops, or stops playing if it
	   reached the end of the sample */
	while(todo>0) {
		if(vnf->flags & SF_REVERSE) {
			/* The sample is playing in reverse */
			if((vnf->flags&SF_LOOP)&&(vnf->current<idxlpos)) {
				/* the sample is looping and has reached the loopstart index */
				if(vnf->flags & SF_BIDI) {
					/* sample is doing bidirectional loops, so 'bounce' the
					   current index against the idxlpos */
					vnf->current = idxlpos+(idxlpos-vnf->current);
					vnf->flags &= ~SF_REVERSE;
					vnf->increment  = -vnf->increment;
				} else
					/* normal backwards looping, so set the current position to
					   loopend index */
					vnf->current=idxlend-(idxlpos-vnf->current);
			} else {
				/* the sample is not looping, so check if it reached index 0 */
				if(vnf->current < 0) {
					/* playing index reached 0, so stop playing this sample */
					vnf->current = vnf->active  = 0;
					break;
				}
			}
		} else {
			/* The sample is playing forward */
			if((vnf->flags & SF_LOOP) && (vnf->current > idxlend)) {
				/* the sample is looping, check the loopend index */
				if(vnf->flags & SF_BIDI) {
					/* sample is doing bidirectional loops, so 'bounce' the
					   current index against the idxlend */
					vnf->flags |= SF_REVERSE;
					vnf->increment = -vnf->increment;
					vnf->current = idxlend-(vnf->current-idxlend);
				} else
					/* normal backwards looping, so set the current position
					   to loopend index */
					vnf->current=idxlpos+(vnf->current-idxlend);
			} else {
				/* sample is not looping, so check if it reached the last
				   position */
				if(vnf->current > idxsize) {
					/* yes, so stop playing this sample */
					vnf->current = vnf->active  = 0;
					break;
				}
			}
		}

		end=(FRACTYPE)(vnf->flags&SF_REVERSE)?(vnf->flags&SF_LOOP)?idxlpos:0:
		     (vnf->flags&SF_LOOP)?idxlend:idxsize;

		/* if the sample is not blocked... */
		if (vnf->increment)
			done=MIN((end-vnf->current)/vnf->increment+1,todo);
		else
			done=0;

		if(!done) {
			vnf->active = 0;
			break;
		}

		if(vnf->vol || vnf->rampvol) {
			if(vc_mode & DMODE_STEREO)
				if((vnf->pan == PAN_SURROUND) && (vc_mode & DMODE_SURROUND))
					vnf->current = MixStereoSurround(s,ptr,vnf->current,vnf->increment,done);
				else
					vnf->current = MixStereoNormal(s,ptr,vnf->current,vnf->increment,done);
			else
				vnf->current = MixMonoNormal(s,ptr,vnf->current,vnf->increment,done);
		} else 
			vnf->lastvalL = vnf->lastvalR = 0;

		todo -= done;
		ptr +=(vc_mode & DMODE_STEREO)?(done<<1):done;
	}
}

void VC_WriteSamples(SBYTE* buf,ULONG todo)
{
	int left,portion=0;
	SBYTE *buffer;
	int t,pan,vol;

	todo*=SAMPLING_FACTOR;

	while(todo) {
		if(!tickleft) {
			if(vc_mode & DMODE_SOFT_MUSIC) md_player();
			tickleft=(md_mixfreq*125L*SAMPLING_FACTOR)/(md_bpm*50L);
			tickleft&=~(SAMPLING_FACTOR-1);
		}
		left = MIN(tickleft, todo);
		buffer    = buf;
		tickleft -= left;
		todo     -= left;
		buf += samples2bytes(left)/SAMPLING_FACTOR;

		while(left) {
			portion = MIN(left, samplesthatfit);
			memset(vc_tickbuf,0,portion<<((vc_mode&DMODE_STEREO)?3:2));
			for(t=0;t<vc_softchn;t++) {
				vnf = &vinf[t];

				if(vnf->kick) {
#ifdef HAS_64BIT_INT
					vnf->current=(FRACTYPE)(vnf->start)<<FRACBITS;
#else
					vnf->current=(FRACTYPE)vnf->start;
#endif
					vnf->kick    = 0;
					vnf->active  = 1;
					vnf->click   = CLICK_BUFFER;
					vnf->rampvol = 0;
				}

				if(!vnf->frq) vnf->active = 0;

				if(vnf->active) {
#ifdef HAS_64BIT_INT
					vnf->increment=((FRACTYPE)(vnf->frq)<<
					            (FRACBITS-SAMPLING_SHIFT))/(FRACTYPE)md_mixfreq;
#else
					vnf->increment=((FRACTYPE)(vnf->frq)/
					            (1<<SAMPLING_SHIFT))/(FRACTYPE)md_mixfreq;
#endif
					if(vnf->flags&SF_REVERSE) vnf->increment=-vnf->increment;
					vol = vnf->vol;  pan = vnf->pan;

					if(vc_mode & DMODE_STEREO) {
						if(pan!=PAN_SURROUND) {
							vnf->oldlvol=vnf->lvolsel;vnf->oldrvol=vnf->rvolsel;
							vnf->lvolsel=(vol*(255-pan))>>8;
							vnf->rvolsel=(vol*pan)>>8;
						} else {
							vnf->oldlvol=vnf->lvolsel;
							vnf->lvolsel=(vol * 256L) / 480;
						}
					} else {
						vnf->oldlvol = vnf->lvolsel;
						vnf->lvolsel = vol;
					}

#ifdef HAS_64BIT_INT
					idxsize=(vnf->size)?((FRACTYPE)(vnf->size)<<FRACBITS)-1:0;
					idxlend=(vnf->repend)?((FRACTYPE)(vnf->repend)<<FRACBITS)-1:0;
					idxlpos=(FRACTYPE)(vnf->reppos)<<FRACBITS;
#else
					idxsize=(FRACTYPE)(vnf->size);
					idxlend=(FRACTYPE)(vnf->repend);
					idxlpos=(FRACTYPE)(vnf->reppos);
#endif
					AddChannel(vc_tickbuf,portion);
				}
			}

			if(md_reverb) MixReverb(vc_tickbuf,portion);

			if(vc_mode & DMODE_16BITS)
				Mix32to16((SWORD*)buffer,vc_tickbuf,portion);
			else
				Mix32to8((SBYTE*)buffer,vc_tickbuf,portion);

			buffer += samples2bytes(portion) / SAMPLING_FACTOR;
			left   -= portion;
		}
	}
}

/* Fill the buffer with 'todo' bytes of silence (it depends on the mixing mode
   how the buffer is filled) */
void VC_SilenceBytes(SBYTE* buf,ULONG todo)
{
	/* clear the buffer to zero (16 bits signed) or 0x80 (8 bits unsigned) */
	if(vc_mode & DMODE_16BITS)
		memset(buf,0,todo);
	else
		memset(buf,0x80,todo);
}

/* Writes 'todo' mixed SBYTES (!!) to 'buf'. It returns the number of SBYTES
   actually written to 'buf' (which is rounded to number of samples that fit
   into 'todo' bytes). */
ULONG VC_WriteBytes(SBYTE* buf,ULONG todo)
{
	if(!vc_softchn) {
		VC_SilenceBytes(buf,todo);
		return todo;
	}

	todo = bytes2samples(todo);
	VC_WriteSamples(buf,todo);

	return samples2bytes(todo);
}

BOOL VC_Init(void)
{
	if(!(Samples=(SWORD**)_mm_calloc(MAXSAMPLEHANDLES,sizeof(SWORD*)))) {
		_mm_errno = MMERR_INITIALIZING_MIXER;
		return 1;
	}
	if(!vc_tickbuf)
		if(!(vc_tickbuf=(SLONG*)_mm_malloc((TICKLSIZE+32)*sizeof(SLONG)))) {
			_mm_errno = MMERR_INITIALIZING_MIXER;
			return 1;
		}

	if(md_mode & DMODE_STEREO) {
		Mix32to16  = Mix32To16_Stereo;
		Mix32to8   = Mix32To8_Stereo;
		MixReverb  = MixReverb_Stereo;
	} else {
		Mix32to16  = Mix32To16_Normal;
		Mix32to8   = Mix32To8_Normal;
		MixReverb  = MixReverb_Normal;
	}
	md_mode |= DMODE_INTERP;
	vc_mode = md_mode;
	return 0;
}

void VC_Exit(void)
{
	if(vc_tickbuf) free(vc_tickbuf);
	if(vinf) free(vinf);
	if(Samples) free(Samples);

	vc_tickbuf = NULL;
	vinf = NULL;
	Samples = NULL;
}

BOOL VC_PlayStart(void)
{
	md_mode|=DMODE_INTERP;

	samplesthatfit = TICKLSIZE;
	if(vc_mode & DMODE_STEREO) samplesthatfit >>= 1;
	tickleft = 0;

#ifndef HAS_64BIT_INT
	RVc1 = (SLONG)(500.0    * md_mixfreq) / REVERBERATION;
	RVc2 = (SLONG)(507.8125 * md_mixfreq) / REVERBERATION;
	RVc3 = (SLONG)(531.25   * md_mixfreq) / REVERBERATION;
	RVc4 = (SLONG)(570.3125 * md_mixfreq) / REVERBERATION;
	RVc5 = (SLONG)(625.0    * md_mixfreq) / REVERBERATION;
	RVc6 = (SLONG)(695.3125 * md_mixfreq) / REVERBERATION;
	RVc7 = (SLONG)(781.25   * md_mixfreq) / REVERBERATION;
	RVc8 = (SLONG)(882.8125 * md_mixfreq) / REVERBERATION;
#else
	RVc1 = (5000L * md_mixfreq) / (REVERBERATION * 10);
	RVc2 = (5078L * md_mixfreq) / (REVERBERATION * 10);
	RVc3 = (5313L * md_mixfreq) / (REVERBERATION * 10);
	RVc4 = (5703L * md_mixfreq) / (REVERBERATION * 10);
	RVc5 = (6250L * md_mixfreq) / (REVERBERATION * 10);
	RVc6 = (6953L * md_mixfreq) / (REVERBERATION * 10);
	RVc7 = (7813L * md_mixfreq) / (REVERBERATION * 10);
	RVc8 = (8828L * md_mixfreq) / (REVERBERATION * 10);
#endif

	if(!(RVbufL1=(SLONG*)_mm_calloc((RVc1+1),sizeof(SLONG)))) return 1;
	if(!(RVbufL2=(SLONG*)_mm_calloc((RVc2+1),sizeof(SLONG)))) return 1;
	if(!(RVbufL3=(SLONG*)_mm_calloc((RVc3+1),sizeof(SLONG)))) return 1;
	if(!(RVbufL4=(SLONG*)_mm_calloc((RVc4+1),sizeof(SLONG)))) return 1;
	if(!(RVbufL5=(SLONG*)_mm_calloc((RVc5+1),sizeof(SLONG)))) return 1;
	if(!(RVbufL6=(SLONG*)_mm_calloc((RVc6+1),sizeof(SLONG)))) return 1;
	if(!(RVbufL7=(SLONG*)_mm_calloc((RVc7+1),sizeof(SLONG)))) return 1;
	if(!(RVbufL8=(SLONG*)_mm_calloc((RVc8+1),sizeof(SLONG)))) return 1;

	if(!(RVbufR1=(SLONG*)_mm_calloc((RVc1+1),sizeof(SLONG)))) return 1;
	if(!(RVbufR2=(SLONG*)_mm_calloc((RVc2+1),sizeof(SLONG)))) return 1;
	if(!(RVbufR3=(SLONG*)_mm_calloc((RVc3+1),sizeof(SLONG)))) return 1;
	if(!(RVbufR4=(SLONG*)_mm_calloc((RVc4+1),sizeof(SLONG)))) return 1;
	if(!(RVbufR5=(SLONG*)_mm_calloc((RVc5+1),sizeof(SLONG)))) return 1;
	if(!(RVbufR6=(SLONG*)_mm_calloc((RVc6+1),sizeof(SLONG)))) return 1;
	if(!(RVbufR7=(SLONG*)_mm_calloc((RVc7+1),sizeof(SLONG)))) return 1;
	if(!(RVbufR8=(SLONG*)_mm_calloc((RVc8+1),sizeof(SLONG)))) return 1;

	RVRindex = 0;
	return 0;
}

void VC_PlayStop(void)
{
	if(RVbufL1) free(RVbufL1);
	if(RVbufL2) free(RVbufL2);
	if(RVbufL3) free(RVbufL3);
	if(RVbufL4) free(RVbufL4);
	if(RVbufL5) free(RVbufL5);
	if(RVbufL6) free(RVbufL6);
	if(RVbufL7) free(RVbufL7);
	if(RVbufL8) free(RVbufL8);
	if(RVbufR1) free(RVbufR1);
	if(RVbufR2) free(RVbufR2);
	if(RVbufR3) free(RVbufR3);
	if(RVbufR4) free(RVbufR4);
	if(RVbufR5) free(RVbufR5);
	if(RVbufR6) free(RVbufR6);
	if(RVbufR7) free(RVbufR7);
	if(RVbufR8) free(RVbufR8);

	RVbufL1=RVbufL2=RVbufL3=RVbufL4=RVbufL5=RVbufL6=RVbufL7=RVbufL8=NULL;
	RVbufR1=RVbufR2=RVbufR3=RVbufR4=RVbufR5=RVbufR6=RVbufR7=RVbufR8=NULL;
}

BOOL VC_SetNumVoices(void)
{
	int t;

	md_mode|=DMODE_INTERP;

	if(!(vc_softchn=md_softchn)) return 0;

	if(vinf) free(vinf);
	if(!(vinf=_mm_calloc(sizeof(VINFO),vc_softchn))) return 1;

	for(t=0;t<vc_softchn;t++) {
		vinf[t].frq=10000;
		vinf[t].pan=(t&1)?0:255;
	}

	return 0;
}

void VC_VoiceSetVolume(UBYTE voice,UWORD vol)
{    
	/* protect against clicks if volume variation is too high */
	if(abs((int)vinf[voice].vol-(int)vol)>32)
		vinf[voice].rampvol=CLICK_BUFFER;
	vinf[voice].vol=vol;
}

void VC_VoiceSetFrequency(UBYTE voice,ULONG frq)
{
	vinf[voice].frq=frq;
}

void VC_VoiceSetPanning(UBYTE voice,ULONG pan)
{
	/* protect against clicks if panning variation is too high */
	if(abs((int)vinf[voice].pan-(int)pan)>48)
		vinf[voice].rampvol=CLICK_BUFFER;
	vinf[voice].pan=pan;
}

void VC_VoicePlay(UBYTE voice,SWORD handle,ULONG start,ULONG size,ULONG reppos,ULONG repend,UWORD flags)
{
	vinf[voice].flags    = flags;
	vinf[voice].handle   = handle;
	vinf[voice].start    = start;
	vinf[voice].size     = size;
	vinf[voice].reppos   = reppos;
	vinf[voice].repend   = repend;
	vinf[voice].kick     = 1;
}

void VC_VoiceStop(UBYTE voice)
{
	vinf[voice].active = 0;
}  

BOOL VC_VoiceStopped(UBYTE voice)
{
	return(vinf[voice].active==0);
}

void VC_VoiceReleaseSustain(UBYTE voice)
{
}

SLONG VC_VoiceGetPosition(UBYTE voice)
{
#ifdef HAS_64BIT_INT
	return(vinf[voice].current>>FRACBITS);
#else
	return((SLONG)vinf[voice].current);
#endif
}

/*========== External mixer interface */

void VC_SampleUnload(SWORD handle)
{
	if (handle<MAXSAMPLEHANDLES) {
		if (Samples[handle])
			free(Samples[handle]);
		Samples[handle]=NULL;
	}
}

SWORD VC_SampleLoad(SAMPLOAD* sload,int type)
{
	SAMPLE *s=sload->sample;
	int handle;
	ULONG t, length,loopstart,loopend;

	if(type==MD_HARDWARE) return -1;

	/* Find empty slot to put sample address in */
	for(handle=0;handle<MAXSAMPLEHANDLES;handle++)
		if(!Samples[handle]) break;

	if(handle==MAXSAMPLEHANDLES) {
		_mm_errno = MMERR_OUT_OF_HANDLES;
		return -1;
	}

	length    = s->length;
	loopstart = s->loopstart;
	loopend   = s->loopend;

	SL_SampleSigned(sload);

	SL_Sample8to16(sload);
	if(!(Samples[handle]=(SWORD*)_mm_malloc((length+20)<<1))) {
		_mm_errno = MMERR_SAMPLE_TOO_BIG;
		return -1;
	}

	if (SL_Load(Samples[handle],sload,length))
		return -1;

	/* Unclick samples: */
	if(s->flags & SF_LOOP) {
		if(s->flags & SF_BIDI)
			for(t=0;t<16;t++)
				Samples[handle][loopend+t]=Samples[handle][(loopend-t)-1];
		else
			for(t=0;t<16;t++)
				Samples[handle][loopend+t]=Samples[handle][t+loopstart];
	} else
		for(t=0;t<16;t++)
			Samples[handle][t+length]=0;

	return handle;
}

ULONG VC_SampleSpace(int type)
{
	return vc_memory;
}

ULONG VC_SampleLength(int type,SAMPLE* s)
{
	return (s->length*((s->flags&SF_16BITS)?2:1))+16;
}

ULONG VC_VoiceRealVolume(UBYTE voice)
{
	ULONG i,s,size;
	int k,j;
	SWORD *smp;
	SLONG t;

#ifdef HAS_64BIT_INT
	t = vinf[voice].current>>FRACBITS;
#else
	t = (vinf[voice].current)/(1L<<FRACBITS);
#endif
	if(!vinf[voice].active) return 0;

	s = vinf[voice].handle;
	size = vinf[voice].size;

	i=64; t-=64; k=0; j=0;
	if(i>size) i = size;
	if(t<0) t = 0;
	if(t+i > size) t = size-i;

	i &= ~1;  /* make sure it's EVEN. */

	smp = &Samples[s][t];
	for(;i;i--,smp++) {
		if(k<*smp) k = *smp;
		if(j>*smp) j = *smp;
	}
	return abs(k-j);
}
