/*
 *  Digital Audio (PCM) abstract layer / Mixing devices
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "minors.h"
#include "info.h"
#include "pmix.h"

#define __KERNEL_SYSCALLS__
static int errno = 0;
#include <asm/unistd.h>

#ifdef LINUX_2_1
EXPORT_NO_SYMBOLS;
#endif

int snd_freq = 44100;	/* default mixing frequency */
int snd_bits = 16;	/* default sample bits */
int snd_channels = 2;	/* default channels */
int snd_buffer_size = 128;
#ifdef MODULE_PARM
MODULE_PARM( snd_freq, "i" );
MODULE_PARM_DESC( snd_freq, "Mixing frequency (44100Hz)." );
MODULE_PARM( snd_bits, "i" );
MODULE_PARM_DESC( snd_bits, "Sample bits (8 or 16)." );
MODULE_PARM( snd_channels, "i" );
MODULE_PARM_DESC( snd_channels, "Channels (1 or 2)." );
MODULE_PARM( snd_buffer_size, "i" );
MODULE_PARM_DESC( snd_buffer_size, "Virtual ring buffer size (4kB-1024kB)." );
#endif

extern struct snd_info_entry_data snd_pmix_fops;
extern struct snd_info_entry_data snd_pmix_oss_fops;

static int snd_pmix_thread( void *data )
{
  unsigned long flags;
  snd_pmix_data_t *pdata = (snd_pmix_data_t *)data;
  snd_pmix_t *pmix;
  snd_minor_t *reg = pdata -> pcm -> reg;
  mm_segment_t fs;
  unsigned int size;
 
  printk( "thread enter ok.. pdata = 0x%lx\n", (long)pdata );
  if ( pdata -> quit ) {
    printk( "thread quit...\n" );
    snd_wakeup( pdata, tstart );
    return 0;
  }
  printk( "thread start wakeup!!\n" );
  snd_wakeup( pdata, tstart );
  while ( 1 ) {
    if ( pdata -> quit ) {
      pdata -> running = 0;
      printk( "thread quit...\n" );
      snd_wakeup( pdata, tstart );
      break;
    }
    pdata -> running = 1;
    /* ok.. prepare buffer for output (fill with silence) */
    memset( pdata -> buffer, pdata -> format == SND_PCM_SFMT_S16_LE ? 0x80 : 0x00, pdata -> fsize );
    snd_mutex_down( pdata, add );
    for ( pmix = pdata -> first; pmix; pmix = pmix -> next ) {
      snd_spin_lock( pmix, playback, &flags );
      if ( (pmix -> flags & SND_PMIX_FLG_TRIGGERA) == SND_PMIX_FLG_TRIGGERA && pmix -> used > 0 ) {
        pmix -> tail++;
        pmix -> used--;
        snd_wakeup( pmix, sleep );
      }
      snd_spin_unlock( pmix, playback, &flags );
    }
    snd_mutex_up( pdata, add );
    fs = snd_enter_user();
    size = reg -> write( &pdata -> file, pdata -> buffer, pdata -> fsize );
    snd_leave_user( fs );
    printk( "write ok - size = %i...\n", size );
  }
  return 0;
}

int snd_pmix_open_device( snd_pmix_data_t *pdata, snd_pmix_t *pmix )
{
  unsigned short minor;
  snd_minor_t *reg;
  int i, err, pid;
  mm_segment_t fs;
  snd_pcm_format_t format;
  snd_pcm_playback_params_t params;
  snd_pcm_playback_status_t status;
  snd_pmix_t *pmix1;
  
  pmix -> next = NULL;
  pmix -> pdata = pdata;
  reg = pdata -> pcm -> reg;
  if ( !reg ) return -ENXIO;
  snd_mutex_down( pdata, open );
  printk( "pdata -> inputs = %i\n", pdata -> inputs );
  if ( pdata -> inputs++ > 0 ) {
    snd_mutex_down( pdata, add );
    for ( pmix1 = pdata -> first; pmix1 -> next; pmix1 = pmix1 -> next );
    pmix1 -> next = pmix;
    snd_mutex_up( pdata, add );
    snd_mutex_up( pdata, open );
    return 0;
  }
  printk( "ok\n" );
  minor = SND_MINOR_PCM + (pdata -> card -> number << 2) + pdata -> device;
  pdata -> file.f_flags = O_WRONLY;	/* playback only */
  if ( (err = reg -> open( minor, pdata -> card -> number, pdata -> device, &pdata -> file )) < 0 ) {
    snd_mutex_up( pdata, open );
    return err;
  }
  /* setup playback format */
  memset( &format, 0, sizeof( format ) );
  format.rate = snd_freq;
  if ( format.rate >= (44100+22050)/2 ) format.rate = 44100; else
  if ( format.rate >= (22050+11025)/2 ) format.rate = 22050; else
                                        format.rate = 11025;
  pdata -> rate = format.rate;
  pdata -> format = format.format = snd_bits == 8 ? SND_PCM_SFMT_U8 : SND_PCM_SFMT_S16_LE;
  pdata -> channels = format.channels = snd_channels;
  if ( pdata -> ssize < 4 * 1024 ) pdata -> ssize = 4 * 1024; else
  if ( pdata -> ssize > 1024 * 1024 ) pdata -> ssize = 1024 * 1024;
  fs = snd_enter_user();
  if ( (err = reg -> ioctl( &pdata -> file, SND_PCM_IOCTL_PLAYBACK_FORMAT, (unsigned long)&format )) < 0 ) {
    reg -> release( minor, pdata -> card -> number, pdata -> device, &pdata -> file );
    snd_leave_user( fs );
    snd_mutex_up( pdata, open );
    snd_printk( "pmix%i%i: unable to setup format (%i)\n", pdata -> card -> number, pdata -> device, err );
    return err;
  }
  snd_leave_user( fs );
  /* setup playback parameters */
  memset( &params, 0, sizeof( params ) );
  params.fragment_size = ( snd_freq * snd_bits * snd_channels ) / ( 100 * 4 );
  for ( i = 31; i >= 0; i-- )
    if ( params.fragment_size & (1 << i) ) break;
  if ( i >= 0 ) i--;
  for ( ; i >= 0; i-- )
    params.fragment_size &= ~(1 << i);
  params.fragments_max = 2;
  params.fragments_room = 1;
  fs = snd_enter_user();
  if ( (err = reg -> ioctl( &pdata -> file, SND_PCM_IOCTL_PLAYBACK_PARAMS, (unsigned long)&params )) < 0 ) {
    reg -> release( minor, pdata -> card -> number, pdata -> device, &pdata -> file );
    snd_leave_user( fs );
    snd_mutex_up( pdata, open );
    snd_printk( "pmix%i%i: unable to setup params (%i)\n", pdata -> card -> number, pdata -> device, err );
    return err;
  }
  snd_leave_user( fs );
  /* get playback status */
  fs = snd_enter_user();
  if ( (err = reg -> ioctl( &pdata -> file, SND_PCM_IOCTL_PLAYBACK_STATUS, (unsigned long)&status )) < 0 ) {
    reg -> release( minor, pdata -> card -> number, pdata -> device, &pdata -> file );
    snd_leave_user( fs );
    snd_mutex_up( pdata, open );
    snd_printk( "pmix%i%i: unable to get status (%i)\n", pdata -> card -> number, pdata -> device, err );
    return err;
  }
  snd_leave_user( fs );
  if ( status.rate != snd_freq ) {
    reg -> release( minor, pdata -> card -> number, pdata -> device, &pdata -> file );
    snd_mutex_up( pdata, open );
    snd_printk( "pmix%i%i: unsupported playback rate %iHz (%i)\n", pdata -> card -> number, pdata -> device, snd_freq, err );
    return -EINVAL;
  }
  pdata -> fsize = status.fragment_size;
  pdata -> bsize = status.fragments * status.fragment_size;
  printk( "fsize = %i, bsize = %i\n", pdata -> fsize, pdata -> bsize );
  pdata -> buffer = (unsigned char *)snd_malloc( pdata -> fsize );
  if ( !pdata -> buffer ) {
    reg -> release( minor, pdata -> card -> number, pdata -> device, &pdata -> file );
    snd_mutex_up( pdata, open );
    return -ENOMEM;
  }
  /* start mixing thread */
  pdata -> quit = pdata -> running = 0;
  snd_sleep_prepare( pdata, tstart );
  snd_sleep_prepare( pdata, tsleep );
  pid = kernel_thread( snd_pmix_thread, (void *)pdata, 0 );
  if ( pid < 0 ) {
    reg -> release( minor, pdata -> card -> number, pdata -> device, &pdata -> file );
    snd_free( pdata -> buffer, pdata -> fsize ); pdata -> buffer = NULL;
    snd_mutex_up( pdata, open );
    snd_printk( "pmix%i%i: unable to start kernel thread\n", pdata -> card -> number, pdata -> device );
    return -EAGAIN;
  }
  pdata -> pid = pid;
  while ( !pdata -> running ) {
    snd_sleep( pdata, tstart, HZ );
  }
  pdata -> first = pmix;
  snd_mutex_up( pdata, open );
  return 0;
}

int snd_pmix_release_device( snd_pmix_data_t *pdata, snd_pmix_t *pmix )
{
  unsigned short minor;
  snd_minor_t *reg;
  snd_pmix_t *pmix1;
  int err;
  
  reg = pdata -> pcm -> reg;
  if ( !reg ) return -ENXIO;
  snd_mutex_down( pdata, open );
  if ( !pdata -> inputs ) {
    snd_mutex_up( pdata, open );
    return -EINVAL;
  }
  if ( --pdata -> inputs > 0 ) {
    if ( pmix == pdata -> first ) {
      pdata -> first = pmix -> next;
    } else {
      for ( pmix1 = pdata -> first; pmix1 && pmix1 -> next != pmix; pmix1 = pmix1 -> next );
      if ( pmix1 ) pmix1 -> next = pmix -> next;
    }
    snd_mutex_up( pdata, open );
    return 0;
  }
  pdata -> quit = 1;
  snd_wakeup( pdata, tsleep );
  while ( pdata -> running ) {
    snd_sleep( pdata, tstart, HZ );
  }
  minor = SND_MINOR_PCM + (pdata -> card -> number << 2) + pdata -> device;
  if ( (err = reg -> release( minor, pdata -> card -> number, pdata -> device, &pdata -> file )) < 0 ) {
    snd_mutex_up( pdata, open );
    return err;
  }
  snd_free( pdata -> buffer, pdata -> fsize );
  snd_mutex_up( pdata, open );
  return 0;
}

/*
 *  ENTRY functions
 */

static int snd_pmix_register_minor( unsigned short native_minor, snd_pcm_t *pcm )
{
  int card, device;
  snd_info_entry_t *entry, *entry1;
  char buf[16];
  snd_pmix_data_t *data;

  card = (native_minor - SND_MINOR_PCM) >> 2;
  device = native_minor & (SND_PCM_DEVICES-1);
  data = (snd_pmix_data_t *)snd_calloc( sizeof( snd_pmix_data_t ) );
  if ( !data ) return -ENOMEM;

  sprintf( buf, "pmix%i%i", card, device );
  entry = snd_info_create_entry( NULL, buf );
  if ( !entry ) {
    snd_free( data, sizeof( snd_pmix_data_t ) );
    return -ENOMEM;
  }
  entry -> type = SND_INFO_ENTRY_DATA;
  entry -> subtype = SND_INFO_ENTRY_SDEVICE;
  entry -> mode = S_IFCHR | S_IRUGO | S_IWUGO;
  memcpy( &entry -> t.data, &snd_pmix_fops, sizeof( snd_pmix_fops ) );
  entry -> private_data = (void *)data;
  if ( snd_info_register( entry ) < 0 ) {
    snd_info_free_entry( entry );
    snd_free( data, sizeof( snd_pmix_data_t ) );
    return -ENOMEM;
  }

  sprintf( buf, "pomix%i%i", card, device );
  entry1 = snd_info_create_entry( NULL, buf );
  if ( !entry1 ) {
    snd_info_free_entry( entry );
    snd_free( data, sizeof( snd_pmix_data_t ) );
    return -ENOMEM;
  }
  entry1 -> type = SND_INFO_ENTRY_DATA;
  entry1 -> subtype = SND_INFO_ENTRY_SDEVICE;
  entry1 -> mode = S_IFCHR | S_IRUGO | S_IWUGO;
  memcpy( &entry1 -> t.data, &snd_pmix_oss_fops, sizeof( snd_pmix_fops ) );
  entry1 -> private_data = (void *)data;
  if ( snd_info_register( entry1 ) < 0 ) {
    snd_info_free_entry( entry1 );
    snd_info_free_entry( entry );
    snd_free( data, sizeof( snd_pmix_data_t ) );
    return -ENOMEM;
  }

  data -> card = snd_cards[ card ];
  data -> device = device;
  data -> pcm = pcm;
  data -> dev = entry;
  data -> oss_dev = entry1;
  data -> ssize = snd_buffer_size * 1024;
  snd_mutex_prepare( data, open );
  snd_mutex_prepare( data, add );
  pcm -> mix_private_data = (void *)data;
  return 0;
}

static int snd_pmix_unregister_minor( unsigned short native_minor, snd_pcm_t *pcm )
{
  int card, device;
  snd_pmix_data_t *data;

  card = (native_minor - SND_MINOR_PCM) >> 2;
  device = native_minor & (SND_PCM_DEVICES-1);
  data = (snd_pmix_data_t *)pcm -> mix_private_data;
  pcm -> mix_private_data = NULL;
  if ( data -> dev )
    snd_info_unregister( data -> dev );
  if ( data -> oss_dev )
    snd_info_unregister( data -> oss_dev );
  snd_free( data, sizeof( snd_pmix_data_t ) );
  return 0;
}

static struct snd_stru_pcm_notify snd_pmix_notify = {
  snd_pmix_register_minor,
  snd_pmix_unregister_minor,
  NULL
};

int init_module( void )
{
  int err;

#ifndef LINUX_2_1
  if ( register_symtab( NULL ) < 0 )
    return -ENOMEM;
#endif
  if ( (err = snd_pcm_notify( &snd_pmix_notify, 0 )) < 0 )
    return err;  
  return 0;
}

void cleanup_module( void )
{
  snd_pcm_notify( &snd_pmix_notify, 1 );
}
