/*
 *  Information interface for ALSA driver
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "minors.h"
#include "info.h"

/*
 *  OSS compatible part
 */

snd_mutex_define_static( strings );
static char *snd_sndstat_strings[ SND_CARDS * SND_OSS_INFO_DEV_COUNT ];

int snd_oss_info_register( int dev, int num, char *string )
{
  char *x;

  if ( dev < 0 || dev >= SND_OSS_INFO_DEV_COUNT ) return -EINVAL;
  if ( num < 0 || num >= SND_CARDS ) return -EINVAL;
  snd_mutex_down_static( strings );
  if ( !string ) {
    if ( (x = snd_sndstat_strings[ dev * SND_CARDS + num ]) != NULL ) {
      snd_free_str( x );
      x = NULL;
    }
  } else {
    x = snd_malloc_strdup( string );
    if ( !x ) {
      snd_mutex_up_static( strings );
      return -ENOMEM;
    }
  }
  snd_sndstat_strings[ dev * SND_CARDS + num ] = x;
  snd_mutex_up_static( strings );
  return 0;
}

static long snd_sndstat_read( struct file *file, char *buffer, long count )
{
  snd_info_buffer_t *buf;
  long size, size1;
  
  buf = (snd_info_buffer_t *)file -> private_data;
  if ( !buf ) return -EIO;
  if ( file -> f_pos >= buf -> size ) return 0;
  size = buf -> size < count ? buf -> size : count;
  size1 = buf -> size - file -> f_pos;
  if ( size1 < size ) size = size1;
  if ( verify_area( VERIFY_WRITE, buffer, size ) ) return -EFAULT;
  copy_to_user( buffer, buf -> buffer + file -> f_pos, size );
  file -> f_pos += size;
  return size;
}

extern void snd_card_info_read_oss( snd_info_buffer_t *buffer );

static int snd_sndstat_show_strings( snd_info_buffer_t *buf, char *id, int dev )
{
  int idx, count, ok = -1;
  char *str;

  snd_iprintf( buf, "\n%s: ", id );
  snd_mutex_down_static( strings );
  for ( idx = dev * SND_CARDS, count = idx + SND_CARDS; idx < count; idx++ ) {
    str = snd_sndstat_strings[ idx ];
    if ( str ) {
      if ( ok < 0 )
        snd_iprintf( buf, "\n" );
      ok = SND_CARDS - (count - idx);
      snd_iprintf( buf, "%i: %s\n", ok, str );
    }
  }
  snd_mutex_up_static( strings );
  if ( ok < 0 )
    snd_iprintf( buf, "NOT ENABLED IN CONFIG\n" );
  return ok;
}

static int snd_sndstat_open( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_info_buffer_t *buf;

  file -> private_data = snd_calloc( sizeof( snd_info_buffer_t ) + 2048 );
  if ( !file -> private_data )
    return -ENOMEM;
  MOD_INC_USE_COUNT;
  buf = (snd_info_buffer_t *)file -> private_data;
  buf -> buffer = buf -> curr = ((char *)file -> private_data) + sizeof( snd_info_buffer_t );
  buf -> len = 2048;
  snd_iprintf( buf, "Sound Driver:3.8.1a-980706 (ALSA v" SND_VERSION " emulation code)\n" );
  snd_iprintf( buf, "Kernel: %s %s %s %s %s\n",
  			system_utsname.sysname,
  			system_utsname.nodename,
  			system_utsname.release,
  			system_utsname.version,
  			system_utsname.machine );
  snd_iprintf( buf, "Config options: 0\n" );
  snd_iprintf( buf, "\nInstalled drivers: \n" );
  snd_iprintf( buf, "Type 10: ALSA emulation\n" );
  snd_iprintf( buf, "\nCard config: \n" );
  snd_card_info_read_oss( buf );
  snd_sndstat_show_strings( buf, "Audio devices", SND_OSS_INFO_DEV_AUDIO );
  snd_sndstat_show_strings( buf, "Synth devices", SND_OSS_INFO_DEV_SYNTH );
  snd_sndstat_show_strings( buf, "Midi devices", SND_OSS_INFO_DEV_MIDI );
  snd_sndstat_show_strings( buf, "Timers", SND_OSS_INFO_DEV_TIMERS );
  snd_sndstat_show_strings( buf, "Mixers", SND_OSS_INFO_DEV_MIXERS );
  return 0;
}

static int snd_sndstat_release( unsigned short minor, int cardnum, int device, struct file *file )
{
  if ( file -> private_data ) {
    snd_free( file -> private_data, sizeof( snd_info_buffer_t ) + 2048 );
    file -> private_data = NULL;
  }
  MOD_DEC_USE_COUNT;
  return 0;
}

static snd_minor_t snd_sndstat_reg = {
  "sndstat",

  NULL,                                 /* unregister */

  NULL,                                 /* lseek */
  snd_sndstat_read,                     /* read */
  NULL,                                 /* write */
  snd_sndstat_open,                     /* open */
  snd_sndstat_release,                  /* release */
  NULL,                                 /* poll or select */
  NULL,                                 /* ioctl */
  NULL                                  /* mmap */
};

int snd_info_minor_register( void )
{
  memset( snd_sndstat_strings, 0, sizeof( snd_sndstat_strings ) );
  return snd_register_minor( SND_MINOR_OSS_SNDSTAT, &snd_sndstat_reg );
}

