package SyncAB;
# Address Book conduit for PilotManager
# 3/17/98 Alan.Harder@Sun.COM
# http://www.moshpit.org/pilotmgr
#
# CSV reading code..
# while ($x) {
#  if ($x =~ s/^"\s*((?:[^"]|"")*)"(?:,|$)// ||
#      $x =~ s/^\s*(.*?)(?:,|$)//)
#  {
#    $y = $1;
#  } else { print "bad\n"; }
#
#  $y =~ s/""/"/g;
#  print "$y\n";
# }

use PilotSync;
use Tk;
use TkUtils;
use Data::Dumper;

my ($VERSION) = ('0.93 PRE-BETA');
my ($RCFILE, $APPINFO_FILE);
my ($gConfigDialog, $gFileLabel, $gFileEntry);
my ($gEntryMap) = ({ 'lastname' => 0,
		     'firstname' => 1,
		     'company' => 2,
		     'phone1' => 3,
		     'phone2' => 4,
		     'phone3' => 5,
		     'phone4' => 6,
		     'phone5' => 7,
		     'address' => 8,
		     'city' => 9,
		     'state' => 10,
		     'zip' => 11,
		     'country' => 12,
		     'title' => 13,
		     'custom1' => 14,
		     'custom2' => 15,
		     'custom3' => 16,
		     'custom4' => 17,
		     'note' => 18,
		     'whichphone' => 'showPhone',
		     'phonetypes' => 'phoneLabel',
		     'category' => 'category',
		     'rolo_id' => 'rolo_id',
		     'updatetop' => 'updatetop'});
my ($gCSVorder) = ([ 'rolo_id', 'lastname', 'firstname', 'company',
		     'phone1', 'phone2', 'phone3', 'phone4', 'phone5',
		     'address', 'city', 'state', 'zip', 'country', 'title',
		     'custom1', 'custom2', 'custom3', 'custom4', 'note',
		     'whichphone', 'phonetypes', 'category' ]);

sub conduitInit
{
    $RCFILE = "SyncAB/SyncAB.prefs";
    $APPINFO_FILE = "SyncAB/pilot.appinfo";
    &loadPrefs;

    $PREFS->{'syncType'} = 'CSV'
	unless (defined $PREFS->{'syncType'});
    $PREFS->{'CSVFile'} = "$ENV{HOME}/.csvAddr"
	unless (defined $PREFS->{'CSVFile'});
    $PREFS->{'vCardFile'} = "$ENV{HOME}/.vCards"
	unless (defined $PREFS->{'vCardFile'});
    $PREFS->{'vCardDir'} = "$ENV{HOME}/.dt/Addresses"
	unless (defined $PREFS->{'vCardDir'});
    $PREFS->{'RoloFile'} = "$ENV{HOME}/.rolo"
	unless (defined $PREFS->{'RoloFile'});
}

sub conduitQuit
{
    &savePrefs;
}

sub conduitInfo
{
    return { 'database' =>
		{
		    'name' => 'AddressDB',
		    'creator' => 'addr',
		    'type' => 'DATA',
		    'flags' => 0,
		    'version' => 0,
		},
	     'version' => $VERSION,
	     'author' => 'Alan Harder',
	     'email' => 'Alan.Harder@Sun.COM' };
}

sub conduitConfigure
{
    my ($this, $wm) = @_;
    my ($frame, $obj, $subfr, @objs);

    unless (defined($gConfigDialog))
    {
	$gConfigDialog = $wm->Toplevel(-title => "Configuring SyncAB");
	$gConfigDialog->transient($wm);
	$frame = $gConfigDialog->Frame;

	$subfr = $frame->Frame;
	@objs = TkUtils::Radiobuttons($subfr, \$PREFS->{'syncType'},
				      'CSV', 'vCard single file',
				      'vCard one per file', 'Rolo');
	$objs[0]->configure(-command => sub{
	    $gFileLabel = 'CSV file:';
	    $gFileEntry->configure(-textvariable => \$PREFS->{'CSVFile'});
	});
	$objs[1]->configure(-command => sub{
	    $gFileLabel = 'vCard file:';
	    $gFileEntry->configure(-textvariable => \$PREFS->{'vCardFile'});
	});
	$objs[2]->configure(-command => sub{
	    $gFileLabel = 'vCard dir:';
	    $gFileEntry->configure(-textvariable => \$PREFS->{'vCardDir'});
	});
	$objs[3]->configure(-command => sub{
	    $gFileLabel = 'Rolo file:';
	    $gFileEntry->configure(-textvariable => \$PREFS->{'RoloFile'});
	});
	$subfr->pack;

	$subfr = $frame->Frame;
	$obj = $subfr->Label(-textvariable => \$gFileLabel,
			     -width => 10, -justify => 'right');
	$obj->pack(-side => 'left', -anchor => 'e');

	$gFileEntry = $subfr->Entry(-relief => 'sunken', -width => 40);
	$gFileEntry->pack;
	$subfr->pack;

	$obj = TkUtils::Button($frame, 'Dismiss',
			       sub{ $gConfigDialog->withdraw });
	$obj->pack;

	$frame->pack;
	PilotMgr::setColors($gConfigDialog);
    }

    if ($PREFS->{'syncType'} eq 'Rolo')
    {
	$gFileLabel = 'Rolo file:';
	$gFileEntry->configure(-textvariable => \$PREFS->{'RoloFile'});
    }
    elsif ($PREFS->{'syncType'} eq 'CSV')
    {
	$gFileLabel = 'CSV file:';
	$gFileEntry->configure(-textvariable => \$PREFS->{'CSVFile'});
    }
    elsif ($PREFS->{'syncType'} eq 'vCard single file')
    {
	$gFileLabel = 'vCard file:';
	$gFileEntry->configure(-textvariable => \$PREFS->{'vCardFile'});
    }
    elsif ($PREFS->{'syncType'} eq 'vCard one per file')
    {
	$gFileLabel = 'vCard dir:';
	$gFileEntry->configure(-textvariable => \$PREFS->{'vCardDir'});
    }

    $gConfigDialog->Popup(-popanchor => 'c', -overanchor => 'c',
			  -popover => $wm);
}

sub conduitSync
{
    my ($this, $dlp, $info) = @_;
    my ($idField, $file, $reader, $writer);

    if (!exists $PREFS->{'lastSyncType'} or
	$PREFS->{'syncType'} ne $PREFS->{'lastSyncType'})
    {
	rename "SyncAB/addr.db", "SyncAB/addr.db.bak";
    }

    if ($PREFS->{'syncType'} eq 'Rolo')
    {
	$idField = 'rolo_id';
	$file = $PREFS->{'RoloFile'};
	$reader = \&readRolo;
	$writer = \&writeRolo;
    }
    elsif ($PREFS->{'syncType'} eq 'CSV')
    {
	$idField = 'rolo_id';
	$file = $PREFS->{'CSVFile'};
	$reader = \&readCSV;
	$writer = \&writeCSV;
    }
    else
    {
	$idField = 'rolo_id';
	PilotMgr::msg(
	    "SyncAB does not yet support type $PREFS->{'syncType'}\n");
	return;
    }

    PilotSync::doSync(	$dlp,
			&conduitInfo->{'database'},
			['entry', 'phoneLabel', 'showPhone', 'category'],
			['categoryName', 'phoneLabel', 'label'],
			$idField,
			"SyncAB/addr.db",
			$file,
			\&titleString,
			$reader,
			$writer,
			\&newRoloId);

    $PREFS->{'lastSyncType'} = $PREFS->{'syncType'};
}

sub loadPrefs
{
    my ($lines);

    open(FD, "<$RCFILE") || return;
    $lines = join('', <FD>);
    close(FD);
    eval $lines;
}

sub savePrefs
{
    $Data::Dumper::Purity = 1;
    $Data::Dumper::Deepcopy = 1;
    $Data::Dumper::Indent = 0;

    if (open(FD, ">$RCFILE"))
    {
	print FD Data::Dumper->Dumpxs([$PREFS], ['PREFS']);
	print FD "1;\n";
	close(FD);
    }
    else
    {
	PilotMgr::msg("Unable to save preferences to $RCFILE!");
    }
}

sub newRoloId
{
    my ($db) = @_;

    return $db->{'NEXT_ID'}++;
}

sub titleString
{
    my ($rec) = @_;
    my ($str, $str2) = ('');

    $str2 = $rec->{'entry'}->[$gEntryMap->{'firstname'}];
    $str = $str2 if (defined $str2 && length $str2);

    $str2 = $rec->{'entry'}->[$gEntryMap->{'lastname'}];
    if (defined $str2 && length $str2)
    {
	$str .= ' ' if (length $str);
	$str .= $str2;
    }
    return $str if (length $str);

    $str2 = $rec->{'entry'}->[$gEntryMap->{'company'}];
    return $str2 if (defined $str2 && length $str2);

    return '-Unnamed-';
}

sub readAppInfoFile
{
    # AppInfo file used by Rolo and CSV formats
    #
    my ($ai, $s) =
	({ 'categoryName' => [], 'label' => [], 'phoneLabel' => [] });

    open(FD, "<$APPINFO_FILE") || return $ai;
    scalar(<FD>);	# read off comment line

    foreach (1..16)
    {
	chomp($s = <FD>);
	push(@{$ai->{'categoryName'}}, $s);
    }
    foreach (1..22)
    {
	chomp($s = <FD>);
	push(@{$ai->{'label'}}, $s);
    }
    foreach (1..8)
    {
	chomp($s = <FD>);
	push(@{$ai->{'phoneLabel'}}, $s);
    }

    close(FD);
    return $ai;
}

sub writeAppInfoFile
{
    my ($ai) = @_;

    open(FD, ">$APPINFO_FILE") || return;
    print FD <<EOW;
WARNING- If you edit this file it will modify your pilot on the next sync!
EOW

    foreach $_ (@{$ai->{'categoryName'}},
		@{$ai->{'label'}},
		@{$ai->{'phoneLabel'}})
    {
	print FD "$_\n";
    }

    close(FD);
}

sub readRolo
{
    my ($ROLOFILE) = @_;
    my ($db, $rec) = ({ 'nonPilot' => [],
			'isPilot' => [],
			'__RECORDS' => [],
			'NEXT_ID' => 0
		      });

    $db->{'__APPINFO'} = &readAppInfoFile;
    open(FD, "<$ROLOFILE") || return $db;

    while (<FD>)
    {
	$rec = { 'topsect' => '' };

	while ($_ !~ /^\*PILOT\*$/ && $_ !~ /^\014/)
	{
	    $rec->{'topsect'} .= $_;
	    $_ = <FD>;
	}

	if ( /^\014/ )
	{
	    push(@{$db->{'isPilot'}}, -1);
	    push(@{$db->{'nonPilot'}}, $rec);
	    next;
	}

	$rec->{'entry'} = [];
	$rec->{'entry'}->[18] = undef;  # ensure right array length
	$rec->{'phoneLabel'} = [0,1,2,3,4];
	$rec->{'showPhone'} = 0;

	for ($_ = <FD>; $_ !~ /^\014/; $_ = <FD>)
	{
	    if ($_ =~ /^([^:]*): ?(.*)$/)
	    {
		$field = $1;
		$value = $2;
		$field =~ tr/A-Z/a-z/;
		$value =~ s/\\n/\n/g;	# translate newlines

		unless (defined $gEntryMap->{$field})
		{
		    print "skipping bad field '$field' in rolo record.\n";
		    next;
		}
		$field = $gEntryMap->{$field};

		if ($field =~ /^\d+$/)
		{
		    $rec->{'entry'}->[$field] = $value;
		}
		elsif ($field eq 'phoneLabel')
		{
		    $rec->{$field} = [split(/ /, $value)];
		}
		else
		{
		    $rec->{$field} = $value;
		}
	    }
	}
	push(@{$db->{'isPilot'}}, $rec->{'rolo_id'});
	push(@{$db->{'__RECORDS'}}, $rec);
	$db->{ $rec->{'rolo_id'} } = $#{$db->{'__RECORDS'}};

	$db->{'NEXT_ID'} = $rec->{'rolo_id'} + 1
	    if ($rec->{'rolo_id'} >= $db->{'NEXT_ID'});
    }
    close(FD);
    return $db;
}

sub writeRolo
{
    my ($ROLOFILE, $db) = @_;
    my ($rec, $which);

    &writeAppInfoFile($db->{'__APPINFO'});
    unless (open(FD, ">$ROLOFILE"))
    {
	PilotMgr::msg("Unable to write to $ROLOFILE.  Help!");
	return;
    }

    foreach $which (@{$db->{'isPilot'}})
    {
	if ($which < 0)
	{
	    # non-pilot rec
	    $rec = shift @{$db->{'nonPilot'}};
	    print FD $rec->{'topsect'}, "\014\n";
	}

	next unless (defined $db->{'__RECORDS'}->[0] &&
		     $which eq $db->{'__RECORDS'}->[0]->{'rolo_id'});
	$rec->{'topsect'} = &makeTopSect($rec, $db->{'__APPINFO'})
	    unless exists $rec->{'topsect'};
	&writeRec(FD, shift @{$db->{'__RECORDS'}});
    }

    while (defined ($rec = shift @{$db->{'__RECORDS'}}))
    {
	$rec->{'topsect'} = &makeTopSect($rec, $db->{'__APPINFO'})
	    unless exists $rec->{'topsect'};
	&writeRec(FD, $rec);
    }

    close(FD);
}

sub writeRec
{
    my ($fd, $rec) = @_;
    my ($key, $val);

    print $fd $rec->{'topsect'} if defined ($rec->{'topsect'});
    print $fd "*PILOT*\n";
    foreach $key (keys %$gEntryMap)
    {
        $val = $gEntryMap->{$key};
        if ($val =~ /^\d+$/)
        {
            next unless (defined ($val = $rec->{'entry'}->[$val]));
            $val =~ s/\n/\\n/g; # translate newlines
            print $fd "$key: $val\n";
        }
        else
        {
            # shouldn't be any newlines to translate down here..
            next unless (defined ($val = $rec->{$val}));
            # for phoneLabel field:
            $val = join(' ', @$val) if (ref($val) eq 'ARRAY');
            print $fd "$key: $val\n";
        }
    }
    print $fd "\014\n";
}

sub isgood
{
    return (defined $_[0] and length($_[0]) > 0);
}

sub makeTopSect
{
    my ($rec, $ai) = @_;
    my ($topsect, $boo, $val, $val2, $i, @phonetypes) = ("", 0);

    $val = $rec->{'entry'}->[ $gEntryMap->{'lastname'} ];
    $val2 = $rec->{'entry'}->[ $gEntryMap->{'firstname'} ];
    $i = $rec->{'entry'}->[ $gEntryMap->{'company'} ];
    if (&isgood($val))
    {
	$topsect .= "$val2 " if (&isgood($val2));
	$topsect .= $val;
    }
    elsif (&isgood($val2))
    {
	$topsect .= $val2;
    }
    elsif (&isgood($i))
    {
	$topsect .= $i;
	$boo = 1;
    }

    $topsect .= "\n";
    $val = $rec->{'entry'}->[ $gEntryMap->{'title'} ];
    $topsect .= $val . "\n" if (&isgood($val));
    $val = $rec->{'entry'}->[ $gEntryMap->{'company'} ];
    $topsect .= $val . "\n" if (!$boo and &isgood($val));
    $topsect .= "\n";

    $val = $rec->{ $gEntryMap->{'phonetypes'} };
    @phonetypes = @$val if (defined $val and ref($val) eq 'ARRAY');

    foreach $i (1..5)
    {
	$val = $rec->{'entry'}->[ $gEntryMap->{"phone$i"} ];
	if (&isgood($val))
	{
	    $topsect .= @phonetypes ? $ai->{'phoneLabel'}->[$phonetypes[$i-1]]
				    : "phone$i";
	    $topsect .= ": $val\n";
	}
    }
    $topsect .= "\n";

    $val = $rec->{'entry'}->[ $gEntryMap->{'address'} ];
    $topsect .= $val . "\n" if (&isgood($val));
    $boo = 0;
    $val = $rec->{'entry'}->[ $gEntryMap->{'city'} ];
    if (&isgood($val))
    {
	$topsect .= $val;
	$boo = 1;
    }
    $val = $rec->{'entry'}->[ $gEntryMap->{'state'} ];
    if (&isgood($val))
    {
	$topsect .= ", " if ($boo);
	$boo = 0;
	$topsect .= "$val  ";
    }
    $val = $rec->{'entry'}->[ $gEntryMap->{'zip'} ];
    if (&isgood($val))
    {
	$topsect .= ', ' if ($boo);
	$topsect .= $val;
    }
    $topsect .= "\n";
    $boo = 0;
    foreach $i (1..4)
    {
	$val = $rec->{'entry'}->[ $gEntryMap->{"custom$i"} ];
	if (&isgood($val))
	{
	    $topsect .= $ai->{'label'}->[$i+13] . ": $val\n";
	    $boo++;
	}
    }
    $topsect .= "\n" if ($boo);

    $val = $rec->{'entry'}->[ $gEntryMap->{'note'} ];
    $topsect .= $val . "\n" if (&isgood($val));

    return $topsect;
}

sub readCSV
{
    my ($CSVFILE) = @_;
    my ($max_id, $db) = (-1, { '__RECORDS' => [] , 'NEXT_ID' => 0 });
    my ($rec, $key, $fld, $val);

    $db->{'__APPINFO'} = &readAppInfoFile;
    open(FD, "<$CSVFILE") || return $db;

    while (<FD>)
    {
	$rec = { 'entry' => [],
		 'showPhone' => 0,
		 'phoneLabel' => [0,1,2,3,4] };
	$rec->{'entry'}->[18] = undef;  # ensure right array length

	foreach $key (@$gCSVorder)
	{
	    $fld = $gEntryMap->{$key};
	    ($val, $_) = &popCSV($_);
	    $val = &CSVToStr($val);
	    $val = undef if ($val eq '');

	    if ($fld =~ /^\d+$/)
	    {
		$rec->{'entry'}->[$fld] = $val;
	    }
	    elsif ($fld eq 'phoneLabel')
	    {
		$rec->{$fld} = [split(/ /, $val)];
	    }
	    else
	    {
		$rec->{$fld} = $val;
	    }
	}

	push(@{$db->{'__RECORDS'}}, $rec);
	$db->{ $rec->{'rolo_id'} } = $#{$db->{'__RECORDS'}};

	$max_id = $rec->{'rolo_id'} if ($rec->{'rolo_id'} > $max_id);
    }
    close(FD);
    $db->{'NEXT_ID'} = $max_id+1;
    return $db;
}

sub writeCSV
{
    my ($CSVFILE, $db) = @_;
    my ($rec, $key, $val, @fields);

    &writeAppInfoFile($db->{'__APPINFO'});
    unless (open(FD, ">$CSVFILE"))
    {
	PilotMgr::msg("Unable to write to $CSVFILE.  Help!");
	return;
    }

    foreach $rec (@{$db->{'__RECORDS'}})
    {
	@fields = ();
	foreach $key (@$gCSVorder)
	{
	    $val = $gEntryMap->{$key};
	    if ($val =~ /^\d+$/)
	    {
		if (defined ($val = $rec->{'entry'}->[$val]))
		{
		    $val = &StrToCSV($val);
		}
	    }
	    else
	    {
		if (defined ($val = $rec->{$val}))
		{
		    $val = join(' ', @$val) if (ref($val) eq 'ARRAY');
		    $val = &StrToCSV($val);
		}
	    }

	    $val = '' unless (defined $val);
	    push(@fields, $val);
	}

	print FD join(',', @fields), "\n";
    }

    close(FD);
##
# Testing!!
#&writeVCardsOneFile("$ENV{HOME}/.vcards", $db);
#&writeVCardsMultipleFiles("$ENV{HOME}/.dt/Addresses", $db);
#
}

sub StrToCSV
{
    my ($str) = @_;

    $str =~ s/(\\*)(n|\n)/'\\' x (2*length($1)) . ($2 eq 'n' ? 'n' : '\\n')/ge;
    if ($str =~ /[,"]/)
    {
	$str =~ s/"/""/g;
	$str = '"' . $str . '"';
    }

    return $str;
}

sub popCSV
{
    my ($str) = @_;

    if ($str =~ s/^("([^"]|"")*")(,|$)//)
    {
	return($1, $str);
    }
    elsif ($str =~ s/^(.*?)(,|$)//)
    {
	return($1, $str);
    }

    return($str, '');
}

sub CSVToStr
{
    my ($str) = @_;

    if ($str =~ /^"(.*)"$/)
    {
	$str = $1;
	$str =~ s/""/"/g;
    }
    $str =~ s/((\\\\)*)(\\)?n/'\\' x (length($1)\/2) . ($3 ? "\n" : 'n')/ge;

    return $str;
}

sub writeVCardsOneFile
{
    my ($VCARDFILE, $db) = @_;
    my ($rec);

    unless (open(FD, ">$VCARDFILE"))
    {
	PilotMgr::msg("Unable to write to $VCARDFILE.  Help!");
	return;
    }

    foreach $rec (@{$db->{'__RECORDS'}})
    {
	&writeVCard($rec, FD);
	print FD "\n";
    }

    close(FD);
}

sub writeVCardsMultipleFiles
{
    my ($VCARDDIR, $db) = @_;
    my ($i, $rec, @oldfiles) = (1);

    # yikes, scary! delete all old files!
    #
#XXX: for now just delete addrfile* files
    @oldfiles = <$VCARDDIR/addrfile*>;
    unlink @oldfiles;

    foreach $rec (@{$db->{'__RECORDS'}})
    {
#XXX: should use better filenames..
	open(FD, ">$VCARDDIR/addrfile" . $i++) || next;
	&writeVCard($rec, FD);
	close(FD);
    }
}

sub writeVCard
{
    my ($rec, $FD) = @_;
    my ($val, $val2, $i);

    #XXX: need to handle newlines or semicolons in ADR and N fields!!

    #sdtname requires a ADR type, HOME/WORK.. we'll default to HOME
    my $defaultAddrPlace = 'HOME';

    print $FD "BEGIN:VCARD\n";

    # FN is just for looks, doesn't store actual data:
    &printEncodedString('FN',
	defined($rec->{'fullname'}) ? $rec->{'fullname'} : &titleString($rec),
	$FD);

    ($val = $rec->{'entry'}->[$gEntryMap->{'lastname'}]) =~ s/;/\\;/g;
    ($val2 = $rec->{'entry'}->[$gEntryMap->{'firstname'}]) =~ s/;/\\;/g;
    if (defined $val || defined $val2)
    {
	print $FD 'N:';
	print $FD $val if (defined $val);
	print $FD ';';
	print $FD $val2 if (defined $val2);
	print $FD "\n";
    }

    $val = $rec->{'entry'}->[$gEntryMap->{'company'}];
    &printEncodedString('ORG', $val, $FD) if (defined $val);

    $val = $rec->{'entry'}->[$gEntryMap->{'title'}];
    &printEncodedString('TITLE', $val, $FD) if (defined $val);

    print $FD 'ADR;';
    $val = $rec->{'addrTypeInfo'};		# vCard info like HOME or WORK
    $val = $defaultAddrPlace unless (defined $val);
    print $FD "$val;X-pilot-field=addr:;;";

    ($val = $rec->{'entry'}->[$gEntryMap->{'address'}]) =~ s/;/\\;/g;
    print $FD $val if (defined $val);
    print $FD ';';

    ($val = $rec->{'entry'}->[$gEntryMap->{'city'}]) =~ s/;/\\;/g;
    print $FD $val if (defined $val);
    print $FD ';';

    ($val = $rec->{'entry'}->[$gEntryMap->{'state'}]) =~ s/;/\\;/g;
    print $FD $val if (defined $val);
    print $FD ';';

    ($val = $rec->{'entry'}->[$gEntryMap->{'zip'}]) =~ s/;/\\;/g;
    print $FD $val if (defined $val);
    print $FD ';';

    ($val = $rec->{'entry'}->[$gEntryMap->{'country'}]) =~ s/;/\\;/g;
    print $FD $val if (defined $val);
    print $FD "\n";

    foreach $i (1..5)
    {
	$val = $rec->{'entry'}->[$gEntryMap->{"phone$i"}];
	next unless (defined $val);

	$val2 = $rec->{'phoneLabel'}->[$i-1];

	#XXX: might want to look at prefs and see what types these
	#     *really* are in case they've been changed..
	if ($val2 == 4)
	{
	    print $FD 'EMAIL;INTERNET';
	}
	else
	{
	    print $FD 'TEL';
	    print $FD ';WORK' if ($val2 == 0);
	    print $FD ';HOME' if ($val2 == 1);
	    print $FD ';FAX' if ($val2 == 2);
	    print $FD ';PREF' if ($val2 == 5);
	    print $FD ';PAGER' if ($val2 == 6);
	    print $FD ';CELL' if ($val2 == 7);
	}

	&printEncodedString(";X-pilot-field=phone$i", $val, $FD);
    }

    foreach $i (1..4)
    {
	$val = $rec->{'entry'}->[$gEntryMap->{"custom$i"}];
	next unless (defined $val);

	&printEncodedString("NOTE;X-pilot-field=custom$i", $val, $FD);
    }

    $val = $rec->{'entry'}->[$gEntryMap->{'note'}];
    &printEncodedString('NOTE;X-pilot-field=note', $val, $FD)
	if (defined $val);

    print $FD "X-pilot-id:$rec->{rolo_id}\n";

    print $FD "END:VCARD\n";
}

sub printEncodedString
{
    # print string value to vcard. use Quoted-Printable if necessary
    # XXX: this needs to be more complete to encode all control chars,etc too
    #
    my ($hdr, $val, $fd) = @_;

    print $fd $hdr if ($hdr);
    if ($val =~ /\n/)
    {
	$val =~ s/=/=3D/g;
	$val =~ s/\n/=0A/g;
	print $fd ";ENCODING=QUOTED-PRINTABLE";
    }
    print $fd ":$val\n";
}

sub readVCardsOneFile
{
    my ($VCARDFILE) = @_;
    my ($max_id, $db, $rec, $i) = (-1, { '__RECORDS' => [], 'NEXT_ID' => 0 });

    $db->{'__APPINFO'} = &readAppInfoFile;
    open(FD, "<$VCARDFILE") || return $db;

    while (<FD>)
    {
	if ( /^\s*BEGIN\s*:\s*VCARD\s*$/ )
	{
	    $rec = &readVCard(FD);
	    push(@{$db->{'__RECORDS'}}, $rec);

	    if (defined $rec->{'rolo_id'})
	    {
		$db->{ $rec->{'rolo_id'} } = $#{$db->{'__RECORDS'}};
		$max_id = $rec->{'rolo_id'} if ($rec->{'rolo_id'} > $max_id);
	    }
	}
    }
    close(FD);
    $db->{'NEXT_ID'} = $max_id+1;

    foreach $i ($[..$#{$db->{'__RECORDS'}})
    {
	$rec = $db->{'__RECORDS'}->[$i];
	unless (defined $rec->{'rolo_id'})
	{
	    $rec->{'rolo_id'} = $db->{'NEXT_ID'}++;
	    $db->{ $rec->{'rolo_id'} } = $i;
	}
    }

    return $db;
}

sub readVCardsMultipleFiles
{
    my ($VCARDDIR) = @_;
    my ($max_id, $db, $rec, $i, @files) =
	(-1, { '__RECORDS' => [], 'NEXT_ID' => 0 });

    $db->{'__APPINFO'} = &readAppInfoFile;
#XXX: just addrfile* files for now
    @files = <$VCARDDIR/addrfile*>;

    foreach $i (@files)
    {
	open(FD, "<$i") || next;
	do { <FD> } until ( /^\s*BEGIN\s*:\s*VCARD\s*$/ || eof(FD));
	next if (eof(FD));
	$rec = &readVCard(FD);
	close(FD);
	push(@{$db->{'__RECORDS'}}, $rec);

	if (defined $rec->{'rolo_id'})
	{
	    $db->{ $rec->{'rolo_id'} } = $#{$db->{'__RECORDS'}};
	    $max_id = $rec->{'rolo_id'} if ($rec->{'rolo_id'} > $max_id);
	}
    }
    $db->{'NEXT_ID'} = $max_id+1;

    foreach $i ($[..$#{$db->{'__RECORDS'}})
    {
	$rec = $db->{'__RECORDS'}->[$i];
	unless (defined $rec->{'rolo_id'})
	{
	    $rec->{'rolo_id'} = $db->{'NEXT_ID'}++;
	    $db->{ $rec->{'rolo_id'} } = $i;
	}
    }

    return $db;
}

sub readVCard
{
    my ($FD) = @_;
    my ($rec, $extra);
    $rec = { 'entry' => [],
	     'showPhone' => 0,
	     'phoneLabel' => [0,1,2,3,4] };
    $rec->{'entry'}->[18] = undef;  # ensure right array length

    while (<$FD>)
    {
	last if ( /^\s*END\s*:\s*VCARD\s*$/ );

	if ( /^\s*FN\s*(;[^:]*)?:\s*(.*)$/ )
	{
	    $extra = $1;
	    $val = $2;
	    if ($extra =~ /ENCODING\s*=\s*QUOTED-PRINTABLE/i)
	    {
		while ($val =~ /=$/)
		{
		    $val .= <$FD>;
		}
		$val =~ s/=0[Aa]/\n/g;
		$val =~ s/=3[Dd]/=/g;
	    }
	    $rec->{'fullname'} = $val;
	}
	elsif ( /^\s*N\s*(;[^:]*)?:\s*(.*)$/ )
	{
	    $extra = $1;
	    $val = $2;

	    #if ($val =~ /^(.*);
	}
    }

    return $rec;
}

1;

