/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: ScsiIf-linux.cc,v $
 * Revision 1.5  1998/09/06 13:34:22  mueller
 * Use 'message()' for printing messages.
 *
 * Revision 1.4  1998/08/13 19:13:28  mueller
 * Added member function 'timout()' to set timeout of SCSI commands.
 *
 * Revision 1.3  1998/08/07 12:36:04  mueller
 * Added enabling command transformation for emulated host adaptor (ATAPI).
 *
 */

static char rcsid[] = "$Id: ScsiIf-linux.cc,v 1.5 1998/09/06 13:34:22 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <scsi/sg.h>
#include <asm/param.h> // for HZ

#include "ScsiIf.h"
#include "util.h"

class ScsiIfImpl {
public:
  char *dev_;
  int fd_;
  
  int maxSendLen_;

  char *outBuf_;
  char *inBuf_;

  struct sg_header *outHd_;
  struct sg_header *inHd_;
};

ScsiIf::ScsiIf(const char *dev)
{
  impl_ = new ScsiIfImpl;

  impl_->dev_ = strdupCC(dev);

#ifdef SG_BIG_BUFF
  impl_->maxSendLen_ = SG_BIG_BUFF;
#else
  impl_->maxSendLen_ = 4096;
#endif

  maxDataLen_ = impl_->maxSendLen_ - sizeof(struct sg_header) - 12;

  impl_->outBuf_ = new char[impl_->maxSendLen_];
  impl_->inBuf_ = new char[impl_->maxSendLen_];

  impl_->outHd_ = (struct sg_header *)impl_->outBuf_;
  impl_->inHd_ = (struct sg_header *)impl_->inBuf_;
  
  impl_->fd_ = -1;

  vendor_[0] = 0;
  product_[0] = 0;
  revision_[0] = 0;
  revisionDate_[0] = 0;
}

ScsiIf::~ScsiIf()
{
  if (impl_->fd_ >= 0) {
    close(impl_->fd_);
  }

  delete[] impl_->outBuf_;
  delete[] impl_->inBuf_;
  delete impl_;
}

// opens and flushes scsi device
// return: 0: OK
//         1: device could not be opened
//         2: inquiry failed
int ScsiIf::init()
{
  int flags;

  if ((impl_->fd_ = open(impl_->dev_, O_RDWR)) < 0) {
    message(-2, "Cannot open generic scsi device '%s': %s",
	    impl_->dev_, strerror(errno));
    return 1;
  }
  
  flags = fcntl(impl_->fd_, F_GETFL);
  fcntl(impl_->fd_, F_SETFL, flags|O_NONBLOCK);

  memset(impl_->inHd_, 0, sizeof(struct sg_header));
  impl_->inHd_->reply_len = sizeof(struct sg_header);

  while (read(impl_->fd_, impl_->inHd_, impl_->maxSendLen_) >= 0 ||
	 errno != EAGAIN) ;

  fcntl(impl_->fd_, F_SETFL, flags);

#if defined(SG_EMULATED_HOST) && defined(SG_SET_TRANSFORM)
  if (ioctl(impl_->fd_, SG_EMULATED_HOST, &flags) == 0 && flags != 0) {
    // emulated host adaptor for ATAPI drives
    // switch on command transformation
    ioctl(impl_->fd_, SG_SET_TRANSFORM, NULL);
  }
#endif

  if (inquiry() != 0) {
    return 2;
  }

  return 0;
}

// Sets given timeout value in seconds and returns old timeout.
// return: old timeout
int ScsiIf::timeout(int t)
{
  int old = ioctl(impl_->fd_, SG_GET_TIMEOUT, NULL);
  int ret;

  t *= HZ;

  if ((ret = ioctl(impl_->fd_, SG_SET_TIMEOUT, &t)) != 0) {
    message(-1, "Cannot set SCSI timeout: %s", strerror(ret));
  }

  return old/HZ;
}

// sends a scsi command and receives data
// return 0: OK
//        1: scsi command failed (os level, no sense data available)
//        2: scsi command failed (sense data available)
int ScsiIf::sendCmd(const unsigned char *cmd, int cmdLen, 
		    const unsigned char *dataOut, int dataOutLen,
		    unsigned char *dataIn, int dataInLen)
{
  int status;
  int sendLen = sizeof(struct sg_header) + cmdLen + dataOutLen;
  int recLen = sizeof(struct sg_header) + dataInLen;

  assert(cmdLen > 0 && cmdLen <= 12);

  assert(sendLen <= impl_->maxSendLen_);

  memset(impl_->outBuf_, 0, impl_->maxSendLen_);

  memcpy(impl_->outBuf_ + sizeof(struct sg_header), cmd, cmdLen);
  if (dataOut != NULL) {
    memcpy(impl_->outBuf_ + sizeof(struct sg_header) + cmdLen, dataOut,
	   dataOutLen);
  }

  impl_->outHd_->reply_len   = recLen;
  impl_->outHd_->twelve_byte = (cmdLen == 12);
  impl_->outHd_->result = 0;

  /* send command */
  status = write(impl_->fd_, impl_->outBuf_, sendLen);
  if (status < 0 || status !=  sendLen) {
    /* some error happened */
    message(-2, "write(generic) result = %d cmd = 0x%x",
	    status, cmd[0] );
    return 1;
  }

  // read result
  memset(impl_->inBuf_, 0, impl_->maxSendLen_);
  
  status = read(impl_->fd_, impl_->inBuf_, recLen);
  if (status < 0) {
    message(-2, "read(generic) failed: %s", strerror(errno));
    return 1;
  }
  else if (status != recLen) {
    message(-2, "read(generic) did not return expected amount of data.");
    return 1;
  }
  else if (impl_->inHd_->result) {
    // scsi command failed
    message(-2, "read(generic) failed: %s", strerror(impl_->inHd_->result));
    return 1;
  }
  else if (impl_->inHd_->sense_buffer[2] != 0) {
    return 2;
  }

  if (dataIn != NULL && dataInLen > 0) {
    memcpy(dataIn, impl_->inBuf_ + sizeof(struct sg_header), dataInLen);
  }

  return 0;
}

const unsigned char *ScsiIf::getSense(int &len) const
{
  len = 15;
  return impl_->inHd_->sense_buffer;
}

int ScsiIf::inquiry()
{
  unsigned char cmd[6];
  unsigned char result[0x2c];
  int i;

  cmd[0] = 0x12; // INQUIRY
  cmd[1] = cmd[2] = cmd[3] = 0;
  cmd[4] = 0x2c;
  cmd[5] = 0;

  switch (sendCmd(cmd, 6, NULL, 0, result, 0x2c)) {
  case 0:
    // OK
    break;

  case 2:
    // check sense key
    message(-2, "Inquiry command failed on '%s': ", impl_->dev_);
    switch ((impl_->inHd_->sense_buffer[2]) & 0x0f) {
    case 0x00:
      message(0, "NO SENSE");
      break;
    case 0x01:
      message(0, "RECOVERED ERROR");
      break;
    case 0x02:
      message(0, "NOT READY");
      break;
    case 0x03:
      message(0, "MEDIUM ERROR");
      break;
    case 0x04:
      message(0, "HARDWARE ERROR");
      break;
    case 0x05:
      message(0, "ILLEGAL REQUEST");
      break;
    case 0x06:
      message(0, "UNIT ATTENTION");
      break;
    case 0x08:
      message(0, "BLANK CHECK");
      break;
    case 0x09:
      message(0, "VENDOR SPECIFIC");
      break;
    case 0x0B:
      message(0, "ABORTED COMMAND");
      break;
    case 0x0D:
      message(0, "VOLUME OVERFLOW");
      break;
    case 0x0E:
      message(0, "MISCOMPARE");
      break;
    default:
      message(0, "unknown sense code");
      break;
    }
    message(0, "");

    return 1;
    break;

  default:
    message (-2, "Inquiry command failed on '%s'.", impl_->dev_);
    return 1;
  }

  strncpy(vendor_, (char *)(result + 0x08), 8);
  vendor_[8] = 0;

  strncpy(product_, (char *)(result + 0x10), 16);
  product_[16] = 0;

  strncpy(revision_, (char *)(result + 0x20), 4);
  revision_[4] = 0;

  strncpy(revisionDate_, (char *)(result + 0x24), 8);
  revisionDate_[8] = 0;

  for (i = 7; i >= 0 && vendor_[i] == ' '; i--) {
    vendor_[i] = 0;
  }

  for (i = 15; i >= 0 && product_[i] == ' '; i--) {
    product_[i] = 0;
  }

  for (i = 3; i >= 0 && revision_[i] == ' '; i--) {
    revision_[i] = 0;
  }

  for (i = 7; i >= 0 && revisionDate_[i] == ' '; i--) {
    revisionDate_[i] = 0;
  }      
  
  return 0;
}


