/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: PlextorReader.cc,v $
 * Revision 1.7  1998/09/27 19:20:05  mueller
 * Added retrieval of control nibbles for track with 'analyzeTrack()'.
 *
 * Revision 1.6  1998/09/08 11:54:22  mueller
 * Extended disk info structure because CDD2000 does not support the
 * 'READ DISK INFO' command.
 *
 * Revision 1.5  1998/09/07 15:20:20  mueller
 * Reorganized read-toc related code.
 *
 * Revision 1.4  1998/09/06 13:34:22  mueller
 * Use 'message()' for printing messages.
 *
 * Revision 1.3  1998/08/30 19:25:43  mueller
 * Added function 'diskInfo()'.
 * Changed sub channel data field in 'readSubChannelData()' to be more
 * compatible to other drives.
 *
 * Revision 1.2  1998/08/25 19:26:07  mueller
 * Moved basic index extraction algorithm to class 'CdrDriver'.
 *
 */

static char rcsid[] = "$Id: PlextorReader.cc,v 1.7 1998/09/27 19:20:05 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>

#include "PlextorReader.h"

#include "Toc.h"
#include "util.h"

PlextorReader::PlextorReader(ScsiIf *scsiIf, Toc *toc) : CdrDriver(scsiIf, toc)
{
  driverName_ = "Plextor CD-ROM Reader - Version 1.1";
  
  speed_ = 0;
  simulate_ = 0;

  memset(&diskInfo_, 0, sizeof(DiskInfo));
  diskInfo_.valid.empty = 1;
}

// static constructor
CdrDriver *PlextorReader::instance(ScsiIf *scsiIf, Toc *toc)
{
  return new PlextorReader(scsiIf, toc);
}

// sets speed
// return: 0: OK
//         1: illegal speed
int PlextorReader::speed(int /*s*/)
{
  return 0;
}

DiskInfo *PlextorReader::diskInfo()
{
  return &diskInfo_;
}

int PlextorReader::initDao()
{
  message(-2, "Writing is not supported by this driver.");
  return 1;
}

int PlextorReader::startDao()
{
  return 1;
}

int PlextorReader::finishDao()
{
  return 1;
}

// tries to read catalog number from disk and adds it to 'toc'
// return: 1 if valid catalog number was found, else 0

int PlextorReader::readCatalog(Toc *toc)
{
  unsigned char cmd[10];
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  char catalog[14];
  int i;

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x02; // get media catalog number
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot read sub channel data.");
    return 0;
  }

  if (data[0x08] & 0x80) {
    for (i = 0; i < 13; i++) {
      catalog[i] = data[0x09 + i];
    }
    catalog[13] = 0;

    if (toc->catalog(catalog) == 0) {
      return 1;
    }
  }

  return 0;
}

// plays one audio block starting from given position
void PlextorReader::playAudioBlock(long start, long len)
{
  unsigned char cmd[10];

  // play one audio block
  memset(cmd, 0, 10);
  cmd[0] = 0x45; // PLAY AUDIO
  cmd[2] = start >> 24;
  cmd[3] = start >> 16;
  cmd[4] = start >> 8;
  cmd[5] = start;
  cmd[7] = len >> 8;
  cmd[8] = len;

  if (sendCmd(cmd, 10, NULL, 0, NULL, 0) != 0) {
    message(-2, "Cannot play audio block.");
    return;
  }
}

int PlextorReader::analyzeTrack(int trackNr, long startLba, long endLba,
				Msf *index, int *indexCnt, long *pregap,
				char *isrcCode, unsigned char *ctl)
{
  int ret = CdrDriver::analyzeTrack(trackNr, startLba, endLba, index,
				    indexCnt, pregap, isrcCode, ctl);

  if (ret == 0)
    readIsrc(trackNr, isrcCode);

  return ret;
}

// tries to read ISRC code of given track number and stores it in
// given buffer.
// return: 1 if valid ISRC code was found, else 0
int PlextorReader::readIsrc(int trackNr, char *buf)
{
  unsigned char cmd[10];
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  int i;

  buf[0] = 0;

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x03; // get ISRC code
  cmd[6] = trackNr;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-1, "Cannot read ISRC code.");
    return 0;
  }

  if (data[0x08] & 0x80) {
    for (i = 0; i < 12; i++) {
      buf[i] = data[0x09 + i];
    }
    buf[12] = 0;

  }

  return 0;
}

// Reads actual track number, index and relative position from sub channel.
// return: 0 OK, 1: SCSI command failed
int PlextorReader::readSubChannelData(int *trackNr, int *indexNr,
				      long *relPos, unsigned char *ctl)
{
  unsigned char cmd[10];
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x01; // get sub Q channel data
  cmd[6] = 0;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot read sub Q channel data.");
  }

  *trackNr = data[6];
  *indexNr = data[7];
  *relPos = 0;
  *relPos |= data[0x0c] << 24;
  *relPos |= data[0x0d] << 16;
  *relPos |= data[0x0e] << 8;
  *relPos |= data[0x0f];
  
  if (ctl != NULL) {
    *ctl = data[5] & 0x0f;
  }

  return 0;
}

// Retrieves track and index at given block address 'lba'.
// Return: 0: OK, 1: SCSI command failed
int PlextorReader::getTrackIndex(long lba, int *trackNr, int *indexNr,
				 unsigned char *ctl)
{
  long relPos;

  playAudioBlock(lba, 1);

  return readSubChannelData(trackNr, indexNr, &relPos, ctl);
}

#if 0
CdToc *PlextorReader::getToc(int *nofTracks) // for debugging
{
  unsigned char cmd[10];
  unsigned short dataLen = 200 * 8 + 4;
  unsigned char data[200 * 8 + 4];
  unsigned char *p = NULL;
  int i, n;
  CdToc *toc;
  int nTracks;

  // read disk toc
  memset(cmd, 0, 10);
  cmd[0] = 0x43; // READ TOC
  cmd[6] = 0; // return info for all tracks
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;
  cmd[9] = 0x80; // return Q subcodes
  
  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot read disk toc.");
    return NULL;
  }

  n = (data[0] << 8) | data[1];

  n /= 11;


  for (i = 0, p = data + 4; i < n; i++, p += 11 ) {
    message(0, "%02x %02d %2x %02x:%02x:%02x %02d %02x:%02x:%02x",
	   p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10]);
  }
  
  return NULL;
}
#endif
