/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: GenericMMC.cc,v $
 * Revision 1.9  1998/10/24 14:48:55  mueller
 * Added retrieval of next writable address. The Panasonic CW-7502 needs it.
 *
 * Revision 1.8  1998/10/03 15:08:44  mueller
 * Moved 'writeZeros()' to base class 'CdrDriver'.
 * Takes 'writeData()' from base class now.
 *
 * Revision 1.7  1998/09/27 19:18:48  mueller
 * Added retrieval of control nibbles for track with 'analyzeTrack()'.
 * Added multi session mode.
 *
 * Revision 1.6  1998/09/22 19:15:13  mueller
 * Removed memory allocations during write process.
 *
 * Revision 1.5  1998/09/08 11:54:22  mueller
 * Extended disk info structure because CDD2000 does not support the
 * 'READ DISK INFO' command.
 *
 * Revision 1.4  1998/09/07 15:20:20  mueller
 * Reorganized read-toc related code.
 *
 * Revision 1.3  1998/09/06 13:34:22  mueller
 * Use 'message()' for printing messages.
 *
 * Revision 1.2  1998/08/30 19:17:56  mueller
 * Fixed cue sheet generation and first writable address after testing
 * with a Yamaha CDR400t.
 *
 * Revision 1.1  1998/08/15 20:44:58  mueller
 * Initial revision
 *
 */

static char rcsid[] = "$Id: GenericMMC.cc,v 1.9 1998/10/24 14:48:55 mueller Exp $";

#include <config.h>

#include <string.h>
#include <assert.h>

#include "GenericMMC.h"

#include "Toc.h"
#include "PWSubChannel96.h"
#include "util.h"

GenericMMC::GenericMMC(ScsiIf *scsiIf, Toc *toc) : CdrDriver(scsiIf, toc)
{
  driverName_ = "Generic SCSI-3/MMC - Version 0.1(alpha)";
  
  speed_ = 0;
  simulate_ = 1;

  scsiTimeout_ = 0;
}

// static constructor
CdrDriver *GenericMMC::instance(ScsiIf *scsiIf, Toc *toc)
{
  return new GenericMMC(scsiIf, toc);
}

// sets speed
// return: 0: OK
//         1: illegal speed
int GenericMMC::speed(int s)
{
  speed_ = s;

  if (selectSpeed(0) != 0)
    return 1;
  
  return 0;
}

int GenericMMC::speed()
{
  DriveInfo di;

  if (driveInfo(&di, 1) != 0) {
    return 0;
  }

  return speed2Mult(di.currentWriteSpeed);

}
// loads ('unload' == 0) or ejects ('unload' == 1) tray
// return: 0: OK
//         1: scsi command failed

int GenericMMC::loadUnload(int unload) const
{
  unsigned char cmd[6];

  memset(cmd, 0, 6);

  cmd[0] = 0x1b; // START/STOP UNIT
  if (unload) {
    cmd[4] = 0x02; // LoUnlo=1, Start=0
  }
  else {
    cmd[4] = 0x03; // LoUnlo=1, Start=1
  }
  
  if (sendCmd(cmd, 6, NULL, 0, NULL, 0) != 0) {
    message(-2, "Cannot load/unload medium.");
    return 1;
  }

  return 0;
}


// sets read/write speed and simulation mode
// return: 0: OK
//         1: scsi command failed
int GenericMMC::selectSpeed(int readSpeed)
{
  unsigned char cmd[12];
  int spd;

  memset(cmd, 0, 12);

  cmd[0] = 0xbb; // SET CD SPEED

  // select maximum read speed
  if (readSpeed == 0) {
    cmd[2] = 0xff;
    cmd[3] = 0xff;
  }
  else {
    spd = mult2Speed(readSpeed);
    cmd[2] = spd >> 8;
    cmd[3] = spd;
  }

  // select maximum write speed
  if (speed_ == 0) {
    cmd[4] = 0xff;
    cmd[5] = 0xff;
  }
  else {
    spd = mult2Speed(speed_);
    cmd[4] = spd >> 8;
    cmd[5] = spd;
  }

  if (sendCmd(cmd, 12, NULL, 0, NULL, 0) != 0) {
    message(-2, "Cannot set cd speed.");
    return 1;
  }

  return 0;
}

// Determins start and length of lead-in and length of lead-out.
// return: 0: OK
//         1: SCSI command failed
int GenericMMC::getSessionInfo()
{
  unsigned char cmd[10];
  unsigned long dataLen = 34;
  unsigned char data[34];

  memset(cmd, 0, 10);
  memset(data, 0, dataLen);

  cmd[0] = 0x51; // READ DISK INFORMATION
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot retrieve disk information.");
    return 1;
  }

  leadInStart_ = Msf(data[17], data[18], data[19]);
  
  leadInLen_ = 450150 - leadInStart_.lba() - 150;

  leadOutLen_ = Msf(1, 30, 0).lba(); // 90 seconds lead-out

  message(2, "Lead-in start: %s length: %ld", leadInStart_.str(),
	  leadInLen_);
  message(2, "Lead-out length: %ld", leadOutLen_);

  return 0;
}

// Sets write parameters via mode page 0x05.
// return: 0: OK
//         1: scsi command failed
int GenericMMC::setWriteParameters()
{
  unsigned char mp[0x38];

  if (getModePage(5/*write parameters mode page*/, mp, 0x38) != 0) {
    message(-2, "Cannot retrieve write parameters mode page.");
    return 1;
  }

  mp[0] &= 0x7f; // clear PS flag

  mp[2] &= 0xe0;
  mp[2] |= 0x02; // write type: Session-at-once
  if (simulate_) {
    mp[2] |= 1 << 4; // test write
  }

  mp[3] &= 0x3f; // Multi-session: No B0 pointer, next session not allowed
  if (multiSession_ != 0)
    mp[3] |= 0x03 << 6; // open next session
 
  mp[4] &= 0xf0; // Data Block Type: raw data, block size: 2352 (I think not
                 // used for session at once writing)

  mp[8] = 0; // session format: CD-DA or CD-ROM

  if (setModePage(mp) != 0) {
    message(-2, "Cannot set write parameters mode page.");
    return 1;
  }

  return 0;
}

long GenericMMC::getNWA()
{
  unsigned char cmd[10];
  int infoblocklen = 16;
  unsigned char info[16];
  long lba = 0;

  cmd[0] = 0x52; // READ TRACK INFORMATION
  cmd[1] = 0x01; // track instead of lba designation
  cmd[2] = 0x00; cmd[3] = 0x00; cmd[4] = 0x00; cmd[5] = 0xff; // invisible track
  cmd[6] = 0x00; // reserved
  cmd[7] = infoblocklen << 8; cmd[8] = infoblocklen; // alloc length
  cmd[9] = 0x00; // Control Byte

  if (sendCmd(cmd, 10, NULL, 0, info, infoblocklen) != 0) {
    message(-1, "Cannot get Track Information Block.");
    return 404850; // this value is not valid as LBA for CD media
  }

  message(2,"Track Information Block");
  for (int i=0;i<infoblocklen;i++) message(2,"byte %02x : %02x",i,info[i]);

  if ((info[6] & 0x40) && (info[7] & 0x01) && !(info[6] & 0xb0))
  {
      message(2,"Disk is Blank, Next Writable Address is valid");
      lba |= info[12] << 24; // MSB of LBA
      lba |= info[13] << 16;
      lba |= info[14] << 8;
      lba |= info[15];       // LSB of LBA
  }

  message(2, "NWA: %ld", lba);

  return lba;
}

// Creates cue sheet for current toc.
// cueSheetLen: filled with length of cue sheet in bytes
// return: newly allocated cue sheet buffer or 'NULL' on error
unsigned char *GenericMMC::createCueSheet(long *cueSheetLen)
{
  const Track *t;
  int trackNr;
  Msf start, end, index;
  unsigned char *cueSheet;
  long len = 3; // entries for lead-in, gap, lead-out
  long n; // index into cue sheet
  unsigned char ctl; // control nibbles of cue sheet entry CTL/ADR
  long i;

  if (toc_->first(start, end) == NULL) {
    return NULL;
  }

  if (toc_->catalogValid()) {
    len += 2;
  }

  for (t = toc_->first(start, end), trackNr = 1;
       t != NULL; t = toc_->next(start, end), trackNr++) {
    len += 1; // entry for track
    if (t->start().lba() != 0 && trackNr > 1) {
      len += 1; // entry for pre-gap
    }
    if (t->isrcValid()) {
      len += 2; // entries for ISRC code
    }
    len += t->nofIndices(); // entry for each index increment
  }

  cueSheet = new (unsigned char)[len * 8];
  n = 0;

  if (toc_->catalogValid()) {
    // fill catalog number entry
    cueSheet[n*8] = 0x02;
    cueSheet[(n+1)*8] = 0x02;
    for (i = 1; i <= 13; i++) {
      if (i < 8) {
	cueSheet[n*8 + i] = toc_->catalog(i-1) + '0';
      }
      else {
	cueSheet[(n+1)*8 + i - 7] = toc_->catalog(i-1) + '0';
      }
    }
    cueSheet[(n+1)*8+7] = 0;
    n += 2;
  }

  // entry for lead-in
  cueSheet[n*8] = 0x01; // CTL/ADR
  cueSheet[n*8+1] = 0;    // Track number
  cueSheet[n*8+2] = 0;    // Index
  cueSheet[n*8+3] = 0x01; // Data Form: CD-DA, generate data by device
  cueSheet[n*8+4] = 0;    // Serial Copy Management System
  cueSheet[n*8+5] = 0;    // MIN
  cueSheet[n*8+6] = 0;    // SEC
  cueSheet[n*8+7] = 0;    // FRAME
  n++;
  
  for (t = toc_->first(start, end), trackNr = 1;
       t != NULL;
       t = toc_->next(start, end), trackNr++) {

    ctl = 0;
    if (t->preEmphasis()) {
      ctl |= 0x10;
    }
    if (t->copyPermitted()) {
      ctl |= 0x20;
    }
    if (t->audioType() == 1) {
      ctl |= 0x80;
    }

    if (t->isrcValid()) {
      cueSheet[n*8] = ctl | 0x03;
      cueSheet[n*8+1] = trackNr;
      cueSheet[n*8+2] = t->isrcCountry(0);
      cueSheet[n*8+3] = t->isrcCountry(1);
      cueSheet[n*8+4] = t->isrcOwner(0);  
      cueSheet[n*8+5] = t->isrcOwner(1);   
      cueSheet[n*8+6] = t->isrcOwner(2);  
      cueSheet[n*8+7] = t->isrcYear(0) + '0';
      n++;

      cueSheet[n*8] = ctl | 0x03;
      cueSheet[n*8+1] = trackNr;
      cueSheet[n*8+2] = t->isrcYear(1) + '0';
      cueSheet[n*8+3] = t->isrcSerial(0) + '0';
      cueSheet[n*8+4] = t->isrcSerial(1) + '0';
      cueSheet[n*8+5] = t->isrcSerial(2) + '0';
      cueSheet[n*8+6] = t->isrcSerial(3) + '0';
      cueSheet[n*8+7] = t->isrcSerial(4) + '0';
      n++;
    }

    if (trackNr == 1) {
      // entry for gap before first track
      cueSheet[n*8]   = ctl | 0x01;
      cueSheet[n*8+1] = 1;    // Track 1
      cueSheet[n*8+2] = 0;    // Index 0
      cueSheet[n*8+3] = 0;    // Data Form: CD-DA
      cueSheet[n*8+4] = 0;    // Serial Copy Management System
      cueSheet[n*8+5] = 0;    // MIN 0
      cueSheet[n*8+6] = 0;    // SEC 0
      cueSheet[n*8+7] = 0;    // FRAME 0
      n++;
    }
    else if (t->start().lba() != 0) {
      // entry for pre-gap
      Msf pstart(start.lba() - t->start().lba() + 150);

      cueSheet[n*8]   = ctl | 0x01;
      cueSheet[n*8+1] = trackNr;
      cueSheet[n*8+2] = 0; // Index 0 indicates pre-gap
      cueSheet[n*8+3] = 0; // CD-DA data
      cueSheet[n*8+4] = 0; // no alternate copy bit
      cueSheet[n*8+5] = pstart.min();
      cueSheet[n*8+6] = pstart.sec();
      cueSheet[n*8+7] = pstart.frac();
      n++;
    }

    Msf tstart(start.lba() + 150);
    
    cueSheet[n*8]   = ctl | 0x01;
    cueSheet[n*8+1] = trackNr;
    cueSheet[n*8+2] = 1; // Index 1
    cueSheet[n*8+3] = 0; // CD-DA data
    cueSheet[n*8+4] = 0; // no alternate copy bit
    cueSheet[n*8+5] = tstart.min();
    cueSheet[n*8+6] = tstart.sec();
    cueSheet[n*8+7] = tstart.frac();
    n++;

    for (i = 0; i < t->nofIndices(); i++) {
      index = tstart + t->getIndex(i);
      cueSheet[n*8]   = ctl | 0x01;
      cueSheet[n*8+1] = trackNr;
      cueSheet[n*8+2] = i+2; // Index
      cueSheet[n*8+3] = 0; // CD-DA data
      cueSheet[n*8+4] = 0; // no alternate copy bit
      cueSheet[n*8+5] = index.min();
      cueSheet[n*8+6] = index.sec();
      cueSheet[n*8+7] = index.frac();
      n++;
    }
  }

  assert(n == len - 1);

  // entry for lead out
  Msf lostart(toc_->length().lba() + 150);
    
  cueSheet[n*8]   = 0x01;
  cueSheet[n*8+1] = 0xaa;
  cueSheet[n*8+2] = 1; // Index 1
  cueSheet[n*8+3] = 0x01; // CD-DA data, data generated by device
  cueSheet[n*8+4] = 0; // no alternate copy bit
  cueSheet[n*8+5] = lostart.min();
  cueSheet[n*8+6] = lostart.sec();
  cueSheet[n*8+7] = lostart.frac();

  message(2, "\nCue Sheet:");
  message(2, "CTL/  TNO  INDEX  DATA  SCMS  MIN  SEC  FRAME");
  message(2, "ADR               FORM");
  for (n = 0; n < len; n++) {
    message(2, "%02x    %02x    %02x     %02x    %02x   %02d   %02d   %02d",
	   cueSheet[n*8],
	   cueSheet[n*8+1], cueSheet[n*8+2], cueSheet[n*8+3], cueSheet[n*8+4],
	   cueSheet[n*8+5], cueSheet[n*8+6], cueSheet[n*8+7]);
  }

  *cueSheetLen = len * 8;
  return cueSheet;
}

int GenericMMC::sendCueSheet()
{
  unsigned char cmd[10];
  long cueSheetLen;
  unsigned char *cueSheet = createCueSheet(&cueSheetLen);

  if (cueSheet == NULL) {
    return 1;
  }

  memset(cmd, 0, 10);

  cmd[0] = 0x5d; // SEND CUE SHEET

  cmd[6] = cueSheetLen >> 16;
  cmd[7] = cueSheetLen >> 8;
  cmd[8] = cueSheetLen;

  if (sendCmd(cmd, 10, cueSheet, cueSheetLen, NULL, 0) != 0) {
    message(-2, "Cannot send cue sheet.");
    delete[] cueSheet;
    return 1;
  }

  delete[] cueSheet;
  return 0;
}

int GenericMMC::initDao()
{
  long n;
  blockLength_ = AUDIO_BLOCK_LEN;
  blocksPerWrite_ = scsiIf_->maxDataLen() / blockLength_;

  assert(blocksPerWrite_ > 0);

  if (selectSpeed(0) != 0 ||
      getSessionInfo() != 0) {
    return 1;
  }

  // allocate buffer for writing zeros
  n = blocksPerWrite_ * blockLength_;
  delete[] zeroBuffer_;
  zeroBuffer_ = new char[n];
  memset(zeroBuffer_, 0, n);

  return 0;
}

int GenericMMC::startDao()
{
  scsiTimeout_ = scsiIf_->timeout(3 * 60);

  if (setWriteParameters() != 0 ||
      sendCueSheet() != 0) {
    return 1;
  }

  //message(0, "Writing lead-in and gap...");

  // It does not hurt if the following command fails.
  // The Panasonic CW-7502 needs it, unfortunately it returns the wrong
  // data so we ignore the returned data and start writing always with
  // LBA 150.
  getNWA();

  long lba = -150;

  if (writeZeros(lba, 150) != 0) {
    return 1;
  }

  
  return 0;
}

int GenericMMC::finishDao()
{
  //message(0, "Writing lead-out...");

  message(0, "Flushing cache...");
  
  if (flushCache() != 0) {
    return 1;
  }

  scsiIf_->timeout(scsiTimeout_);

  delete[] zeroBuffer_, zeroBuffer_ = NULL;

  return 0;
}

DiskInfo *GenericMMC::diskInfo()
{
  unsigned char cmd[10];
  unsigned long dataLen = 34;
  unsigned char data[34];
  static DiskInfo di;
  char spd;

  memset(&di, 0, sizeof(DiskInfo));

  memset(cmd, 0, 10);
  memset(data, 0, dataLen);

  cmd[0] = 0x51; // READ DISK INFORMATION
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-1, "Cannot retrieve disk information.");
    return &di;
  }

  di.empty = (data[2] & 0x03) == 0 ? 1 : 0;
  di.valid.empty = 1;
  di.cdrw = (data[2] & 0x10) ? 1 : 0;
  di.valid.cdrw = 1;

  if (data[21] != 0xff || data[22] != 0xff || data[23] != 0xff) {
    di.valid.capacity = 1;
    di.capacity = Msf(data[21], data[22], data[23]).lba() - 150;
  }

  if (di.empty) {
    di.valid.manufacturerId = 1;
    
    di.manufacturerId = Msf(data[17], data[18], data[19]);
  }

  if (di.empty) {
    memset(cmd, 0, 10);
    memset(data, 0, dataLen);

    cmd[0] = 0x43; // READ TOC/PMA/ATIP
    cmd[2] = 4; // get ATIP
    cmd[7] = 0;
    cmd[8] = 28; // data length

    if (sendCmd(cmd, 10, NULL, 0, data, 28, 0) == 0) {
      if (data[6] & 0x04) {
	di.valid.recSpeed = 1;
	spd = (data[16] >> 4) & 0x07;
	di.recSpeedLow = spd == 1 ? 2 : 0;

	spd = (data[16] & 0x0f);
	di.recSpeedHigh = spd >= 1 && spd <= 4 ? spd * 2 : 0;
      }
    }
    else {
      message(-1, "Cannot read ATIP data.");
    }
  }

  return &di;
}

int GenericMMC::setPlayParameters()
{
  unsigned char mp[16];

  if (getModePage(0x0e/*audio control mode page*/, mp, 16) != 0) {
    message(-2, "Cannot retrieve audio control mode page.");
    return 1;
  }

  mp[0] &= 0x7f; // clear PS flag

  // save mode page
  memcpy(audioModePage_, mp, 16);

  mp[2] &= 0xf9; // clear Immed and SOTC flags

  // muting of output channels
  mp[8] &= 0xf0;
  mp[10] &= 0xf0;

  if (setModePage(mp) != 0) {
    message(-2, "Cannot set audio control mode page.");
    return 1;
  }
  return 0;
}

int GenericMMC::restorePlayParameters()
{
  if (setModePage(audioModePage_) != 0) {
    message(-1, "Cannot restore audio control mode page.");
    return 1;
  }
  return 0;
}

int GenericMMC::getTrackIndex(long lba, int *trackNr, int *indexNr)
{
  unsigned char cmd[10];
  unsigned char data[16];

  memset(cmd, 0, 10);

  cmd[0] = 0x45; // PLAY AUDIO
  cmd[2] = lba >> 24;
  cmd[3] = lba >> 16;
  cmd[4] = lba >> 8;
  cmd[5] = lba;
  cmd[8] = 1; // play 1 block
  
  if (sendCmd(cmd, 10, NULL, 0, NULL, 0) != 0) {
    message(-2, "Cannot play audio block.");
    return 1;
  }

  memset(cmd, 0, 10);
  memset(data, 0, 16);

  cmd[0] = 0x42; // READ SUB-CHANNEL
  cmd[2] = 1 << 6;
  cmd[3] = 0x01; // CD-ROM current position
  cmd[8] = 16;

  if (sendCmd(cmd, 10, NULL, 0, data, 16) != 0) {
    message(-2, "Cannot read sub Q channel data.");
    return 1;
  }

  *trackNr = data[6];
  *indexNr = data[7];

  return 0;
}

// tries to read catalog number from disk and adds it to 'toc'
// return: 1 if valid catalog number was found, else 0
int GenericMMC::readCatalog(Toc *toc)
{
  unsigned char cmd[10];
  unsigned char data[24];
  char catalog[14];
  int i;

  memset(cmd, 0, 10);
  memset(data, 0, 24);

  cmd[0] = 0x42; // READ SUB-CHANNEL
  cmd[2] = 1 << 6;
  cmd[3] = 0x02; // get media catalog number
  cmd[8] = 24;   // transfer length

  if (sendCmd(cmd, 10, NULL, 0, data, 24) != 0) {
    message(-2, "Cannot get catalog number.");
    return 0;
  }

  if (data[8] & 0x80) {
    for (i = 0; i < 13; i++) {
      catalog[i] = data[0x09 + i];
    }
    catalog[13] = 0;

    if (toc->catalog(catalog) == 0) {
      return 1;
    }
  }

  return 0;
}

int GenericMMC::readIsrc(int trackNr, char *buf)
{
  unsigned char cmd[10];
  unsigned char data[24];
  int i;

  buf[0] = 0;

  memset(cmd, 0, 10);
  memset(data, 0, 24);

  cmd[0] = 0x42; // READ SUB-CHANNEL
  cmd[2] = 1 << 6;
  cmd[3] = 0x03; // get media catalog number
  cmd[6] = trackNr;
  cmd[8] = 24;   // transfer length

  if (sendCmd(cmd, 10, NULL, 0, data, 24) != 0) {
    message(-2, "Cannot get ISRC code.");
    return 0;
  }

  if (data[8] & 0x80) {
    for (i = 0; i < 12; i++) {
      buf[i] = data[0x09 + i];
    }
    buf[12] = 0;
  }

  return 0;
}

int GenericMMC::analyzeTrack(int trackNr, long startLba,
			     long endLba, Msf *indexIncrements,
			     int *indexIncrementCnt, long *pregap,
			     char *isrcCode, unsigned char *ctl)
{
  unsigned char cmd[12];
  unsigned char *data;
  int blockLength = AUDIO_BLOCK_LEN + 96;
  int blocksPerRead = scsiIf_->maxDataLen() / blockLength;
  int n, i;
  int actIndex = 1;
  long length;
  long crcErrCnt = 0;
  long timeCnt = 0;
  int ctlSet = 0;

  readIsrc(trackNr, isrcCode);

  if (pregap != NULL)
    *pregap = 0;

  *indexIncrementCnt = 0;
  *ctl = 0;

  data = new (unsigned char)[blocksPerRead * blockLength];

  startLba -= 75;
  if (startLba < 0) {
    startLba = 0;
  }
  length = endLba - startLba;

  memset(cmd, 0, 12);
  cmd[0] = 0xbe;  // READ CD
  cmd[9] = 1 << 4; // user data (is transmitted anyway)
  cmd[10] = 0x01;  // raw sub-channel data

  while (length > 0) {
    n = (length > blocksPerRead ? blocksPerRead : length);

    cmd[2] = startLba >> 24;
    cmd[3] = startLba >> 16;
    cmd[4] = startLba >> 8;
    cmd[5] = startLba;
    cmd[6] = n >> 16;
    cmd[7] = n >> 8;
    cmd[8] = n;

    if (sendCmd(cmd, 12, NULL, 0, data, n * blockLength) != 0) {
      delete[] data;
      return 1;
    }

    for (i = 0; i < n; i++) {
      PWSubChannel96 chan(data + i * blockLength + AUDIO_BLOCK_LEN);
      
      if (chan.checkCrc()) {
	if (chan.type() == SubChannel::QMODE1DATA) {
	  int t = chan.trackNr();
	  Msf time(chan.min(), chan.sec(), chan.frame()); // track rel time

	  if (timeCnt > 74) {
	    message(0, "%s\r", time.str());
	    timeCnt = 0;
	  }

	  if (t == trackNr && !ctlSet) {
	    *ctl = chan.ctl();
	    *ctl |= 0x80;
	    ctlSet = 1;
	  }
	  if (t == trackNr && chan.indexNr() > actIndex) {
	    actIndex = chan.indexNr();
	    message(0, "Found index %d at: %s", actIndex, time.str());
	    if ((*indexIncrementCnt) < 98) {
	      indexIncrements[*indexIncrementCnt] = time;
	      *indexIncrementCnt += 1;
	    }
	  }
	  else if (t == trackNr + 1) {
	    if (chan.indexNr() == 0) {
	      if (pregap != NULL)
		*pregap = time.lba();
	    }

	    if (crcErrCnt != 0)
	      message(0, "Found %ld Q sub-channels with CRC errors.",
		      crcErrCnt);
	    
	    delete[] data;
	    return 0;
	  }
	}
      }
      else {
	crcErrCnt++;
#if 0
	if (chan.type() == SubChannel::QMODE1DATA) {
	  message(0, "Q sub-channel data at %02d:%02d:%02d failed CRC check - ignored.",
		 chan.min(), chan.sec(), chan.frame());
	}
	else {
	  message(0, "Q sub-channel data failed CRC check - ignored.");
	}
	chan.print();
#endif
      }

      timeCnt++;
    }

    length -= n;
    startLba += n;
  }

  if (crcErrCnt != 0)
    message(0, "Found %ld Q sub-channels with CRC errors.", crcErrCnt);

  delete[] data;
  return 0;
}

int GenericMMC::driveInfo(DriveInfo *info, int showErrorMsg)
{
  unsigned char mp[22];

  if (getModePage(0x2a, mp, 22, showErrorMsg) != 0) {
    if (showErrorMsg) {
      message(-2, "Cannot retrieve CD capabilities mode page.");
    }
    return 1;
  }

  info->accurateAudioStream = mp[5] & 0x02 ? 1 : 0;

  info->maxReadSpeed = (mp[8] << 8) | mp[9];
  info->currentReadSpeed = (mp[14] << 8) | mp[15];

  info->maxWriteSpeed = (mp[18] << 8) | mp[19];
  info->currentWriteSpeed = (mp[20] << 8) | mp[21];

  return 0;
}
