/*
Copyright (c) 1998 Peter Zelezny.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "style.h"
#include "xchat.h"
#include <netdb.h>
#include <arpa/inet.h>
#include <errno.h>
#include <signal.h>

extern GSList *sess_list;
extern struct xchatprefs prefs;

extern void clear_user_list(struct session *sess);
extern void PrintText(struct session *sess, unsigned char *text);
extern void read_data(struct session *sess, gint sok);
extern char *errorstring(int err);
extern int waitline(int sok, char *buf, int bufsize);


void connected_signal(struct session *sess, gint sok)
{
   char tbuf[128];
   char outbuf[256];

   waitline(sess->server->childread, tbuf, 128);

   if(!sess->server->used)
   {
      gdk_input_remove(sess->server->iotag);
      close(sess->server->childread);
      close(sess->server->sok);
      return;
   }

   switch(tbuf[0])
     {
      case '1':// unknown host
	gdk_input_remove(sess->server->iotag);
	close(sess->server->childread);
	close(sess->server->sok);
	sess->server->flags = 0;
	PrintText(sess, STARTOFF" Unknown host. Maybe you misspelled it?\n");
	break;
      case '3':// gethostbyname finished
	{
	   char host[100];
	   char ip[100];
	   int port;

	   waitline(sess->server->childread, tbuf, 128);
	   strcpy(host, tbuf);

	   waitline(sess->server->childread, tbuf, 128);
	   strcpy(ip, tbuf);

	   waitline(sess->server->childread, tbuf, 128);
	   port = atoi(tbuf);

	   sprintf(outbuf, STARTOFF" Connecting to %s (%s) port %d..\n",
		host, ip, port);
	   PrintText(sess, outbuf);
	}
	break;
      case '4':// success
	 gdk_input_remove(sess->server->iotag);
	 close(sess->server->childread);
	 sess->server->flags |= 1<<0;
	 sess->server->flags &= ~(1<<1);
	 if(!(sess->server->flags&(1<<2)))
	 {
	    PrintText(sess, STARTOFF" Connected. Now logging in..\n");
	    if(sess->server->password[0])
	    {
	       sprintf(outbuf, "PASS %s\r\n", sess->server->password);
	       send(sess->server->sok, outbuf, strlen(outbuf), 0);
	    }
	    sprintf(outbuf, "NICK %s\r\n", sess->server->nick);
	    send(sess->server->sok, outbuf, strlen(outbuf), 0);
	    sprintf(outbuf, "USER %s 0 0 :%s\r\n",
		    prefs.username, prefs.realname);
	    send(sess->server->sok, outbuf, strlen(outbuf), 0);
	 } else
	  PrintText(sess, STARTOFF" Connected.\n");
	 sess->server->iotag = gdk_input_add(sess->server->sok, GDK_INPUT_READ, (GdkInputFunction)read_data, sess);
 	 break;
      default:// connect() failed
	waitline(sess->server->childread, tbuf, 128);
	sprintf(outbuf, STARTOFF" Connection failed. Error: %s\n", errorstring(atoi(tbuf)));
	PrintText(sess, outbuf);
	gdk_input_remove(sess->server->iotag);
	close(sess->server->childread);
	close(sess->server->sok);
	sess->server->flags = 0;
     }
}

void disconnect_server(struct session *sess, int sendquit)
{
   struct server *serv = sess->server;
   GSList *list;

   PrintText(sess, STARTOFF" Disconnected.\n");

   if(sendquit)
   {
      char tbuf[128];
      sprintf(tbuf, "QUIT :%s\r\n", prefs.quitreason);
      send(serv->sok, tbuf, strlen(tbuf), 0);
   }
   gdk_input_remove(serv->iotag);
   shutdown(serv->sok, 2);
   close(serv->sok);
   serv->sok = -1;
   serv->pos = 0;
   serv->flags = 0;

   list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 gtk_window_set_title(GTK_WINDOW(sess->window), "X-Chat ("VERSION")");
	 sess->channel[0] = 0;
	 gtk_entry_set_text(GTK_ENTRY(sess->topicgad), "");
	 gtk_label_set(GTK_LABEL(sess->namelistinfo), "");
	 gtk_label_set(GTK_LABEL(sess->changad), "#nonejoined");
	 clear_user_list(sess);
      }
      list = list->next;
   }
}

void connect_server(struct session *sess, char *server, int port, int quiet)
{
   int sok;
   unsigned char tbuf[256];

   if(!server[0]) return;

   if(sess->server->flags&(1<<1)) // if CONNECTING
   {
      kill(sess->server->childpid, SIGKILL);
      sprintf(tbuf, STARTOFF" Stopping previous connection attempt (pid=%d)\n", sess->server->childpid);
      PrintText(sess, tbuf);
      gdk_input_remove(sess->server->iotag);
      close(sess->server->childread);
      close(sess->server->sok);
      /*sprintf(tbuf, "\0033\0032XCHAT\0033 \003 Already connecting, please wait.\n");
      PrintText(sess, tbuf);
      return;*/
   }

   if(sess->server->flags&(1<<0)) // if CONNECTED
      disconnect_server(sess, TRUE);

   sprintf(tbuf, STARTOFF" Looking up %s..\n", server);
   PrintText(sess, tbuf);

   sok = socket(AF_INET, SOCK_STREAM, 0);
   if(sok == -1) return;

   sess->server->nickcount = 1;
   sess->server->flags = 2;
   sess->server->sok = sok;
   sess->server->port = port;
   
   if(quiet) sess->server->flags |= 1<<2;

   {
      int pid, write_des[2], read_des[2];

      socketpair(AF_UNIX, SOCK_STREAM, 0, (write_des));
      socketpair(AF_UNIX, SOCK_STREAM, 0, (read_des));
      signal(SIGCHLD, SIG_IGN);
      switch(pid = fork())
      {
       case -1: return;

       case 0:
	 {
	    struct sockaddr_in SAddr;
	    struct hostent *HostAddr;

	    dup2(read_des[1], 1);
	    dup2(write_des[0], 0);
	    close(read_des[0]);
	    close(write_des[1]);
	    setuid(getuid());
	    HostAddr = gethostbyname(server);
	    if(HostAddr)
	    {
	       printf("3\n%s\n%s\n%d\n", HostAddr->h_name,
		       inet_ntoa(*((struct in_addr *) HostAddr->h_addr)),
		       port);
	       fflush(stdout);
	       memset(&SAddr, 0, sizeof(SAddr));
	       SAddr.sin_port = htons(port);
	       SAddr.sin_family = AF_INET;
	       memcpy((char *)&SAddr.sin_addr, HostAddr->h_addr, HostAddr->h_length);
	       if(connect(sok, (struct sockaddr *)&SAddr, sizeof(SAddr)) < 0)
		  printf("2\n%d\n", errno);
	       else
		  printf("4\n");
	    } else
	       printf("1\n");
	    fflush(stdout);
	    _exit(0);
	    
       default:
	 close(read_des[1]);
	 close(write_des[0]);
	 break;
	 }
      }
      sess->server->childpid = pid;
      sess->server->iotag = gdk_input_add(read_des[0], GDK_INPUT_READ,
			     (GdkInputFunction)connected_signal, sess);
      sess->server->childread = read_des[0];
   }
}
