/*
  $Id: request.c,v 1.17 1997/01/16 15:04:19 luik Exp $

  request.c - handle/process request queue.
  Copyright (C) 1996-1997, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: request.c,v 1.17 1997/01/16 15:04:19 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#include "libomirr.h"
#include "debug.h"
#include "error.h"
#include "request.h"


static void reqFreeRequest(ReqRequest r);

/*
 * Request Queue handling functions.
 */

/* reqAllocQueue - allocate and initialize a new request queue.  */
ReqQueue reqAllocQueue(void)
{
    ReqQueue q;

    if ((q = malloc(sizeof (ReqQueueRec)))) {
	q->head = NULL;
	q->tail = NULL;
    }
    return q;
}

/* reqFreeQueue - free a request queue completely, including all
   requests in the queue.  */
void reqFreeQueue(ReqQueue q)
{
    ReqRequest r;
    ReqRequest next;

    for (r = q->head; r; r = next) {
	next = r->next;
	reqFreeRequest(r);
    }
    free(q);
}

/* reqAllocRequest - allocate a new request, initialize it and append
   it to the end of the request queue.  */
ReqRequest reqAllocRequest(ReqQueue q)
{
    ReqRequest r;

    if ((r = malloc(sizeof (ReqRequestRec)))) {
	if (q->tail)
	    q->tail->next = r;
	q->tail = r;
	if (q->head == NULL)
	    q->head = r;
	r->next = NULL;
	r->queue = q;
	r->path.ompath = NULL;
	r->path.root_found = 0;
	r->path.pwd_found = 0;
	r->path.file_found = 0;
	r->path.file = NULL;
	r->path.path = NULL;
	r->newpath.ompath = NULL;
	r->newpath.root_found = 0;
	r->newpath.pwd_found = 0;
	r->newpath.file_found = 0;
	r->newpath.file = NULL;
	r->newpath.path = NULL;
	r->s = NULL;
    }
    return r;
}

/* reqFreeRequest - free the dynamically allocated elements of a
   request and the request structure itself.  */
static void reqFreeRequest(ReqRequest r)
{
    if (r->path.ompath)
	free(r->path.ompath);
    if (r->path.path && *r->path.path)
	free(r->path.path);
    if (r->newpath.ompath)
	free(r->newpath.ompath);
    if (r->newpath.path && *r->newpath.path)
	free(r->newpath.path);
    if (r->s)
	free(r->s);
    free(r);
}

/* reqGetFirstRequest - return the first request in the specified queue.  */
ReqRequest reqGetFirstRequest(ReqQueue q)
{
    return q->head;
}

/* reqGetNextRequest - return the requests following `r' in the queue.  */
ReqRequest reqGetNextRequest(ReqRequest r)
{
    return r->next;
}

/* reqDropFirstRequest - remove and free the first queue entry.  */
void reqDropFirstRequest(ReqQueue q)
{
    ReqRequest r = q->head;

    if (r)
	q->head = r->next;
    if (r == q->tail)
	q->tail = NULL;

    reqFreeRequest(r);
}


/*
 * Filename quoting and unquoting
 */

static char quote_src[] = "\\ \t\n\f"; /* characters to be quoted by backslashes */
static char quote_dst[] = "\\stnf"; /* replacement characters */

/* quote_count - returns the number of characters that need to be
   quoted in filename. */
static int quote_count(const char *filename)
{
    int count = 0;
    const char *cp;

    for (cp = filename; *cp; cp++)
	if (strchr(quote_src, *cp))
	    count++;
    return count;
}

/* quote_len - returns the new length of character string `filename' after
   it is quoted. This equals `strlen(filename) + quote_count(filename).  */
static int quote_len(const char *filename)
{
    int count = 0;
    const char *cp;

    for (cp = filename; *cp; cp++)
	if (strchr(quote_src, *cp))
	    count += 2;
	else count++;
    return count;
}

/* quote_name - quotes all characters listed in `quote_src' which
   occur in filename using backslashes. The filename data is
   overwritten and must be long enough for `quote_count(filename)'
   additional characters. Returns `filename'.  */
static char *quote_name(char *filename)
{
    int count;
    int len;
    char *src;
    char *dst;
    char *q;			/* points to character to quote in quote_src */

    count = quote_count(filename);
    if (count) {
	len = strlen(filename);
	for (src = filename + len, dst = src + count; src != dst; src--, dst--) {
	    if (*src && ((q = strchr(quote_src, *src)))) {
		*dst-- = quote_dst[q - quote_src];
		*dst = '\\';
	    }
	    else {
		*dst = *src;
	    }
	}
    }
    return (filename);
}

/* unquote_name - removes the backslash escapes in `filename'. All
   two-character sequences starting with a backslash are replaced by
   the second character, converted according the `quote_dst' -
   `quote_src' mapping. The filename data is overwritten but naturally
   the length will not increase. Returns `filename'.  */
static char *unquote_name(char *filename)
{
    char *src;
    char *dst;
    char *q;			/* points to character to unquote in quote_dst */

    for (dst = src = filename; *src; src++, dst++) {
	if (*src == '\\') {
	    if ((q = strchr(quote_dst, *++src)))
		*dst = quote_src[q - quote_dst];
	    else if (*src == '\0')
		break;
	    else *dst = *src;
	}
	else {
	    *dst = *src;
	}
    }
    *dst = '\0';
    return (filename);
}

/* canonilize_name - convert absolute pathname `filename' to canonic
   pathname: (1) if any of the partial filenames (from the beginning
   to a slash) is a symbolic link, the link is resolved; (2) if
   `follow_link' is true and the complete pathname is a symbolic link,
   it is resolved, too, otherwise if `follow_link' is false, this
   final resolving is suppressed; (3) remove duplicate and trailing
   slashes, remove "/." and "/.." constructs.  The original filename
   is not modified.  Returns a pointer to a malloc'd area containing
   the new pathname `newname' which is long enough for additional
   quoting (see `quote_name' above).  */
static char *canonilize_name(char *filename, int follow_link)
{
    char *src;
    char *dst;
    int len;
    char buf[8192];		/* XXX fixed length */
    char oldname[8192];		/* XXX fixed length */
    static char newname[8192];	/* XXX fixed length */

    assert(*filename == '/');
    strcpy(oldname, filename);	/* copy to work buffer */
    if (follow_link) {
	strcat(oldname, "/");	/* ensure we have a trailing slash */
    }
    else {
	src = strchr(oldname, '\0');
	while (*--src == '/' && src > oldname)
	    *src = '\0';	/* remove all trailing slashes */
    }

    for (dst = newname, src = oldname; *src; src++) {
      again:
	if (*src == '/') {
	    *src = '\0';
	    /* Try whether oldname up to *src is a symbolic link. If
               yes, replace path (or pathname component) with link
               target.  */
	    if ((len = readlink(oldname, buf, sizeof(buf))) != -1) {
		*src = '/';
		strcpy(buf + len, src); /* append rest */
		if (*buf == '/') { /* absolut, replace everything */
		    strcpy(oldname, buf);
		    dst = newname, src = oldname;
		}
		else {
		    /* The following code asserts that we already have
                       a slash in newname before dst and in oldname
                       before src, which must be true (unless "/" is a
                       symbolic link which is very unlikely :-).  */
		    while (*--src != '/' && src > oldname)
			;
		    strcpy(src + 1, buf);
		    while (*--dst != '/' && dst > newname)
			;
		}
		goto again;
	    }
	    *src = '/';
	    if (strneq(src + 1, "../", 3) || streq(src + 1, "..")) {
		while (dst > newname && *--dst != '/')
		    ;
		src += 2;	/* skip "/.." */
	    }
	    else if (strneq(src + 1, "./", 2) || streq(src + 1, ".")) {
		src++;		/* skip "/." */
	    }
	    else if (src[1] == '/' || src[1] == '\0') {
		;		/* skip duplicate or trailing '/' */
	    }
	    else *dst++ = *src;	/* append "/" */
	}
	else *dst++ = *src;	/* append character */
    }
    if (dst == newname)
	*dst++ = (*oldname == '/' ? '/' : '.');
    *dst = *src;

    if ((dst = malloc(quote_len(newname) + 1)))
	strcpy(dst, newname);
    return dst;
}



void reqError(char *errmsg, char *line)
{
    if (strlen(line) > 24)
	warning("%s: message `%30.30s ...' ignored\n", errmsg, line);
    else
	warning("%s: message `%s' ignored\n", errmsg, line);
}



/*
 * Request Parser (converts message to request).
 */

#define parse_dev_ino(L, D, I, N)			\
    (sscanf((L), "%i,%i%n", (D), (I), (N)) == 2)

#define parse_name_dev_ino(L, P, D, I, N)		\
    (sscanf((L), "%s %i,%i%n", (P), (D), (I), (N)) == 3)

#define parse_path(L, T, P, N)				\
    (sscanf((L), "%li %s%n", (T), (P), (N)) == 2)

#define parse_path_path(L, T, P1, P2, N)		\
    (sscanf((L), "%li %s %s%n", (T), (P1), (P2), (N)) == 3)

#define parse_path_int_int(L, T, P, I1, I2, N)		\
    (sscanf((L), "%li %s %i %i%n", (T), (P), (I1), (I2), (N)) == 4)


ReqRequest reqParseMessage(ReqQueue q, char *line)
{
    char *cp;			/* current position in line */
    int n;			/* number of bytes matched */
    long timestamp;		/* parsed timestamp */
    char *path;			/* parsed pathname (malloc'ed) */
    char *newpath;		/* parsed pathname (malloc'ed) for newpath */
    int i1, i2;			/* parsed multi-purpose integer values */
    ReqRequest r = NULL;	/* new request */
    char protocol_error = 0;	/* protocol error flag, checked at end */

    cp = line;

    switch (*cp++) {
      case 'A':			/* A dev,ino - increment inode counter */
	if (parse_dev_ino(cp, &i1, &i2, &n) && cp[n] == '\0') {
	    cp += n;
	    if ((r = reqAllocRequest(q))) {
		r->op = *line;	/* set operation */
		r->di.dev = i1;
		r->di.ino = i2;
	    }
	}
	else protocol_error++;
	break;

      case 'F':			/* F dev,ino - decrement inode counter, free */
	if (parse_dev_ino(cp, &i1, &i2, &n) && cp[n] == '\0') {
	    cp += n;
	    if ((r = reqAllocRequest(q))) {
		r->op = *line;	/* set operation */
		r->di.dev = i1;
		r->di.ino = i2;
	    }
	}
	else protocol_error++;
	break;

      case 'M':			/* M ts path mode 0 - set file mode */
      case 'O':			/* O ts path owner group - set file owner/group */
      case 'U':			/* U ts path actime modtime - set file times */
      case 'W':			/* W ts inopath mode size - write file contents */
      case 'c':			/* c ts fpath mode 0 - creat */
      case 'd':			/* d ts fpath mode 0 - mkdir */
      case 'n':			/* n ts fpath mode dev - mknod */
	if ((path = malloc(strlen(line)))) {
	    if ((parse_path_int_int(cp, &timestamp, path, &i1, &i2, &n)
		 && cp[n] == '\0')) {
		cp += n;
		if ((r = reqAllocRequest(q))) {
		    r->op = *line; /* set operation */
		    r->path.ompath = unquote_name(path);
		    r->i1 = i1;
		    r->i2 = i2;
		    r->timestamp = timestamp;
		}
	    }
	    else protocol_error++;
	}
	break;

      case 'N':			/* N fpath dev,ino - assoc inode with pathname */
	if ((path = malloc(strlen(line)))) {
	    if (parse_name_dev_ino(cp, path, &i1, &i2, &n) && cp[n] == '\0') {
		cp += n;
		if ((r = reqAllocRequest(q))) {
		    r->op = *line; /* set operation */
		    r->path.ompath = unquote_name(path);
		    r->di.dev = i1;
		    r->di.ino = i2;
		}
	    }
	    else protocol_error++;
	}
	break;

      case 'r':			/* r ts fpath - rmdir */
      case 'u':			/* u ts fpath - unlink */
	if ((path = malloc(strlen(line)))) {
	    if (parse_path(cp, &timestamp, path, &n) && cp[n] == '\0') {
		cp += n;
		if ((r = reqAllocRequest(q))) {
		    r->op = *line; /* set operation */
		    r->path.ompath = unquote_name(path);
		    r->timestamp = timestamp;
		}
	    }
	    else protocol_error++;
	}
	break;

      case 'l':			/* l ts name newname - link */
      case 'm':			/* m ts name newname - rename (move) */
      case 's':			/* s ts string name - symlink */
	if ((path = malloc(strlen(line))) && (newpath = malloc(strlen(line)))) {
	    if ((parse_path_path(cp, &timestamp, path, newpath, &n)
		 && cp[n] == '\0')) {
		cp += n;
		if ((r = reqAllocRequest(q))) {
		    r->op = *line; /* set operation */
		    if (*line == 's') {	/* symlink has special arguments */
			r->s = unquote_name(path); /* not really a pathname */
			r->path.ompath = unquote_name(newpath);
		    }
		    else {
			r->path.ompath = unquote_name(path);
			r->newpath.ompath = unquote_name(newpath);
		    }
		    r->timestamp = timestamp;
		}
	    }
	    else protocol_error++;
	}
	break;

      case 'D':			/* Dv string - data */
	cp++;
	while (isspace(*cp))
	    cp++;

	if (*cp) {
	    if ((r = reqAllocRequest(q))) {
		r->op = *line;	/* set operation */
		r->subop = line[1];
		r->s = xstrdup(cp);
	    }
	}
	else protocol_error++;
	break;

      case 'E':			/* E - end-of-data */
      case 'Z':			/* Z - overflow - resync needed */
	if (*cp == '\0') {
	    if ((r = reqAllocRequest(q))) {
		r->op = *line;	/* set operation */
	    }
	}
	else protocol_error++;
	break;

      default:
	protocol_error++;
    }

    if (protocol_error)
	reqError("protocol error", line);
    else if (!r)		/* handle out-of-memory error from above */
	reqError("memory exhausted", line);

    return r;
}


/*
 * Request completor which fills missing parts of requests and especially
 * converts ompaths to pathnames. XXX should perhaps be moved into omirrk.c.
 */

#define parse_devino(L, D, I, N)			\
    (sscanf((L), "[%i,%i]%n", (D), (I), (N)) == 2)

static void ompath_to_path(ReqPath pathp)
{
    char *cp;
    int n;
    int i1, i2;
    int len;
    char *root;
    char *pwd;
    InamStatus root_status;
    InamStatus pwd_status;
    InamStatus file_status;

    if (!pathp->ompath)
	return;			/* nothing to resolve */

    if (pathp->path)
	return;			/* already resolved */

    if (((pathp->root_found | pathp->pwd_found | pathp->file_found) == 0
	 && pathp->file == NULL)) {
	/* transition REQ_INCOMPLETE -> REQ_PROCESSING */

	cp = pathp->ompath;
	if (*cp == 'R') {
	    if (parse_devino(++cp, &i1, &i2, &n)) {
		pathp->root_found = 1;
		pathp->root_di.dev = i1;
		pathp->root_di.ino = i2;
		cp += n;
	    }
	    else /* XXX parse error */;
	}

	if (*cp == 'P') {
	    if (parse_devino(++cp, &i1, &i2, &n)) {
		pathp->pwd_found = 1;
		pathp->pwd_di.dev = i1;
		pathp->pwd_di.ino = i2;
		cp += n;
	    }
	    else /* XXX parse error */;
	}
	else /* XXX assert(*cp == '/' || *cp == '[') */;

	if (*cp == '[') {
	    if (parse_devino(cp, &i1, &i2, &n)) {
		pathp->file_found = 1;
		pathp->file_di.dev = i1;
		pathp->file_di.ino = i2;
		cp += n;
		/* XXX assert(*cp == '\0') */
	    }
	    else /* XXX parse error */;
	}
	else pathp->file = cp;
    }

    root_status = INAM_SUCCESS;
    file_status = INAM_SUCCESS;
    pwd_status = INAM_SUCCESS;
    if (pathp->root_found)
	root_status = inamLookupDevIno(&pathp->root_di, &root);
    if (pathp->file_found)
	file_status = inamLookupDevIno(&pathp->file_di, &pathp->file);
    else if (pathp->pwd_found && *pathp->file != '/')
	pwd_status = inamLookupDevIno(&pathp->pwd_di, &pwd);

    if ((root_status == INAM_INPROCESS
	 || file_status == INAM_INPROCESS
	 || pwd_status == INAM_INPROCESS))
	return;			/* still processing */

    /* transition REQ_PROCESSING -> REQ_COMPLETE */
    if ((pathp->root_found && root == NULL
	 || pathp->file_found && pathp->file == NULL)) {
	pathp->path = "";
	return;			/* error */
    }

    len = pathp->root_found ? quote_len(root) : 0;
    len += quote_len(pathp->file);
    if (*pathp->file != '/') {
	if (pathp->pwd_found && pwd == NULL) {
	    pathp->path = "";
	    return;		/* error */
	}
	len += pathp->pwd_found ? quote_len(pwd) + 1 : 0;
    }

    if (len && (pathp->path = malloc(len + 1))) {
	*pathp->path = '\0';
	if (pathp->root_found)
	    strcat(pathp->path, root);
	if (*pathp->file != '/' && pathp->pwd_found) {
	    strcat(pathp->path, pwd);
	    strcat(pathp->path, "/");
	}
	strcat(pathp->path, pathp->file);
    }
    else pathp->path = "";
}


ReqState reqFillRequest(ReqRequest r)
{
    ReqState state = REQ_COMPLETE;

    if (r->path.ompath) {
	if (!r->path.path)
	    ompath_to_path(&r->path);
	if (!r->path.path)
	    state = REQ_PROCESSING;
	else if (!*r->path.path)
	    state = REQ_ERROR;
    }

    if (r->newpath.ompath) {
	if (!r->newpath.path)
	    ompath_to_path(&r->newpath);
	if (!r->newpath.path)
	    state = REQ_PROCESSING;
	else if (!*r->newpath.path)
	    state = REQ_ERROR;
    }

    return (state);
}


/*
 * Request Processor/Writer (processes request or converts request to message).
 */

/* reqProcessKernelRequest - process the requests which are special
   for the kernel interface, i.e. the requests that modify the
   inode-to-name cache.  Returns 0 if the request is completly
   processed, i.e. it is not neccessary to send the request from
   omirrk to omirrd, and 1 if the requests needs further processing,
   i.e. needs to be sent to omirrd.  */
int reqProcessKernelRequest(ReqRequest r)
{
    char *path;			/* canonic and quoted path (malloc'd) */
    char *newpath;		/* canonic and quoted newpath (malloc'd) */
    int follow_link;

    switch (r->op) {
      case 'A':			/* A dev,ino - increment inode counter */
	inamUseDevIno(&r->di);
	return 0;

      case 'F':			/* F dev,ino - decrement inode counter, free */
	inamFreeDevIno(&r->di);
	return 0;

      case 'N':			/* N fpath dev,ino - assoc inode with pathname */
	if ((path = canonilize_name(r->path.path, 0)))
	    inamAssocNameDevIno(path, &r->di);
	xfree(path);
	return 0;

      case 'm':			/* m ts name newname - rename (move) */
	newpath = NULL;
	follow_link = 0;
	if ((path = canonilize_name(r->path.path, follow_link)))
	    if ((newpath = canonilize_name(r->newpath.path, 0))) 
		inamRename(path, newpath);
	    else inamRemove(path);
	else inamRemove(r->path.path);
	break;			/* further processing needed */
    }
    return 1;
}


char *reqRequestToMessage(ReqRequest r)
{
    char *line = NULL;		/* new message (malloc'd) */
    char *path;			/* canonic and quoted path (malloc'd) */
    char *newpath;		/* canonic and quoted newpath (malloc'd) */
    int follow_link;

    switch (r->op) {
      case 'M':			/* M ts path mode 0 - set file mode */
      case 'O':			/* O ts path owner group - set file owner/group */
      case 'U':			/* U ts path actime modtime - set file times */
      case 'W':			/* W ts inopath mode size - write file contents */
      case 'c':			/* c ts fpath mode 0 - creat */
      case 'd':			/* d ts fpath mode - mkdir */
      case 'n':			/* n ts fpath mode dev - mknod */
	follow_link = (r->op == 'd' || r->op == 'n' ? 0 : 1);
	if (((path = canonilize_name(r->path.path, follow_link)))
	    && quote_name(path)
	    && ((line = malloc(strlen(path) + 5 + 3 * 20 + 1))))
	    sprintf(line, "%c %ld %s %d %d", r->op, (long) r->timestamp,
		    path, r->i1, r->i2);
	else error("memory exhausted: request `%c %s %d %d' ignored\n",
		   r->op, path ? path : r->path.path, r->i1, r->i2);
	xfree(path);
	break;

      case 'r':			/* r ts fpath - rmdir */
      case 'u':			/* u ts fpath - unlink */
	follow_link = 0;
	if (((path = canonilize_name(r->path.path, follow_link)))
	    && quote_name(path)
	    && ((line = malloc(strlen(path) + 3 + 20 + 1))))
	    sprintf(line, "%c %ld %s", r->op, (long) r->timestamp, path);
	else error("memory exhausted: request `%c %s' ignored\n",
		   r->op, path ? path : r->path.path);
	xfree(path);
	break;

      case 'l':			/* l ts name newname - link */
      case 'm':			/* m ts name newname - rename (move) */
	newpath = NULL;
	follow_link = (r->op == 'l' ? 1 : 0);
	if (((path = canonilize_name(r->path.path, follow_link)))
	    && ((newpath = canonilize_name(r->newpath.path, 0)))
	    && quote_name(path)
	    && quote_name(newpath)
	    && ((line = malloc(strlen(path) + strlen(newpath) + 4 + 20 + 1))))
	    sprintf(line, "%c %ld %s %s", r->op, (long) r->timestamp,
		    path, newpath);
	else error("memory exhausted: request `%c %s %s' ignored\n",
		   r->op, path ? path : r->path.path,
		   newpath ? newpath : r->newpath.path);
	xfree(newpath);
	xfree(path);
	break;

      case 's':			/* s ts string name - symlink */
	if (((path = canonilize_name(r->path.path, 0)))
	    && quote_name(path)
	    && quote_name(r->s)
	    && ((line = malloc(strlen(r->s) + strlen(path) + 4 + 20 + 1))))
	    sprintf(line, "%c %ld %s %s", r->op, (long) r->timestamp,
		    r->s, path);
	else error("memory exhausted: request `%c %s %s' ignored\n",
		   r->op, r->s, path ? path : r->path.path);
	xfree(path);
	break;

      case 'Z':			/* Z - overflow - resync needed */
	if ((line = malloc(2 + 1)))
	    sprintf(line, "%c", r->op);
	else error("memory exhausted: request `%c' ignored\n", r->op);
	break;

      case 'D':			/* Dv string - data */
      case 'E':			/* E - end-of-data */
      default:
	error("internal: illegal request op `%d'\n", r->op);
	break;
    }

    return line;
}

