/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#ifndef _ATALK_ADOUBLE_H
#define _ATALK_ADOUBLE_H

#include <unistd.h>

#if defined(sun) && defined(__svr4__)
#include </usr/ucbinclude/sys/file.h>
#else
#include <sys/file.h>
#endif
#include <fcntl.h>
#include <sys/cdefs.h>
#include <sys/types.h>
#include <netatalk/endian.h>

#ifdef _POSIX_MAPPED_FILES
#include <sys/mman.h>
#endif

/*
 * AppleDouble entry IDs. 
 */
#define ADEID_DFORK		1
#define ADEID_RFORK		2
#define ADEID_NAME		3
#define ADEID_COMMENT		4
#define ADEID_ICONBW		5
#define ADEID_ICONCOL		6
#define ADEID_FILEI		7  /* v1, replaced by: */
#define ADEID_FILEDATESI	8  /* this */
#define ADEID_FINDERI		9
#define ADEID_MACFILEI		10 /* and this */
#define ADEID_PRODOSFILEI	11
#define ADEID_MSDOSFILEI	12
#define ADEID_SHORTNAME		13
#define ADEID_AFPFILEI		14
#define ADEID_DID		15

#define ADEID_MAX		16

/*
 * AppleDouble entry default offsets.
 * The layout looks like this:
 *
 * this is the v1 layout:
 *	  255	  200		  16	  32		  N
 *	|  NAME	|    COMMENT	| FILEI	|    FINDERI	| RFORK	|
 *
 * we need to change it to look like this:
 *
 * v2 layout:
 * field       length (in bytes)
 * NAME        255
 * COMMENT     200
 * FILEDATESI  16     replaces FILEI
 * FINDERI     32  
 * MACFILEI    32     new
 * PRODOSI      8     new
 * MSDOSI       2     new
 * SHORTNAME   12     8.3
 * AFPFILEI     4     new
 * DID          4     new
 * RFORK        N
 * 
 * so, all we need to do is replace FILEI with FILEDATESI, move RFORK
 * + 62 bytes, and add in the new fields.
 *
 * NOTE: the HFS module will need some modifications to interact with
 * afpd correctly.
 */
 
/* v2 fields */
#define ADEDOFF_RFORK_V2	651
#define ADEDOFF_FILEDATESI	541
#define ADEDOFF_MACFILEI        589
#define ADEDOFF_PRODOSFILEI     621
#define ADEDOFF_MSDOSFILEI      629
#define ADEDOFF_SHORTNAME       631
#define ADEDOFF_AFPFILEI        643
#define ADEDOFF_DID		647

#define ADEDLEN_FILEDATES	16
#define ADEDLEN_MACFILEI        32
#define ADEDLEN_PRODOSFILEI      8
#define ADEDLEN_MSDOSFILEI       2
#define ADEDLEN_SHORTNAME        0 /* length up to 8.3 */
#define ADEDLEN_AFPFILEI         4
#define ADEDLEN_DID              4

/* v1 fields */
#define ADEDOFF_RFORK_V1	589
#define ADEDOFF_RFORK    	ADEDOFF_RFORK_V1
#define ADEDOFF_NAME	86
#define ADEDOFF_COMMENT	341
#define ADEDOFF_FILEI	541
#define ADEDOFF_FINDERI	557

#define ADEDLEN_RFORK	0
#define ADEDLEN_NAME	0
#define ADEDLEN_COMMENT	0
#define ADEDLEN_FILEI	16
#define ADEDLEN_FINDERI	32

#define AD_APPLESINGLE_MAGIC 0x00051600
#define AD_APPLEDOUBLE_MAGIC 0x00051607
#define AD_MAGIC	AD_APPLEDOUBLE_MAGIC
#define AD_VERSION1	0x00010000
#define AD_VERSION2	0x00020000
#define AD_VERSION	AD_VERSION1

#define AD_DATASZ_V1	589
#define AD_DATASZ_V2	651
#define AD_DATASZ	AD_DATASZ_V1

/*
 * The header of the AppleDouble Header File looks like this:
 *
 *	NAME			SIZE
 *	====			====
 *	Magic			4
 *	Version			4
 * 	Home File System	16  (this becomes filler in ad v2)
 *	Number of Entries	2
 *	Entry Descriptors for each entry:
 *		Entry ID	4
 *		Offset		4
 *		Length		4
 */

struct ad_entry {
    u_int32_t	ade_off;
    u_int32_t	ade_len;
};

struct ad_fd {
    int		 adf_fd;
    u_int32_t	 adf_off;
    int		 adf_flags;
    struct flock adf_lock;
    struct ad_fd *adf_same;
};

struct adouble {
    u_int32_t		ad_magic;
    u_int32_t		ad_version;
  /* i'm intentionally breaking this to make sure that i upgrade
   * things correctly. */
#if AD_VERSION == AD_VERSION1 
    char		ad_homefs[ 16 ];
#else
    char		ad_filler[ 16 ];
#endif
    struct ad_entry	ad_eid[ ADEID_MAX ];
    char		ad_data[ AD_DATASZ ];
    struct ad_fd	ad_df, ad_hf;
};

#define ADFLAGS_DF	(1<<0)
#define ADFLAGS_HF	(1<<1)
#define ADFLAGS_DIR	(1<<2)

/* Bits in the finderinfo data. 
 * see etc/afpd/{directory.c,file.c} for the finderinfo structure
 * layout. */
#define FINDERINFO_CUSTOMICON 0x4
#define FINDERINFO_CLOSEDVIEW 0x100

/* offsets in finderinfo */
#define FINDERINFO_FLAGSOFF    8
#define FINDERINFO_FRVIEWOFF  14

/* lock flags */
#define ADLOCK_CLR      0x0
#define ADLOCK_RD       0x1
#define ADLOCK_WR       0x2

#define ad_dfileno(ad)		((ad)->ad_df.adf_fd)
#define ad_dlock(ad)            ((ad)->ad_df.adf_lock)
#define ad_hfileno(ad)		((ad)->ad_hf.adf_fd)
#define ad_hlock(ad)            ((ad)->ad_hf.adf_lock)
#define ad_getversion(ad)	((ad)->ad_version)
#define ad_gethomefs(ad)	((ad)->ad_homefs);
#define ad_sethomefs(ad,buf) \
	(bcopy((buf),(ad)->ad_homefs,sizeof((ad)->ad_homefs)))
#define ad_getentrylen(ad,eid)	((ad)->ad_eid[(eid)].ade_len)
#define ad_setentrylen(ad,eid,len) \
	((ad)->ad_eid[(eid)].ade_len = (len))
#define ad_entry(ad,eid)	(((ad)->ad_eid[(eid)].ade_off + \
	(ad)->ad_eid[(eid)].ade_len > AD_DATASZ ) ? 0 : \
	(caddr_t)(ad)->ad_data + (ad)->ad_eid[(eid)].ade_off)
#define ad_getoflags(ad,adf)	(((adf)&ADFLAGS_HF) ? \
	(ad)->ad_hf.adf_flags : (ad)->ad_df.adf_flags)

/* ad_flush.c */
extern int ad_flush __P((struct adouble *, int));

/* ad_lock.c */
extern int ad_flock_lock __P((struct adouble *, const u_int32_t /*eid*/,
			      const int /*type*/, off_t /*offset*/,
			      const int /*endflag*/, const off_t /*len*/));
extern int ad_fcntl_lock __P((struct adouble *, const u_int32_t /*eid*/,
			      const int /*type*/, off_t /*offset*/,
			      const int /*endflag*/, const off_t /*len*/));

extern int ad_flock_tmplock __P((struct adouble *, const u_int32_t /*eid*/,
				 const int /*type*/, off_t /*offset*/,
				 const int /*end*/, const off_t /*len*/));
extern int ad_fcntl_tmplock __P((struct adouble *, const u_int32_t /*eid*/,
				 const int /*type*/, off_t /*offset*/,
				 const int /*end*/, const off_t /*len*/));

#ifdef USE_FCNTL_LOCKS
#define ad_lock ad_fcntl_lock
#define ad_tmplock ad_fcntl_tmplock
#else
#define ad_lock ad_flock_lock
#define ad_tmplock ad_flock_tmplock
#endif

/* ad_open.c */
extern char *ad_path __P((char *, int));
extern int ad_mode __P((char *, int));
extern int ad_mkdir __P((char *, int));
extern int ad_open __P((char *, int, int, int, struct adouble *));
extern int ad_refresh __P((struct adouble *));

/* ad_read.c/ad_write.c */
extern ssize_t ad_read __P((struct adouble *, const u_int32_t, 
			    const u_int32_t, char *, const size_t));
extern ssize_t ad_write __P((struct adouble *, const u_int32_t, u_int32_t,
			     const int, const char *, const size_t));
extern int ad_dtruncate __P((struct adouble *, const size_t));
extern int ad_rtruncate __P((struct adouble *, const size_t));

/* ad_size.c */
extern off_t ad_size __P((struct adouble *, const u_int32_t ));

/* ad_mmap.c */
extern void *ad_mmapread __P((struct adouble *, const u_int32_t,
			      const off_t, const size_t));
extern void *ad_mmapwrite __P((struct adouble *, const u_int32_t,
			       const off_t, const int, const size_t));
#define ad_munmap(buf, len)  (munmap((buf), (len)))

#endif
