/*
   Pathetic Writer
   Copyright (C) 1997, 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * fileio_pw.c
 * 970706 File format snatched from Siag
 * 970707 New tag .sty to distinguish Pw styles from Siag styles
 * 970714 New tag .ad for adjustment. But this sucks!
 * 980720 .plugin added, structured file format
 */

/*
   Brief description of the file format:

   #comment                     A comment; ignored
   .sw width                    Standard column width
   .sh height                   Standard row height
   .sf format                   Standard format code
   .cw col width                        Column width in pixels
   .rh row height                       Column height in pixels
   .st row style			Style
   .ad					Adjust
   .fmtN				Format N
   .styN				Style N
   row col format       #               Empty cell
   row col format       "label          Cell containing a label

   Leading or trailing white space is not allowed
   Empty lines are ignored
   All widths and heights are in pixels
   Format codes are in decimal

   This is the same format as Siag with only labels and with cells
   replaced by segments.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../common/common.h"
#include "../common/fonts.h"
#include "../common/cmalloc.h"

#include "pw.h"

/* returns the name of a component in a structured file. Caller must free */
static char *file_name(buffer *buf, char *n)
{
        char b[1024];
        plugin_basedir(buf, b);
        strcat(b, "/");
        strcat(b, n);
        return cstrdup(b);
}

static int save_flat(char *fn, buffer * buf)
/* Returns: 0 if successful, otherwise 1 */
{
	int i, lr;
	FILE *fp;
	int sh = styles[STY_DEFAULT].height, sf = styles[STY_DEFAULT].format;

	if ((fp = fopen(fn, "w")) == (FILE *) 0)
		return 1;

	/* start by saving standard values */
	fprintf(fp, "# Creator: %s\n", VERSION);
	fprintf(fp, ".sh %d\n", sh);
	fprintf(fp, ".sf %d\n", sf);
	for (i = STY_DEFAULT; i <= STY_EMBED; i++)
		fprintf(fp, ".sty%d %s\n", i, fmt_get(NULL, i));

	for (i = 0; i < buf->nplugin; i++) {
		fprintf(fp, ".plugin %ld %ld %s\n",
			buf->plugin[i].row, buf->plugin[i].col,
			buf->plugin[i].name);
	}

	lr = line_last_used(buf);
	fprintf(fp, "# %s\n# %d lines\n#\n", fn, lr);

	for (i = 1; i <= lr; i++) {
		int height = cell_height(buf, i);
		if (height != sh)
			fprintf(fp, ".rh %d %d\n", i, height);
		if (buf->text[i].sty != STY_DEFAULT)
			fprintf(fp, ".st %d %d\n", i, buf->text[i].sty);
		if (buf->text[i].adj != HADJ_LEFT)
			fprintf(fp, ".ad %d %d\n", i, buf->text[i].adj);
	}
	for (i = 1; i <= lr; i++) {
		int j = 0, col = 1;
		rich_char *line = buf->text[i].p;
		if (rc_strlen(line) == 0) continue;	/* empty line */
		fprintf(fp, "# Line %d", i);

		while (line[j].c) {
			if (j == 0 || line[j].fmt != line[j-1].fmt)
				fprintf(fp, "\n%d %d %ld\t\"",
					i, col++, line[j].fmt);
			putc(line[j++].c, fp);
		}
		putc('\n', fp);
	}
	fprintf(fp, "# End of file %s\n", fn);
	fclose(fp);
	return 0;
}				/* save_slat */

static int load_flat(char *fn, buffer * buf)
/* Returns: 0 if successful, otherwise 1 */
{
	int i, j;
	long f;

	FILE *fp;
	char instring[256], *p, *contents;
	int sf = styles[STY_DEFAULT].format;

	if ((fp = fopen(fn, "r")) == NULL)
		return 1;

	while (fgets(instring, 250, fp) != NULL) {
		switch (instring[0]) {
		case '\0':
		case '#':
			break;
		case '.':
			if (!strncmp(instring, ".sf", 3)) {
				sscanf(instring, ".sf %d", &sf);
			} else if (!strncmp(instring, ".rh", 3)) {
				int h;
				sscanf(instring, ".rh %d %d", &i, &h);
				alloc_line(buf, i);
				buf->text[i].height = h;
			} else if (!strncmp(instring, ".st ", 4)) {
				int sty;
				sscanf(instring, ".st %d %d", &i, &sty);
				alloc_line(buf, i);
				buf->text[i].sty = sty;
			} else if (!strncmp(instring, ".ad ", 4)) {
				int adj;
				sscanf(instring, ".ad %d %d", &i, &adj);
				alloc_line(buf, i);
				buf->text[i].adj = adj;
			} else if (!strncmp(instring, ".sty", 4)) {
				int h;
				char b[256];
				sscanf(instring, ".sty%d %[^\n]", &h, b);
				fmt_put(b, h);
			} else if (!strncmp(instring, ".plugin", 7)) {
				int n;
				long row, col;
				char name[1024];
				plugin_t plugin;
				sscanf(instring, ".plugin %ld %ld %s",
					&row, &col, name);
				plugin.row = row;
				plugin.col = col;
				plugin.name = cstrdup(name);
				p = file_name(buf, name);
				plugin.ph = pw_plugin_start(p);
				cfree(p);
				plugin.displayed = 0;
				if (plugin.ph != -1) {
					n = buf->nplugin++;
					buf->plugin = crealloc(buf->plugin,
                                                buf->nplugin*sizeof(plugin_t));
                                        buf->plugin[n] = plugin;
				}
			}
			break;
		default:	/* cell definition */
			i = j = 0;
			i = strtol(instring, &p, 10);
			if (i < 1 || i > 1000000) break;
			j = strtol(p, &p, 10);
			if (j < 1 || j > 1000000) break;
			f = strtol(p, &p, 10);
			if (p[0] != '\t' || p[1] != '"') break;
			contents = p+2;
			if ((p = strchr(contents, '\n'))) *p = '\0';
			ins_text(buf, make_position(i, line_length(buf, i)),
					(unsigned char *)contents, f);
			break;
		}
	}

	fclose(fp);
	return 0;
} /* load_flat */

#define PW_MAGIC "# Creator: Pathetic Writer"

static int flatfile(char *fn)
{
	char b[100];
	FILE *fp = fopen(fn, "r");
	if (!fp) return 0;
	if (fgets(b, sizeof b, fp) &&
			!strncmp(b, PW_MAGIC, strlen(PW_MAGIC))) {
		fclose(fp);
		return 1;
	}
	return 0;
}

static int tryuntar(char *fn)
{
	char b[1024];
	int result;
        struct stat statbuf;

        sprintf(b,
                "mkdir -p %s/untar;"
                "cat %s|(cd %s/untar;tar xf - INDEX.pw) 2> /dev/null",
                siag_basedir, fn, siag_basedir);
        system(b);
        sprintf(b, "%s/untar/INDEX.pw", siag_basedir);
        result = !stat(b, &statbuf);
        sprintf(b, "rm -rf %s/untar", siag_basedir);
        system(b);
        return result;
}

static int save(char *fn, buffer *buf)
{
        char cmd[1024];
        char *dir, *b, *p;
        int i, result;

        if (buf->nplugin == 0) {        /* no plugins, save as usual */
                return save_flat(fn, buf);
        }

        /* save all the plugins */
        dir = file_name(buf, "");
        p = strrchr(dir, '/');
        if (p) *p = '\0';
        mkdir(dir, 0700);
        for (i = 0; i < buf->nplugin; i++) {
                b = file_name(buf, buf->plugin[i].name);
                pw_plugin_save(buf->plugin[i].ph, b);
                cfree(b);
        }
        /* save the main file */
        b = file_name(buf, "INDEX.pw");
        result = save_flat(b, buf);
        cfree(b);
        /* tar up the lot */
        sprintf(cmd, "(cd %s;tar cf - *)>%s", dir, fn);
        system(cmd);
        cfree(dir);
        return result;
}

/* this is flawed: the archive isn't untarred in the right directory,
   so a single Siag process can hose itself by unpacking archives on
   top of each other. Furthermore, the file is loaded under the name
   INDEX.pw, which will get stored into the buffer name. FIX IT! */
static int load(char *fn, buffer *buf)
{
        char cmd[1024];
        char *p;
        int n;

        /* old files don't have the MAGIC but we still want to load them.
           So first try flat, then tar, then flat again. */
        if (flatfile(fn) || !tryuntar(fn))
                return load_flat(fn, buf);

        p = plugin_basedir(buf, NULL);
        sprintf(cmd,
		"mkdir -p %s;"
		"cat %s|(cd %s;tar xf -)",
		p, fn, p);
        system(cmd);
        p = file_name(buf, "INDEX.pw");
        n = load_flat(p, buf);
        cfree(p);
        return n;
}
 

/* Conservative file format guessing:
   1. extension ".pw"
   2. starts with the string "# Creator: Pathetic Writer"
*/
static int myformat(char *fn)
{
        char *ext;
        FILE *fp;
        char b[100];

        ext = strrchr(fn, '.');
        if (!ext) return 0;     /* no extension */
        if (cstrcasecmp(ext, ".pw"))
                return 0;       /* wrong extension */
	if (flatfile(fn))
		return 1;	/* old format */
        if ((fp = fopen(fn, "r")) == NULL)
                return 1;       /* new file */
        if (fgets(b, sizeof b, fp) &&
			!strncmp(b, PW_MAGIC, strlen(PW_MAGIC))) {
                fclose(fp);
                return 1;
        }
        fclose(fp);
	if (tryuntar(fn)) return 1;
        return 0;
}

void fileio_pw_init()
{
	register_format(load, save, myformat, "Pathetic Writer (*.pw)");
}

