/* BeroList 2.5.9                                                    */
/* An easy-to-use mailing list server                                */
/* (c) 1996-98 by Bernhard Rosenkraenzer <bero@linux.net.eu.org>     */
/* Fixed by Jarno Paananen <jpaana@kalahari.ton.tut.fi>              */
/*          Stefano Torricella <stetor@ronchiato.it>                 */
/*          Christoph Lameter <clameter@waterf.org>                  */
/*          Andrus Kangro <andrusk@ml.ee>                            */
/* CVS: $Id: list.c,v 1.1.1.1 1998/08/21 18:11:02 root Exp $ */

#define _GNU_SOURCE

#include "list.h"

/* Set default values for variables if they aren't set in list.h */
#if !defined(USE_SENDMAIL) && !defined(SMTP_SERVER)
	#define SMTP_SERVER "localhost"
#endif
#ifdef HAS_NNTP
	#ifndef DEFAULT_NNTP_SERVER
		#define DEFAULT_NNTP_SERVER "localhost"
	#endif
	#ifndef DEFAULT_DISTRIBUTION
		#define DEFAULT_DISTRIBUTION "local"
	#endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "tool.h"
#ifdef HAVE_GETHOSTBYNAME
	#include <netdb.h>
#endif
#if defined(HAVE_GDBM) && defined(ARCHIVE)
	#include <gdbm.h>
#endif
#if defined(HAS_NNTP) || !defined(USE_SENDMAIL)
	#include "socket.h"
#endif
#ifndef HOST
	char *hostname;
	#define HOST hostname
	#define GET_HOST
#endif
#ifdef LOG
	FILE *log;
#endif
char *listaddress;
int retval=0;

/* Function prototypes... */

void send_mail(char *to, char *from, char *subject, char *message, char *footfile);
void sendtolist(char *header, char *listname, char *from, char *subject, char *message);
void subscribe(char *listname, char *user, char *operator);
void unsubscribe(char *listname, char *user, char *operator);

/* Functions... */

void send_mail(char *to, char *from, char *subject, char *message, char *footfile)
{
	char *footer;
	char *tmp=NULL;
	#ifdef USE_SENDMAIL
		FILE *mailer;
		char *mailer_cmd;
	#else
		FILE *socket;
	#endif
	#ifdef DEBUG
		puts("proc send_mail");
	#endif
	puts("Sending...");
	
	if(strchr(from,'@')==NULL) { /* Add @HOST to local address */
		tmp=from;
		from=salloc(2*slen(from)+slen(HOST)+2); /* 2 x slen(from) because of realname */
		sprintf(from,"%s <%s@%s>",realname(tmp),email(tmp),HOST);
	}
	
	#ifdef USE_SENDMAIL     /* USE_SENDMAIL: Use the 1.x.x way */
		#ifdef DEBUG
			puts("using sendmail");
			#if defined(LOG) && defined(DEBUG)
				if(log!=NULL)
					fputs("* Sending a message using sendmail\n",log);
			#endif
		#endif
		#ifdef TRUSTED
			#ifdef DEBUG
				puts("trusted");
			#endif
			from=email(from);
			mailer_cmd=salloc(slen(SENDMAIL)+slen(from)+slen(to)+7);
			sprintf(mailer_cmd,"%s -f%s '%s'",SENDMAIL,from,to);
			#ifdef LOG
				if(log!=NULL)
					fprintf(log,"* %s\n",mailer_cmd);
			#endif
			mailer=popen(mailer_cmd,"w");
		#else
			mailer_cmd=salloc(slen(SENDMAIL)+slen(to)+4);
			sprintf(mailer_cmd,"%s '%s'",SENDMAIL,to);
			#ifdef LOG
				if(log!=NULL)
					fprintf(log,"* %s\n",mailer_cmd);
			#endif
			mailer=popen(mailer_cmd,"w");
			fprintf(mailer,"From: %s\n",from);
		#endif
		if(subject==NULL && strncasecmp(message,"subject:",8)==0) {
			/* Subject: line @ beginning of msg body... */
			subject=salloc(slen(message)-6);
			strcpy(subject,message+8);
			subject=noleadingspaces(subject);
			*strchr(subject,'\n')=0;
			message=strchr(message,'\n')+1;
		}
		if(subject!=NULL)	fprintf(mailer,"Subject: %s\n",subject);
		fprintf(mailer,"\n%s\n",message);
		footer=readfile(footfile);
		if(footer!=NULL)	fprintf(mailer,"%s\n",footer);
		fprintf(mailer,"\n.\n");
		if(footer!=NULL)	free(footer);
		pclose(mailer);
	#else
		#ifdef DEBUG
			puts("using SMTP");
		#endif
		#if defined(LOG) && defined(DEBUG)
			if(log!=NULL)
				fputs("* Sending a message using SMTP\n",log);
		#endif
		socket=SockOpen(SMTP_SERVER,25);
		if(socket==NULL) {
			printf("Can't connect to server %s.\n",SMTP_SERVER);
			exit(1);
		}
		printf("Connected to %s.\n",SMTP_SERVER);
		if(Reply(socket)!=220)
			abandon(9,"Server fails to initiate contact.");

		#ifdef NT_SMTP_SERVER
			if(Send(socket,"HELO %s\r\n",HOST)!=250)
		#else
			if(Send(socket,"HELO %s\n",HOST)!=250)
		#endif
			abandon(9,"Server doesn't accept HELO.");

		#ifdef NT_SMTP_SERVER
			if(Send(socket,"MAIL FROM: <%s>\r\n",email(from))!=250)
		#else
			if(Send(socket,"MAIL FROM: <%s>\n",email(from))!=250)
		#endif
			abandon(9,"Server doesn't accept MAIL FROM.");
			
		#ifdef NT_SMTP_SERVER
			if(Send(socket,"RCPT TO: <%s>\r\n",email(to))!=250)
		#else
			if(Send(socket,"RCPT TO: <%s>\n",email(to))!=250)
		#endif
			abandon(9,"Server doesn't accept RCPT TO.");
			
		#ifdef NT_SMTP_SERVER
			if(Send(socket,"DATA\r\n")!=354)
		#else
			if(Send(socket,"DATA\n")!=354)
		#endif
			abandon(9,"Server doesn't accept DATA.");

		SockPrintf(socket,"From: %s\r\n",from);
		SockPrintf(socket,"To: %s\r\n",to);
		if(listaddress!=NULL)
			SockPrintf(socket,"Errors-To: %s\r\n",listaddress);

		if(subject==NULL && strncasecmp(message,"subject:",8)==0) {
			/* Subject: line @ beginning of msg body... */
			subject=salloc(slen(message)-6);
			strcpy(subject,message+8);
			subject=noleadingspaces(subject);
			*strchr(subject,'\n')=0;
			message=strchr(message,'\n')+1;
		}
		if(subject!=NULL)	SockPrintf(socket,"Subject: %s\r\n",subject);
		SockPrintf(socket,"\r\n");
		SockPuts(socket,message);
		footer=readfile(footfile);
		if(footer!=NULL)	{
			SockPuts(socket,footer);
			free(footer);
		}

		#ifdef NT_SMTP_SERVER
			if(Send(socket,"\r\n.\r\n")!=250)
		#else
			if(Send(socket,"\n.\n")!=250)
		#endif
			abandon(9,"Message rejected by server.");

		#ifdef NT_SMTP_SERVER
			if(Send(socket,"QUIT\r\n")!=221) {
		#else
			if(Send(socket,"QUIT\n")!=221) {
		#endif
			puts("SMTP Server doesn't accept QUIT, probably non-fatal error.");
			puts("Your message has probably arrived; notify list administrator.");
			retval=10;
		}
		fclose(socket);
	#endif
	if(tmp!=NULL) { /* Clean up address if we had to modify it */
		free(from);
		from=tmp;
	}
}

void sendtolist(char *header, char *listname, char *from, char *subject, char *message)
{
	char *listfile, *footfile, *conffile, *member;
	char *oldmsg, *limits, *limiterr=NULL;
	char *config, *sender, *replyto;
	char *prefix, *real_subject;
	unsigned long limit;
	#ifdef KILLFILE
		FILE *killfile;
		char *killmember,*temp;
		char is_killed=0;
	#endif
	FILE *list;
	#if !defined(USE_SENDMAIL) || defined(HAS_NNTP)
		FILE *socket;
		char *footer;
	#endif
	#ifdef HAS_NNTP
		char *nntp_server;
		char *newsgroup;
		char *distribution;
		int post_nntp=1;
	#endif
	#if defined(HAVE_GDBM) && defined(ARCHIVE)
		char *archivefile;
		int use_archive=1;
		GDBM_FILE db;
		datum key,content;
		char *d_key,*d_content;
		unsigned long num;
		time_t utc;
		struct tm *servertime;
		static char *weekday[7]={ "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
		static char *month[12]={ "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Nov", "Dec" };
		char *msgdat;
	#endif
	#ifndef USE_SENDMAIL
		char *errors;
	#endif

	puts("Sending to list...");

	#ifdef KILLFILE
		killfile=fopen(KILLFILE,"r");
		if(killfile!=NULL) {
			while(!feof(killfile)) {
				killmember=readline(killfile);
				if((slen(killmember)>2) && (killmember[0]!='#')) {
					if(killmember[0]=='@') {
						temp=email(from);
						if(strcasestr(temp,killmember)==temp+slen(temp)-slen(killmember)) is_killed=1;
					} else if(killmember[slen(killmember)-1]=='@') {
						temp=email(from);
						if(strcasestr(temp,killmember)==temp) is_killed=1;
					} else
						if(strcasestr(killmember,email(from))!=NULL) is_killed=1;
				}
				free(killmember);
			}
		}
		if(is_killed==0) {
	#endif

	listfile=salloc(slen(LISTDIR)+slen(listname)+10);
	sprintf(listfile,"%s/%s.members",LISTDIR,listname);
	footfile=salloc(slen(LISTDIR)+slen(listname)+9);
	sprintf(footfile,"%s/%s.footer",LISTDIR,listname);
	conffile=salloc(slen(LISTDIR)+slen(listname)+9);
	sprintf(conffile,"%s/%s.config",LISTDIR,listname);
	config=readfile(conffile);

	#if defined(HAVE_GDBM) && defined(ARCHIVE)
		archivefile=extract(config,"archive=",'\n');
		if(archivefile==NULL) {
			archivefile=salloc(slen(LISTDIR)+slen(listname)+10);
			sprintf(archivefile,"%s/%s.archive",LISTDIR,listname);
		}
		if(scasecmp(archivefile,"none")==0)
			use_archive=0;
		if(use_archive!=0 && archivefile[0]!='/') {
			archivefile=(char *) realloc(archivefile,slen(archivefile)+slen(LISTDIR)+2);
			sprintf(archivefile,"%s/%s\0",LISTDIR,archivefile);
		}
	#endif

	sender=extract(config,"sender=",'\n');
	if(scasecmp(sender,"original")==0 || scasecmp(sender,"sender")==0 || sender==NULL)
		sender=from;
	if(scasecmp(sender,"list")==0)
		sender=listaddress;

	replyto=extract(config,"replyto=",'\n');
	if(scasecmp(replyto,"original")==0 || scasecmp(replyto,"sender")==0)
		replyto=from;
	if(scasecmp(replyto,"list")==0)
		replyto=listaddress;

	limits=extract(config,"limit=",'\n');
	if(limits!=NULL) {
		limit=atol(limits);
		if(limit!=0 && slen(message)>limit) {
			if(strchr(limits,',')!=NULL)
				limiterr=readfile(strchr(limits,',')+1);
			if(limiterr==NULL) {
				limiterr=salloc(100);
				sprintf(limiterr, "Sorry, you are not allowed to post messages longer than %u bytes\nto this list.",limit);
			}
			puts(limiterr);
			send_mail(from,"MAILER-DAEMON","Size limit exceeded",limiterr,NULL);
			free(limiterr);
			return;
		}
	}

	prefix=extract(config,"prefix=",'\n');
	if(prefix!=NULL && subject!=NULL) {
		real_subject=salloc(slen(prefix)+3);
		sprintf(real_subject,"[%s]",prefix);
		if(strstr(subject,real_subject)==NULL) {
			free(real_subject);
			real_subject=salloc(slen(subject)+slen(prefix)+4);
			sprintf(real_subject,"[%s] %s\0",prefix,subject);
		} else {
			free(real_subject);
			real_subject=strdup(subject);
		}
	} else if(prefix!=NULL) {
		real_subject=salloc(slen(prefix)+16);
		sprintf(real_subject,"[%s] (no subject)\0",prefix);
	} else if(subject!=NULL) {
		real_subject=strdup(subject);
	} else {
		real_subject=NULL;
	}

	#ifndef NO_SENDER_ADDRESS
		if(scasecmp(email(sender),email(from))) { /* sender != original sender */
			oldmsg=message;
			message=salloc(slen(from)+slen(oldmsg)+15);
			sprintf(message,"* From: %s\n\n%s",from,oldmsg);
			free(oldmsg);
		}
	#endif
	#ifndef NO_LIST_ADDRESS
		if(scasecmp(listaddress,email(sender))) { /* sender != list */
			oldmsg=message;
			message=salloc(slen(listaddress)+slen(oldmsg)+15);
			sprintf(message,"* List: %s\n\n%s",listaddress,oldmsg);
			free(oldmsg);
		}
	#endif

	list=fopen(listfile,"r");

	#ifdef USE_SENDMAIL
		#if defined(LOG) && defined(DEBUG)
			if(log!=NULL)
				fputs("* Sending to list using sendmail\n",log);
		#endif
		while(!feof(list)) {
			member=readline(list);
			if((slen(member)>2) && (member[0]!='#'))
				send_mail(member,sender,real_subject,message,footfile);
			free(member);
		}
	#else
		#if defined(LOG) && defined(DEBUG)
			if(log!=NULL)
				fputs("* Sending to list using SMTP\n",log);
		#endif
		socket=SockOpen(SMTP_SERVER,25);
		if(socket==NULL) {
			printf("Can't connect to server %s.\n",SMTP_SERVER);
			exit(1);
		}
		printf("Connected to %s.\n",SMTP_SERVER);
		if(Reply(socket)!=220)
			abandon(7,"Server fails to initiate contact.");
		
		#ifdef NT_SMTP_SERVER
			if(Send(socket,"HELO %s\r\n",HOST)!=250)
		#else
			if(Send(socket,"HELO %s\n",HOST)!=250)
		#endif
			abandon(7,"Server doesn't accept HELO.");
			
		#ifdef NT_SMTP_SERVER
			if(Send(socket,"MAIL FROM: <%s>\r\n",email(sender))!=250)
		#else
			if(Send(socket,"MAIL FROM: <%s>\n",email(sender))!=250)
		#endif
			abandon(7,"Server doesn't accept MAIL FROM.");
			
		while(!feof(list)) {
			member=readline(list);
			if((slen(member)>2) && (member[0]!='#')) {
				#ifdef NT_SMTP_SERVER
					switch(Send(socket,"RCPT TO: <%s>\r\n",email(member))) {
				#else
					switch(Send(socket,"RCPT TO: <%s>\n",email(member))) {
				#endif
				case 250:
					/* Everything's OK */
					break;
				case 550:
					/* User Unknown */
					printf("User %s unknown to SMTP server\n",member);
					errors=extract(config,"errors=",'\n');
					if(scasecmp(errors,"forward")!=0) {
						#ifdef LOG
							if(log!=NULL)
								fprintf(log,"Automatically unsubscribing %s (Unknown to SMTP Server)\n",member);
						#endif
						unsubscribe(listname,member,NULL);
					}
					break;
				default:
					printf("User %s causes unknown SMTP server response.\n",member);
				}
			}
		}
		#ifdef NT_SMTP_SERVER
			if(Send(socket,"DATA\r\n")!=354)
		#else
			if(Send(socket,"DATA\n")!=354)
		#endif
			abandon(7,"Server doesn't accept DATA.");
			
		SockPrintf(socket,"From: %s\r\n",sender);
		if(replyto!=NULL) SockPrintf(socket,"Reply-To: %s\r\n",replyto);
		if(listaddress!=NULL)
			SockPrintf(socket,"Errors-To: %s\r\n",listaddress);
		SockPrintf(socket,"To: %s\r\n",listaddress);
		SockPrintf(socket,"Message-ID: <%X.%s@%s>\r\n",(unsigned long) time(NULL),VERSION,HOST);
		if(header!=NULL)	SockPrintf(socket,"%s",header);
		if(real_subject!=NULL)	SockPrintf(socket,"Subject: %s\r\n",real_subject);
		             else	SockPuts(socket,"Subject: (no subject)\r\n");
		SockPrintf(socket,"\r\n");
		SockPuts(socket,message);
		footer=readfile(footfile);
		if(footer!=NULL)	SockPrintf(socket,"%s\r\n",footer);

		#ifdef NT_SMTP_SERVER
			if(Send(socket,".\r\n")!=250)
		#else
			if(Send(socket,".\n")!=250)
		#endif
			abandon(7,"Message rejected by server.");
			
		#ifdef NT_SMTP_SERVER
			if(Send(socket,"QUIT\r\n")!=221) {
		#else
			if(Send(socket,"QUIT\n")!=221) {
		#endif
			puts("SMTP Server doesn't accept QUIT, probably non-fatal error.");
			puts("Your message has probably arrived; notify list administrator.");
			retval=8;
		}
		fclose(socket);
		if(footer!=NULL)	free(footer);
	#endif /* !USE_SMTP */
	#ifdef HAS_NNTP
		newsgroup=extract(config,"newsgroup=",'\n');
		if(newsgroup==NULL)
		post_nntp=0;
		if(post_nntp!=0) {
			#ifdef LOG
				if(log!=NULL)
					fprintf(log,"Posting to newsgroup %s.\n",newsgroup);
			#endif
			nntp_server=extract(config,"nntp-server=",'\n');
			if(nntp_server==NULL) {
				nntp_server=salloc(slen(DEFAULT_NNTP_SERVER)+1);
				strcpy(nntp_server,DEFAULT_NNTP_SERVER);
			}
			distribution=extract(config,"distribution=",'\n');
			if(distribution==NULL) {
				distribution=salloc(slen(DEFAULT_DISTRIBUTION)+1);
				strcpy(distribution,DEFAULT_DISTRIBUTION);
			}
			socket=SockOpen(nntp_server,119);
			if(socket==NULL) {
				printf("Can't connect to server %s.\n",nntp_server);
				exit(1);
			}
			printf("Connected to %s.\n",nntp_server);
			if(Reply(socket)!=200)
				abandon(11,"NNTP server fails to initiate contact.");
			if(Send(socket,"MODE READER\r\n")!=200)
				abandon(11,"NNTP Server doesn't accept MODE.");
			if(Send(socket,"POST\r\n")!=340)
				abandon(11,"NNTP Server doesn't accept POST.");
			SockPrintf(socket,"From: %s\n",sender);
			SockPrintf(socket,"Newsgroups: %s\n",newsgroup);
			SockPrintf(socket,"Distribution: %s\n",distribution);
			if(real_subject==NULL && strncasecmp(message,"subject:",8)==0) {
				/* Subject: line @ beginning of msg body... */
				real_subject=salloc(slen(message)+slen(prefix)-2);
				strcpy(real_subject,message+8);
				while(strchr(real_subject,' ')==real_subject || strchr(real_subject,'\t')==real_subject) real_subject++;
				*strchr(real_subject,'\n')=0;
				message=strchr(message,'\n')+1;
				sprintf(real_subject,"[%s] %s\0",prefix,real_subject);
			}
			if(real_subject!=NULL)
				SockPrintf(socket,"Subject: %s\n",real_subject);
			else
				SockPuts(socket,"Subject: (no subject)\n");

			/* The following line is required for MIME, but might crash the news server... */
			/* if(header!=NULL) SockPrintf(socket,"%s",header); */
			SockPrintf(socket,"\n");
			SockPuts(socket,message);
			footer=readfile(footfile);
			if(footer!=NULL)	SockPrintf(socket,"%s\r\n",footer);
			if(Send(socket,".\r\n")!=240)
				abandon(11,"Message rejected by NNTP server.");
			if(Send(socket,"QUIT\r\n")!=205)
				abandon(12,"NNTP Server doesn't accept QUIT, probably non-fatal error.");
			fclose(socket);
			if(footer!=NULL) free(footer);
		}
	#endif
	#if defined(HAVE_GDBM) && defined(ARCHIVE)
		if(use_archive!=0) {
			#if defined(LOG) && defined(DEBUG)
				if(log!=NULL)
					fputs("* Storing to archive\n",log);
			#endif

			/* Store message to archive */
			/* Check for existing messages */
			db=gdbm_open(archivefile,0,GDBM_READER,00664,0);
			if(db==NULL)
				num=0;
			else {
				d_key=salloc(4);
				strcpy(d_key,"num");
				key.dptr=d_key;
				key.dsize=3;
				content=gdbm_fetch(db,key);
				free(d_key);
				num=atoi(content.dptr);
				gdbm_close(db);
			}
			db=gdbm_open(archivefile,0,GDBM_WRCREAT,00664,NULL);
			if(db==NULL) {
				puts("Error: Couldn't write to message archive.");
				puts("Your message has been sent to the list, anyway.");
				abandon(12,"Please contact the list operator.");
			}
			d_key=salloc(12);
			sprintf(d_key,"s%u",(unsigned short) num);
			key.dptr=d_key;
			key.dsize=slen(d_key);
			content.dptr=subject;
			content.dsize=slen(subject);
			gdbm_store(db,key,content,GDBM_INSERT);
			sprintf(d_key,"d%u",(unsigned short) num);
			key.dptr=d_key;
			key.dsize=slen(d_key);
			utc=time(NULL);
			servertime=localtime(&utc);
			msgdat=salloc(26);
			sprintf(msgdat,"%s, %u %s %u %02u:%02u:%02u",weekday[servertime->tm_wday],servertime->tm_mday,month[servertime->tm_mon-1],servertime->tm_year,servertime->tm_hour,servertime->tm_min,servertime->tm_sec);
			content.dptr=msgdat;
			content.dsize=slen(msgdat);
			gdbm_store(db,key,content,GDBM_INSERT);
			free(msgdat);
			sprintf(d_key,"f%u",(unsigned short) num);
			key.dptr=d_key;
			key.dsize=slen(d_key);
			content.dptr=from;
			content.dsize=slen(from);
			gdbm_store(db,key,content,GDBM_INSERT);
			free(d_key);
			d_key=salloc(12);
			sprintf(d_key,"%u",(unsigned short) num);
			key.dptr=d_key;
			key.dsize=slen(d_key);
			content.dptr=message;
			content.dsize=slen(message);
			gdbm_store(db,key,content,GDBM_INSERT);
			free(d_key);
			d_key=salloc(4);
			strcpy(d_key,"num");
			key.dptr=d_key;
			key.dsize=3;
			d_content=salloc(12);
			sprintf(d_content,"%u",(unsigned short) ++num);
			content.dptr=d_content;
			content.dsize=slen(d_content);
			gdbm_store(db,key,content,GDBM_REPLACE);
			free(d_key);
			gdbm_close(db);
		}
	#endif
	#if defined(LOG) && defined(DEBUG)
		if(log!=NULL)
			fputs("* done.\n",log);
	#endif

	fclose(list);
	free(config); free(listfile); free(footfile); free(conffile);
	free(real_subject);
	#ifdef KILLFILE
		}
		free(killfile);
	#endif
}

void subscribe(char *listname, char *user, char *operator)
{
	FILE *list;
	char *mail,*from;
	char *listfile,*footfile,*welcfile;
	char *members;

	#ifdef LOG
		if(log!=NULL) {
			if(operator==NULL)
				fprintf(log,"* subscribing %s\n",user);
			else
				fprintf(log,"* %s subscribed by %s\n",user,operator);
		}
	#endif

	listfile=salloc(slen(LISTDIR)+slen(listname)+10);
	sprintf(listfile,"%s/%s.members",LISTDIR,listname);
	welcfile=salloc(slen(LISTDIR)+slen(listname)+10);
	sprintf(welcfile,"%s/%s.welcome",LISTDIR,listname);
	footfile=salloc(slen(LISTDIR)+slen(listname)+9);
	sprintf(footfile,"%s/%s.footer",LISTDIR,listname);

	members=readfile(listfile);
	if(members==NULL) {
		members=salloc(sizeof(char));
		members[0]='\0';
	}

	if(strcasestr(members,email(user))!=NULL) { /* User is already subscribed */
		mail=salloc(2048);
		if(operator==NULL) { /* User subscribed himheritself */
			sprintf(mail,"You are already subscribed to this list.\n");
			send_mail(user,listname,"List subscription...",mail,footfile);
		} else {
			sprintf(mail,"You tried to subscibe a user (%s) who is already subscribed to the list.\n",user);
			send_mail(operator,listname,"List subscription...",mail,footfile);
		}
		free(mail);
	} else {
		list=fopen(listfile,"a");
		if(list==NULL)
			abandon(13,"Can't write to members file - contact list operator.");
		fprintf(list,"%s\n",user);
		fclose(list);
		mail=readfile(welcfile);
		if(mail!=NULL) {
			#ifdef LOG
				if(log!=NULL)
					fprintf(log,"* sending welcome message\n");
			#endif
			if(operator!=NULL)
				send_mail(user,operator,NULL,mail,footfile);
			else {
				from=salloc(1024);
				sprintf(from,"%s@%s",listname,HOST);
				send_mail(user,from,NULL,mail,footfile);
				free(from);
			}
			free(mail);
		}
	}
	free(footfile);  free(welcfile);  free(listfile);
}

void unsubscribe(char *listname, char *user, char *operator)
{
	char *listfile;
	FILE *list;
	register int members,i;
	char *member[MAX_MEMBERS];

	listfile=salloc(slen(LISTDIR)+slen(listname)+10);
	sprintf(listfile,"%s/%s.members",LISTDIR,listname);

	#ifdef LOG
		if(log!=NULL) {
			if(operator==NULL)
				fprintf(log,"* %s unsubscribed\n",user);
			else
				fprintf(log,"* %s unsubscribed by %s\n",user,operator);
		}
	#endif

	list=fopen(listfile,"r");
	members=0;
	while(!feof(list)) {
		member[members]=readline(list);
		if(scasecmp(email(member[members]),email(user))!=0) members++;
	}
	members--;
	fclose(list);
	list=fopen(listfile,"w");
	for(i=0;i<=members;i++)
		if(slen(member[i])>2) fprintf(list,"%s\n",member[i]);
	fclose(list);
	free(listfile);
}

int main(int argc, char *argv[])
{
	int pos,flen,wsize,i,control;
	char *mail,*header,*msg,*to,*from,*errors,*subject,*recipient;
	char *conffile,*listfile,*opfile,*footfile;
	char *header_copy,*additional_header,*header_line,*save_header;
	char *config;
	char *a;
	unsigned char temp;
	char *addr_tmp,*conf_errors;
	#ifdef LOG
		char *logfile;
		time_t utc;
		struct tm *servertime;
		static char *weekday[7]={"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
	#endif

	#ifdef GET_HOST
		#ifdef HAVE_GETHOSTBYNAME
			struct hostent *x;
			hostname=salloc(1024);
			gethostname(hostname,1024);
			x=gethostbyname(hostname);
			if (x) strcpy(hostname,x->h_name); else strcpy(hostname,"localhost");
		#else   /* Try to determine hostname anyway */
			hostname=salloc(1024);
			gethostname(hostname,1024);
			hostname[slen(hostname)+1]='\0';
			hostname[slen(hostname)]='.';
			getdomainname(hostname+slen(hostname),1024-slen(hostname)-1);
		#endif
	#endif
	#ifdef USE_SENDMAIL
		signal(SIGPIPE,SIG_IGN);
	#endif

	printf("%s ready...\n\n",VERSION);
	chdir(LISTDIR);

	to=NULL; control=0; /* Default settings... Process argv */
	if(argc>1) {
		for(i=1;i<argc;i++) {
			#ifdef DEBUGONLY
				printf("i=%u, argc=%u, argv[i]=%s\n",i,argc,argv[i]);
			#endif
			if(argv[i][0]=='-') {
				if(scasecmp(argv[i],"-control")==0 || scasecmp(argv[i],"--control")==0) {
					if(control==0)
						control=1;
					else if(control==1)
						puts("Warning: -control specified twice.");
					else {
						puts("Warning: contradictory arguments -control and -nocontrol.");
						control=0;
					}
				} else if(scasecmp(argv[i],"-nocontrol")==0 || scasecmp(argv[i],"--nocontrol")==0) {
					if(control==0)
						control=2;
					else if(control==2)
						puts("Warning: -nocontrol specified twice.");
					else {
						puts("Warning: contradictory arguments -control and -nocontrol.");
						control=0;
					}
				} else
					printf("Warning: unknown option %s\n",argv[i]);
			} else {
				if(to==NULL) {
					to=salloc(slen(argv[i])+1);
					strcpy(to,argv[i]);
				} else {
					puts("Warning: too many arguments for list program.");
					puts("Usage: list [-control|-nocontrol] listname");
				}
			}		
		}
	}

	#ifdef DEBUG
		puts("reading message...");
	#endif

	flen=0; wsize=1024;
	mail=salloc(sizeof(char)*(wsize+2));
	while(!feof(stdin)) {
		mail[flen]=(char) fgetc(stdin);
		if(flen==wsize) {
			mail[flen+1]='\0'; wsize+=1024;
			mail=(char *) realloc(mail,sizeof(char)*(wsize+2));
		}
		flen++;
	}
	flen--;
	mail[flen]='\0';

	#ifdef DEBUG
		puts("Extracting header...");
	#endif

	header=salloc(slen(mail) + 1);
	strcpy(header,mail);

	pos=strstr(header,"\n\n")-header;
	header[pos]='\0';

	msg=salloc(slen(mail+pos+2)+1);
	strcpy(msg,mail+pos+2);

	/* Get miscellaneous information from the message header... */
	if(to==NULL) {
		/* If called without parameters, try to determine our address. */
		/* Warn about it... */
		puts("WARNING: listname not specified in arguments.");
		puts("Usage: list [-control|-nocontrol] listname");
		to=downcase(email(extract(header,"To: ",'\n')));
		if(strchr(to,'@')!=NULL) *strchr(to,'@')=0;
		if(strncasecmp(to+slen(to)-8,"-request",8)==0)
			to[slen(to)-8]='\0';
	}

	from=extract(header,"From: ",'\n');
	if(from==NULL)
		abandon(200,"ERROR: No From: line."); 

	subject=extract(header,"Subject: ",'\n');
	if(subject==NULL) {
		subject=salloc(13);
		strcpy(subject,"(no subject)\0");
	}

	if(to[0]=='"') to++;

	/* Get all other information from the message header...
	   Permit for MIME, etc. */

	#ifdef DEBUG
		puts("Extracting information...");
	#endif

	header_copy=salloc(slen(header)+2);
	strcpy(header_copy,header);
	header_copy[slen(header)]='\n';
	header_copy[slen(header)+1]='\0';
	save_header=header_copy;
	additional_header=salloc(slen(header)+2);
	header_line=salloc(slen(header)+2);
	while(strchr(header_copy,'\n')!=NULL) {
		strcpy(header_line,header_copy);

		/* Handle multi-line headers (Received, etc.) */
		while(*(strchr(header_copy,'\n')+1)==9) { /* Multi Line */
			strcpy(strchr(header_copy,'\n'),strchr(header_copy,'\n')+2);
		}

		header_copy=strchr(header_copy,'\n')+1;
		*(strchr(header_line,'\n')+1)=0;
		while(strchr(header_line,'\t')!=NULL)
			*strchr(header_line,'\t')=' ';
		if(strncasecmp(header_line,"From: ",6)!=0 &&
		   strncasecmp(header_line,"To: ",4)!=0 &&
		   strncasecmp(header_line,"Subject: ",9)!=0 &&
		   strncasecmp(header_line,"From ",5)!=0 &&
		   strncasecmp(header_line,"Return-Path",11)!=0 &&
		   strncasecmp(header_line,"Received ",9)!=0 &&
		   strncasecmp(header_line,"Received: ",10)!=0 &&
		   strncasecmp(header_line,"Message-Id",10)!=0 &&
		   strncasecmp(header_line,"Reply-To: ",10)!=0 &&
		   strncasecmp(header_line,"Errors-To: ",11)!=0) {
			strcpy(additional_header+slen(additional_header),header_line);
		}
	}
	free(header_line);
	free(save_header);

	listaddress=salloc(slen(to)+slen(HOST)+2);
	sprintf(listaddress,"%s@%s",to,HOST);

	/* Find filenames for list... */

	listfile=salloc(slen(LISTDIR)+slen(to)+10);
	sprintf(listfile,"%s/%s.members",LISTDIR,to);
	opfile=salloc(slen(LISTDIR)+slen(to)+12);
	sprintf(opfile,"%s/%s.operators",LISTDIR,to);
	conffile=salloc(slen(LISTDIR)+slen(to)+9);
	sprintf(conffile,"%s/%s.config",LISTDIR,to);
	footfile=salloc(slen(LISTDIR)+slen(to)+9);
	sprintf(footfile,"%s/%s.footer",LISTDIR,to);

	/* read config file... */

	config=readfile(conffile);

	#ifdef LOG
		a=extract(config,"log=",'\n');
		if(a!=NULL && strcasecmp(a,"no")!=0) {
			if(strchr(a,'/')==NULL) {
				logfile=salloc(slen(LOGDIR)+slen(a)+2);
				sprintf(logfile, "%s/%s", LOGDIR, a);
			} else {
				logfile=salloc(slen(a)+1);
				strcpy(logfile,a);
			}
			log=fopen(logfile,"a");
			if(log==NULL) {
				printf("can't access log file %s :(\nPlease contact the list operator.\n",logfile);
				printf("Make sure user %d (group %d) has read/write permissions\n",geteuid(),getegid());
				puts("to the logfile (or the log directory, if logfile is to be created).\n");
				retval=98;
			}
			utc=time(NULL);
			servertime=localtime(&utc);
			fprintf(log,"=========================%20s=========================\n",to);
			fprintf(log,"%s %2u/%2u/%2u, %2u:%2u:%2u: Message '%s'\nfrom %s...\n",weekday[servertime->tm_wday],servertime->tm_year,servertime->tm_mon+1,servertime->tm_mday,servertime->tm_hour,servertime->tm_min,servertime->tm_sec,subject,from);
		} else
			log=NULL;
	#endif


	if(control!=2) { /* Process control messages */
		/* Permit "subscribe someone@somewhere" for the user - by converting it
		   to a reasonable form ("subscribe"). */
		if(strncasecmp(noleadingspaces(subject),"subscribe ",10)==0) {
			recipient=subject+10;
			if(scasecmp(email(from),email(recipient))==0)
				subject[9]='\0';
		}
		if(strncasecmp(noleadingspaces(subject),"unsubscribe ",12)==0) {
			recipient=subject+12;
			if(scasecmp(email(from),email(recipient))==0)
			subject[11]='\0';
		}

		if(strncasecmp(noleadingspaces(subject),"subscribe ",10)==0) {
			recipient=subject+10;
			a=readfile(opfile);
			if((char *) strcasestr(a,email(from))==NULL) { /* NO OPERATOR */
				#ifdef LOG
					if(log!=NULL)
						fprintf(log,"* attempt to subscribe %s by %s\n",recipient,from);
				#endif
				free(mail);
				mail=salloc(1024);
				sprintf(mail,"You must be list operator in order to add new users to the list %s@%s.\n",to,HOST);
				send_mail(from,listaddress,"List subscription",mail,footfile);
			} else
				subscribe(to,recipient,from);
			control=1;
		} else if(strncasecmp(noleadingspaces(subject),"unsubscribe ",12)==0) {
			recipient=subject+12;
			a=readfile(opfile);
			if((char *) strcasestr(a,email(from))==NULL) { /* NO OPERATOR */
				#ifdef LOG
					if(log!=NULL)
						fprintf(log,"* attempt to unsubscribe %s by %s\n",recipient,from);
				#endif
				free(mail);
				mail=salloc(1024);
				sprintf(mail,"You must be list operator in order to unsubscribe users.\n");
				send_mail(from,listaddress,"List unsubscription",mail,footfile);
			} else {
				unsubscribe(to,recipient,from);
			}
			control=1;
		} else if(scasecmp(noleadingspaces(subject),"subscribe")==0) {
			a=extract(config,"newusers=",'\n');
			if(scasecmp(a,"no")==0 && a!=NULL) {  /* Do not accept new users */
				a=extract(config,"contact=",'\n');
				if(a==NULL) {
					a=salloc(6+slen(HOST));
					sprintf(a,"root@%s",HOST);
				}
				#ifdef LOG
					if(log!=NULL)
						fprintf(log,"* %s tried to subscribe\n",from);
				#endif
				free(mail);
				mail=salloc(1024);
				sprintf(mail,"%s is a closed mailing list. If you wish to subscribe, contact\nthe list operator at %s.\n",to,a);
				send_mail(from,listaddress,"List subscribtion",mail,footfile);
			} else {
				subscribe(to,from,NULL);
			}
			control=1;
		} else if(scasecmp(noleadingspaces(subject),"unsubscribe")==0) {
			unsubscribe(to,from,NULL);
			control=1;
		} else if(control==1) {
			/* Mode is -control, but command couldn't be processed... */
			free(mail);
			mail=salloc(2048);
			sprintf(mail,"Your command, \"%s\" is not understood by BeroList.\nAll commands have to be given in the subject line of your message.\n\nIf your message was intended to be sent to the list, you have used a wrong address.\n",subject);
			send_mail(from,listaddress,"BeroList: Unknown command",mail,footfile);
		}
	}
	if(control!=1) {  /* Mail for list */
		errors=extract(config,"errors-to=",'\n');
		if(errors==NULL) {
			errors=extract(config,"contact=",'\n');
			if(errors==NULL) {
				errors=salloc(6+slen(HOST));
				sprintf(errors,"root@%s",HOST);
			}
		}
		#ifndef NO_POSTMASTER
			if(scasecmp(username(from),"mailer-daemon")==0) {
		#else
			if(scasecmp(username(from),"mailer-daemon")==0 || scasecmp(username(from),"postmaster")==0) {
		#endif
			#ifdef DEBUG
				puts("Error message...");
			#endif
			#ifdef LOG
				if(log!=NULL)
					fprintf(log,"* Error message - forwarding to %s.\n",errors);
			#endif
			conf_errors=extract(config,"errors=",'\n');
			if(scasecmp(conf_errors,"forward")!=0) {
				/* Try to determine and eliminate
				   source of error. */
				retval=254;
				a=strstr(msg,"----- The following addresses had permanent fatal errors -----");
				if(a!=NULL) {
					do {
						a=strchr(a,'\n')+1;
						if(a!=NULL) {
							addr_tmp=salloc(slen(a)+1);
							strcpy(addr_tmp,a);
							*strchr(addr_tmp,'\n')=0;
							if(addr_tmp[0]=='\r') addr_tmp++;
							temp=addr_tmp[0];
							if(temp!='\n' && temp!=0) {
								if(temp==' ') {
									/* Expanded from?? */
									if(strstr(addr_tmp,"(expanded from:")) {
										addr_tmp=strstr(addr_tmp,"(expanded from:")+15;
										while(addr_tmp[0]==' ') addr_tmp++;
										if(strchr(addr_tmp,')')!=NULL) { /* End of expanded from: */
											*strchr(addr_tmp,')')=0;
											#ifdef LOG
												fprintf(log,"Automatically unsubscribing %s (MAILER-DAEMON)\n",addr_tmp);
											#endif
											unsubscribe(to,addr_tmp,NULL);
										}
									}
								} else {
									#ifdef LOG
										fprintf(log,"Automatically unsubscribing %s\n",addr_tmp);
									#endif
									unsubscribe(to,addr_tmp,NULL);
								}
							}
							/* free(addr_tmp); */
						}
					} while(a!=NULL && temp!='\n' && temp!=0);
				}
			}
			if(scasecmp(conf_errors,"forward")==0 || scasecmp(conf_errors,"both")==0) {
			a=salloc(slen(VERSION)+slen(msg)+slen(to)+31);
			sprintf(a,"%s\nError from mailing list %s:\n\n%s\n",VERSION,to,msg);
			free(mail);
			mail=a;
			send_mail(errors,from,subject,mail,NULL);
			} /* if errors=forward || errors=both */
		} else {
			a=extract(config,"closed=",'\n');
			if(scasecmp(a,"yes")==0) {
				a=readfile(listfile);
			if((char *) strcasestr(a,email(from))==NULL) { /* NOT member of list */
				#ifdef LOG
					if(log!=NULL)
						fprintf(log,"* Message rejected - %s not member of list.\n",from);
				#endif
				a=extract(config,"newusers=",'\n');
				free(mail); mail=salloc(2048);
				if(scasecmp(a,"no")==0) {
					a=extract(config,"contact=",'\n');
					if(a==NULL) {
						a=salloc(6+slen(HOST));
						sprintf(a,"root@%s",HOST);
					}
					sprintf(mail,"You may send mail to %s@%s only if you are\nsubscribed to the list.\nContact the list operator, %s.\n",to,HOST,a);
					send_mail(from,listaddress,"Sending mail to this list",mail,footfile);
				} else {
					sprintf(mail,"You may send mail to %s@%s only if you are\nsubscribed to the list.\nTo subscribe, send a message to %s@%s\nwith the Subject set to 'subscribe'.\n",to,HOST,to,HOST);
					send_mail(from,listaddress,"Sending mail to this list",mail,footfile);
				}
			} else
				sendtolist(additional_header,to,from,subject,msg);
			} else if(scasecmp(a,"operators")==0) {
				a=readfile(opfile);
				if((char *) strcasestr(a,email(from))==NULL) { /* NOT list operator */
					#ifdef LOG
						if(log!=NULL)
							fprintf(log,"* Message rejected - %s not list operator.\n",from);
					#endif
					free(mail); mail=salloc(2048);
					sprintf(mail,"This is an announcements list.\nOnly list operators can send messages to it.\n");
					send_mail(from,listaddress,"This mailing list",mail,footfile);
				} else
					sendtolist(additional_header,to,from,subject,msg);
			} else
				sendtolist(additional_header,to,from,subject,msg);
		}
	}

	#ifdef LOG
		if(log!=NULL) {
			fclose(log);
			free(logfile);
		}
	#endif

	free(mail); free(header);
	free(to); free(from); free(subject);
	free(listfile); free(opfile);
	free(config); free(listaddress);
	if(additional_header!=NULL) free(additional_header);
	#ifdef GET_HOST
		free(hostname);
	#endif
	#ifdef DEBUGONLY
		abandon(99,"Error message sent for debugging purposes only.");
	#endif
	return retval;
}
