/* 
 *  bezier curve drawing
 *
 *  Copyright (C) 1998 Thomas Tanner. See CREDITS for details.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <math.h>
#include "lib.h"
#include "util.h"
#include "fpoint.h"

/*--------------------------------------------------------------------------*/

/*

a bezier curve between 4 poggi_ss is defined by 

x(t) = c*t^3 + b*t^2 + a*t + x1
y(t) = f*t^3 + e*t^2 + d*t + y1

with

x2 = x1 + a / 3
y2 = y1 + d / 3
x3 = (a + b) / 3
y3 = (d + e) / 3
x4 = x1 + a + b + c
y4 = y1 + d + e + f

*/

static void	bezier(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2, 
			ggi_sint x3, ggi_sint y3, ggi_sint x4, ggi_sint y4)
{
	ggi_sint x, y, x0, y0;
	fpoint	a, b, c, d, e, f;
	fpoint	t, tinc;
	void 	(*line)(ggi_visual_t, ggi_sint, ggi_sint, ggi_sint, ggi_sint) 
		= GGI2D_FUNC(vis)->line;

	a = FPNewInt(3*(x2-x1));	d = FPNewInt(3*(y2-y1));
	b = FPNewInt(3*(x3-x2))-a;	e = FPNewInt(3*(y3-y2))-d;
	c = FPNewInt(x4-x1)-b-a;	f = FPNewInt(y4-y1)-e-d;
	x = abs(FPInt(b+3*c));	y = abs(FPInt(e+3*f));
	if (x < abs(FPInt(b))) x = abs(FPInt(b));
	if (y < abs(FPInt(e))) y = abs(FPInt(e));
	t = (x < y) ? FPNewFloat(sqrt(y)) : FPNewFloat(sqrt(x));
	if (!t) t = FP_ONE;
	tinc = FPDiv(FP_ONE,t);
	t = FP_ZERO; x0 = x1; y0 = y1;
	while (t <= 1) {
		x = FPRound( FPMul( FPMul( FPMul(c,t)+b, t)+a, t)) + x1;
		y = FPRound( FPMul( FPMul( FPMul(f,t)+e, t)+d, t)) + y1;
		line(vis, x0, y0, x, y);
		x0 = x; y0 = y; t += tinc;
	}
}

static void	bezier_st(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2, 
		ggi_sint x3, ggi_sint y3, ggi_sint x4, ggi_sint y4)
{
	int	append = GGI2D_GC_APPEND(vis);
	if (!append)	ggi2dSetAppendMode(vis, 1);
	bezier(vis, x1, y1, x2, y2, x3, y3, x4, y4);
	if (!append)	ggi2dSetAppendMode(vis, 0);
}

void	(*GGIEXP(bezier)[2])(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2, 
			ggi_sint x3, ggi_sint y3, ggi_sint x4, ggi_sint y4) =
{
	bezier,
	bezier_st
};

/*----------------------------------------------------------------------*/
