/*
 * shmdemo.c - (c) 1998 Andreas Beck   becka@ggi-project.org
 *
 * This is a demonstration of LibGGI's functions and can be used as a 
 * reference programming example.
 *
 *   This software is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 */

/* Include the necessary headers used for e.g. error-reporting.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

/* Include the LibGGI declarations.
 */
#include <ggi/gii.h>
#include <ggi/ggi.h>

/* We do shm here !
 */
#include <sys/ipc.h>
#include <sys/shm.h>
#include <signal.h>

#define NUMSHMEM	6
/* A cube has 6 sides ... :-) */
                     
/* In case we were called with wrong parameters, give an explanation.
 */
void usage(const char *prog)
{
	fprintf(stderr, "Usage:\n\n"
		        "%s [<xsize>x<ysize>[#<virtx>x<virty>]"
			"['['<bpp>']']\n\n"
		       "Example: %s 320x200[8]\n"
		"For textmode, try %s 80x25T\n",prog,prog,prog);
	exit(1);
}

static void Setup_Palette(ggi_visual_t vis,ggi_color *palette)
{
	ggi_mode mode;

	int depth, numcols;

	ggi_color black  = { 0x0000, 0x0000, 0x0000 };
	ggi_color white  = { 0xffff, 0xffff, 0xffff };
	ggi_color blue   = { 0x0000, 0x0000, 0xffff };
	ggi_color yellow = { 0xffff, 0xffff, 0x0000 };


	ggiGetMode(vis, &mode);

	depth = GT_DEPTH(mode.graphtype);
	numcols = 1 << depth;

	if (depth < 3) {

		palette[0] = black;
		palette[1] = white;
		palette[2] = blue;
		palette[3] = yellow;

	} else {
		int i;

		int rnum = (depth+1) / 3;
		int gnum = (depth+2) / 3;
		int bnum = (depth)   / 3;

		int rmask = (1 << rnum) - 1;
		int gmask = (1 << gnum) - 1;
		int bmask = (1 << bnum) - 1;

		for (i=0; i < numcols; i++) {

			int j=i, r, g, b;
			
			b = j & bmask;  j >>= bnum;
			g = j & gmask;  j >>= gnum;
			r = j & rmask;

			palette[i].r = r * 0xffff / rmask;
			palette[i].g = g * 0xffff / gmask;
			palette[i].b = b * 0xffff / bmask;
		}
        }

        ggiSetPalette(vis, 0, numcols, palette);
}


/* The main routine.
 * It will set up a graphics mode as requested on the commandline and 
 * do an extensive test using about all graphics primitives LibGGI
 * knows.
 */
int main(int argc, char **argv)
{
	/* First we define a bunch of variables we will access throughout the
	 * main() function. Most of them are pretty meaningless loop-counters
	 * and helper-variables.
	 */

	const char *prog;	/* Make an alias for the program name */
	
	/* This is an enum holding the requested type of graphics mode.
	 * See the mode setting code below for details.
	 */
	 
	ggi_graphtype type;

	/* This holds the palette we will soon set for GT_8BIT type
	 * graphics modes
	 */
	 
	ggi_color pal[256];

	/* The depth is used for chosing a graphics type.  it is mapped
	 * to the right GT_* by a switch-statement.
	 */
	 
	int depth=0;

	/* These holde the visible screen size (sx,sy) and
	 * the virtual size (vx,vy). On most targets you can 
	 * request a certain larger area on which the visible
	 * area can be placed.
	 */

	int sx,sy,vx,vy;
	
	/* Pretty meaningless helpers used everywhere. */

	int err, x, keeprunning, is_escape;
	char text[128];

 	ggi_visual_t vis;
 	
 	ggi_visual_t memvis[NUMSHMEM];
 	int shmid[NUMSHMEM];

	/* This is a struct containing visual and virtual screen size as
	 * well as the bpp/textmode information 
	 */

	ggi_mode mode = { /* This will cause the default mode to be set */
		1,                      /* 1 frame */
		{GGI_AUTO,GGI_AUTO},    /* Default size */
		{GGI_AUTO,GGI_AUTO},    /* Virtual */
		{0,0},                  /* size in mm don't care */
		GT_AUTO,               /* Mode */
		{GGI_AUTO,GGI_AUTO}     /* Font size */
	};

	/* Get the arguments from the command line. 
	 * Set defaults for optional arguments.
	 */

	prog=*argv; argv++; argc--;
	if (strrchr(prog, '/') != NULL) prog=strrchr(prog, '/')+1;

	if ((argc > 0) &&
	    ((strcmp(*argv, "-h")==0) || 
	     (strcmp(*argv, "--help")==0))) {

		usage(prog);	/* This is not an allowed call format. */
		return 1;	/* Tell the user how to call and fail. */
	}

	if (argc > 1) {
		usage(prog);	/* This is not an allowed call format. */
		return 1;	/* Tell the user how to call and fail. */
	}
	
	/* Set up the random number generator. */
	srandom(time(NULL));

	for(x=0;x<NUMSHMEM;x++)
	{
		shmid[x]=shmget(ftok("/dev/null",'0'+x), 1024*1024, IPC_CREAT|0666);
		printf("shmid[%d]=%d\n",x,shmid[x]);
	}

	/* Initialize the GGI library. This must be called before any other 
	 * GGI function. Otherwise behaviour is undefined.
	 */
	if (ggiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize libggi, exiting.\n",
			prog);
		exit(1);
	}

	/* Open the default visual. This is automatically selected depending
	 * on the calling environment (Linux console, X, ...). You can 
	 * explicitly set this in the LIBGGI_DISPLAY environment variable.
	 */
	if ((vis=ggiOpen(NULL)) == NULL) {
		fprintf(stderr,
			"%s: unable to open default visual, exiting.\n",
			prog);
		ggiExit();
		exit(1);
	}
	ggiSetFlags(vis,GGIFLAG_ASYNC);

	if (argc == 1) { /* A mode-string was given - so we parse it */
		if (ggiParseMode(*argv, &mode) != 0) {
			/* Error parsing mode */
			fprintf(stderr,
				"Error parsing mode: %s\n"
				"Valid modes are:\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames (T) [bpp] + XOFF + YOFF\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames [graphtype] + XOFF + YOFF\n\n"
"  Everything (!) can be omitted, all ommitted values default to GGI_AUTO,\n"
"  with the exception of a textmode with unspecified size which defaults\n"
"  to TEXT16.\n  Whitespace is ignored.\n"
				, *argv);
			exit(1);
		}
		/* mode now holds the requested visible 
		 * and virtual size of the screen.
		 */
	}

	/* that's what we try. See what we get ... */
	printf("Trying mode ");
	ggiPrintMode(&mode);
	printf("\n");

	/* Is the mode possible ? If not, a better one will be
	 * suggested. 
	 */

	ggiCheckMode(vis,&mode);

	printf("Suggested mode ");
	ggiPrintMode(&mode);
	printf("\n");

       	err=ggiSetMode(vis,&mode);   /* now try it. it *should* work! */
	ggiSetFlags(vis,GGIFLAG_ASYNC);
	
	/* Check if there were errors. Almost all LibGGI functions
	 * return 0 on success and an error code otherwise. No messing
	 * with errno, as this is not thread- friendly. The functions
	 * that are supposed to return a pointer normally return NULL on
	 * error.
	 */

	if (err) {
		fprintf(stderr,"Can't set mode\n");
		ggiClose(vis);
		ggiExit();
		return 2;
	}


	/* Now we read back the set mode, as it might have been
	 * autoselected or changed.
	 */

	type=mode.graphtype;
	vx=mode.virt.x;    vy=mode.virt.y;
	sx=mode.visible.x; sy=mode.visible.y;
	depth=GT_DEPTH(mode.graphtype);

	/* Set a colorful palette for the tests.
	 */
	 
	if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
		Setup_Palette(vis,pal);
	}
	

	for(x=0;x<NUMSHMEM;x++)
	{
		/* Open a shared "memory-visual" which is simply a simulated 
		 * display in shared memory.
		 */
		sprintf(text,"display-memory:-input:shmid:%d",shmid[x]);

		if ((memvis[x]=ggiOpen(text, NULL)) == NULL) {
			ggiPanic("Ouch - can't open the shmem target %d !",x);
		}

		/* Set it to the same mode as the main visual */
		err=ggiSetMode(memvis[x],&mode);
		
		if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
			Setup_Palette(memvis[x],pal);
		}

		/* Check for errors */
		if (err) ggiPanic("Ouch - can't setmode on shmem target %d !",x);

	}

	printf("mode: %s\n",text);
	printf("Server is running - start another GGI demo (e.g. flying_ggis)\n");
	printf("with the following variable settings:\n");
	printf("GGI_DISPLAY=\"display-memory:-input:keyfile:%d:[0-%d]:%s\"\n",1024*1024,NUMSHMEM-1,"/dev/null");
	ggiSPrintMode(text,&mode);
	printf("GGI_DEFMODE=\"%s\"\n",text);
	printf("export GGI_DEFMODE GGI_DISPLAY\n");


	{
		ggi_color black  = { 0x0000, 0x0000, 0x0000 };
		ggi_color white  = { 0xffff, 0xffff, 0xffff };
		ggiSetGCForeground(memvis[0],ggiMapColor(memvis[0],&white));
		ggiSetGCBackground(memvis[0],ggiMapColor(memvis[0],&black));
	}
	ggiPuts(memvis[0],0,  0,"Keyboard:");
	ggiPuts(memvis[0],0, 10,"#   : Escape char. Twice to send it itself.");
	ggiPuts(memvis[0],0, 20,"#0-5: Select screen to view.");
	ggiPuts(memvis[0],0, 30,"#+/-: Next/previous screen.");
	ggiPuts(memvis[0],0, 40,"#'r': Pick random screen.");

#if 0
	y=0;dy=5;
	while(1)
	{
		ggiCrossBlit(memvis[0],0,vy-1-y,vx,   y,vis,0,0);
		ggiCrossBlit(memvis[1],0,     0,vx,vy-y,vis,0,y);
		ggiFlush(vis);
		y+=dy;
		if (y<=   0) { dy=-dy; y+=dy; }
		if (y>=vy-1) { dy=-dy; y+=dy; }
		usleep(50000);
	}
#else
	x=0;keeprunning=1;is_escape=0;
	while(keeprunning)
	{
		struct timeval t={0,50000};
		
		if (ggiEventPoll(vis,emKey,&t))
		{
			ggi_event event;
			ggiEventRead(vis,&event,emKey);
			if (event.any.type==evKeyPress)
			{
				switch(event.key.sym)
				{
					case '#': is_escape=!is_escape;
						  if (!is_escape) goto sendit;
						  break;
					case '0': if (is_escape) x=0;else goto sendit;
						is_escape=0;break;
					case '1': if (is_escape) x=1;else goto sendit;
						is_escape=0;break;
					case '2': if (is_escape) x=2;else goto sendit;
						is_escape=0;break;
					case '3': if (is_escape) x=3;else goto sendit;
						is_escape=0;break;
					case '4': if (is_escape) x=4;else goto sendit;
						is_escape=0;break;
					case '5': if (is_escape) x=5;else goto sendit;
						is_escape=0;break;
					case '+': if (!is_escape) goto sendit;
						  is_escape=0;x++;if (x>=NUMSHMEM)  x=0;break;
					case '-': if (!is_escape) goto sendit;
						  is_escape=0;x--;if (x<0) x=NUMSHMEM-1;break;
					case 'r': if (!is_escape) goto sendit;
						  is_escape=0;x=rand()%NUMSHMEM;break;
					case 'q': if (!is_escape) goto sendit;
						  is_escape=0;keeprunning=0;break;
					default:
					sendit:
						giiEventSend(ggiJoinInputs(memvis[x],NULL),&event);
						is_escape=0;break;
				}
			}
		}
		ggiCrossBlit(memvis[x],0,0,vx,vy,vis,0,0);
		ggiFlush(vis);
	}
#endif

	for(x=0;x<NUMSHMEM;x++)
	{
		ggiClose(memvis[x]);

		/* this makes the memory detach when all programs using this memory
		 * complete (according to SVR4 standard).
		 * note - this may not work on all platforms....
		 * note2: This may _not be called at an earlier place.
		 * Calling before our own shmat() will destroy _instantly_,
		 * calling before the other programs are running will deny
		 * the shmem to them.
		 */
		shmctl(shmid[x],IPC_RMID, NULL);
	}

	ggiClose(vis);
	
	/* Now close down LibGGI. */
	ggiExit();	

	/* Terminate the program.*/
	return 0;
}
