// fl_color_win32.C

// The fltk "colormap".  This allows ui colors to be stored in 8-bit
// locations, and provides a level of indirection so that global color
// changes can be made.  Not to be confused with the X colormap, which
// I try to hide completely.

// SGI compiler seems to have problems with unsigned char arguments
// being used to index arrays.  So I always copy them to an integer
// before use.

#include <FL/Fl.H>
#include <FL/win32.H>
#include <FL/fl_draw.H>

static unsigned fl_cmap[256] = {
#include "fl_cmap.h" // this is a file produced by "cmap.C":
};

// Translations to win32 data structures:
Fl_XMap fl_xmap[256];

Fl_XMap* fl_current_xmap;

static void clear_xmap(Fl_XMap& xmap) {
  if (xmap.pen) {
    DeleteObject((HGDIOBJ)(xmap.pen));
    xmap.pen = 0;
    xmap.brush = -1;
  }
}

static void set_xmap(Fl_XMap& xmap, uchar r, uchar g, uchar b) {
  xmap.rgb = RGB(r, g, b);
  xmap.pen = CreatePen(PS_SOLID, 1, xmap.rgb);
  xmap.brush = -1;
}

static void use_xmap(Fl_XMap& xmap) {
  fl_current_xmap = &xmap;
  SelectObject(fl_gc, (HGDIOBJ)(xmap.pen));
}

uchar fl_color_;

void fl_color(uchar ii) {
  int i = ii;
  fl_color_ = ii;
  Fl_XMap &xmap = fl_xmap[i];
  if (!xmap.pen) {
    unsigned c = fl_cmap[i];
    set_xmap(xmap, c>>24, c>>16, c>>8);
  }
  use_xmap(xmap);
}

void fl_color(uchar r, uchar g, uchar b) {
  static Fl_XMap xmap;
  // should probably check if color is the same...
  clear_xmap(xmap);
  set_xmap(xmap, r, g, b);
  use_xmap(xmap);
}

// Wonko: we use some statistics to cache only a limited number
// of brushes:

#define FL_N_BRUSH 16
struct Fl_Brush {
  HBRUSH brush;
  unsigned short usage;
  Fl_XMap* backref;
};
static Fl_Brush brushes[FL_N_BRUSH];

// returns the matching brush to the current color
HBRUSH fl_brush() {
  Fl_XMap *xmap = fl_current_xmap;
  char i = xmap->brush; // find the associated brush
  if (i != -1) { // if the brush was allready allocated
    if (brushes[i].brush == NULL) goto CREATE_BRUSH;
    if ( (++brushes[i].usage) > 32000 ) { // keep a usage statistic
      for (int j=0; j<FL_N_BRUSH; j++) {
	if (brushes[j].usage>16000)
	  brushes[j].usage -= 16000;
	else 
	  brushes[j].usage = 0;
      }
    }
    return brushes[i].brush;
  } else {
    int umin = 32000, imin = 0;
    for (i=0; i<FL_N_BRUSH; i++) {
      if (brushes[i].brush == NULL) goto CREATE_BRUSH;
      if (brushes[i].usage<umin) {
	umin = brushes[i].usage;
	imin = i;
      }
    }
    i = imin;
    DeleteObject(brushes[i].brush);
    brushes[i].brush = NULL;
    brushes[i].backref->brush = -1;
  }
CREATE_BRUSH:
  brushes[i].brush = CreateSolidBrush(xmap->rgb);
  brushes[i].usage = 0;
  brushes[i].backref = xmap;
  xmap->brush = i;
  return brushes[i].brush;
}

uchar Fl::set_color(uchar ii, unsigned c) {
  int i = ii;
  if (fl_xmap[i].pen && fl_cmap[i] != c) clear_xmap(fl_xmap[i]);
  fl_cmap[i] = c;
  return i;
}

unsigned Fl::get_color(uchar ii) {
  int i = ii;
  return fl_cmap[i];
}

uchar Fl::set_color(uchar i, uchar red, uchar green, uchar blue) {
  return Fl::set_color(i,
	((unsigned)red<<24)+((unsigned)green<<16)+((unsigned)blue<<8));
}

void Fl::get_color(uchar ii, uchar &red, uchar &green, uchar &blue) {
  int i = ii;
  unsigned c = fl_cmap[i];
  red   = uchar(c>>24);
  green = uchar(c>>16);
  blue  = uchar(c>>8);
}

// 'fl_make_palette()' - Make a color palette for 8-bit displays if necessary
// Thanks to Michael Sweet @ Easy Software Products for this

HPALETTE
fl_make_palette(void)
{
  // Find out if we need to define a color palette...

  PIXELFORMATDESCRIPTOR pfd;
  DescribePixelFormat(fl_gc, GetPixelFormat(fl_gc), sizeof(pfd), &pfd);

  if (!(pfd.dwFlags & PFD_NEED_PALETTE)) return NULL;
  int nColors = 1 << pfd.cColorBits;
  if (nColors > 256) return NULL;
  // this will try to work on < 256 color screens, but will probably
  // come out quite badly.

  // I lamely try to get this variable-sized object allocated on stack:
  ulong foo[(sizeof(LOGPALETTE)+256*sizeof(PALETTEENTRY))/sizeof(ulong)+1];
  LOGPALETTE *pPal = (LOGPALETTE*)foo;

  pPal->palVersion    = 0x300;
  pPal->palNumEntries = nColors;

  // Build 256 colors from the standard FLTK colormap...

  for (int i = 0; i < nColors; i ++) {
    pPal->palPalEntry[i].peRed   = (fl_cmap[i] >> 24) & 255;
    pPal->palPalEntry[i].peGreen = (fl_cmap[i] >> 16) & 255;
    pPal->palPalEntry[i].peBlue  = (fl_cmap[i] >>  8) & 255;
    pPal->palPalEntry[i].peFlags = 0;
  };

  // Create the palette:

  return CreatePalette(pPal);
}
