// fl_boxtype.c

// Box drawing code for the common box types and the table of
// boxtypes.  Other box types are in seperate files so they are not
// linked in if not used.

#include <FL/Fl.H>
#include <FL/Fl_Widget.H>
#include <FL/fl_draw.H>

////////////////////////////////////////////////////////////////

void fl_no_box(int, int, int, int, uchar) {}

void fl_thin_down_frame(int x, int y, int w, int h, uchar) {
  if (w >= 1 && h >= 1) {
    fl_color(FL_DARK3); fl_yxline(x, y+h-1, y, x+w-1);
    if (w >= 2 && h >= 2)
      fl_color(FL_LIGHT3); fl_xyline(x+1, y+h-1, x+w-1, y+1);
  }
}

void fl_thin_down_box(int x, int y, int w, int h, uchar c) {
  fl_thin_down_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+1, y+1, w-2, h-2);
}

void fl_thin_up_frame(int x, int y, int w, int h, uchar) {
  if (w >= 1 && h >= 1) {
    fl_color(FL_LIGHT3); fl_yxline(x, y+h-1, y, x+w-1);
    if (w >= 2 && h >= 2)
      fl_color(FL_DARK3); fl_xyline(x+1, y+h-1, x+w-1, y+1);
  }
}

void fl_thin_up_box(int x, int y, int w, int h, uchar c) {
  fl_thin_up_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+1, y+1, w-2, h-2);
}

void fl_up_frame(int x, int y, int w, int h, uchar) {
  if (w >= 3 && h >= 3) {
    fl_color(FL_LIGHT3); fl_xyline(x+1, y+1, x+w-2);
    fl_color(FL_LIGHT2); fl_yxline(x+1, y+h-2, y+2, x+w-3);
    fl_color(FL_LIGHT1); fl_yxline(x+2, y+h-3, y+3/*, x+w-3*/);
    fl_color(FL_DARK1); fl_xyline(x+2, y+h-3, x+w-3, y+3);
    fl_color(FL_DARK2); fl_xyline(x+1, y+h-2, x+w-2, y+2);
  }
  fl_color(FL_GRAY0); fl_rect(x,y,w,h);
}

void fl_up_box(int x, int y, int w, int h, uchar c) {
  fl_up_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+3,y+3,w-6,h-6);
}

void fl_down_frame(int x, int y, int w, int h, uchar) {
  if (w >= 3 && h >= 3) {
    fl_color(FL_DARK1); fl_yxline(x+0, y+h-1, y, x+w-1);
    fl_color(FL_DARK2); fl_yxline(x+1, y+h-2, y+1, x+w-2);
    fl_color(FL_LIGHT3); fl_xyline(x+2, y+h-2, x+w-3);
    fl_color(FL_LIGHT2); fl_xyline(x+1, y+h-1, x+w-2, y+2);
    fl_color(FL_LIGHT1); fl_yxline(x+w-1, y+h-1, y);
    fl_color(FL_GRAY0); fl_rect(x+2, y+2, w-4, h-4);
  }
}

void fl_down_box(int x, int y, int w, int h, uchar c) {
  fl_down_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+3, y+3, w-6, h-6);
}

void fl_engraved_frame(int x, int y, int w, int h, uchar) {
  fl_color(FL_LIGHT3); fl_rect(x+1, y+1, w-1, h-1);
  fl_color(FL_DARK3);
  fl_yxline(x, y+h-2, y, x+w-2);
  fl_xyline(x+2, y+h-2, x+w-2, y+2);
}

void fl_engraved_box(int x, int y, int w, int h, uchar c) {
  fl_engraved_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+2, y+2, w-4, h-4);
}

void fl_embossed_frame(int x, int y, int w, int h, uchar) {
  fl_color(FL_DARK3); fl_rect(x+1, y+1, w-1, h-1);
  fl_color(FL_LIGHT3);
  fl_yxline(x, y+h-1, y, x+w-1);
  fl_xyline(x+2, y+h-2, x+w-2, y+2);
}

void fl_embossed_box(int x, int y, int w, int h, uchar c) {
  fl_embossed_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+2, y+2, w-4, h-4);
}

void fl_rectbound(int x, int y, int w, int h, uchar bgcolor) {
//   if (w<0) {x += w; w = -w;}
//   if (h<0) {y += h; h = -h;}
//   if (!w) w = 1;
//   if (!h) h = 1;
  fl_color(bgcolor); fl_rectf(x+1, y+1, w-2, h-2);
  fl_color(FL_BLACK); fl_rect(x, y, w, h);
}
#define fl_border_box fl_rectbound

void fl_rectf(int x, int y, int w, int h, uchar c) {
  fl_color(c);
  fl_rectf(x, y, w, h);
}

void fl_border_frame(int x, int y, int w, int h, uchar c) {
  fl_color(c);
  fl_rect(x, y, w, h);
}

////////////////////////////////////////////////////////////////

static struct {
  Fl_Box_Draw_F *f;
  uchar dx, dy, dw, dh;
} fl_box_table[] = {
// must match list in Enumerations.H!!!
  {fl_no_box,		0,0,0,0},		
  {fl_rectf,		0,0,0,0}, // FL_FLAT_BOX
  {fl_up_box,		3,3,6,6},
  {fl_down_box,		3,3,6,6},
  {fl_up_frame,		3,3,6,6},
  {fl_down_frame,	3,3,6,6},
  {fl_thin_up_box,	1,1,2,2},
  {fl_thin_down_box,	1,1,2,2},
  {fl_thin_up_frame,	1,1,2,2},
  {fl_thin_down_frame,	1,1,2,2},
  {fl_engraved_box,	2,2,4,4},
  {fl_embossed_box,	2,2,4,4},
  {fl_engraved_frame,	2,2,4,4},
  {fl_embossed_frame,	2,2,4,4},
  {fl_border_box,	1,1,2,2},
  {fl_border_box,	1,1,2,2}, // _FL_SHADOW_BOX,
  {fl_border_frame,	1,1,2,2},
  {fl_border_frame,	1,1,2,2}, // _FL_SHADOW_FRAME,
  {fl_border_box,	4,4,8,8}, // _FL_ROUNDED_BOX,
  {fl_border_box,	4,4,8,8}, // _FL_RSHADOW_BOX,
  {fl_border_frame,	4,4,8,8}, // _FL_ROUNDED_FRAME
  {fl_rectf,		4,4,8,8}, // _FL_RFLAT_BOX,
  {fl_up_box,		3,3,6,6}, // _FL_ROUND_UP_BOX
  {fl_down_box,		3,3,6,6}, // _FL_ROUND_DOWN_BOX,
  {fl_up_box,		0,0,0,0}, // _FL_DIAMOND_UP_BOX
  {fl_down_box,		0,0,0,0}, // _FL_DIAMOND_DOWN_BOX
  {fl_border_box,	1,1,2,2}, // _FL_OVAL_BOX,
  {fl_border_box,	1,1,2,2}, // _FL_OVAL_SHADOW_BOX,
  {fl_border_frame,	1,1,2,2}, // _FL_OVAL_FRAME
  {fl_rectf,		0,0,0,0}, // _FL_OVAL_FLAT_BOX,
  {fl_up_box,		3,3,6,6}, // FL_FREE_BOX+0
  {fl_down_box,		3,3,6,6}, // FL_FREE_BOX+1
  {fl_up_box,		3,3,6,6}, // FL_FREE_BOX+2
  {fl_down_box,		3,3,6,6}, // FL_FREE_BOX+3
  {fl_up_box,		3,3,6,6}, // FL_FREE_BOX+4
  {fl_down_box,		3,3,6,6}, // FL_FREE_BOX+5
  {fl_up_box,		3,3,6,6}, // FL_FREE_BOX+6
  {fl_down_box,		3,3,6,6}, // FL_FREE_BOX+7
};

int Fl::box_dx(uchar t) {return fl_box_table[t].dx;}
int Fl::box_dy(uchar t) {return fl_box_table[t].dy;}
int Fl::box_dw(uchar t) {return fl_box_table[t].dw;}
int Fl::box_dh(uchar t) {return fl_box_table[t].dh;}

void fl_internal_boxtype(int t, Fl_Box_Draw_F* f) {fl_box_table[t].f = f;}

uchar Fl::set_boxtype(uchar t, Fl_Box_Draw_F* f,
		      uchar a, uchar b, uchar c, uchar d) {
  fl_box_table[t].f=f;
  fl_box_table[t].dx = a;
  fl_box_table[t].dy = b;
  fl_box_table[t].dw = c;
  fl_box_table[t].dh = d;
  return t;
}

uchar Fl::set_boxtype(uchar t, uchar f) {
  fl_box_table[t] = fl_box_table[f];
  return t;
}

void fl_draw_box(uchar t,int x,int y,int w,int h,uchar c) {
  if (t) fl_box_table[t].f(x,y,w,h,c);
}

extern Fl_Widget *fl_boxcheat; // hack set by Fl_Window.C

void Fl_Widget::draw_box() const {
  int t = box_;
  if (!t) return;
  if (this == fl_boxcheat) {
    fl_boxcheat = 0;
    if (t == FL_FLAT_BOX) return;
    t += 2; // convert box to frame
  }
  fl_box_table[t].f(x_,y_,w_,h_,color_);
}

void Fl_Widget::draw_box(uchar b,uchar c) const {
  if (b) fl_box_table[b].f(x_,y_,w_,h_,c);
}
