/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>
#include <stdio.h>	/* sprintf, vsnprintf */
#include <stdarg.h>	/* vsnprintf */

#include <gtk/gtk.h>

#include "xqf.h"
#include "stat.h"
#include "source.h"
#include "pref.h"
#include "filter.h"
#include "skin.h"
#include "dialogs.h"
#include "utils.h"
#include "server.h"
#include "qrun.h"
#include "psearch.h"
#include "dns.h"
#include "xutils.h"
#include "srv-list.h"
#include "srv-prop.h"
#include "rc.h"
#include "pixmaps.h"
#include "rcon.h"


GtkWidget *main_window = NULL;
GtkWidget *server_clist = NULL;
GtkWidget *player_clist = NULL;
GtkWidget *serverinfo_clist = NULL;

struct server *cur_server = NULL;

int psort_column = 1;
int ssort_column = 2;
int isort_column = 0;

int show_hostnames = TRUE;

static const char *show_hide[2] = { "Show Host Names", "Hide Host Names" };

static GtkWidget *server_menu = NULL;

static GtkWidget *connect_menu_item = NULL;
static GtkWidget *observe_menu_item = NULL;
static GtkWidget *record_menu_item = NULL;
static GtkWidget *favadd_menu_item = NULL;
static GtkWidget *add_menu_item = NULL;
static GtkWidget *delete_menu_item = NULL;
static GtkWidget *refresh_menu_item = NULL;
static GtkWidget *refrsel_menu_item = NULL;
static GtkWidget *resolve_menu_item = NULL;
static GtkWidget *hostnames_menu_item = NULL;
static GtkWidget *find_player_menu_item = NULL;
static GtkWidget *properties_menu_item = NULL;
static GtkWidget *rcon_menu_item = NULL;

static GtkWidget *file_quit_menu_item = NULL;

static GtkWidget *edit_add_menu_item = NULL;
static GtkWidget *edit_delete_menu_item = NULL;
static GtkWidget *edit_selall_menu_item = NULL;
static GtkWidget *edit_invsel_menu_item = NULL;

static GtkWidget *view_refresh_menu_item = NULL;
static GtkWidget *view_refrsel_menu_item = NULL;
static GtkWidget *view_update_menu_item = NULL;
static GtkWidget *view_hostnames_menu_item = NULL;

static GtkWidget *server_connect_menu_item = NULL;
static GtkWidget *server_observe_menu_item = NULL;
static GtkWidget *server_record_menu_item = NULL;
static GtkWidget *server_favadd_menu_item = NULL;
static GtkWidget *server_resolve_menu_item = NULL;
static GtkWidget *server_properties_menu_item = NULL;
static GtkWidget *server_rcon_menu_item = NULL;

static GtkWidget *update_button = NULL;
static GtkWidget *refresh_button = NULL;
static GtkWidget *refrsel_button = NULL;
static GtkWidget *stop_button = NULL;
static GtkWidget *source_option_menu = NULL;
static GSList    *filter_buttons = NULL;

static GtkWidget *player_skin_popup = NULL;
static GtkWidget *player_skin_popup_preview = NULL;

static GtkWidget *status_bar = NULL;
static GtkWidget *progress_bar = NULL;

static unsigned status_bar_context_id;

static GSList *xqf_windows = NULL;
static GtkWidget *target_window = NULL;

static struct master *cur_master = NULL;

static struct condef *con = NULL;

static struct stat_job *stat_process = NULL;

struct clist_coldef server_clist_columns[7] = {
  { "Name", 	250, 	GTK_JUSTIFY_LEFT,	SORT_SERVER_NAME,     NULL },
  { "Address",	150, 	GTK_JUSTIFY_LEFT,	SORT_SERVER_ADDRESS,  NULL },
  { "Ping",	50,  	GTK_JUSTIFY_RIGHT,	SORT_SERVER_PING,     NULL },
  { "TO",	40,  	GTK_JUSTIFY_RIGHT,	SORT_SERVER_TO,	      NULL },
  { "Players",	70,  	GTK_JUSTIFY_RIGHT,	SORT_SERVER_PLAYERS,  NULL },
  { "Map",	60,  	GTK_JUSTIFY_LEFT,	SORT_SERVER_MAP,      NULL },
  { "Game",	60,	GTK_JUSTIFY_LEFT,	SORT_SERVER_GAME,     NULL }
};

struct clist_coldef player_clist_columns[6] = {
  { "Name",	120,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_NAME,     NULL },
  { "Frags",	60,	GTK_JUSTIFY_RIGHT,      -SORT_PLAYER_FRAGS,   NULL },
  { "Colors",	60,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_COLOR,    NULL },
  { "Skin",	70,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_SKIN,     NULL },
  { "Ping",	50,	GTK_JUSTIFY_RIGHT,      SORT_PLAYER_PING,     NULL },
  { "Time",	50,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_TIME,     NULL }
};

struct clist_coldef serverinfo_clist_columns[2] = {
  { "Rule",	70,	GTK_JUSTIFY_LEFT,       SORT_INFO_RULE,	      NULL },
  { "Value",	160,	GTK_JUSTIFY_LEFT,       SORT_INFO_VALUE,      NULL }
};


static void print_status (char *fmt, ...) {
  char buf[1024];
  va_list ap;

  if (status_bar) {
    buf[0] = ' ';		/* indent */
    buf[1] = '\0';

    if (fmt) {
      va_start (ap, fmt);
      vsnprintf (buf + 1, 1024 - 1, fmt, ap);
      va_end (ap);
    }

    gtk_statusbar_pop (GTK_STATUSBAR (status_bar), status_bar_context_id);
    gtk_statusbar_push (GTK_STATUSBAR (status_bar), status_bar_context_id, 
                                                                         buf);
#ifdef DEBUG
    fprintf (stderr, "Status: %s\n", buf);
#endif
  }
}


void update_progress_bar (float pct) {
  float delta;

  if (progress_bar && pct >= 0) {
    delta = pct - GTK_PROGRESS_BAR (progress_bar)->percentage;
    if ((int) (delta * 256) != 0)
      gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_bar), pct);
  }
}


int window_delete_event_callback (GtkWidget *widget, gpointer data) {
  target_window = widget;
  gtk_widget_destroy ((GtkWidget *) (xqf_windows->data));
  return TRUE;
}


void register_window (GtkWidget *window) {
  xqf_windows = g_slist_prepend (xqf_windows, window);
}


void unregister_window (GtkWidget *window) {
  GSList *first;

  first = xqf_windows;
  xqf_windows = g_slist_next (xqf_windows);
  g_slist_free_1 (first);

  if (target_window && target_window != window)
    gtk_widget_destroy ((GtkWidget *) (xqf_windows->data));
  else
    target_window = NULL;
}


GtkWidget *top_window (void) {
  if (xqf_windows)
    return (GtkWidget *) xqf_windows->data;
  else
    return NULL;
}


void set_widgets_sensitivity (void) {
  GList *selected = GTK_CLIST (server_clist)->selection;
  GSList *list;

  if (!stat_process && cur_server) {
    gtk_widget_set_sensitive (connect_menu_item, TRUE);
    gtk_widget_set_sensitive (server_connect_menu_item, TRUE);

    gtk_widget_set_sensitive (observe_menu_item, 
                               (cur_server->type == QW_SERVER)? TRUE : FALSE);
    gtk_widget_set_sensitive (server_observe_menu_item,
                               (cur_server->type == QW_SERVER)? TRUE : FALSE);

    gtk_widget_set_sensitive (record_menu_item, TRUE);
    gtk_widget_set_sensitive (server_record_menu_item, TRUE);

    gtk_widget_set_sensitive (properties_menu_item, TRUE);
    gtk_widget_set_sensitive (server_properties_menu_item, TRUE);

    gtk_widget_set_sensitive (rcon_menu_item, TRUE);
    gtk_widget_set_sensitive (server_rcon_menu_item, TRUE);
  }
  else {
    gtk_widget_set_sensitive (connect_menu_item, FALSE);
    gtk_widget_set_sensitive (server_connect_menu_item, FALSE);

    gtk_widget_set_sensitive (observe_menu_item, FALSE);
    gtk_widget_set_sensitive (server_observe_menu_item, FALSE);

    gtk_widget_set_sensitive (record_menu_item, FALSE);
    gtk_widget_set_sensitive (server_record_menu_item, FALSE);

    gtk_widget_set_sensitive (properties_menu_item, FALSE);
    gtk_widget_set_sensitive (server_properties_menu_item, FALSE);

    gtk_widget_set_sensitive (rcon_menu_item, FALSE);
    gtk_widget_set_sensitive (server_rcon_menu_item, FALSE);
  }

  if (!stat_process && selected) {
    gtk_widget_set_sensitive (favadd_menu_item, 
                                     (cur_master != favorites)? TRUE : FALSE);
    gtk_widget_set_sensitive (server_favadd_menu_item, 
                                     (cur_master != favorites)? TRUE : FALSE);

    gtk_widget_set_sensitive (delete_menu_item,
                   (!cur_master->host && !cur_master->sources)? TRUE : FALSE);
    gtk_widget_set_sensitive (edit_delete_menu_item,
                   (!cur_master->host && !cur_master->sources)? TRUE : FALSE);

    gtk_widget_set_sensitive (refrsel_menu_item, TRUE);
    gtk_widget_set_sensitive (view_refrsel_menu_item, TRUE);

    gtk_widget_set_sensitive (resolve_menu_item, TRUE);
    gtk_widget_set_sensitive (server_resolve_menu_item, TRUE);
  }
  else {
    gtk_widget_set_sensitive (favadd_menu_item, FALSE);
    gtk_widget_set_sensitive (server_favadd_menu_item, FALSE);

    gtk_widget_set_sensitive (delete_menu_item, FALSE);
    gtk_widget_set_sensitive (edit_delete_menu_item, FALSE);

    gtk_widget_set_sensitive (refrsel_menu_item, FALSE);
    gtk_widget_set_sensitive (view_refrsel_menu_item, FALSE);

    gtk_widget_set_sensitive (resolve_menu_item, FALSE);
    gtk_widget_set_sensitive (server_resolve_menu_item, FALSE);
  }

  if (!stat_process && cur_master->servers) {
    gtk_widget_set_sensitive (find_player_menu_item, TRUE);

    gtk_widget_set_sensitive (refresh_menu_item, TRUE);
    gtk_widget_set_sensitive (view_refresh_menu_item, TRUE);

    gtk_widget_set_sensitive (hostnames_menu_item, TRUE);
    gtk_widget_set_sensitive (view_hostnames_menu_item, TRUE);
  }
  else {
    gtk_widget_set_sensitive (find_player_menu_item, FALSE);

    gtk_widget_set_sensitive (refresh_menu_item, FALSE);
    gtk_widget_set_sensitive (view_refresh_menu_item, FALSE);

    gtk_widget_set_sensitive (hostnames_menu_item, FALSE);
    gtk_widget_set_sensitive (view_hostnames_menu_item, FALSE);
  }

  if (!stat_process && 
                     (cur_master == unbound_qw || cur_master == unbound_q2)) {
    gtk_widget_set_sensitive (add_menu_item, TRUE);
    gtk_widget_set_sensitive (edit_add_menu_item, TRUE);
  }
  else {
    gtk_widget_set_sensitive (add_menu_item, FALSE);
    gtk_widget_set_sensitive (edit_add_menu_item, FALSE);
  }

  if (stat_process) {
    gtk_widget_set_sensitive (source_option_menu, FALSE);

    gtk_widget_set_sensitive (update_button, FALSE);
    gtk_widget_set_sensitive (view_update_menu_item, FALSE);

    gtk_widget_set_sensitive (refresh_button, FALSE);
    gtk_widget_set_sensitive (refrsel_button, FALSE);
    gtk_widget_set_sensitive (stop_button, TRUE);

    for (list = filter_buttons; list; list = list->next)
      gtk_widget_set_sensitive (GTK_WIDGET (list->data), FALSE);
  }
  else {
    gtk_widget_set_sensitive (source_option_menu, TRUE);

    gtk_widget_set_sensitive (update_button, 
                     (cur_master->host || cur_master->sources)? TRUE : FALSE);
    gtk_widget_set_sensitive (view_update_menu_item,
                     (cur_master->host || cur_master->sources)? TRUE : FALSE);

    gtk_widget_set_sensitive (refresh_button, 
                                         (cur_master->servers)? TRUE : FALSE);
    gtk_widget_set_sensitive (refrsel_button, (selected)? TRUE : FALSE);
    gtk_widget_set_sensitive (stop_button, FALSE);

    for (list = filter_buttons; list; list = list->next)
      gtk_widget_set_sensitive (GTK_WIDGET (list->data), TRUE);
  }
}


static void redraw_server_clist (int preserve_selection, int update_filter) {
  GSList *selected = NULL;

  if (!cur_master)
    return;

  if (preserve_selection)
    selected = server_clist_selected_servers ();

  if (update_filter)
    free_filters_data ();

  if (!cur_filter->built)
    build_filtered_list (cur_filter, cur_master);

  if (cur_filter->servers)
    populate_server_clist (&cur_filter->servers, selected);
  else
    gtk_clist_clear (GTK_CLIST (server_clist));

  if (selected)
    free_servers (selected);

  sync_selection ();
}


static void set_filter_callback (GtkWidget *widget, struct filter *filter) {
  int preserve_selection = FALSE;

  /* it's called "by hand" if widget == NULL */

  if (filter == cur_filter) {
    if (widget != NULL)
      return;
    preserve_selection = TRUE;
  }

  if (cur_filter->widget) {
    gtk_arrow_set (GTK_ARROW (cur_filter->widget), GTK_ARROW_RIGHT, 
                                                              GTK_SHADOW_OUT);
  }

  if (filter->widget) {
    gtk_arrow_set (GTK_ARROW (filter->widget), GTK_ARROW_RIGHT, 
                                                               GTK_SHADOW_IN);
  }

  cur_filter = filter;
  redraw_server_clist (preserve_selection, FALSE);
}


static void find_player_callback (GtkWidget *widget, gpointer data) {

  if (stat_process || !cur_master->servers)
    return;

  switch (psearch_dialog ()) {
    
  case PSEARCH_OK:
    free_filter_data (&filters[3]);
    set_filter_callback (NULL, &filters[3]);

    /* select first server */

    gtk_clist_select_row (GTK_CLIST (server_clist), 0, 0);
    sync_selection ();

    break;

  case PSEARCH_USELAST:
    set_filter_callback (NULL, &filters[3]);
    break;

  case PSEARCH_CANCEL:
  default:
    break;

  }
}


static void start_preferences_dialog (GtkWidget *widget, int page_num) {
  preferences_dialog (page_num);
  rc_save ();

  if (filter_prefs_changed) {
    filter_prefs_changed = FALSE;
    redraw_server_clist (TRUE, TRUE);
    if (stat_process) {
      stat_process->redraw = FALSE;
    }
  }
}


static void stat_lists_close_handler (struct stat_job *job, int killed) {
  int multiple_sources = (int) job->data;

  if (main_window) {
    if (job->redraw || multiple_sources) {
      if (cur_master->sources) {
	free_servers (cur_master->servers);
	cur_master->servers = 
           collate_server_lists (NULL, cur_master->sources, cur_master->type);
      }
      redraw_server_clist (TRUE, TRUE);
    }

    server_clist_selection_visible ();

    if (killed) {
      print_status ("Aborted.");
    }
    else {
      if (job->redraw && !job->single)
	print_status ("Done.");
    }	

    stat_job_free (stat_process);
    stat_process = NULL;

    update_progress_bar (0.0);
    set_widgets_sensitivity ();
  }
}


static void stat_lists_report_master (struct stat_job *job, 
                                                           struct master *m) {

  if (job->conn && job->conn->master)
    m = job->conn->master;

  if (m) {
    print_status ("Updating server list from master %s [%s:%d]: %d/%d", 
                                           m->name, m->host->address, m->port,
                                           job->progress.subtasks_done, 
                                           job->progress.subtasks_total);
  }
  else {
    print_status ("Updating server list...");
  }
  update_progress_bar (stat_job_progress (job));
}


static void stat_lists_report_server (struct stat_job *job, 
                                                           struct server *s) {
  char *status;
  struct master *m = job->conn->master;

  if (job->single) {
    if (s) {
      status = (s->ping < MAX_PING)? "Ok" :
                            (s->ping == MAX_PING + 1)? "down" : "unreachable";
      print_status ("%s:%d is %s.",
	  (s->host->name)? s->host->name : s->host->address, s->port, status);
    }
    else {
      if (job->servers) {
	s = (struct server *) job->servers->data;
	print_status ("Refreshing %s:%d...",
	          (s->host->name)? s->host->name : s->host->address, s->port);
      }
    }
  }
  else {
    if (m) {
      stat_lists_report_master (job, m);
    }
    else {
      print_status ("Refreshing servers: %d/%d", 
                   job->progress.subtasks_done, job->progress.subtasks_total);
      update_progress_bar (stat_job_progress (job));
    }
  }
}


static void stat_lists_report_host (struct stat_job *job, struct host *h) {

  if (job->single) {
    if (h) {
      if (h->name)
	print_status ("%s is resolved to %s.", h->address, h->name);
      else
	print_status ("Host %s has no name.", h->address);
    }
    else {
      if (job->hosts) {
	h = (struct host *) job->hosts->data;
	print_status ("Resolving %s...", h->address);
      }
    }
  }
  else {
    print_status ("Resolving host addresses: %d/%d", 
		   job->progress.subtasks_done, job->progress.subtasks_total);
    update_progress_bar (stat_job_progress (job));
  }
}


static void stat_lists_state_handler (struct stat_job *job, 
                                                      enum stat_state state) {
  switch (state) {

  case STAT_SOURCE:
    stat_lists_report_master (job, NULL);
    break;

  case STAT_SERVER:
    stat_lists_report_server (job, NULL);
    break;

  case STAT_RESOLVE:
    stat_lists_report_host (job, NULL);
    break;

  }
}


static void stat_lists_master_handler (struct stat_job *job, 
                                              struct master *m, int servers) {
  int multiple_sources;
  GSList *list;

  multiple_sources = (int) job->data;

  if (servers >= 0) {
    stat_lists_report_master (job, m);
  }
  else {
    print_status ("%s [%s:%d] is not responding.", m->name, 
		                                   m->host->address, m->port);
    update_progress_bar (stat_job_progress (job));
  }
  
  if (servers < 0) {
    if (multiple_sources) {
      if (m->servers) {
	list = g_slist_prepend (NULL, m);
	job->servers = collate_server_lists (job->servers, list, ANY_SERVER);
	g_slist_free (list);
      }
      job->redraw = TRUE;
    }
  }
  else {
    job->redraw = TRUE;
  }
}


static void stat_lists_server_handler (struct stat_job *job, 
                                                           struct server *s) {
  if (default_always_resolve)
    job->hosts = merge_host_to_resolve (job->hosts, s->host);

  if ((job->single && !job->hosts) || server_clist_refresh_server (s))
    job->redraw = TRUE;

  stat_lists_report_server (job, s);
}


static void stat_lists_host_handler (struct stat_job *job, 
                                                   struct host *h, char *id) {
  if (h->name)
    server_clist_show_hostname (h);

  stat_lists_report_host (job, h);
}


static void stat_lists (GSList *masters, GSList *servers, GSList *hosts) {

  if (stat_process || (!masters && !servers && !hosts))
    return;

  stat_process = stat_job_create (masters, servers, hosts, NULL);

  stat_process->state_handlers = g_slist_prepend (
                      stat_process->state_handlers, stat_lists_state_handler);
  stat_process->close_handlers = g_slist_prepend (
                      stat_process->close_handlers, stat_lists_close_handler);

  stat_process->master_handlers = g_slist_append (
                    stat_process->master_handlers, stat_lists_master_handler);
  stat_process->server_handlers = g_slist_append (
                    stat_process->server_handlers, stat_lists_server_handler);
  stat_process->host_handlers = g_slist_append (
                        stat_process->host_handlers, stat_lists_host_handler);

  stat_process->single = FALSE;
  stat_process->redraw = FALSE;
  stat_process->data = (gpointer) FALSE;

  if (masters) {
    if (masters->next || servers)			/* multiple sources */
      stat_process->data = (gpointer) TRUE;
  }
  else {
    if ((servers && !servers->next && !hosts) ||	/* single host */
        (hosts && !hosts->next && !servers))
      stat_process->single = TRUE;
  }

  stat_start (stat_process);
  set_widgets_sensitivity ();
}


static void stat_one_server (struct server *s) {

  if (stat_process || !s)
    return;

  s->ref_count++;
  stat_lists (NULL, g_slist_prepend (NULL, s), NULL);
}


static void launch_close_handler (struct stat_job *job, int killed) {

  /* Reduced version of stat_lists_close_handler() */

  if (main_window) {
    if (killed)
      print_status ("Aborted.");

    stat_job_free (stat_process);
    stat_process = NULL;

    set_widgets_sensitivity ();
  }
}


static void launch_server_handler (struct stat_job *job, struct server *s) {
  struct server_props *props;
  int launch = FALSE;
  int spectator;
  int save = 0;

  stat_lists_report_server (job, s);
  redraw_server_clist (TRUE, TRUE);

  spectator = (int) job->data;
  props = properties (s);

  if (s->ping >= MAX_PING) {
    launch = dialog_yesno (NULL, 1, "Launch", "Cancel",
                      "Server %s:%d is unreachable.\n\nLaunch client anyway?",
                      (s->host->name)? s->host->name : s->host->address, 
                      s->port);
    if (!launch) {
      condef_free (con);
      con = NULL;
      return;
    }
  }

  if (!quake_config_is_valid (s->type)) {
    start_preferences_dialog (NULL, PREF_PAGE_GAMEPLAY);
    condef_free (con);
    con = NULL;
    return;
  }

  if (spectator && s->type == Q2_SERVER) {
    dialog_ok (NULL, "Quake2 doesn't support spectators");
    condef_free (con);
    con = NULL;
    return;
  }

  if (!launch && s->curplayers == s->maxplayers && !spectator) {
    launch = dialog_yesno (NULL, 1, "Launch", "Cancel",
			 "Server %s:%d is full.\n\nLaunch client anyway?",
			 (s->host->name)? s->host->name : s->host->address,
			 s->port);
    if (!launch) {
      condef_free (con);
      con = NULL;
      return;
    }
  }

  if (spectator) {
    if ((s->flags & SERVER_SP_PASSWORD) == 0) {
      con->observe = g_strdup ("1");
    }
    else {
      if (props && props->spectator_password) {
	con->observe = g_strdup (props->spectator_password);
      }
      else {
	con->observe = enter_string_with_option_dialog (
                               "Save Password", &save, "Spectator Password:");
	if (!con->observe) {
	  condef_free (con);
	  con = NULL;
	  return;
	}
	if (save) {
	  if (!props)
	    props = properties_new (s->host, s->port);
	  props->spectator_password = g_strdup (con->observe);
	  props_save ();
	}
      }
    }
  }
  else {
    if (s->flags & SERVER_PASSWORD) {
      if (props && props->server_password) {
	con->password = g_strdup (props->server_password);
      }
      else {
	con->password = enter_string_with_option_dialog (
                                  "Save Password", &save, "Server Password:");
	if (!con->password) {
	  condef_free (con);
	  con = NULL;
	  return;
	}
	if (save) {
	  if (!props)
	    props = properties_new (s->host, s->port);
	  props->server_password = g_strdup (con->password);
	  props_save ();
	}
      }
    }
  }

  con->server = g_malloc (strlen (s->host->address) + 1 + 5 + 1);
  sprintf (con->server, "%s:%5d", s->host->address, s->port);

  con->gamedir = g_strdup (s->game);

  if (props && props->rcon_password)
    con->rcon_password = g_strdup (props->rcon_password);

  if (props && props->custom_cfg)
    con->custom_cfg = g_strdup (props->custom_cfg);

  if (default_terminate) {
    con->s->ref_count++;
    gtk_widget_destroy (main_window);
  }
  else {
    if (default_iconify)
      iconify_window (main_window->window);
    launch_quake (con, TRUE);
    condef_free (con);
    con = NULL;
  }

  return;
}


static void launch_callback (GtkWidget *widget, enum launch_mode mode) {
  int spectator = FALSE;
  char *demo = NULL;
  GSList *servers;

  if (stat_process || !cur_server || 
           (cur_server->type != QW_SERVER && cur_server->type != Q2_SERVER)) {
    return;
  }

  switch (mode) {

  case LAUNCH_NORMAL:
    break;

  case LAUNCH_SPECTATE:
    spectator = TRUE;
    break;

  case LAUNCH_RECORD:
    switch (cur_server->type) {

    case QW_SERVER:
      demo = enter_string_with_option_dialog ("Spectator", &spectator, 
                                                                "Demo name:");
      break;

    case Q2_SERVER:
      demo = enter_string_dialog ("Demo name:");
      break;

    default:
      return;
    }

    if (!demo)
      return;

    break;

  }

  con = condef_new (cur_server);
  con->demo = demo;

  if (!stat_process) {
    servers = g_slist_prepend (NULL, cur_server);
    cur_server->ref_count++;

    stat_process = stat_job_create (NULL, servers, NULL, NULL);

    stat_process->close_handlers = g_slist_prepend (
                          stat_process->close_handlers, launch_close_handler);

    stat_process->server_handlers = g_slist_append (
                        stat_process->server_handlers, launch_server_handler);

    stat_process->single = TRUE;
    stat_process->data = (gpointer) spectator;

    stat_start (stat_process);
    set_widgets_sensitivity ();

    stat_lists_report_server (stat_process, NULL);
  }
}


static void server_clist_column_set_title (int col, int append) {
  char tmp[64];
  char *ptr;
  int sortmode = server_clist_columns[col].sortmode;

  if (append) {
    sprintf (tmp, "%s %c", server_clist_columns[col].name, 
                                                   (sortmode > 0)? '<' : '>');
    ptr = tmp;
  }
  else {
    ptr = server_clist_columns[col].name;
  }
  gtk_label_set (GTK_LABEL (server_clist_columns[col].widget), ptr);
}


static void server_clist_click_column_callback (GtkWidget *widget, int col) {
  int sortmode = server_clist_columns[col].sortmode;

  server_clist_column_set_title (ssort_column, FALSE);

  if (ssort_column == col)
    server_clist_columns[col].sortmode = -sortmode;
  else
    ssort_column = col;

  server_clist_column_set_title (ssort_column, TRUE);

  if (cur_filter->servers && sortmode != SORT_SERVER_NONE) {
    redraw_server_clist (TRUE, FALSE);
    if (stat_process) {
      stat_process->redraw = FALSE;
    }
  }
}


static void player_clist_column_set_title (int col, int append) {
  char tmp[64];
  char *ptr;
  int sortmode = player_clist_columns[col].sortmode;

  if (append) {
    sprintf (tmp, "%s %c", player_clist_columns[col].name, 
                                                   (sortmode > 0)? '<' : '>');
    ptr = tmp;
  }
  else {
    ptr = player_clist_columns[col].name;
  }
  gtk_label_set (GTK_LABEL (player_clist_columns[col].widget), ptr);
}


static void player_clist_click_column_callback (GtkWidget *widget, int col) {
  int sortmode;

  sortmode = player_clist_columns[col].sortmode;

  player_clist_column_set_title (psort_column, FALSE);

  if (psort_column == col)
    player_clist_columns[col].sortmode = -sortmode;
  else 
    psort_column = col;

  player_clist_column_set_title (psort_column, TRUE);

  if (cur_server && sortmode != SORT_PLAYER_NONE)
    populate_player_clist (cur_server);
}


static void serverinfo_clist_column_set_title (int col, int append) {
  char tmp[64];
  char *ptr;
  int sortmode = serverinfo_clist_columns[col].sortmode;

  if (append) {
    sprintf (tmp, "%s %c", serverinfo_clist_columns[col].name, 
                                                   (sortmode > 0)? '<' : '>');
    ptr = tmp;
  }
  else {
    ptr = serverinfo_clist_columns[col].name;
  }
  gtk_label_set (GTK_LABEL (serverinfo_clist_columns[col].widget), ptr);
}


static void serverinfo_clist_click_column_callback (GtkWidget *widget, 
                                                                    int col) {
  int sortmode;

  sortmode = serverinfo_clist_columns[col].sortmode;

  serverinfo_clist_column_set_title (isort_column, FALSE);

  if (isort_column == col)
    serverinfo_clist_columns[col].sortmode = -sortmode;
  else 
    isort_column = col;

  serverinfo_clist_column_set_title (isort_column, TRUE);

  if (cur_server && sortmode != SORT_INFO_NONE)
    populate_serverinfo_clist (cur_server);
}



static void select_source_callback (GtkWidget *widget, struct master *m) {

  if (stat_process)
    return;

  if (m->host)
    print_status ("Source: %s [%s:%d].", m->name, m->host->address, m->port);
  else
    print_status ("Source: %s.", m->name);

  if (cur_master != m) {
    if (m->sources) {
      free_servers (m->servers);
      m->servers = collate_server_lists (NULL, m->sources, m->type);
    }
    cur_master = m;
    redraw_server_clist (FALSE, TRUE);
  }
}


static void update_master_callback (GtkWidget *widget, gpointer data) {
  GSList *masters = NULL;
  GSList *servers = NULL;
  GSList *unbound = NULL;
  GSList *tmp;
  struct master *m;

  if (stat_process || (!cur_master->host && !cur_master->sources))
    return;

  if (!cur_master->sources) {
    masters = g_slist_prepend (masters, cur_master);
  }
  else {
    for (tmp = cur_master->sources; tmp; tmp = tmp->next) {
      m = (struct master *) tmp->data;
      if (m->host)
	masters = g_slist_append (masters, m);
      else
	unbound = g_slist_append (unbound, m);
    }

    servers = collate_server_lists (NULL, unbound, cur_master->type);
    g_slist_free (unbound);
  }

  stat_lists (masters, servers, NULL);
}


static void refresh_selected_callback (GtkWidget *widget, gpointer data) {
  GSList *list;

  if (stat_process)
    return;

  list = server_clist_selected_servers ();
  if (list) {
    stat_lists (NULL, list, NULL);
  }
}


static void refresh_callback (GtkWidget *widget, gpointer data) {
  struct server *s;
  GSList *list = NULL;
  GSList *tmp;

  if (stat_process || !cur_master->servers)
    return;

  if (cur_master->sources) {
    free_servers (cur_master->servers);
    cur_master->servers = collate_server_lists (NULL, cur_master->sources, 
                                                            cur_master->type);
  }

  for (tmp = cur_master->servers; tmp; tmp = tmp->next) {
    s = (struct server *) tmp->data;
    list = g_slist_append (list, s);
    s->ref_count++;
  }

  stat_lists (NULL, list, NULL);
}


static void stop_callback (GtkWidget *widget, gpointer data) {
  if (stat_process) {
    stat_stop (stat_process);
    stat_process = NULL;
  }
}


static void add_to_favorites_callback (GtkWidget *widget, gpointer data) {
  GList *selected = GTK_CLIST (server_clist)->selection;
  GSList *list;
  GSList *tmp;

  if (stat_process || !selected || cur_master == favorites)
    return;

  list = server_clist_selected_servers ();
  if (list) {
    for (tmp = list; tmp; tmp = tmp->next) {
      add_server_to_master (favorites, (struct server *) tmp->data);
    }
    save_server_list (favorites);
    free_servers (list);
  }
}


static void real_add_server (struct server *s) {
  int row;

  add_server_to_master (cur_master, s);
  save_server_list (cur_master);

  if (cur_filter != &filters[0]) {
    free_filters_data ();
    set_filter_callback (NULL, &filters[0]);
  }
  else {
    server_clist_refresh_server (s);
  }

  row = slist_index (cur_filter->servers, s);
  server_clist_select_row (row, 0);
  server_clist_selection_visible ();

  stat_one_server (s);
}


struct resolve_data {
  int port;
  struct server *server;
};


static void add_server_host_handler (struct stat_job *job, struct host *h, 
                                                                char *idstr) {
  struct resolve_data *rd;

  if (h) {
    rd = (struct resolve_data *) stat_process->data;
    print_status ("%s is resolved to %s", idstr, h->address);
    rd->server = server_add (h, rd->port, cur_master->type);
    rd->server->ref_count++;
  }
  else {
    print_status (NULL);
    dialog_ok (NULL, "Host \"%s\" does not exist.", idstr);
  }
}


static void add_server_close_handler (struct stat_job *job, int killed) {
  struct resolve_data *rd;

  rd = (struct resolve_data *) stat_process->data;

  stat_job_free (stat_process);
  stat_process = NULL;

  if (rd) {
    if (main_window) {
      if (rd->server) {
	real_add_server (rd->server);
      }
      else {
	if (killed)
	  print_status ("Aborted.");
	set_widgets_sensitivity ();
      }
    }

    if (rd->server)
      server_unref (rd->server);
    g_free (rd);
  }
}


static void add_server_callback (GtkWidget *widget, gpointer data) {
  struct host *h;
  char *str;
  char *addr = NULL;
  int port;
  GSList *names;
  struct resolve_data *rd;
  struct server *s;

  if (stat_process || (cur_master != unbound_qw && cur_master != unbound_q2))
    return;

  str = enter_string_dialog ("Server:");
  if (!str || !*str)
    return;

  if (!parse_address (str, &addr, &port)) {
    dialog_ok (NULL, "\"%s\" is not valid host[:port] combination.", str);
    g_free (str);
    return;
  }

  g_free (str);

  h = host_add (addr);

  if (h) {
    g_free (addr);

    h->ref_count++;
    s = server_add (h, port, cur_master->type);
    real_add_server (s);
    host_unref (h);

    return;
  }

  names = g_slist_prepend (NULL, addr);

  stat_process = stat_job_create (NULL, NULL, NULL, names);
  stat_process->host_handlers = 
        g_slist_append (stat_process->host_handlers, add_server_host_handler);
  stat_process->close_handlers = 
     g_slist_prepend (stat_process->close_handlers, add_server_close_handler);

  rd = g_malloc (sizeof (struct resolve_data));
  rd->port = port;
  rd->server = NULL;
  stat_process->data = (gpointer) rd;

  print_status ("Resolving %s...", addr);

  stat_start (stat_process);
  set_widgets_sensitivity ();
}


static void del_server_callback (GtkWidget *widget, gpointer data) {
  GSList *list;
  GSList *tmp;

  if (stat_process || cur_master->host || cur_master->sources)
    return;

  list = server_clist_selected_servers ();
  if (list) {
    for (tmp = list; tmp; tmp = tmp->next) {
      remove_server_from_master (cur_master, (struct server *) tmp->data);
    }
    save_server_list (cur_master);
    free_servers (list);

    server_clist_remove_selected ();
  }
}


static void select_callback (GtkWidget *widget, gpointer data) {
  int invert = (int) data;

  if (!cur_filter || !cur_filter->servers)
    return;

  server_clist_select_all (invert);
}


static void show_hide_hostnames_callback (GtkWidget *widget, gpointer data) {
  GList *children;

  if (stat_process || !cur_master->servers)
    return;

  show_hostnames = TRUE - show_hostnames;

  children = gtk_container_children (GTK_CONTAINER (hostnames_menu_item));
  if (children && GTK_IS_LABEL (children->data)) {
    gtk_label_set (GTK_LABEL (children->data), show_hide[show_hostnames]);
  }

  children = gtk_container_children (
                                    GTK_CONTAINER (view_hostnames_menu_item));
  if (children && GTK_IS_LABEL (children->data))
    gtk_label_set (GTK_LABEL (children->data), show_hide[show_hostnames]);

  server_clist_show_hostnames ();
}


static void resolve_callback (GtkWidget *widget, gpointer data) {
  GSList *selected;
  GSList *hosts;
  struct server *s;

  if (stat_process)
    return;

  if (!show_hostnames)
    show_hide_hostnames_callback (NULL, NULL);

  selected = server_clist_selected_servers ();
  if (!selected)
    return;

  if (selected->next) {
    hosts = merge_hosts_to_resolve (NULL, selected);
  }
  else {
    s = (struct server *) selected->data;
    hosts = g_slist_prepend (NULL, s->host);
    s->host->ref_count++;
  }

  free_servers (selected);

  stat_lists (NULL, NULL, hosts);
}


static void properties_callback (GtkWidget *widget, gpointer data) {

  if (stat_process)
    return;

  if (cur_server) {
    properties_dialog (cur_server);
    set_widgets_sensitivity ();
  }
}


static void rcon_callback (GtkWidget *widget, gpointer data) {
  struct server_props *sp;
  char *passwd = NULL;
  int save = 0;

  if (stat_process || !cur_server)
    return;

  sp = properties (cur_server);

  if (!sp || !sp->rcon_password) {
    passwd = enter_string_with_option_dialog ("Save Password", &save, 
                                                          "Server Password:");
    if (!passwd)	/* canceled */
      return;

    if (save) {
      sp = properties_new (cur_server->host, cur_server->port);
      sp->rcon_password = passwd;
      props_save ();
      passwd = NULL;
    }
  }

  rcon_dialog (cur_server, (passwd)? passwd : sp->rcon_password);

  if (passwd)
    g_free (passwd);
}


static int server_clist_event_callback (GtkWidget *widget, GdkEvent *event) {
  GdkEventButton *bevent = (GdkEventButton *) event;
  GList *selection;
  int row;

  if (event->type == GDK_BUTTON_PRESS || 
      event->type == GDK_2BUTTON_PRESS ||
      event->type == GDK_3BUTTON_PRESS) {

    if (bevent->window == GTK_CLIST (server_clist)->clist_window) {

      if (bevent->type == GDK_BUTTON_PRESS && bevent->button == 3) {

	if (gtk_clist_get_selection_info (GTK_CLIST (server_clist), 
                                          bevent->x, bevent->y, &row, NULL)) {
	  selection = GTK_CLIST (server_clist)->selection;
	  if (!g_list_find (selection, (gpointer) row) && 
                 (bevent->state & (GDK_CONTROL_MASK | GDK_SHIFT_MASK)) == 0) {
	    server_clist_select_row (row, 0);
	  }
	}

	gtk_menu_popup (GTK_MENU (server_menu), NULL, NULL, NULL, NULL,
	                                        bevent->button, bevent->time);
      }
      else {
	if (gtk_clist_get_selection_info (GTK_CLIST (server_clist), 
                                          bevent->x, bevent->y, &row, NULL)) {

	  server_clist_select_row (row, bevent->state);

	  if (!(bevent->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) && 
                                                              !stat_process) {

	    if (bevent->type == GDK_2BUTTON_PRESS && bevent->button == 1)
	      launch_callback (NULL, LAUNCH_NORMAL);

	    if (bevent->type == GDK_BUTTON_PRESS && bevent->button == 2)
	      stat_one_server (cur_server);
	  }
	}
      }
      return TRUE;
    }
  }
  return FALSE;
}


static void player_skin_preview_popup_show (guchar *skin, 
                                          int top, int bottom, int x, int y) {
  GtkWidget *frame;
  int win_x, win_y, scr_w, scr_h;

  if (!player_skin_popup) {
    player_skin_popup = gtk_window_new (GTK_WINDOW_POPUP);
    gtk_window_set_policy (GTK_WINDOW (player_skin_popup), FALSE, FALSE, TRUE);
  
    frame = gtk_frame_new (NULL);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
    gtk_container_add (GTK_CONTAINER (player_skin_popup), frame);
    gtk_widget_show (frame);

    player_skin_popup_preview = gtk_preview_new (GTK_PREVIEW_COLOR);
    gtk_container_add (GTK_CONTAINER (frame), player_skin_popup_preview);
    gtk_preview_size (GTK_PREVIEW (player_skin_popup_preview), 320, 200);
    gtk_widget_show (player_skin_popup_preview);
  }
  else {
    gtk_widget_hide (player_skin_popup);
  }

  gdk_window_get_origin (GTK_CLIST (player_clist)->clist_window, 
                                                              &win_x, &win_y);
  x += win_x;
  y += win_y;
  scr_w = gdk_screen_width ();
  scr_h = gdk_screen_height ();
  x = (x + 320 > scr_w)? scr_w - 320 : x;
  y = (y + 200 > scr_h)? scr_h - 200 : y;

  gtk_widget_popup (player_skin_popup, x, y);

  draw_qw_skin (player_skin_popup_preview, skin, 
                     fix_qw_player_color (top), fix_qw_player_color (bottom));
}


static int player_clist_event_callback (GtkWidget *widget, GdkEvent *event) {
  GdkEventButton *bevent = (GdkEventButton *) event;
  guchar *skindata;
  char *skin;
  int row;

  switch (event->type) {

  case GDK_BUTTON_PRESS:
    if (bevent->window == GTK_CLIST (player_clist)->clist_window) {
      if (gtk_clist_get_selection_info (GTK_CLIST (player_clist), 
                                          bevent->x, bevent->y, &row, NULL)) {
	if (cur_server && cur_server->type == QW_SERVER && 
                                  row >= 0 && row <= cur_server->curplayers) {

	  skin = cur_server->players[row]->skin;
	  if (!skin || skin[0] == '\0') {
	    skin = "base";
	  }

	  skindata = get_skin (skin, real_quake_dir, FALSE);
	  gdk_pointer_grab (GTK_CLIST (player_clist)->clist_window, FALSE,
                              GDK_POINTER_MOTION_HINT_MASK |
                              GDK_BUTTON1_MOTION_MASK |
                              GDK_BUTTON_RELEASE_MASK,
                              NULL, NULL, bevent->time);
	  player_skin_preview_popup_show (skindata,
				       cur_server->players[row]->shirt,
				       cur_server->players[row]->pants,
				       bevent->x, bevent->y);
	  if (skindata)
	    g_free (skindata);
	}
      }
    }
    break;

  case GDK_BUTTON_RELEASE:
    if (player_skin_popup) {
      gdk_pointer_ungrab (bevent->time);
      gtk_widget_hide (player_skin_popup);
    }
    break;

  default:
    break;
  }

  return FALSE;
}


static void about_dialog (GtkWidget *widget, gpointer data) {
  dialog_ok ("About XQF", 
	     "X11 QuakeWorld/Quake2 FrontEnd %s\n"
	     "Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>\n\n"
	     "http://www.linuxgames.com/xqf/", XQF_VERSION);
}


struct menuitem {
  char *label;
  char accel_key;
  int accel_mods;
  GtkWidget **widget;
  GtkSignalFunc	callback;
  gpointer user_data;
};

static const struct menuitem srvopt_menu_items[] = {
  { 
    "Connect",		0,	0,	&connect_menu_item,
    GTK_SIGNAL_FUNC (launch_callback), (gpointer) LAUNCH_NORMAL
  },
  { 
    "Observe",		0,	0, 	&observe_menu_item,
    GTK_SIGNAL_FUNC (launch_callback), (gpointer) LAUNCH_SPECTATE
  },
  { 
    "Record Demo",	0,	0, 	&record_menu_item,
    GTK_SIGNAL_FUNC (launch_callback), (gpointer) LAUNCH_RECORD
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "Add to Favorites",	0,	0, 	&favadd_menu_item,
    GTK_SIGNAL_FUNC (add_to_favorites_callback), NULL
  },
  { 
    "Add...",		0,   	0,	&add_menu_item,
    GTK_SIGNAL_FUNC (add_server_callback), NULL
  },
  { 
    "Delete",		0,   	0, 	&delete_menu_item,
    GTK_SIGNAL_FUNC (del_server_callback), NULL
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "Refresh",		0,	0,   	&refresh_menu_item,
    GTK_SIGNAL_FUNC (refresh_callback), NULL
  },
  { 
    "Refresh Selected",	0,	0, 	&refrsel_menu_item,
    GTK_SIGNAL_FUNC (refresh_selected_callback), NULL
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "DNS Lookup",	0,	0, 	&resolve_menu_item,
    GTK_SIGNAL_FUNC (resolve_callback), NULL
  },
  { 
    "",			0,	0, 	&hostnames_menu_item,
    GTK_SIGNAL_FUNC (show_hide_hostnames_callback), NULL
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "Find Player...",	0,	0, 	&find_player_menu_item,
    GTK_SIGNAL_FUNC (find_player_callback), NULL
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "RCon...",   	0,	0, 	&rcon_menu_item,
    GTK_SIGNAL_FUNC (rcon_callback), 	NULL
  },
  { 
    "Properties...",   	0,	0, 	&properties_menu_item,
    GTK_SIGNAL_FUNC (properties_callback), NULL
  },
  { NULL, 		0, 	0, 	NULL, NULL, NULL }
};

static const struct menuitem file_menu_items[] = {
  { 
    "Quit",		'Q',	GDK_CONTROL_MASK, &file_quit_menu_item,
    NULL,		NULL
  },
  { NULL, 		0, 	0, 	NULL, NULL, NULL }
};

static const struct menuitem edit_menu_items[] = {
  { 
    "Add...",		'N',	GDK_CONTROL_MASK, &edit_add_menu_item,
    GTK_SIGNAL_FUNC (add_server_callback), NULL
  },
  { 
    "Delete",		'D',   	GDK_CONTROL_MASK, &edit_delete_menu_item,
    GTK_SIGNAL_FUNC (del_server_callback), NULL
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "Select All",	'A',   	GDK_CONTROL_MASK, &edit_selall_menu_item,
    GTK_SIGNAL_FUNC (select_callback), (gpointer) FALSE
  },
  { 
    "Invert Selection",	'I',   	GDK_CONTROL_MASK, &edit_invsel_menu_item,
    GTK_SIGNAL_FUNC (select_callback), (gpointer) TRUE
  },
  { NULL, 		0, 	0, 	NULL, NULL, NULL }
};

static const struct menuitem view_menu_items[] = {
  { 
    "Refresh",		'R',	GDK_CONTROL_MASK, &view_refresh_menu_item,
    GTK_SIGNAL_FUNC (refresh_callback), NULL
  },
  { 
    "Refresh Selected",	'S',	GDK_CONTROL_MASK, &view_refrsel_menu_item,
    GTK_SIGNAL_FUNC (refresh_selected_callback), NULL
  },
  { 
    "Update From Master", 'U',	GDK_CONTROL_MASK, &view_update_menu_item,
    GTK_SIGNAL_FUNC (update_master_callback), NULL
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "",			'H',	GDK_CONTROL_MASK, &view_hostnames_menu_item,
    GTK_SIGNAL_FUNC (show_hide_hostnames_callback), NULL
  },
  { NULL, 		0, 	0, 	NULL, NULL, NULL }
};

static const struct menuitem server_menu_items[] = {
  { 
    "Connect",		0,	0,	&server_connect_menu_item,
    GTK_SIGNAL_FUNC (launch_callback), (gpointer) LAUNCH_NORMAL
  },
  { 
    "Observe",		0,	0, 	&server_observe_menu_item,
    GTK_SIGNAL_FUNC (launch_callback), (gpointer) LAUNCH_SPECTATE
  },
  { 
    "Record Demo",	0,	0, 	&server_record_menu_item,
    GTK_SIGNAL_FUNC (launch_callback), (gpointer) LAUNCH_RECORD
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "Add to Favorites",	'F',	GDK_CONTROL_MASK, &server_favadd_menu_item,
    GTK_SIGNAL_FUNC (add_to_favorites_callback), NULL
  },
  { 
    "DNS Lookup",	'L',	GDK_CONTROL_MASK, &server_resolve_menu_item,
    GTK_SIGNAL_FUNC (resolve_callback), NULL
  },
  { "-", 		0, 	0, 	NULL, NULL, NULL },
  { 
    "RCon...",   	0,	0, 	&server_rcon_menu_item,
    GTK_SIGNAL_FUNC (rcon_callback), 	NULL
  },
  { 
    "Properties...",   	0,	0, 	&server_properties_menu_item,
    GTK_SIGNAL_FUNC (properties_callback), NULL
  },
  { NULL, 		0, 	0, 	NULL, NULL, NULL }
};

static const struct menuitem preferences_menu_items[] = {
  { 
    "Player Profile...",	0,	0, 	NULL,
    GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_PLAYER
  },
  { 
    "Gameplay...",	0,	0, 	NULL,
    GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_GAMEPLAY
  },
  { 
    "Filtering...",	0,	0, 	NULL,
    GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_FILTER
  },
  { 
    "General...",	0,	0, 	NULL,
    GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_GENERAL
  },
  { NULL, 		0, 	0, 	NULL, NULL, NULL }
};

static const struct menuitem help_menu_items[] = {
  { 
    "About...",		0,	0, 	NULL,
    GTK_SIGNAL_FUNC (about_dialog),	NULL
  },
  { NULL, 		0, 	0, 	NULL, NULL, NULL }
};


static GtkWidget *create_menu (const struct menuitem *items, 
                                           GtkAcceleratorTable *accel_table) {
  GtkWidget *menu;
  GtkWidget *menu_item;

  menu = gtk_menu_new ();
  
  while (items->label) {
    if (*items->label == '-') {		/* separator */
      menu_item = gtk_menu_item_new ();
      gtk_widget_set_sensitive (menu_item, FALSE);
      gtk_menu_append (GTK_MENU (menu), menu_item);
      gtk_widget_show (menu_item);
    }
    else {
      menu_item = gtk_menu_item_new_with_label (items->label);

      if (items->callback) {
	gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
 	                 GTK_SIGNAL_FUNC (items->callback), items->user_data);
      }

      if (accel_table && items->accel_key) {
	gtk_widget_install_accelerator (menu_item, accel_table, "activate",
                                         items->accel_key, items->accel_mods);
      }

      gtk_menu_append (GTK_MENU (menu), menu_item);
      gtk_widget_show (menu_item);

      if (items->widget)
	*items->widget = menu_item;
    }
    items++;
  }

  return menu;
}


static void append_source_menu_item (GtkWidget *menu, struct master *m) {
  GtkWidget *menu_item;
  GtkWidget *label;
  GtkWidget *pixmap;
  GtkWidget *hbox;

  if (m) {
    switch (m->type) {

    case QW_SERVER:
      pixmap = gtk_pixmap_new (q_pix, q_pix_mask);
      break;

    case Q2_SERVER:
      pixmap = gtk_pixmap_new (q2_pix, q2_pix_mask);
      break;

    default:
      pixmap = NULL;
    }

    if (pixmap) {
      menu_item = gtk_menu_item_new ();

      hbox = gtk_hbox_new (FALSE, 4);
      gtk_container_add (GTK_CONTAINER (menu_item), hbox);

      gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 0);
      gtk_widget_show (pixmap);

      label = gtk_label_new (m->name);
      gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
      gtk_widget_show (label);

      gtk_widget_show (hbox);
    }
    else {
      menu_item = gtk_menu_item_new_with_label (m->name);
    }

    gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
	              GTK_SIGNAL_FUNC (select_source_callback), (gpointer) m);
  }
  else {
    menu_item = gtk_menu_item_new ();		/* separator */
    gtk_widget_set_sensitive (menu_item, FALSE);
  }
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);
}


static GtkWidget *create_source_menu (void) {
  GtkWidget *menu;
  GSList *list;

  menu = gtk_menu_new ();

  append_source_menu_item (menu, favorites);

  append_source_menu_item (menu, NULL); 	/* separator */
  append_source_menu_item (menu, NULL); 	/* separator */

  append_source_menu_item (menu, qw_sources);

  append_source_menu_item (menu, NULL); 	/* separator */

  for (list = qw_masters; list; list = list->next) {
    append_source_menu_item (menu, (struct master *) list->data);
  }

  append_source_menu_item (menu, NULL); 	/* separator */

  append_source_menu_item (menu, unbound_qw);

  append_source_menu_item (menu, NULL); 	/* separator */
  append_source_menu_item (menu, NULL); 	/* separator */

  append_source_menu_item (menu, q2_sources);

  append_source_menu_item (menu, NULL); 	/* separator */

  for (list = q2_masters; list; list = list->next) {
    append_source_menu_item (menu, (struct master *) list->data);
  }

  append_source_menu_item (menu, NULL); 	/* separator */

  append_source_menu_item (menu, unbound_q2);

  append_source_menu_item (menu, NULL); 	/* separator */
  append_source_menu_item (menu, NULL); 	/* separator */

  append_source_menu_item (menu, allsources);

  return menu;
}


static GtkWidget *create_clist_coldef (int num, struct clist_coldef *cols) {
  GtkWidget *clist;
  GtkWidget *alignment;
  GtkWidget *label;
  int i;

  clist = gtk_clist_new (num);
  GTK_CLIST_SET_FLAGS (clist, CLIST_SHOW_TITLES);

  for (i = 0; i < num; i++) {
    gtk_clist_set_column_width (GTK_CLIST (clist), i, cols[i].width);
    if (cols[i].justify != GTK_JUSTIFY_LEFT) {
      gtk_clist_set_column_justification (GTK_CLIST (clist), i, 
                                                             cols[i].justify);
    }

    alignment = gtk_alignment_new (0.0, 0.5, 0.0, 0.0);

    label = gtk_label_new (cols[i].name);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
    gtk_container_add (GTK_CONTAINER (alignment), label);
    gtk_widget_show (label);

    cols[i].widget = label;

    gtk_clist_set_column_widget (GTK_CLIST (clist), i, alignment);
    gtk_widget_show (alignment);
  }

  return clist;
}


static GtkWidget *create_main_window_menu_bar (
                                           GtkAcceleratorTable *accel_table) {
  GtkWidget *menu;
  GtkWidget *root_menu;
  GtkWidget *menu_bar;
  GList *children;

  menu_bar = gtk_menu_bar_new ();

  /*
   *  File
   */

  root_menu = gtk_menu_item_new_with_label ("File");
  menu = create_menu (file_menu_items, accel_table);

  gtk_signal_connect_object (GTK_OBJECT (file_quit_menu_item), "activate",
	      GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (main_window));

  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  /* 
   *  Edit
   */

  root_menu = gtk_menu_item_new_with_label ("Edit");
  menu = create_menu (edit_menu_items, accel_table);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  /* 
   *  View
   */

  root_menu = gtk_menu_item_new_with_label ("View");
  menu = create_menu (view_menu_items, accel_table);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  children = gtk_container_children (
                                    GTK_CONTAINER (view_hostnames_menu_item));
  if (children && GTK_IS_LABEL (children->data))
    gtk_label_set (GTK_LABEL (children->data), show_hide[show_hostnames]);

  /* 
   *  Server
   */

  root_menu = gtk_menu_item_new_with_label ("Server");
  menu = create_menu (server_menu_items, accel_table);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  /*
   *  Preferences
   */

  root_menu = gtk_menu_item_new_with_label ("Preferences");
  menu = create_menu (preferences_menu_items, accel_table);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  /*
   *  Help
   */

  root_menu = gtk_menu_item_new_with_label ("Help");
  menu = create_menu (help_menu_items, accel_table);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_item_right_justify (GTK_MENU_ITEM (root_menu));
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  return menu_bar;
}


static GdkColor tooltips_color;

static void set_tooltips_color (GtkWidget *window, GtkTooltips *tooltips) {
  GdkColormap *colormap;

  tooltips_color.red = 65535;
  tooltips_color.green = 65535;
  tooltips_color.blue = 53052;

  colormap = gdk_window_get_colormap (window->window);
  gdk_color_alloc (colormap, &tooltips_color);
  gtk_tooltips_set_colors (tooltips, &tooltips_color, 
                                        &window->style->fg[GTK_STATE_NORMAL]);
}


void create_main_window (void) {
  GtkWidget *main_vbox;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *hpaned;
  GtkWidget *vpaned;
  GtkWidget *button;
  GtkWidget *menu_bar;
  GtkWidget *label;
  GtkWidget *hbox2;
  GtkWidget *vseparator;
  GtkWidget *arrow;
  GtkWidget *handlebox;
  GtkTooltips *tooltips;
  GtkAcceleratorTable *accel_table;
  struct filter *filter;
  GList *children;

  accel_table = gtk_accelerator_table_new ();

  server_menu = create_menu (srvopt_menu_items, accel_table);

  children = gtk_container_children (GTK_CONTAINER (hostnames_menu_item));
  if (children && GTK_IS_LABEL (children->data)) {
    gtk_label_set (GTK_LABEL (children->data), show_hide[show_hostnames]);
  }

  main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
		      GTK_SIGNAL_FUNC (window_delete_event_callback), NULL);
  gtk_signal_connect (GTK_OBJECT (main_window), "destroy",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_window_set_title (GTK_WINDOW (main_window), "XQF");

  register_window (main_window);

  gtk_widget_realize (main_window);
  init_pixmaps (main_window);

  main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (main_window), main_vbox);

  handlebox = gtk_handle_box_new ();

  menu_bar = create_main_window_menu_bar (accel_table);
  gtk_container_add (GTK_CONTAINER (handlebox), menu_bar);
  gtk_widget_show (menu_bar);

  gtk_box_pack_start (GTK_BOX (main_vbox), handlebox, FALSE, FALSE, 0);
  gtk_widget_show (handlebox);

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_box_pack_start (GTK_BOX (main_vbox), vbox, TRUE, TRUE, 0);

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  /*
   *  Source
   */

  label = gtk_label_new ("Source");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  source_option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (source_option_menu), 
                                                       create_source_menu ());
  gtk_option_menu_set_history (GTK_OPTION_MENU (source_option_menu), 0);
  gtk_box_pack_start (GTK_BOX (hbox), source_option_menu, FALSE, FALSE, 0);
  gtk_widget_show (source_option_menu);   

  tooltips = gtk_tooltips_new ();

  update_button = gtk_button_new_with_label (" U ");
  gtk_box_pack_start (GTK_BOX (hbox), update_button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (update_button), "clicked",
		      GTK_SIGNAL_FUNC (update_master_callback), NULL);
  gtk_widget_show (update_button);

  gtk_tooltips_set_tip (tooltips, update_button, 
                                      "Update server list from master", NULL);

  refresh_button = gtk_button_new_with_label (" R ");
  gtk_box_pack_start (GTK_BOX (hbox), refresh_button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (refresh_button), "clicked",
		      GTK_SIGNAL_FUNC (refresh_callback), NULL);
  gtk_widget_show (refresh_button);

  gtk_tooltips_set_tip (tooltips, refresh_button, 
                                                "Refresh current list", NULL);

  refrsel_button = gtk_button_new_with_label ("RS");
  gtk_box_pack_start (GTK_BOX (hbox), refrsel_button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (refrsel_button), "clicked",
		      GTK_SIGNAL_FUNC (refresh_selected_callback), NULL);
  gtk_widget_show (refrsel_button);

  gtk_tooltips_set_tip (tooltips, refrsel_button, 
                                            "Refresh selected servers", NULL);

  stop_button = gtk_button_new_with_label (" S ");
  gtk_box_pack_start (GTK_BOX (hbox), stop_button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (stop_button), "clicked",
	 	      GTK_SIGNAL_FUNC (stop_callback), NULL);
  gtk_widget_show (stop_button);

  gtk_tooltips_set_tip (tooltips, stop_button, "Stop", NULL);

  /*
   * Separator 
   */

  hbox2 = gtk_hbox_new (FALSE, 1);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 8);

  vseparator = gtk_vseparator_new (); 
  gtk_box_pack_start (GTK_BOX (hbox2), vseparator, FALSE, FALSE, 0);
  gtk_widget_show (vseparator);

  vseparator = gtk_vseparator_new (); 
  gtk_box_pack_start (GTK_BOX (hbox2), vseparator, FALSE, FALSE, 0);
  gtk_widget_show (vseparator);

  gtk_widget_show (hbox2);

  /*
   *  Filter buttons
   */

  filter_buttons = NULL;

  for (filter = filters; filter->name; filter++) {
    button = gtk_button_new ();
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);

    hbox2 = gtk_hbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (button), hbox2);

    arrow = gtk_arrow_new (GTK_ARROW_RIGHT, 
                      (filter == cur_filter)? GTK_SHADOW_IN : GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (hbox2), arrow, FALSE, FALSE, 2);
    gtk_widget_show (arrow);

    filter->widget = arrow;

    label = gtk_label_new (filter->name);
    gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 2);
    gtk_widget_show (label);

    gtk_widget_show (hbox2);

    if (filter != &filters[3]) {
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (set_filter_callback), (gpointer) filter);
    }
    else {
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (find_player_callback), NULL);
    }
    gtk_widget_show (button);

    filter_buttons = g_slist_prepend (filter_buttons, button);
  }

  gtk_widget_show (hbox);

  vpaned = gtk_vpaned_new ();
  gtk_box_pack_start (GTK_BOX (vbox), vpaned, TRUE, TRUE, 0);

  /*
   *  Server CList
   */

  server_clist = create_clist_coldef (7, server_clist_columns);
  gtk_clist_set_selection_mode (GTK_CLIST (server_clist), 
                                                      GTK_SELECTION_MULTIPLE);
  gtk_widget_set_usize (server_clist, 750, 260);
  gtk_clist_set_policy (GTK_CLIST (server_clist), 
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (server_clist), "click_column",
                  GTK_SIGNAL_FUNC (server_clist_click_column_callback), NULL);
  gtk_signal_connect (GTK_OBJECT(server_clist), "event",
                         GTK_SIGNAL_FUNC (server_clist_event_callback), NULL);
  gtk_paned_add1 (GTK_PANED (vpaned), server_clist);
  gtk_widget_show (server_clist);

  server_clist_column_set_title (ssort_column, TRUE);

  gtk_clist_set_row_height (GTK_CLIST (server_clist), 
                            MAX (pixmap_height (q_pix), 
				 server_clist->style->font->ascent + 
				 server_clist->style->font->descent + 1));

  hpaned = gtk_hpaned_new ();
  gtk_paned_add2 (GTK_PANED (vpaned), hpaned);

  /*
   *  Player CList
   */

  player_clist = create_clist_coldef (6, player_clist_columns);
  gtk_clist_set_selection_mode (GTK_CLIST (player_clist), 
                                                      GTK_SELECTION_EXTENDED);
  gtk_widget_set_usize (player_clist, 480, 220);
  gtk_clist_set_policy (GTK_CLIST (player_clist),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (player_clist), "click_column",
                  GTK_SIGNAL_FUNC (player_clist_click_column_callback), NULL);
  gtk_signal_connect (GTK_OBJECT(player_clist), "event",
                         GTK_SIGNAL_FUNC (player_clist_event_callback), NULL);
  gtk_paned_add1 (GTK_PANED (hpaned), player_clist);
  gtk_widget_show (player_clist);

  player_clist_column_set_title (psort_column, TRUE);

  /*
   *  Server Info CList
   */

  serverinfo_clist = create_clist_coldef (2, serverinfo_clist_columns);
  gtk_clist_set_selection_mode (GTK_CLIST (serverinfo_clist), 
                                                      GTK_SELECTION_EXTENDED);
  gtk_widget_set_usize (serverinfo_clist, 270, 220);
  gtk_clist_set_policy (GTK_CLIST (serverinfo_clist),
		 	          GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (serverinfo_clist), "click_column",
              GTK_SIGNAL_FUNC (serverinfo_clist_click_column_callback), NULL);
  gtk_paned_add2 (GTK_PANED (hpaned), serverinfo_clist);
  gtk_widget_show (serverinfo_clist);

  serverinfo_clist_column_set_title (isort_column, TRUE);

  gtk_widget_show (hpaned);
  gtk_widget_show (vpaned);

  /*
   *  Status Bar & Progress Bar
   */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  status_bar = gtk_statusbar_new ();
  gtk_box_pack_start (GTK_BOX (hbox), status_bar, TRUE, TRUE, 0);
  gtk_widget_show (status_bar);

  status_bar_context_id = gtk_statusbar_get_context_id (
					   GTK_STATUSBAR (status_bar), "XQF");

  progress_bar = gtk_progress_bar_new ();
  gtk_widget_set_usize (progress_bar, 200, -1);
  gtk_box_pack_end (GTK_BOX (hbox), progress_bar, FALSE, FALSE, 0);
  gtk_widget_show (progress_bar);

  gtk_widget_show (hbox);

  gtk_widget_show (vbox);

  gtk_widget_show (main_vbox);

  gtk_widget_show (main_window);

  gtk_window_add_accelerator_table (GTK_WINDOW (main_window), accel_table);
  gtk_accelerator_table_unref (accel_table);

  set_tooltips_color (main_window, tooltips);
}


static int init_user_info (void) {
  user.name = get_user_name ();

  user.home = get_user_home ();
  if (user.home == NULL) {
    fprintf (stderr, "You are homeless!\n");
    return 0;
  }

  user.rcdir  = file_in_dir (user.home, RC_DIR);
  return 1;
}


static void free_user_info (void) {
  if (user.name)    { g_free (user.name); user.name = NULL; }
  if (user.home)    { g_free (user.home); user.home = NULL; }
  if (user.rcdir)   { g_free (user.rcdir); user.rcdir = NULL; }
}


static void init_styles (void) {
  char *local_gtk_config;

  gtk_preview_set_gamma (1.2);

  gtk_rc_parse ("/etc/gtkrc");

  local_gtk_config = file_in_dir (user.home, ".gtkrc");
  gtk_rc_parse (local_gtk_config);
  g_free (local_gtk_config);

  local_gtk_config = file_in_dir (user.rcdir, "gtkrc");
  gtk_rc_parse (local_gtk_config);
  g_free (local_gtk_config);
}


int main (int argc, char *argv[]) {

  gtk_init (&argc, &argv);

  if (!init_user_info ())
    return 1;

  rc_check_dir ();
  init_styles ();

  if (rc_parse () < 0)
    user_fix_defaults ();

  init_masters ();
  props_load ();

  client_init ();
  ignore_sigpipe ();

  create_main_window ();

  select_source_callback (NULL, favorites);
  print_status (NULL);

  if (default_auto_favorites)
    refresh_callback (NULL, NULL);

  gtk_main ();

  unregister_window (main_window);
  main_window = NULL;

  if (stat_process)
    stop_callback (NULL, NULL);

  if (server_menu) {
    gtk_widget_destroy (server_menu);
    server_menu = NULL;
  }

  if (player_skin_popup) {
    gtk_widget_destroy (player_skin_popup);
    player_skin_popup = NULL;
  }

  free_filters_data ();
  free_masters ();
  props_free_all ();
  free_user_info ();
  free_pixmaps ();
  psearch_free_data ();
  rcon_history_free ();
  client_detach_all ();

#ifdef DEBUG
  fprintf (stderr, "total servers: %d\n", servers_total ());
  fprintf (stderr, "total hosts: %d\n", hosts_total ());
#endif

  if (con && con->server) {
    launch_quake (con, FALSE);
    server_unref (con->s);
    condef_free (con);
    con = NULL;
  }

  return 0;
}

