/*
    CrossFire, A Multiplayer game for X-windows

    Copryight (C) 1994 Mark Wedel
    Copyright (C) 1992 Frank Tore Johansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be reached via e-mail to master@rahul.net
*/

#include <ctype.h>	/* needed for isdigit */
#include <includes.h>	/* prototypes most standard functions */
#include <client.h>
#include <item.h>
#include <newclient.h>

static item *free_items;	/* the list of free (unused) items */
static item *player, *map;	/* these lists contains rest of items */
				/* player = pl->ob, map = pl->below */

#define NROF_ITEMS 50		/* how many items are reserved initially */
				/* for the item spool */


/* Stolen from common/item.c */
/*
 * get_number(integer) returns the text-representation of the given number
 * in a static buffer.  The buffer might be overwritten at the next
 * call to get_number().
 * It is currently only used by the query_name() function.
 */

char *get_number(int i) {
static char numbers[21][20] = {
  "no","a","two","three","four","five","six","seven","eight","nine","ten",
  "eleven","twelve","thirteen","fourteen","fifteen","sixteen","seventeen",
  "eighteen","nineteen","twenty"
};


  if(i<=20)
    return numbers[i];
  else {
    static char buf[MAX_BUF];
    sprintf(buf,"%d",i);
    return buf;
  }
}


/*
 *  new_item() returns pointer to new item which
 *  is allocated and initialized correctly
 */
static item *new_item () 
{
    item *op = malloc (sizeof(item));

    if (! op) 
	exit(0);

    op->next = op->prev = NULL;
    copy_name (op->name, "");
    copy_name (op->o_name, "");
    op->inv = NULL;
    op->env = NULL;
    op->tag = 0;
    op->face = 0;
    op->weight = 0;
    op->magical = op->cursed = op->damned = 0;
    op->unpaid = op->locked = op->applied = 0;
    op->flagsval=0;
    op->animation_id=0;
    op->last_anim=0;
    op->anim_state=0;
    op->nrof=0;
    op->open=0;
    return op;
};

/*
 *  alloc_items() returns pointer to list of allocated objects
 */
static item *alloc_items (int nrof) {
    item *op, *list;
    int i;

    list = op = new_item();

    for (i=1; i<nrof; i++) {
	op->next = new_item();
	op->next->prev = op;
	op = op->next;
    }
    return list;
}

/*
 *  free_items() frees all allocated items from list
 */
void free_all_items (item *op) {
    item *tmp;

    while (op) {
	if (op->inv)
	    free_all_items (op->inv);  
	tmp = op->next;
	free(op);
	op = tmp;
    }
}

/*
 *  Recursive function, used by locate_item()
 */
static item *locate_item_from_item (item *op, sint32 tag)
{
    item *tmp;

    for (; op; op=op->next)
	if (op->tag == tag)
	    return op;
	else if (op->inv && (tmp = locate_item_from_item (op->inv, tag)))
	    return tmp;

    return NULL;
}

/*
 *  locate_item() returns pointer to the item which tag is given 
 *  as parameter or if item is not found returns NULL
 */
item *locate_item (sint32 tag)
{
    item *op;

    if (tag == 0)
	return map;

    if ((op=locate_item_from_item(map->inv, tag)) != NULL)
	return op;
    if ((op=locate_item_from_item(player, tag)) != NULL)
	return op;

    return NULL;
}

/*
 *  remove_item() inserts op the the list of free items
 *  Note that it don't clear all fields in item
 */
void remove_item (item *op) 
{
    /* IF no op, or it is the player */
    if (!op || op==player || op==map) return;
    op->env->inv_updated = 1;

    /* Do we really want to do this? */
    if (op->inv)
	remove_item_inventory (op);

    if (op->prev) {
	op->prev->next = op->next;
    } else {
	op->env->inv = op->next;
    }
    if (op->next) {
	op->next->prev = op->prev;
    }

    /* add object to a list of free objects */
    op->next = free_items;
    if (op->next != NULL)
	op->next->prev = op;
    free_items = op;
    op->prev = NULL;
    op->env = NULL;
    op->tag = 0;
}

/*
 *  remove_item_inventory() recursive frees items inventory
 */
void remove_item_inventory (item *op)
{
    op->inv_updated = 1;
    while (op->inv)
	remove_item (op->inv);
}

/*
 *  add_item() adds item op to end of the inventory of item env
 */
static void add_item (item *env, item *op) 
{
    item *tmp;
    
    for (tmp = env->inv; tmp && tmp->next; tmp=tmp->next)
	;

    op->next = NULL;
    op->prev = tmp;
    op->env = env;
    if (!tmp) {
	env->inv = op;
    } else {
	if (tmp->next)
	    tmp->next->prev = op;
	tmp->next = op;
    }
/*    fprintf(stderr,"Added object %s to %s\n", op->name, env->name);*/
}

/*
 *  create_new_item() returns pointer to a new item, inserts it to env 
 *  and sets its tag field and clears locked flag (all other fields
 *  are unitialized and may contain random values)
 */
item *create_new_item (item *env, sint32 tag)
{
    item *op;

    if (!free_items)
	free_items = alloc_items (NROF_ITEMS);

    op = free_items;
    free_items = free_items->next;
    if (free_items)
	free_items->prev = NULL;

    op->tag = tag;
    op->locked = 0;
    add_item (env, op);
    return op;
}

/*
 *  Hardcoded now, server could send these at initiation phase.
 */
static char *apply_string[] = {
    "", " (readied)", " (wielded)", " (worn)", " (active)", " (applied)"
};

static void set_flag_string (item *op)
{
    op->flags[0] = 0;

    if (op->locked) 
	strcat (op->flags, " *");
    if (op->apply_type) {
	if (op->apply_type <= sizeof (apply_string) / sizeof(apply_string[0])) 
	    strcat (op->flags, apply_string[op->apply_type]);
	else 
	    strcat (op->flags, " (undefined)");
    }
    if (op->open)
	strcat (op->flags, " (open)");
    if (op->damned)
	strcat (op->flags, " (damned)");
    if (op->cursed)
	strcat (op->flags, " (cursed)");
    if (op->magical)
	strcat (op->flags, " (magic)");
    if (op->unpaid)
	strcat (op->flags, " (unpaid)");
}

static void get_flags (item *op, uint16 flags)
{
    op->was_open = op->open;
    op->open    = flags & F_OPEN    ? 1 : 0;
    op->damned  = flags & F_DAMNED  ? 1 : 0;
    op->cursed  = flags & F_CURSED  ? 1 : 0;
    op->magical = flags & F_MAGIC   ? 1 : 0;
    op->unpaid  = flags & F_UNPAID  ? 1 : 0;
    op->applied = flags & F_APPLIED ? 1 : 0;
    op->locked  = flags & F_LOCKED  ? 1 : 0;
    op->flagsval= flags;
    op->apply_type = flags & F_APPLIED;
    set_flag_string(op);
}


/*
 *  get_nrof() functions tries to get number of items from the item name
 */
static sint32 get_nrof(char *name) 
{
    static char *numbers[21] = {
	"no ","a ","two ","three ","four ","five ","six ","seven ","eight ",
	"nine ","ten ","eleven ","twelve ","thirteen ","fourteen ","fifteen ",
	"sixteen ","seventeen ","eighteen ","nineteen ","twenty "
    };
    static char *numbers_10[10] = {
	"zero ","ten ","twenty ","thirty ","fourty ","fifty ","sixty ",
	"seventy ","eighty ","ninety "
    };
    sint32 nrof = 0;
    int i;

    if (isdigit (*name))
	nrof = atol (name);
    else if (strncmp (name, "a ", 2) == 0 || strncmp (name, "an ", 3) == 0)
	nrof = 1;
    else {
	for (i=1; i<sizeof(numbers)/sizeof(numbers[0]); i++)
	    if (strncmp (name, numbers[i], strlen (numbers[i])) == 0) {
		nrof = i;
		break;
	    }
	if ( !nrof ) {
	    for (i=1; i<sizeof(numbers_10)/sizeof(numbers_10[0]); i++)
		if (strncmp(name, numbers_10[i], strlen(numbers_10[i])) == 0) {
		    nrof = i * 10;
		    break;
		}
	}
    }
    
    return nrof ? nrof : 1; 
}

void set_item_values (item *op, char *name, sint32 weight, uint16 face, 
		      uint16 flags, uint16 anim, uint16 animspeed,
		      sint32 nrof) 
{
    if (!op) {
	printf ("Error in set_item_values(): item pointer is NULL.\n");
	return;
    }
    if (nrof<0) {
	copy_name (op->name, name);
	op->nrof = get_nrof(name);
    } else { /* we have a nrof - item1 command */
	int need_new_name=0;

	/* Program always expect at least 1 object internall */
	if (nrof==0) nrof=1;

	/* Little hack to force it to make a new name if nrof changes */
	if (nrof != op->nrof) need_new_name=1;

	op->nrof = nrof;

	/* Bunch of hacks here.  First, the UpdItem command passes the
	 * same name to us, and nrof has been set by the first item command.
	 * So, if the name passed matches the name we used, don't 
	 * recopy..
	 */
	if (strcmp(name, op->name)) {
	    copy_name (op->o_name, name);
	    sprintf(op->name,"%s %s", get_number(nrof), op->o_name);
	}
	else if (need_new_name)
	    sprintf(op->name,"%s %s", get_number(nrof), op->o_name);
    }

    op->env->inv_updated = 1;
    op->weight = (float) weight / 1000;
    op->face = face;
    op->animation_id = anim;
    op->anim_speed=animspeed;
    get_flags (op, flags);
}

void toggle_locked (item *op)
{
    SockList sl;
    char    buf[MAX_BUF];

    if (op->env->tag == 0)
	return;	/* if item is on the ground, don't lock it */

    sl.buf = buf;
    strcpy(sl.buf, "lock ");
    sl.len=5;
    if (op->locked) sl.buf[sl.len++]=0;
    else sl.buf[sl.len++]=1;
    SockList_AddInt(&sl, op->tag);
    send_socklist(csocket.fd, sl);
}

void send_mark_obj (item *op) {
    SockList sl;
    char    buf[MAX_BUF];

    if (op->env->tag == 0)
	return;	/* if item is on the ground, don't mark it */

    sl.buf = buf;
    strcpy(sl.buf, "mark ");
    sl.len=5;
    SockList_AddInt(&sl, op->tag);
    send_socklist(csocket.fd, sl);
}


item *player_item ()
{
    player = new_item(); 
    return player;
}

item *map_item ()
{
    map = new_item();
    map->weight = -1;
    return map;
}

/* This makes debugging much easier. declare it static so it isn't
 * multiple defined at the link stage.
 */
void update_item(int tag, int loc, char *name, int weight, int face, int flags,
		 int anim, int animspeed, int nrof)
{
    item *ip = locate_item(tag), *env=locate_item(loc);

    /* Need to do some special handling if this is the player that is
     * being updated.
     */
    if (player->tag==tag) {
	copy_name (player->name, name);
	player->nrof = get_nrof(name);
	player->weight = (float) weight / 1000;
	player->face = face;
	get_flags (player, flags);
	player->inv->inv_updated = 1;
	player->animation_id = anim;
	player->anim_speed = animspeed;
	player->nrof = nrof;
    }
    else { 
	if (ip && ip->env != env) {
	    remove_item(ip);
	    ip=NULL;
	}
	set_item_values(ip?ip:create_new_item(env,tag), name, weight, face, flags,
			anim, animspeed,nrof);
    }
}


/*
 *  Prints players inventory, contain extra information for debugging purposes
 */
void print_inventory (item *op)
{
    char buf[MAX_BUF];
    char buf2[MAX_BUF];
    item *tmp;
    static int l = 0;
    int info_width = get_info_width();

    if (l == 0) {
	sprintf (buf, "%s's inventory (%d):", op->name, op->tag);
	sprintf (buf2, "%-*s%6.1f kg", info_width - 10, buf, op->weight);
	draw_info (buf2,NDI_BLACK);
    }

    l += 2;
    for (tmp = op->inv; tmp; tmp=tmp->next) {
	sprintf (buf, "%*s- %d %s%s (%d)", l - 2, "", tmp->nrof, tmp->name,
		 tmp->flags, tmp->tag);
	sprintf (buf2, "%-*s%6.1f kg", info_width - 8 - l, buf, tmp->nrof*tmp->weight);
	draw_info (buf2,NDI_BLACK);
	if (tmp->inv)
	    print_inventory (tmp);
    }
    l -= 2;
}

/* Check the objects, animate the ones as necessary */
void animate_objects()
{
    item *ip;
    int got_one=0;

    /* For now, only the players inventory needs to be animated */
    for (ip=player->inv; ip; ip=ip->next) {
	if (ip->animation_id>0 && ip->anim_speed) {
	    ip->last_anim++;
	    if (ip->last_anim>=ip->anim_speed) {
		ip->anim_state++;
		if (ip->anim_state >= animations[ip->animation_id].num_animations)
		    ip->anim_state=0;
		ip->face = animations[ip->animation_id].faces[ip->anim_state];
		ip->last_anim=0;
		got_one=1;
	    }
	}
    }
    if (got_one) player->inv_updated=1;
}

