//  UString.cpp version 1.1
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998  Gaspar Sinai
// 
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//
//
//

#include "UString.h"
#include "UCache.h"
#include <string.h>

static 		UCache<UCS2Converter*>* cache=0;

UString::~UString ()
{
	if (converterName)
	{
		cache->unuseItem (converterName);
		delete converterName;
	}
	if (string) delete string;
}

UString::UString ()
{
	converterName=0;
	converter=0;
	string=0;
	setConverter (DEFAULT_CONVERTER);
}


UString::UString (const unsigned char* str)
{
	converterName=0;
	converter=0;
	string=0;
	setConverter (DEFAULT_CONVERTER);
	string = decode (str);
}

UString::UString (const UCS2* str)
{
	converterName=0;
	converter=0;
	string=0;
	setConverter (DEFAULT_CONVERTER);
	string = new UCS2[size (str) +1];
	CHECKNULL (string);
	memcpy (string, str, sizeof (UCS2) * size (str)+1);
}

//
// This one sets the converter and adds the string.
//
UString::UString (const unsigned char* str, const char* converterIn)
{
	converterName=0;
	converter=0;
	string=0;
	setConverter (converterIn);
	string = decode (str);
}

//
// setting a unititialized converter
//
UString::UString (UCS2Converter* converterIn, const char *converterNameIn) 
{
	string=0;
	converter=0;
	converterName=0;
	if (addConverter (converterIn, converterNameIn) != OK)
	{
		setConverter (DEFAULT_CONVERTER);
	}
}

UString::UStatus
UString::setConverter (const char* nameIn)
{
	U8BitConv		*deflt8;
	UMap			*map8;

	if (cache == 0)
	{
		cache = new UCache<UCS2Converter*>;
		CHECKNULL (cache);
		UInitUString ();
	}
	if (converterName)
	{
		cache->unuseItem (converterName);
		delete converterName;
		converterName=0;
	}
	converter = cache->getItem (nameIn);
	if (converter != 0)
	{
		converterName = new char[ strlen (nameIn) +1];
		CHECKNULL (converterName);
		strcpy (converterName, nameIn);
		cache->useItem (converterName);
		return OK;
	}
	// Try to load it as an 8bit converter. It will 
	// succeed if we can get the map.
	map8 = new UMap();
	CHECKNULL (map8);
	if (map8->rename (nameIn)==UMap::ERROR)
	{
		cerr << "error: can not load '" << nameIn 
			<< "' even as an 8bit umap. Loading default.\n";
		setConverter (DEFAULT_CONVERTER);
		delete map8;
		return ERROR;
	}
	deflt8 = new U8BitConv (nameIn);
	CHECKNULL (deflt8);
	if (addConverter (deflt8, nameIn) != OK)
	{
		setConverter (DEFAULT_CONVERTER);
		delete map8;
		delete deflt8;
		return ERROR;
	}
	return OK;
}


unsigned char*
UString::getString ()
{
	return encode (string);
}

UCS2*
UString::getUString ()
{
	UCS2*	ret_vle;

	if (string==0) return 0;
	ret_vle =  new UCS2 [size (string)+1];
	CHECKNULL (ret_vle);
	memcpy (ret_vle, string, sizeof (UCS2) * (size(string) +1));
	return ret_vle;
}

void 
UString::putString (const unsigned char* str)
{
	if (string) delete string;
	string=decode (str);
}

void 
UString::putUString (const UCS2* str)
{
	if (string) delete string;
	if (str==0)
	{
		string =  new UCS2[1];
		CHECKNULL (string);
		string[0] = 0;
		return;
	}
	string =  new UCS2[size (str)+1];
	CHECKNULL (string);
	memcpy (string, str, sizeof (UCS2) * (size (str)+1));
}

//------------------------------------------------------------------------------
// private functions
//------------------------------------------------------------------------------
int
UString::size (const UCS2* str) const
{
	register int	i=0;
	while (str[i]) i++;
	return i;
}

unsigned char *
UString::encode (const UCS2* str) 
{
	if (converter == 0) return 0;
	return converter->encode (str);
}

UCS2 *
UString::decode (const unsigned char* str)
{
	if (converter == 0) return 0;
	return converter->decode (str);
}

//
// add a new converter
//
UString::UStatus
UString::addConverter (UCS2Converter* converterIn, const char* nameConverter)
{
	UCache<UCS2Converter*>::UStatus	status;

	if (cache == 0)
	{
		cache = new UCache<UCS2Converter*>;
		CHECKNULL (cache);
		UInitUString ();
	}
	
	if (converterName) delete converterName;
	converterName = new char[strlen (nameConverter) +1];
	CHECKNULL (converterName);
	strcpy (converterName, nameConverter);

	status = cache->addItem (converterName, converterIn);
	converter = cache->getItem (converterName);
	switch (status)
	{
	case UCache<UCS2Converter*>::DUPLICATE:
			delete converterIn;
			cache->useItem (converterName);
	case UCache<UCS2Converter*>::OK:
			return OK;
	case UCache<UCS2Converter*>::ERROR:
			(void) setConverter (DEFAULT_CONVERTER);
			return ERROR;
	}
	return ERROR;
}

UString::operator
const UCS2 * () const
{
	return string;
}

//------------------------------------------------------------------------------
// Defaults. we could have saved some space if we did it in a init routine.
//------------------------------------------------------------------------------
static UCS2Converter*	_utf8_conv;
static UString*		_utf8_string;
static UUTF7Conv*	_utf7_conv;
static UString*		_utf7_string;
static UJavaConv*	_java_conv;
static UString*		_java_string;
static UJISConv*	_jis_conv;
static UString*		_jis_string;
static USJISConv*	_sjis_conv;
static UString*		_sjis_string;
static UJISConv*	_iso_2022_conv;
static UString*		_iso_2022_string;
static UEUC_JPConv*	_euc_jp_conv;
static UString*		_euc_jp_string;
static CTEXT_JAConv*	_ctext_ja_conv;
static UString*		_ctext_ja_string;

static UKSC5601Conv*	_ksc5601_conv;
static UString*		_ksc5601_string;
static UEUC_KRConv*	_euc_kr_conv;
static UString*		_euc_kr_string;
static UJOHABConv*	_johab_conv;
static UString*		_johab_string;
static UUHCConv*	_uhc_conv;
static UString*		_uhc_string;

static UGB2312_7Conv*	_gb2312_7_conv;
static UString*		_gb2312_7_string;
static UGB2312_8Conv*	_gb2312_8_conv;
static UString*		_gb2312_8_string;

static UHZConv*		_hz_conv;
static UString*		_hz_string;

static UBIG5Conv*	_big5_conv;
static UString*		_big5_string;

static U16BEConv*	_u16be_conv;
static UString*		_u16be_string;
static U16LEConv*	_u16le_conv;
static UString*		_u16le_string;
static UUniConv*	_uni_conv;
static UString*		_uni_string;
static UMSTXTConv*	_mstxt_conv;
static UString*		_mstxt_string;

void
UInitUString ()
{
	if (_utf8_conv!=0) return;

	_utf8_conv = new UCS2Converter();
	CHECKNULL (_utf8_conv);
	_utf8_string = new UString (_utf8_conv, "UTF8");
	CHECKNULL (_utf8_string);

	_utf7_conv = new UUTF7Conv();
	CHECKNULL (_utf7_conv);
	_utf7_string = new UString (_utf7_conv, "UTF7");
	CHECKNULL (_utf7_string);

	_java_conv = new UJavaConv();
	CHECKNULL (_java_conv);
	_java_string = new UString (_java_conv, "JAVA");
	CHECKNULL (_java_string);


	_jis_conv = new UJISConv();
	CHECKNULL (_jis_conv);
	_jis_string = new UString (_jis_conv, "JIS");
	CHECKNULL (_jis_string);

	_sjis_conv = new USJISConv();
	CHECKNULL (_sjis_conv);
	_sjis_string = new UString (_sjis_conv, "SJIS");
	CHECKNULL (_sjis_string);

	_iso_2022_conv = new CTEXT_JAConv();
	CHECKNULL (_iso_2022_conv);
	_iso_2022_string = new UString (_iso_2022_conv, "ISO_2022_JP");
	CHECKNULL (_iso_2022_string);

	_euc_jp_conv = new UEUC_JPConv();
	CHECKNULL (_euc_jp_conv);
	_euc_jp_string = new UString (_euc_jp_conv, "EUC_JP");
	CHECKNULL (_euc_jp_string);

	_ctext_ja_conv = new CTEXT_JAConv();
	CHECKNULL (_ctext_ja_conv);
	_ctext_ja_string = new UString (_ctext_ja_conv, "CTEXT_JA");
	CHECKNULL (_ctext_ja_string);

	_ksc5601_conv = new UKSC5601Conv();
	CHECKNULL (_ksc5601_conv);
	_ksc5601_string = new UString (_ksc5601_conv, "CTEXT_KR");
	CHECKNULL (_ksc5601_string);

	_euc_kr_conv = new UEUC_KRConv();
	CHECKNULL (_euc_kr_conv);
	_euc_kr_string = new UString (_euc_kr_conv, "EUC_KR");
	CHECKNULL (_euc_kr_string);

	_johab_conv = new UJOHABConv();
	CHECKNULL (_johab_conv);
	_johab_string = new UString (_johab_conv, "JOHAB");
	CHECKNULL (_johab_string);

	_uhc_conv = new UUHCConv();
	CHECKNULL (_uhc_conv);
	_uhc_string = new UString (_uhc_conv, "UHC");
	CHECKNULL (_uhc_string);

	_gb2312_7_conv = new UGB2312_7Conv();
	CHECKNULL (_gb2312_7_conv);
	_gb2312_7_string = new UString (_gb2312_7_conv, "GB2312_7");
	CHECKNULL (_gb2312_7_string);

	_gb2312_8_conv = new UGB2312_8Conv();
	CHECKNULL (_gb2312_8_conv);
	_gb2312_8_string = new UString (_gb2312_8_conv, "GB2312_8");
	CHECKNULL (_gb2312_8_string);

	_hz_conv = new UHZConv();
	CHECKNULL (_hz_conv);
	_hz_string = new UString (_hz_conv, "HZ");
	CHECKNULL (_hz_string);

	_big5_conv = new UBIG5Conv();
	CHECKNULL (_big5_conv);
	_big5_string = new UString (_big5_conv, "BIG5");
	CHECKNULL (_big5_string);

	_u16be_conv = new U16BEConv("UCS2BE");
	CHECKNULL (_u16be_conv);
	_u16be_string = new UString (_u16be_conv, "UCS2BE");
	CHECKNULL (_u16be_string);

	_u16le_conv = new U16LEConv("UCS2LE");
	CHECKNULL (_u16le_conv);
	_u16le_string = new UString (_u16le_conv, "UCS2LE");
	CHECKNULL (_u16le_string);

	_uni_conv = new UUniConv("UNI");
	CHECKNULL (_uni_conv);
	_uni_string = new UString (_uni_conv, "UNI");
	CHECKNULL (_uni_string);

	_mstxt_conv = new UMSTXTConv("MSTXT");
	CHECKNULL (_mstxt_conv);
	_mstxt_string = new UString (_mstxt_conv, "MSTXT");
	CHECKNULL (_mstxt_string);
	UTuneUString (0);
}

void
UDeinitUString ()
{
	if (_utf8_conv == 0) return;
	delete (_utf8_string);
	delete (_utf7_string);
	delete (_java_string);
	delete (_jis_string);
	delete (_sjis_string);
	delete (_iso_2022_string);
	delete (_euc_jp_string);
	delete (_ctext_ja_string);
	delete (_ksc5601_string);
	delete (_euc_kr_string);
	delete (_johab_string);
	delete (_uhc_string);
	delete (_gb2312_7_string);
	delete (_gb2312_8_string);
	delete (_hz_string);
	delete (_big5_string);
	delete (_u16be_string);
	delete (_u16le_string);
	delete (_uni_string);
	delete (_mstxt_string);

	_utf8_string=0;
	_utf7_string=0;
	_java_string=0;
	_jis_string=0;
	_iso_2022_string=0;
	_euc_jp_string=0;
	_ctext_ja_string=0;
	_ksc5601_string=0;
	_euc_kr_string=0;
	_johab_string=0;
	_uhc_string=0;
	_gb2312_7_string=0;
	_gb2312_8_string=0;
	_hz_string=0;
	_big5_string=0;
	_u16be_string=0;
	_u16le_string=0;
	_uni_string=0;
	_mstxt_string=0;

	delete (_utf8_conv);
	delete (_utf7_conv);
	delete (_java_conv);
	delete (_jis_conv);
	delete (_sjis_conv);
	delete (_iso_2022_conv);
	delete (_euc_jp_conv);
	delete (_ctext_ja_conv);
	delete (_ksc5601_conv);
	delete (_euc_kr_conv);
	delete (_johab_conv);
	delete (_uhc_conv);
	delete (_gb2312_7_conv);
	delete (_gb2312_8_conv);
	delete (_hz_conv);
	delete (_big5_conv);
	delete (_u16be_conv);
	delete (_u16le_conv);
	delete (_uni_conv);
	delete (_mstxt_conv);

	_utf8_conv=0;
	_utf7_conv=0;
	_java_conv=0;
	_jis_conv=0;
	_sjis_conv=0;
	_iso_2022_conv=0;
	_euc_jp_conv=0;
	_ctext_ja_conv=0;
	_ksc5601_conv=0;
	_euc_kr_conv=0;
	_johab_conv=0;
	_uhc_conv=0;
	_gb2312_7_conv=0;
	_gb2312_8_conv=0;
	_hz_conv=0;
	_big5_conv=0;
	_u16be_conv=0;
	_u16le_conv=0;
	_mstxt_conv=0;
	_uni_conv=0;

	delete cache;
	cache = 0;
}

void
UTuneUString (int cacheSize, int batchSize)
{
	if (cache)
	{
		cache->tune (cacheSize, batchSize);
	}
}
//
// 8 bit converters are automatically loaded when not found.
