//  MainWidget.cpp version 1.1
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998  Gaspar Sinai
// 
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <MainWidget.h>
#include <Yudit.h>
#include <USeparator.h>
#include <UTextMenu.h>
#include <Translate.h>
#include <string.h>
#include <stdlib.h>
#include <strings.h>
#include <math.h>

MainWidget::MainWidget (const Config& configIn, 
	const char* fileNameIn,
	const char* encodingIn,
	const UWidget::UEditable editableIn) 
	: UShell (GetTopLevel(), "Yudit"), config (configIn) 
{
	UBestSize	bSize;
	const char*	str;
	AString		fullFileName;
	UColor		color(top->display);

	language = 0;
	fileDialog = 0;
	findDialog = 0;
	printDialog = 0;
	promptDialog = 0;

	setFont (GetDefaultFont());
	buttonFont = GetDefaultFont();
	textFont = GetDefaultFont();

	if (encodingIn!=0)
	{
		config.setChecked (Config::ENCODING, encodingIn);
	}
	editArea = new UTextArea (this);
	editArea->textFrame->uSetConverter ("UTF8");
	editArea->textFrame->setBackground("White");
	//
	// Set up edit area...
	//
	editFont = UNewFont (top->display, top->screen, 
		(const char*) config.getChecked (Config::FONTNAME));
	if (editFont==0)
	{
		cerr << "error: can not load font map: '" <<
		 (const char*) config.getChecked (Config::FONTNAME) << "'\n";
		editFont = GetDefaultFont();
	}
	editFont->setPixel (atoi((const char*)config.getChecked(Config::FONTSIZE)));
	editFont->setWeight ((const char*)config.getChecked(Config::FONTWEIGHT));
	editFont->setSlant ((const char*)config.getChecked(Config::FONTSLANT));
	editFont->setSpacing ((const char*)config.getChecked(Config::FONTSPACING));

	editArea->textFrame->uSetFont (editFont);
	if (editArea->textFrame->uSetConverter(
		(const char*)config.getChecked(Config::ENCODING)) == UString::ERROR)
	{
		cerr << "myudit: can not find converter for encoding '"  <<
			(const char*)config.getChecked(Config::ENCODING) << "'\n";
	}
	str = (const char*)config.getChecked(Config::INPUT);
	lastInput = str;

	if (strcasecmp (str, "Straight") ==0
		|| strcasecmp (str, "None")==0)
	{
		editArea->textFrame->uSetInput (0);
	}
	else if (editArea->textFrame->uSetInput(str) == UWidget::ERROR)
	{
		cerr << "myudit: can not set input '" << str << "'\n";
	}
	editArea->textFrame->uSetTabSize (
		atoi((const char*) config.getChecked (Config::TABSIZE)));
	color = (const char*) config.getChecked (Config::WINDOW_BACKGROUND);
	editArea->textFrame->uSetBackground (color.getPixel());

	color = (const char*) config.getChecked (Config::WINDOW_FOREGROUND);
	editArea->textFrame->uSetForeground (color.getPixel());

	color = (const char*) config.getChecked (Config::CURSOR_COLOR);
	editArea->textFrame->uSetCursorColor (color.getPixel());

	setXInput ((const char*) config.getChecked (Config::XINPUT));

	menuBar = new UMenuBar (this, 0);
	buildMenu ();
	bSize = menuBar->getBestSize ();
	menuBar->place (0, 0, 0, -bSize.height);
	editArea->place (0, bSize.height , 0, 0);

	if (fileNameIn!=0)
	{
		fullFileName = "file:";
		fullFileName+=fileNameIn;
		editArea->textFrame->uSetEditable (UWidget::EDITABLE);
		loadFile ((const char*) fullFileName);
		setTitle (fileName.isNull () ? TITLE : (const char*)fileName);
	}
	isEditable = editableIn;
	editArea->textFrame->uSetEditable (editableIn);
	editArea->textFrame->uClearModified ();
	editArea->textFrame->setFocus();

	str = (const char*) config.getChecked (Config::LANGUAGE);
	setLanguage (str);

	resize (560, 370);
}

MainWidget::~MainWidget ()
{
	if (findDialog) delete findDialog;
	if (printDialog) delete printDialog;
	if (promptDialog) delete promptDialog;
	if (fileDialog) delete fileDialog;

	// components are autodelete
	if (editFont && editFont != GetDefaultFont() ) delete editFont;
	if (buttonFont && buttonFont != GetDefaultFont()) delete buttonFont;
	if (textFont && textFont != GetDefaultFont()) delete textFont;
}

void
MainWidget::setLanguage (const char* lang_)
{
	UFont*			txt;
	UFont*			bt;
	UBestSize		bSize;
	const Language*		lng;

	if (lang_==0) return;
	lng = config.getLanguage (lang_);
	if (lng==0)
	{
		cerr << "warn: can not find language '" <<
			lang_ << "'\n";
		return;
	}
	language = lng;
	txt=textFont;
	bt=buttonFont;
	if (findDialog!=0) findDialog->hide();

	textFont = UNewFont (top->display, top->screen, 
			(const char*) language->list[Language::TEXT_FONT]);
	if (textFont)
	{
		textFont->setPixel (atoi ((const char*) language->list[Language::TEXT_SIZE]));
		textFont->setWeight ((const char*) language->list[Language::TEXT_WEIGHT]);
		textFont->setSlant ((const char*) language->list[Language::TEXT_SLANT]);
		textFont->setSpacing ((const char*) language->list[Language::TEXT_SPACING]);
	}
	else
	{
		cerr << "warn: can not find language '" << lang_ << "'\n";
		textFont=GetDefaultFont();
	}
	buttonFont = UNewFont (top->display, top->screen, 
			(const char*) language->list[Language::BUTTON_FONT]);
	if (buttonFont)
	{
		buttonFont->setPixel (atoi ((const char*) language->list[Language::BUTTON_SIZE]));
		buttonFont->setWeight ((const char*) language->list[Language::BUTTON_WEIGHT]);
		buttonFont->setSlant ((const char*) language->list[Language::BUTTON_SLANT]);
		buttonFont->setSpacing ((const char*) language->list[Language::BUTTON_SPACING]);
	}
	else
	{
		cerr << "warn: can not find language '" << lang_ << "'\n";
		textFont=GetDefaultFont();
	}
	
	if (txt && txt != GetDefaultFont()) delete txt;
	if (bt && bt != GetDefaultFont()) delete bt;
	menuBar->setFont(buttonFont);

	TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
	// remake the menubar
	translateMenuBar ();
	bSize = menuBar->getBestSize ();
	menuBar->place (0, 0, 0, -bSize.height);
	editArea->place (0, bSize.height , 0, 0);
	pack ();
}

MainWidget::MenuId
MainWidget::getMenuId (UPopUp* pup)
{
	int		i;
	MenuId		mid;
	for (i=(int)UFILE; i<(int)UMENUMAX; ++i)
	{	
		if (popUp[i] == pup) break;
	}
	mid = (MenuId) i;
	return mid;
}

void
MainWidget::buildMenu ()
{
	//
	// File menu
	//
	popUp[UFILE] = new UPopUp (menuBar, "File", Translate ("File"));
	(void) new UTextMenu (popUp[UFILE], "Open...", 
		Translate ("Open..."), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[UFILE], "Append...", 
		Translate ("Append..."), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[UFILE], "Re-Load", 
		Translate ("Re-Load"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[UFILE], "New Window", 
		Translate ("New Window"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new USeparator (popUp[UFILE]);
	(void) new UTextMenu (popUp[UFILE], "Save", 
		Translate ("Save"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[UFILE], "Save As...", 
		Translate ("Save As..."), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new USeparator (popUp[UFILE]);
	(void) new UTextMenu (popUp[UFILE], "Config", 
		Translate ("Config"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[UFILE], "Print...", 
		Translate ("Print..."), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new USeparator (popUp[UFILE]);
	(void) new UTextMenu (popUp[UFILE], "Close", 
		Translate ("Close"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[UFILE], "Exit All", 
		Translate ("Exit All"), UMenu::NO_LEFT, UMenu::NO_ARROW);

	// Encoding
	popUp[UENCODING] = new UPopUp (menuBar, "Encoding", Translate ("Encoding"));
	addButtons (UENCODING, config.getCheckList (Config::ENCODING), UMenu::NO_ARROW);
	checkItem (UENCODING, config.getChecked (Config::ENCODING));

	// Input, Xinput
	popUp[UINPUT] = new UPopUp (menuBar, "Input", Translate ("Input"));
	popUp[UXINPUT] = new UPopUp (popUp[UINPUT], "XInput", Translate ("XInput"), 
			UMenu::LEFT_BLANK, UMenu::RIGHT);

	addButtons (UINPUT, config.getCheckList (Config::INPUT), UMenu::ARROW_BLANK);
	checkItem (UINPUT, config.getChecked (Config::INPUT));
	addButtons (UXINPUT, config.getCheckList (Config::XINPUT), UMenu::NO_ARROW);
	checkItem (UXINPUT, config.getChecked (Config::XINPUT));

	// Font, Size, Weight, Slant, Spacing
	popUp[UFONTNAME] = new UPopUp (menuBar, "Font", Translate ("Font"));
	popUp[UFONTSIZE] = new UPopUp (popUp[UFONTNAME], "Size", Translate ("Size"), 
			UMenu::LEFT_BLANK, UMenu::RIGHT);
	popUp[UFONTWEIGHT] = new UPopUp (popUp[UFONTNAME], "Weight", Translate ("Weight"), 
			UMenu::LEFT_BLANK, UMenu::RIGHT);
	popUp[UFONTSLANT] = new UPopUp (popUp[UFONTNAME], "Slant", Translate ("Slant"), 
			UMenu::LEFT_BLANK, UMenu::RIGHT);
	popUp[UFONTSPACING] = new UPopUp (popUp[UFONTNAME], "Spacing", Translate ("Spacing"), 
			UMenu::LEFT_BLANK, UMenu::RIGHT);
	addButtons (UFONTSIZE, config.getCheckList (Config::FONTSIZE), UMenu::NO_ARROW);
	checkItem (UFONTSIZE, config.getChecked (Config::FONTSIZE));

	addButtons (UFONTWEIGHT, config.getCheckList (Config::FONTWEIGHT), UMenu::NO_ARROW);
	checkItem (UFONTWEIGHT, config.getChecked (Config::FONTWEIGHT));

	addButtons (UFONTSLANT, config.getCheckList (Config::FONTSLANT), UMenu::NO_ARROW);
	checkItem (UFONTSLANT, config.getChecked (Config::FONTSLANT));

	addButtons (UFONTSPACING, config.getCheckList (Config::FONTSPACING), UMenu::NO_ARROW);
	checkItem (UFONTSPACING, config.getChecked (Config::FONTSPACING));


	(void) new USeparator (popUp[UFONTNAME]);
	addButtons (UFONTNAME, config.getCheckList (Config::FONTNAME), UMenu::ARROW_BLANK);
	checkItem (UFONTNAME, config.getChecked (Config::FONTNAME));

	// Window: Tabsize, Background, Foreground, CursorColor
	popUp[UWINDOW] = new UPopUp (menuBar, "Window", Translate ("Window"));
	popUp[UTABSIZE] = new UPopUp (popUp[UWINDOW], "Tab Size", 
			Translate ("Tab Size"), UMenu::NO_LEFT, UMenu::RIGHT);
	popUp[UWINDOW_BACKGROUND] = new UPopUp (popUp[UWINDOW], "Background", 
			Translate ("Background"), UMenu::NO_LEFT, UMenu::RIGHT);
	popUp[UWINDOW_FOREGROUND] = new UPopUp (popUp[UWINDOW], "Foreground", 
			Translate ("Foreground"), UMenu::NO_LEFT, UMenu::RIGHT);
	popUp[UCURSOR_COLOR] = new UPopUp (popUp[UWINDOW], "Cursor Color", 
			Translate ("Cursor Color"), UMenu::NO_LEFT, UMenu::RIGHT);
	popUp[ULANGUAGE] = new UPopUp (popUp[UWINDOW], "Language", 
			Translate ("Language"), UMenu::NO_LEFT, UMenu::RIGHT);

	addButtons (UTABSIZE, config.getCheckList (Config::TABSIZE), UMenu::NO_ARROW);
	checkItem (UTABSIZE, config.getChecked (Config::TABSIZE));

	addButtons (UWINDOW_BACKGROUND, config.getCheckList (Config::WINDOW_BACKGROUND), UMenu::NO_ARROW);
	checkItem (UWINDOW_BACKGROUND, config.getChecked (Config::WINDOW_BACKGROUND));

	addButtons (UWINDOW_FOREGROUND, config.getCheckList (Config::WINDOW_FOREGROUND), UMenu::NO_ARROW);
	checkItem (UWINDOW_FOREGROUND, config.getChecked (Config::WINDOW_FOREGROUND));

	addButtons (UCURSOR_COLOR, config.getCheckList (Config::CURSOR_COLOR), UMenu::NO_ARROW);
	checkItem (UCURSOR_COLOR, config.getChecked (Config::CURSOR_COLOR));
	addButtons (ULANGUAGE, config.getCheckList (Config::LANGUAGE), UMenu::NO_ARROW);
	checkItem (ULANGUAGE, config.getChecked (Config::LANGUAGE));

	// Search
	popUp[USEARCH] = new UPopUp (menuBar, "Search", Translate ("Search"));
	(void) new UTextMenu (popUp[USEARCH], "Find/Replace", 
		Translate ("Find/Replace"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[USEARCH], "Go To", 
		Translate ("Go To"), UMenu::NO_LEFT, UMenu::NO_ARROW);

	// Help
	popUp[UHELP] = new UPopUp (menuBar, "Help", Translate ("Help"));
	(void) new UTextMenu (popUp[UHELP], "About...", 
		Translate ("About..."), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new USeparator (popUp[UHELP]);
	(void) new UTextMenu (popUp[UHELP], "General", 
		Translate ("General"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new UTextMenu (popUp[UHELP], "Config", 
		Translate ("Config"), UMenu::NO_LEFT, UMenu::NO_ARROW);
	(void) new USeparator (popUp[UHELP]);
	(void) new UTextMenu (popUp[UHELP], "Test Page", 
		Translate ("Test Page"), UMenu::NO_LEFT, UMenu::NO_ARROW);

	//translateMenuBar ();
}

void
MainWidget::addButtons (MenuId id, 
		const AStringList& list, UMenu::UMenuArrow arrow_)
{
	int		i;
	UTextLabel*	empty;

	for (i=0; i<list.getSize(); i++)
	{
		(void) new UTextMenu (popUp[id], (const char*) list.at (i), 
			Translate ((const char*) list.at (i)), 
			UMenu::CHECK, arrow_);
	}
	if (i==0)
	{
		empty = new UTextLabel (popUp[id], Translate ("Not configured"));
	}
}

void
MainWidget::checkItem (MenuId id, const char* item)
{
	int		i;
	UComponent*	comp;
	UTextMenu*	tmenu;
	UMenuItem*	menuItem;

	popUp[id]->uncheckAllItems();
	if (item==0) return;
	for (i=0; i<popUp[id]->getSize(); i++)
	{
		comp = popUp[id]->itemAt (i);
		if (!comp->isA (UComponent::TEXT_MENU)) continue;
		tmenu = (UTextMenu*) comp;
		if (strcmp (tmenu->getName(), item)) continue;
		menuItem = (UMenuItem*) tmenu->getMenu();
		if (menuItem==0) continue;
		menuItem->check();
		break;
	}
}

void
MainWidget::translateMenuBar ()
{
	int		i;
	UTextMenu*	tmenu;

	for (i=(int) UFILE; i< (int) UMENUMAX; i++)
	{
		tmenu = (UTextMenu*) popUp[i]->cascade;
		tmenu->setText (Translate (tmenu->getName()));
		translateMenu ((MenuId)i);
	}
}
	
void
MainWidget::translateMenu (MenuId id)
{
	int		i;
	UComponent*	comp;
	UTextMenu*	tmenu;

	for (i=0; i<popUp[id]->getSize(); i++)
	{
		comp = popUp[id]->itemAt (i);
		if (!comp->isA (UComponent::TEXT_MENU)) continue;
		tmenu = (UTextMenu*) comp;
		tmenu->setText (Translate (tmenu->getName()));
	}
}

void
MainWidget::eventUp (UEvent* event)
{
	UShell::eventUp (event);
	switch (event->type)
	{
	case  UEvent::MENU_SELECTION:
		if (event->client != menuBar) return;
		menuAction (getMenuId ((UPopUp*) event->popUp), event->value);
		return;
	default:
		break;
	}
}

void
MainWidget::quit ()
{
	if (editArea->textFrame->uIsModified())
	{
		if (language)
		{
			TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
		}
		if(!ask ("Buffer has been modified.\nDo you really want to quit?"))
		{
			return;
		}
	}
	delete this;
}

int
MainWidget::setXInput (const char* nameIn)
{
	const XInputMap*        xInputMap;

	xInputMap = config.getXInputMap (nameIn);
	if (xInputMap==0)
	{
		cerr << "error: can not get XInputMap.\n";
		return -1;
	}
	return (editArea->textFrame->setXInput (nameIn,
		(const char*) xInputMap->list[XInputMap::TYPE],
		(const char*) xInputMap->list[XInputMap::STYLE],
		(const char*) xInputMap->list[XInputMap::TEXT_TYPE],
		(const char*) xInputMap->list[XInputMap::ENCODING],
		(const char*) xInputMap->list[XInputMap::FONTMAP],
		atof ((const char*)xInputMap->list[XInputMap::VERSION])));
}

// ---- these will appear in gettext ---
// Translate ("Hungarian");
// Translate ("English");
// Translate ("Japanese");
// Translate ("Chinese");
// Translate ("Chinese-CJ");
// Translate ("Any");
// Translate ("Roman");
// Translate ("Regular");
// Translate ("Medium");
// Translate ("Demibold");
// Translate ("Bold");
// Translate ("Italic");
// Translate ("Oblique");
// Translate ("Monospace");
// Translate ("Condensed");
// Translate ("Proportional");
// Translate ("European");
// Translate ("Kana");
// Translate ("Straight");
// Translate ("Russian");
// Translate ("Greek");
// Translate ("Lithuanian");
// Translate ("Unicode");
// Translate ("None");
// Translate ("black");
// Translate ("white");
// Translate ("darkGray");
// Translate ("gray");
// Translate ("lightGray");
// Translate ("red");
// Translate ("green");
// Translate ("blue");
// Translate ("cyan");
// Translate ("magenta");
// Translate ("yellow");
// Translate ("darkRed");
// Translate ("darkGreen");
// Translate ("darkBlue");
// Translate ("darkCyan");
// Translate ("darkMagenta");
// Translate ("darkYellow");
// Translate ("General.help");
// Translate ("Config.help");
// Translate ("TestPage.help");
