//  UTextList.cpp version 1.1
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998  Gaspar Sinai
// 
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
#include "UTextList.h"
#include <UString.h>

UTextList::UTextList (UFrame* parent_) :
	UTextLabel (parent_)
{
	select (ButtonPressMask |ButtonReleaseMask | Button1MotionMask);
	selectedItem = -1;
}

UTextList::~UTextList ()
{
}

void
UTextList::replace (const AStringList& str)
{
	int	i;
	int	to;
	UString ustr;
	ustr.setConverter ("UTF8");
	// clean previous
	setText ((const UCS2*) 0);
	selectedItem = -1;

	lineCount = str.getSize();
	if (lineCount==0) 
	{
		setScroll();
		if (isShown (this))
		{
			XClearArea (top->display, window, 0, 0, 
				rectangle.width, rectangle.height, True);
		}
		return;
	}
	line = new UTextLine* [lineCount];
	
	for (i=0; i<lineCount;i++)
	{
		ustr.putString ((const unsigned char*) str.at (i));
		to = UCS2Len ((const UCS2*)ustr);
		line[i] = new UTextLine ((const UCS2*)ustr, 0, to);
	}
	calculateBestSize();
	setScroll();
	if (isShown (this))
	{
		XClearArea (top->display, window, 0, 0, 
			rectangle.width, rectangle.height, True);
	}
	return;
}

void
UTextList::eventDown (UEvent* event)
{
	int		newSelect;
	UEvent		newEvent;
	switch (event->type)
	{
	case UEvent::X:
		switch (event->xevent.type)
		{
		case ButtonRelease:
			if (selectedItem >=0)
			{
				newEvent.type = UEvent::SELECTED;
				newEvent.client = this;
				newEvent.value = selectedItem;
				parent->eventUp (&newEvent);
			}
			break;
		case ButtonPress:
			if (event->xevent.xbutton.button != Button1) return;
			if (event->xevent.xbutton.x < 0
				|| event->xevent.xbutton.x > rectangle.width
				|| event->xevent.xbutton.y < 0
				|| event->xevent.xbutton.y > rectangle.height)
			{
				return;
			}
			if (font==0 || font->fontHeight<1) return;
			newSelect = (event->xevent.xbutton.y + vertOffset) /
				font->fontHeight;
			if (newSelect<0 || newSelect>=lineCount)
			{
				unselectLine (selectedItem);
				selectedItem=-1;
				return;
			}
			if (selectedItem==newSelect) return;
			unselectLine (selectedItem);
			selectedItem = newSelect;
			selectLine (selectedItem);
			selectedVisible ();
			break;
		case MotionNotify:
			// compress event
			if (event->xevent.xmotion.state & Button1MotionMask==0)
				 return;
			XCheckTypedWindowEvent (top->display,
				event->xevent.xany.window,
				event->xevent.type,
				&event->xevent);
			// no need to check bounds
			if (font==0 || font->fontHeight<1) return;
			newSelect = (event->xevent.xbutton.y + vertOffset) /
				font->fontHeight;
			if (newSelect<0 || newSelect>=lineCount)
			{
				unselectLine (selectedItem);
				selectedItem=-1;
				return;
			}
			if (selectedItem==newSelect) return;
			unselectLine (selectedItem);
			selectedItem = newSelect;
			selectLine (selectedItem);
			selectedVisible ();
			break;

		default:
			UTextLabel::eventDown(event);
		}
		break;
	default:
		UTextLabel::eventDown(event);
	}
}

void
UTextList::unselect()
{
	unselectLine (selectedItem);
	selectedItem=-1;
}

void
UTextList::selectedVisible ()
{
	UEvent		event;
	
	event.client = this;
	event.type = UEvent::SCROLL_RANGE_VERTICAL;
	event.step = 0;
	event.page = 0;
	event.max = 0;

	if (selectedItem < 0 || selectedItem >= lineCount) return;
	// make selected line visible.
	if (font->fontHeight*selectedItem-vertOffset < 0)
	{
		scrollVertical (font->fontHeight*selectedItem);
		event.value = vertOffset;
		parent->eventUp (&event);
		return;
	}
	if (font->fontHeight*(selectedItem+1)-vertOffset > rectangle.height)
	{
		scrollVertical (font->fontHeight*(selectedItem+1)-rectangle.height);
		event.value = vertOffset;
		parent->eventUp (&event);
		return;
	}
}
void
UTextList::resize (int width, int height)
{
	UTextLabel::resize (width, height);
	setScroll ();
}

void
UTextList::setScroll ()
{
	UEvent		event;
	
	event.client = this;
	event.type = UEvent::SCROLL_RANGE_VERTICAL;
	event.value = vertOffset;
	event.step = (font==0) ? 1 : font->fontHeight;
	event.page = rectangle.height;
	event.max = bestSize.height;
	parent->eventUp (&event);
}

void
UTextList::add (const char* str, int pos_)
{
	UString 	ustr;
	int		newSize = lineCount+1;
	UTextLine** 	newLine;
	int		newPos=0;
	int		to;
	int		i;

	if (str==0 || pos_ > lineCount) return;
	ustr.setConverter ("UTF8");
	ustr.putString ((const unsigned char*) str);
	to = UCS2Len ((const UCS2*)ustr);
	newLine = new UTextLine* [newSize];
	if (lineCount>0)
	{
		memcpy (newLine, line, lineCount * sizeof (UTextLine*));
		delete line;
	}
	newPos = (pos_<0) ? lineCount : pos_;
	if (selectedItem >=0 && selectedItem >= newPos) selectedItem++;
	line = newLine;
	lineCount = newSize;

	for (i=newSize; i>newPos; i--)
	{
		line[i] = line[i-1];
	}
	line[newPos] = new UTextLine ((const UCS2*)ustr, 0, to);
	calculateBestSize();
	setScroll();
	if (isShown (this))
	{
		XClearArea (top->display, window, 0, 0, 
			rectangle.width, rectangle.height, True);
	}
}

void
UTextList::deleteAll ()
{
	setText ((const UCS2*) 0);
}

int
UTextList::isA (UComponent::UType type)
{
	if (type==TEXT_LIST) return 1;
	return (UTextLabel::isA (type));
}

void
UTextList::scrollVertical (int vle)
{
	int     diff;

	// the value is the offset itself
	if (vle==vertOffset) return;

	diff = vertOffset-vle;
	vertOffset = vle;
	if (diff>0)
	{
		// scroll down
		if (rectangle.height -diff > 0)
		{
			XCopyArea (top->display, window, window,
				fgGC, 0, 0,
				rectangle.width, rectangle.height-diff,
				0, diff);
			XClearArea (top->display, window,
				0, 0, rectangle.width, diff, False);
			redraw (0, 0, rectangle.width, diff);
		}
		else
		{
			XClearArea (top->display, window,
				0, 0, rectangle.width, rectangle.height, False);
			redraw (0, 0, rectangle.width, rectangle.height);
		}
	}
	else
	{
		// scroll up
		if (rectangle.height + diff > 0)
		{
			XCopyArea (top->display, window, window,
				fgGC, 0, -diff,
				rectangle.width, rectangle.height+diff,
				0, 0);
			XClearArea (top->display, window,
				0, rectangle.height+diff, rectangle.width, 
				-diff, False);
			redraw (0, rectangle.height+diff, rectangle.width, 
				-diff);
		}
		else
		{
			XClearArea (top->display, window,
				0, 0, rectangle.width, rectangle.height, False);
			redraw (0, 0, rectangle.width, rectangle.height);
		}
	}
}
