/* Quinn Diff, Compares two Packages files looking for differences between Archs */
/* Copyright (C) 1997, 1998, James Troup <james@nocrew.org> */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <string.h>
#include <db.h>
#include "common.h"
#include "error.h"
#include "xmalloc.h"
#include "hash.h"
#include "source.h"
#include "utils.h"
#include "main.h"
#include "output.h"

HASHINFO *output_hash_info;
DB *output_table = NULL;
DBT *output_dbt, *output_key;
char package_key[PACKAGE_NAME_LENGTH];

void
output_create(void)
{

  output_hash_info = xmalloc (sizeof(*output_hash_info));

  output_hash_info->bsize = 256;
  output_hash_info->cachesize = 1048576;
  output_hash_info->ffactor = 8;
  output_hash_info->hash = NULL; 
  output_hash_info->lorder = 0;
  output_hash_info->nelem = 1;

  output_table = dbopen (NULL, O_RDWR, 0, DB_HASH, output_hash_info);

  if (output_table == NULL)
    fubar (SYSERR, "Unable to create hash table for output buffer");

  output_dbt = xmalloc (sizeof(*output_dbt));
  output_dbt->size = sizeof (output_info);

  output_key = xmalloc (sizeof(*output_key));
  output_key->data = package_key;

}

int 
output_add (const char *package_name, const char *primary_version, const char *secondary_version, 
	    const char *priority, const char *section, const boolean needs_x, 
	    const int compare_versions_result, const boolean multi_binary_source_package, 
	    const char *source_version)
{

  output_info output_to_hash;
  int i;

  debug(debug_output, "output_add: Adding \"%s\" (%s:%s) [Versions %s and %s] to output hash table", 
	package_name, priority, section, primary_version, secondary_version);
  
  strcpy (output_to_hash.package_name, package_name);  
  strcpy (output_to_hash.primary_version, primary_version);  
  strcpy (output_to_hash.secondary_version, secondary_version);  
  strcpy (output_to_hash.priority, priority);  
  strcpy (output_to_hash.section, section);  
  output_to_hash.needs_x = needs_x;
  output_to_hash.compare_versions_result = compare_versions_result;
  output_to_hash.multi_binary_source_package = multi_binary_source_package;  
  if (source_version)
    strcpy (output_to_hash.source_version, source_version);
  else 
    output_to_hash.source_version[0] = '\0';
  output_to_hash.errors = 0;
  
  output_dbt->data = (output_info *) &output_to_hash;
  
  strcpy (package_key, package_name);
  output_key->size = strlen (package_name) + 1;
  
  debug(debug_output, "output_add: key is %s and key size is %d", 
	(char *) output_key->data, output_key->size);
  debug(debug_output, "output_add: data is at %p and size is %d", 
	output_dbt->data, output_dbt->size);

  i = output_table->put (output_table, output_key, output_dbt, R_NOOVERWRITE);

  return i;

}

void output_mbs_update (const char *priority, const int needs_x, 
			output_info *source_package, const char *source_version)
{

  int current_priority, new_priority, i;
  
  /* Check source version */

  if (source_version && source_version[0] != '\0' && source_package->source_version[0] != '\0')
    {
      if (strcmp (source_package->source_version, source_version))
	{
	  debug (debug_warn, "AIEE!  Bogus source version (%s vs %s)", source_package->source_version, source_version);
	  source_package->errors |= SOURCE_VERSION_MISMATCH;
	}
    }
  else if (source_package->source_version[0] == '\0' && source_version && source_version[0] != '\0')
    strcpy (source_package->source_version, source_version);

  /* Fix priority */

  current_priority = priority_to_int (source_package->priority);
  new_priority =  priority_to_int (priority);
  if (new_priority > current_priority)
    {
      debug (debug_mbs, "%s > %s, priority becomes %s.", 
	     priority,  source_package->priority, priority);
      strcpy (source_package->priority, priority);
    }
  else
    debug (debug_mbs,  "%s <= %s, priority remains %s.", 
	   priority, source_package->priority, source_package->priority);
    
  /* Fix needs_x.  A single binary of an mbs is sufficent for requiring X */

  debug (debug_mbs, "package needs_x = %d... ", source_package->needs_x);
  source_package->needs_x |= needs_x;
  debug (debug_mbs, "becomes needs_x = %d", source_package->needs_x);

  output_dbt->data = (output_info *) source_package;
  
  strcpy (package_key, source_package->package_name);
  output_key->size = strlen (source_package->package_name) + 1;
  
  debug(debug_mbs, "output_mbs_update: key is %s and key size is %d", 
	(char *) output_key->data, output_key->size);
  debug(debug_mbs, "output_mbs_update: data is at %p and size is %d", 
	output_dbt->data, output_dbt->size);

  i = output_table->put (output_table, output_key, output_dbt, 0);
  if (i)
    fubar (NONSYS, "couldn't update \"%s\" in the output hash table (put returned %d)", 
	   source_package->package_name, i);

}


output_info *
output_retrieve (const char *package) 
{ 

   output_info *output_details; 
#ifdef CPU_BUS_ERRORS_ON_BAD_ALIGNS
   output_info *output_details_realigned;
   char *p, *q;
#endif /* CPU_BUS_ERRORS_ON_BAD_ALIGNS */
   
   strcpy ( output_key->data, package ); 
   output_key->size = strlen (package) + 1; 

  debug(debug_output, "output_retrieve: key is %s and key size is %d", 
	(char *) output_key->data, output_key->size);

   /* RQ: WTF do I do with flags?  Presently set to 0. */

   if (output_table->get (output_table, output_key, output_dbt, 0) != 0)
     output_details = NULL; 
   else 
     output_details = (output_info *) output_dbt->data; 

#ifdef CPU_BUS_ERRORS_ON_BAD_ALIGNS

   /* libdb often returns the data at a poorly aligned address, on a
      RISC machine like sparc based ones this causes a bus error.
      CISC machines handle it (albeit with a penalty performance).
      Here we hack around this problem by memcpy()ing the data struct
      to a freshly allocated chunk of memory.  We don't use memcpy(),
      becasue it appears to be broken on Solaris *sigh* */

   if (output_details) 
     {
       output_details_realigned = malloc (sizeof(*output_details_realigned)); 
       for(p = (char *) output_details_realigned + sizeof(output_info), 
	     q = (char *) output_details + sizeof(output_info); 
	   p > (char *) output_details_realigned; )
	 *--p = *--q;
       return (output_details_realigned); 
     }
#endif /* CPU_BUS_ERRORS_ON_BAD_ALIGNS */

   return (output_details);

} 

void 
output_display(void)
{

  int result = 0;
  int i;
  output_info *package; 
#ifdef CPU_BUS_ERRORS_ON_BAD_ALIGNS
  output_info *package_realigned; 
  char *p, *q;
#endif /* CPU_BUS_ERRORS_ON_BAD_ALIGNS */
  source_packages_info source_package;

  char needs_x[5];
  char status[15];
  char errors[22];
  
  /* Sequentially walk through the output hash table */
  /* The aforementioned hash table might be better of as a btree, at
     least then we can go backwards through it */

  while (result == 0)
    {

      result = output_table->seq (output_table, output_key, output_dbt, R_NEXT);

      if (result == 0)
	{

	  package = (output_info *) output_dbt->data;

#ifdef CPU_BUS_ERRORS_ON_BAD_ALIGNS

	  /* libdb often returns the data at a poorly aligned address, on a
	     RISC machine like sparc based ones this causes a bus error.
	     CISC machines handle it (albeit with a penalty performance).
	     Here we hack around this problem by memcpy()ing the data struct
	     to a freshly allocated chunk of memory.  We don't use memcpy(),
	     becasue it appears to be broken on Solaris *sigh* */
	  
	  if (package) 
	    {
	      package_realigned = malloc (sizeof(*package_realigned)); 
	      for(p = (char *) package_realigned + sizeof(output_info), 
		    q = (char *) package + sizeof(output_info); 
		  p > (char *) package_realigned; )
		*--p = *--q;
	      package = package_realigned;
	    }
#endif /* CPU_BUS_ERRORS_ON_BAD_ALIGNS */

	  i = source_packages_retrieve (package->package_name, &source_package);
	  
	  if (i == 0)
	    {
	      debug(debug_warn, "%s: aiee; no source found.", package->package_name);
	      /* Kludge round lack of source package entry, these values may be wrong */
	      strcpy(source_package.section, package->section);
	      strcpy(source_package.version, package->primary_version);
	    }
	  else
	    {
	      /* Check that the version number of the source matches the binary */
	      if (strcmp (source_package.version, package->primary_version))
		{
		  char *version;
		  /* May be a mismatch due to epoch's, as source packages don't have them */
		  
		  version = strchr (package->primary_version, ':');
		  
		  if (version && strcmp (source_package.version, version+1))
		    debug  (debug_warn, "%s: %s (source) != %s (primary)", 
			    package->package_name, source_package.version, version+1);
		  else if (!version)
		    {
		      if (!strcmp (source_package.version, package->secondary_version))
			debug (debug_warn, "%s: source version and secondary match, primary does not.", 
			       source_package.package_name);
		      debug  (debug_warn, "%s: %s (source) != %s (primary)", 
			      package->package_name, source_package.version, package->primary_version);
		    }
		}

	      /* This fails when the source version is different to
		 That of the source version listed in the Packages file; if someone has
		 done an NMU of a package with an explict source version in the
		 Packages file and this NMU has yet to be recompiled for the primary
		 architecture, then quinn diff will falsley identify this as a FCUKed mirror */

	      if (package->source_version[0] != '\0' && strcmp (source_package.version, package->source_version))
		{
/* 		  package->errors |= SOURCE_VERSION_MISMATCH; */
		  debug (debug_warn, "%s: %s (source) != %s (source version from primary)", 
			 package->package_name, source_package.version, package->source_version);
		}
	    }

	  /* If the package appears erroneous and visible warning isn't set, skip it */
	  if (package->errors <= 0 || package->errors <= visible_warn_level)
	    {
	      if (package->errors > 0 && package->errors <= visible_warn_level)
		sprintf (errors, "** WARNING [%o] ** ", package->errors);
	      else
		strcpy (errors, "");

	      if (package->needs_x)  
		strcpy (needs_x, ":X");  
	      else  
		strcpy (needs_x, "");  

	      if (!(strcmp(package->secondary_version, "")))
		strcpy (status, "uncompiled");
	      else
		strcpy (status, "out-of-date");

	      if (!(strcmp(package->secondary_version, "")) ||
		  package->compare_versions_result < 0)
		printf ("%s%s/%s_%s.dsc [%s:%s%s]\n", 
			errors, source_package.section, package->package_name, 
			source_package.version, package->priority, status,
			needs_x);
	      else if (package->compare_versions_result == 0)
		fubar (NONSYS, "%s [%s] %s/%s: %s and %s are the same? I'm very confused.  Tell James. (%d)\n", 
			errors, package->priority, source_package.section,
			package->package_name, 
			package->primary_version,
			package->secondary_version, package->compare_versions_result);
	    }
	}
      else if (result != 1) /* 1 means no more packages, if result isn't 0 or 1 then we're in trouble */
	fubar (SYSERR, "couldn't retrieve next package in output");
    }
}

void
output_destroy(void)
{

 if (output_table)
   {  
     if (output_table->close(output_table))
       fubar (SYSERR, "Couldn't destroy output hash table");
     
     free (output_key);
     free (output_dbt);
     free (output_hash_info);
   }
 else
   error ("output_destroy: output_table is null.  Tell James.");

}


