/*
 * MLMATR.C - matrix operations including
 *          - create a matrix, destroy a matrix,
 *          - addition, subtraction, and multiplication,
 *          - transpose, invert, lu decompose, solve,
 *          - take upper and lower diagonal parts,
 *          - print
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pml.h"

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_CREATE - create a matrix with nrow rows and ncol columns */

PM_matrix *PM_create(nrow, ncol)
   int nrow, ncol;
   {PM_matrix *mp;
    REAL *ap;

    mp = FMAKE(PM_matrix, "PM_CREATE:mp");

    mp->nrow = nrow;
    mp->ncol = ncol;
    
    ap = FMAKE_N(REAL, nrow*ncol, "PM_CREATE:ap");
    mp->array = ap;

    return(mp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_DESTROY - release the storage associated with the given matrix */

int PM_destroy(mp)
   PM_matrix *mp;
   {if (mp == NULL)
       return(FALSE);

    SFREE(mp->array);
    SFREE(mp);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_ZERO - zero a matrix with nrow rows and ncol columns */

PM_matrix *PM_zero(a)
   PM_matrix *a;
   {int i, j, nrow, ncol;

    nrow = a->nrow;
    ncol = a->ncol;

    for (i = 1; i <= nrow; i++)
        {for (j = 1; j <= ncol; j++)
             PM_element(a, i, j) = 0. ;};

    return(a);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_IDENT - set diagonals of a matrix to unity */

PM_matrix *PM_ident(a)
   PM_matrix *a;
   {int i, nrow, ncol;

    nrow = a->nrow;
    ncol = a->ncol;

    if (nrow > ncol) nrow = ncol;

    for (i = 1; i <= nrow; i++)  PM_element(a, i, i) = 1. ;

    return(a);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_NEGATIVE - replace a matrix of nrow rows and ncol columns with the
 *                negative of the matrix
 */

PM_matrix *PM_negative(m, a)
   PM_matrix *m, *a;
   {int i, j, nrow, ncol;

    nrow = a->nrow;
    ncol = a->ncol;

    for (i = 1; i <= nrow; i++)
        for (j = 1; j <= ncol; j++)
            PM_element(m, i, j) = -PM_element(a, i, j);

    return(m);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_COPY - copy a matrix of nrow rows and ncol columns to a matrix
 *           of the same size
 */

PM_matrix *PM_copy(to, from)
   PM_matrix *to, *from;
   {int i, j, nrow, ncol;

    nrow = from->nrow;
    ncol = from->ncol;

    for (i = 1; i <= nrow; i++)
        for (j = 1; j <= ncol; j++)
            PM_element(to, i, j) = PM_element(from, i, j);

    return(to);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_TRANSPOSE - transpose the given matrix */

PM_matrix *_PM_transpose(m, a)
   PM_matrix *m, *a;
   {int i, j, ncol, nrow;

    if ((m == NULL) || (a == NULL))
       {sprintf(PM_error, "NULL MATRIX - _PM_TRANSPOSE");
        return(NULL);};

    nrow = a->nrow;
    ncol = a->ncol;

    for (i = 1; i <= nrow; i++)
        {for (j = 1; j <= ncol; j++)
             PM_element(m, j, i) = PM_element(a, i, j);};

    return(m);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_TRANSPOSE - transpose the given matrix */

PM_matrix *PM_transpose(a)
   PM_matrix *a;
   {int ncol, nrow;

    if (a == NULL)
       {sprintf(PM_error, "NULL MATRIX - PM_TRANSPOSE");
        return(NULL);};

    nrow = a->nrow;
    ncol = a->ncol;

    return(_PM_transpose(PM_create(ncol, nrow), a));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_TIMES - multiply two matrices M = A.B */

PM_matrix *_PM_times(m, a, b)
   PM_matrix *m, *a, *b;
   {int mrow, mcol, arow, acol, brow, bcol;
    REAL sum, ce, *va, *vb;
    int k, j, i;

    if ((m == NULL) || (a == NULL) || (b == NULL))
       {sprintf(PM_error, "NULL MATRIX - _PM_TIMES");
        return(NULL);};
        
    mrow = m->nrow;
    mcol = m->ncol;
    arow = a->nrow;
    acol = a->ncol;
    brow = b->nrow;
    bcol = b->ncol;

/* handle error */
    if ((acol != brow) || (mrow != arow) || (mcol != bcol))
       {sprintf(PM_error, "ROW COLUMN MISMATCH - _PM_TIMES");
        return(NULL);};

    for (i = 1; i <= arow; i++)
        {for (j = 1; j <= bcol; j++)
	     {va = a->array + (i - 1)*acol;
	      vb = b->array + j - 1;
              sum = 0.0;
	      for (k = 0; k < acol; k++, va++, vb += bcol)
		  {ce   = (*va)*(*vb);
		   sum += ce;};

	      PM_element(m, i, j) = sum;};};

    return(m);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_TIMES - multiply two matrices */

PM_matrix *PM_times(a, b)
   PM_matrix *a, *b;
   {int arow, bcol;

    if ((a == NULL) || (b == NULL))
       {sprintf(PM_error, "BAD MATRIX - PM_TIMES");
        return(NULL);};
        
    arow = a->nrow;
    bcol = b->ncol;

    return(_PM_times(PM_create(arow, bcol), a, b));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_NEGATIVE_TIMES - return negative product of two matrices, M = -AB  */

PM_matrix *_PM_negative_times(m, a, b)
   PM_matrix *m, *a, *b;
   {int mrow, mcol, arow, acol, brow, bcol;
    REAL sum, ce, *va, *vb;
    int k, j, i;

    if ((m == NULL) || (a == NULL) || (b == NULL))
       {sprintf(PM_error, "NULL MATRIX - _PM_TIMES");
        return(NULL);};
        
    mrow = m->nrow;
    mcol = m->ncol;
    arow = a->nrow;
    acol = a->ncol;
    brow = b->nrow;
    bcol = b->ncol;

/* handle error */
    if ((acol != brow) || (mrow != arow) || (mcol != bcol))
       {sprintf(PM_error, "ROW COLUMN MISMATCH - _PM_TIMES");
        return(NULL);};

    for (i = 1; i <= arow; i++)
        {for (j = 1; j <= bcol; j++)
	     {va = a->array + (i - 1)*acol;
	      vb = b->array + j - 1;
              sum = 0.0;
	      for (k = 0; k < acol; k++, va++, vb += bcol)
		  {ce   = (*va)*(*vb);
		   sum += ce;};

	      PM_element(m, i, j) = -sum;};};

    return(m);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_NEGATIVE_TIMES - multiply two matrices */

PM_matrix *PM_negative_times(a, b)
   PM_matrix *a, *b;
   {int arow, bcol;

    if ((a == NULL) || (b == NULL))
       {sprintf(PM_error, "BAD MATRIX - PM_TIMES");
        return(NULL);};
        
    arow = a->nrow;
    bcol = b->ncol;

    return(_PM_negative_times(PM_create(arow, bcol), a, b));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_PLUS - add the given matrices C = A + B */

PM_matrix *_PM_plus(c, a, b)
   PM_matrix *c, *a, *b;
   {int i, j, row, col;

    row = a->nrow;
    col = a->ncol;
    if ((row != b->nrow) || (col != b->ncol))
       {sprintf(PM_error, "ROW COLUMN MISMATCH - PM_PLUS");
        return(NULL);};

    for (i = 1; i <= row; i++)
        for (j = 1; j <= col; j++)
            PM_element(c, i, j) = PM_element(a, i, j) + PM_element(b, i, j);

    return(c);}
                
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_PLUS - add the given matrices */

PM_matrix *PM_plus(a, b)
   PM_matrix *a, *b;
   {int row, col;

    row = a->nrow;
    col = a->ncol;
    if ((row != b->nrow) || (col != b->ncol))
       {sprintf(PM_error, "ROW COLUMN MISMATCH - PM_PLUS");
        return(NULL);}

    else
       return(_PM_plus(PM_create(row, col), a, b));}
                
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_MINUS - subtract the given matrices C = A - B */

PM_matrix *_PM_minus(c, a, b)
   PM_matrix *c, *a, *b;
   {int i, j, row, col;

    row = a->nrow;
    col = a->ncol;
    if ((row != b->nrow) || (col != b->ncol))
       {sprintf(PM_error, "ROW COLUMN MISMATCH - PM_MINUS");
        return(NULL);};

    for (i = 1; i <= row; i++)
        for (j = 1; j <= col; j++)
            PM_element(c, i, j) = PM_element(a, i, j) - PM_element(b, i, j);

    return(c);}
                
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_MINUS - subtract the given matrices */

PM_matrix *PM_minus(a, b)
   PM_matrix *a, *b;
   {int row, col;

    row = a->nrow;
    col = a->ncol;
    if ((row != b->nrow) || (col != b->ncol))
       {sprintf(PM_error, "ROW COLUMN MISMATCH - PM_MINUS");
        return(NULL);}

    else
       return(_PM_minus(PM_create(row, col), a, b));}
                
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

