/*
 * GSCRV.C - 1D renderings for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"
#include "gsrast.h"

#define LABEL_TEXT_SIZE         30
#define GRAPH_LABEL_TEXT_SIZE   45

int
 PG_hide_rescale  = FALSE,
 PG_plot_labels   = TRUE,
 _PG_hist_start   = 0,
 _PG_plot_type    = CARTESIAN,
 _PG_logical_plot = FALSE,
 _PG_scatter_plot = FALSE;

REAL
 _PG_error_cap_size = 0.018;

char
 *_PG_text_format = NULL;

static void
 SC_DECLARE(PG_error_plot,
         (PG_device *dev, REAL *x, REAL *y, int n, int lncol,
          double lnwid, int lnsty, int scatter, int marker, int l,
	  pcons *info));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_CURVE - setup a window for the 1D-1D renderings
 *                        - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_curve(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int j, n, rendering, plt, bare, datafl, change;
    REAL xmx, xmn, ymx, ymn;
    REAL xmax, xmin, ymax, ymin;
    REAL nvxmn, nvxmx, nvymn, nvymx;
    REAL rco, tco, xco, yco, dy, tol, avg;
    REAL extr[2], *dextr, *rextr;
    REAL *vwprt, *rc, *tc;
    PM_mapping *pf;
    PM_set *domain, *range;
    PG_graph *g;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;
    static REAL HC = 0.0;

    if (HC == 0.0)
       HC = HUGE_REAL*HUGE_REAL*HUGE_REAL;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    pd->palette_fl = FALSE;
    pd->legend_fl  = FALSE;

    rendering = pd->rendering;
    alst      = pd->alist;
    pri       = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = PLOT_CURVE;};};

    plt = pd->pl_type;

    pd->palette_fl = FALSE;
    pd->legend_fl  = FALSE;

/* setup the viewport */
    vwprt = pd->viewport;
    if (vwprt != NULL)
       {nvxmn = vwprt[0];
        nvxmx = vwprt[1];
        nvymn = vwprt[2];
        nvymx = vwprt[3];
        bare  = FALSE;}
    else
       {nvxmn = 0.175;
        nvxmx = (plt == POLAR) ? 0.825 : 0.925;
        nvymn = 0.175;
        nvymx = 0.825;
        bare  = dev->print_labels;};

    if (change)
       PG_set_viewport(dev, nvxmn, nvxmx, nvymn, nvymx);

    pd->curve_fl = bare;

/* find the data extrema */
    xmin =  HUGE_REAL;
    ymin =  HUGE_REAL;
    xmax = -HUGE_REAL;
    ymax = -HUGE_REAL;

    datafl = PG_render_data_type(data);
    if (datafl)
       {for (g = data; g != NULL; g = g->next)
	    {for (pf = g->f; pf != NULL; pf = pf->next)
	         {domain = pf->domain;
		  range = pf->range;
		  if (range == NULL)
		     continue;

		  PM_array_real(domain->element_type, domain->extrema, 2, extr);

		  xmn = extr[0];
		  xmx = extr[1];

		  PM_array_real(range->element_type, range->extrema, 2, extr);

		  ymn = extr[0];
		  ymx = extr[1];

		  if ((xmn > HC) || (xmx > HC) || (ymn > HC) || (ymx > HC) ||
		      (xmn < -HC) || (xmx < -HC) || (ymn < -HC) || (ymx < -HC))
		     continue;

		  xmin = min(xmin, xmn);
		  xmax = max(xmax, xmx);
		  ymin = min(ymin, ymn);
		  ymax = max(ymax, ymx);};};

	domain = data->f->domain;
	range  = data->f->range;

	if (rendering == LOGICAL)
	   {xmin = 1.0;
	    xmax = 1.0;
	    for (g = data; g != NULL; g = g->next)
	        {for (pf = g->f; pf != NULL; pf = pf->next)
		     {n    = pf->domain->n_elements;
		      xmax = max(xmax, n);};};};

	dextr = PM_get_limits(domain);
	rextr = PM_get_limits(range);}

    else
       {PG_image_picture_info(data, NULL,
			      &xmin, &xmax, &ymin, &ymax,
			      NULL, NULL);
	dextr = NULL;
	rextr = NULL;};

    if (xmin >= xmax)
       {xmin = 0.0;
        xmax = 1.0;};

    if ((ymin == ymax) && (ymin != 0.0))
       {dy = 0.005*(ymin + ymax);
        ymin -= dy;
        ymax += dy;}

    else if (ymin >= ymax)
       {ymin = 0.0;
        ymax = 1.0;}

/* set the plot limits and set the range and domain controls */
/*    if ((dev->autorange == TRUE) || (rextr == NULL)) */
    if (rextr == NULL)
       {ymn = ymin;
        ymx = ymax;}
    else
       {ymn = rextr[0];
        ymx = rextr[1];};

    if ((dev->autodomain == TRUE) || (dextr == NULL) || (rendering == LOGICAL))
       {xmn = xmin;
	xmx = xmax;}
    else
       {xmn = dextr[0];
        xmx = dextr[1];};

    if (datafl && (plt == POLAR))
       {xmin =  HUGE_REAL;
        ymin =  HUGE_REAL;
        xmax = -HUGE_REAL;
        ymax = -HUGE_REAL;
        for (g = data; g != NULL; g = g->next)
	    {for (pf = g->f; pf != NULL; pf = pf->next)
	         {domain = pf->domain;
		  range  = pf->range;

/* GOTCHA: we are not format converting here because it is not clear
 *         what the correct conversion strategy is wrt the conversion
 *         that must be done below. Do something later if necessary.
 */
		  tc = *((REAL **) domain->elements);
		  rc = *((REAL **) range->elements);
		  n  = domain->n_elements;
		  for (j = 0; j < n; j++)
		      {rco = rc[j];
		       tco = tc[j];
		       if (((tco < xmn) || (xmx < tco)) ||
			   ((rco < ymn) || (ymx < rco)))
			  continue;
		       xco = rco*cos(tco);
		       yco = rco*sin(tco);
		       xmin = min(xmin, xco);
		       xmax = max(xmax, xco);
		       ymin = min(ymin, yco);
		       ymax = max(ymax, yco);};};};

        xmn = xmin - 0.0005*(xmax - xmin);
        xmx = xmax + 0.0005*(xmax - xmin);
        ymn = ymin - 0.0005*(ymax - ymin);
        ymx = ymax + 0.0005*(ymax - ymin);};

    if (change)

/* setup the range limits */
       {extr[0] = ymn;
	extr[1] = ymx;
	PG_register_range_extrema(dev, 1, extr);

/* setup the window limits */
	avg = 0.5*(xmn + xmx);
	tol = (avg == 0.0) ? TOLERANCE : ABS(0.001*avg);
	if (xmn + tol > xmx)
	   {xmn -= tol;
	    xmx += tol;};

	avg = 0.5*(ymn + ymx);
	tol = (avg == 0.0) ? TOLERANCE : ABS(0.001*avg);
	if (ymn + tol > ymx)
	   {ymn -= tol;
	    ymx += tol;};

	PG_set_window(dev, xmn, xmx, ymn, ymx);

	PG_set_palette(dev, "standard");};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_CURVE_PLOT - plot graphs with 1D domains and ranges */

#ifdef PCC

void PG_curve_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_curve_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {int j, n, rendering;
    int color, plt;
    REAL xmx, xmn, ymx, ymn;
    REAL dextr[2];
    REAL *x, *y, *ix;
    PM_set *domain, *range;
    PM_mapping *pf;
    PG_graph *g;
    PG_picture_desc *pd;
    pcons *info;

    data->rendering = PLOT_CURVE;

    pd = PG_setup_picture(dev, data, FALSE, TRUE, TRUE);

/* plot all of the current curves */
    for (g = data; g != NULL; g = g->next)
        {for (pf = g->f; pf != NULL; pf = pf->next)
	     {domain = pf->domain;
	      range  = pf->range;

	      if (pf != g->f)
                 {REAL rmn, rmx;

		  PM_array_real(domain->element_type, domain->extrema,
				2, dextr);

		  PG_get_viewport_WC(dev, &xmn, &xmx, &rmn, &rmx);

		  xmn = max(xmn, dextr[0]);
		  xmx = min(xmx, dextr[1]);

		  WtoS(dev, xmn, rmn);
		  WtoS(dev, xmx, rmx);

		  PG_clear_region_NDC(dev, xmn, xmx, ymn, ymx, 0);};

	      n = domain->n_elements;
	      x = NULL;
	      y = PM_array_real(range->element_type,
				DEREF(range->elements), n, NULL);

	      if (data->info_type != NULL)
		 {if (strcmp(data->info_type, SC_PCONS_P_S) == 0)
		     {int *pc;

		      info = (pcons *) data->info;
		      SC_assoc_info(info,
				    "LINE-COLOR", &pc,
				    NULL);

		      color = (pc == NULL) ? dev->BLUE : *pc;};}

	      else
		 {info  = NULL;
		  color = dev->BLUE;};

	      if (rendering == LOGICAL)
		 {ix = FMAKE_N(REAL, n, "PG_CURVE_PLOT:ix");
		  for (j = 1; j <= n; j++)
		      ix[j-1] = (double) j;

		  PG_plot_curve(dev, ix, y, n, info, 1);

		  if ((dev->data_id == TRUE) && (plt != INSEL))
		     {PG_set_color_text(dev, color, TRUE);
		      PG_draw_data_ids(dev, ix, y,
				       (int) domain->n_elements,
				       g->identifier, info);
		      PG_set_color_text(dev, dev->WHITE, TRUE);};

		  SFREE(ix);}

	      else
		 {x = PM_array_real(domain->element_type,
				    DEREF(domain->elements), n, NULL);

		  PG_plot_curve(dev, x, y, n, info, 1);

		  if ((dev->data_id == TRUE) && (plt != INSEL))
		     {PG_set_color_text(dev, color, TRUE);
		      PG_draw_data_ids(dev, x, y,
				       (int) domain->n_elements,
				       g->identifier, info);
		      PG_set_color_text(dev, dev->WHITE, TRUE);};

		  SFREE(x);};

	      SFREE(y);

	      PG_draw_domain_boundary(dev, pf);};};

    pd->curve_fl &= dev->print_labels;

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PRINT_GRAPH_LABELS - print the curve labels at the bottom
 *                        - of the screen/page
 */

void _PG_print_graph_labels(dev, data)
   PG_device *dev;
   PG_graph *data;
   {int k, color, size, datafl;
    PG_graph *pd;
    REAL xmn, xmx, ymn, ymx;
    REAL yoffset, xofflmt, xofffil, dx, dy, xoffset;
    REAL dextr[2], rextr[2];
    int xlog, ylog;
    char label[MAXLINE], *face, *style;
    PM_set *domain, *range;
    PG_image *im;

    if (!PG_plot_labels)
       return;

    PG_get_font(dev, &face, &style, &size);
    PG_get_text_color(dev, &color);
    PG_set_color_text(dev, dev->WHITE, TRUE);

    xlog = dev->ifxlog;
    ylog = dev->ifylog;

    PG_set_axis_log_scale(dev, FALSE, FALSE);
    PG_set_font(dev, face, style, 8);
    PG_set_clipping(dev, FALSE);

    PG_get_viewport_NDC(dev, &xmn, &xmx, &ymn, &ymx);

    dx = xmx - xmn;
    dy = ymx - ymn;

    yoffset = ymn - 0.15*dy;
    dy     *= 0.016;

    xoffset = dev->fxmin + 0.05;
    xofflmt = xoffset + 0.5*dx;
    xofffil = xoffset + 1.15*dx;

#ifdef HAVE_JPEGLIB
    if (JPG_DEVICE(dev))
	xofffil = xoffset + 1.2*(xmx - xmn);
#endif

    PG_write_NDC(dev, xoffset, yoffset,
                 "CURVE  LABEL                   ");
    PG_write_NDC(dev, xofflmt, yoffset,
                 "NPTS     XMIN       XMAX        YMIN       YMAX");
    PG_write_NDC(dev, xofffil, yoffset,
                 "  FILE");
    yoffset -= 2*dy;

    datafl = PG_render_data_type(data);
    if (datafl)
       {for (pd = data; pd != NULL; pd = pd->next)
	    {domain = pd->f->domain;
	     PM_array_real(domain->element_type, domain->extrema, 2, dextr);

	     range = pd->f->range;
	     PM_array_real(range->element_type, range->extrema, 2, rextr);

	     if (pd->f->name == NULL)
	        for (k = 0; k < GRAPH_LABEL_TEXT_SIZE; label[k++] = ' ');
	     else
	        strncpy(label, pd->f->name, GRAPH_LABEL_TEXT_SIZE);
	     label[GRAPH_LABEL_TEXT_SIZE] = '\0';
	     PG_write_NDC(dev, xoffset, yoffset,
			  "   %c    %s  ", pd->identifier, label);

	     PG_write_NDC(dev, xofflmt, yoffset,
			  "%4d  %10.2e %10.2e  %10.2e %10.2e",
			  pd->f->domain->n_elements,
			  dextr[0], dextr[1],
			  rextr[0], rextr[1]);
	     if (pd->f->file != NULL)
	        PG_write_NDC(dev, xofffil, yoffset, "%s", pd->f->file);

#ifdef HAVE_JPEGLIB
	     if (JPG_DEVICE(dev))
	        yoffset -= 0.5*dy;
#endif
	     yoffset -= dy;};}

    else
       {im = (PG_image *) data;

	strncpy(label, im->label, GRAPH_LABEL_TEXT_SIZE);
	label[GRAPH_LABEL_TEXT_SIZE] = '\0';
	PG_write_NDC(dev, xoffset, yoffset,
		     "   A    %s  ", label);

	PG_write_NDC(dev, xofflmt, yoffset,
		     "      %10.2e %10.2e  %10.2e %10.2e",
		     im->xmin, im->xmax,
		     im->ymin, im->ymax);};

    PG_set_clipping(dev, TRUE);

    PG_set_color_text(dev, color, TRUE);
    PG_set_axis_log_scale(dev, xlog, ylog);
    PG_set_font(dev, face, style, size);

    SFREE(face);
    SFREE(style);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_PLOT_CURVE - dispatch to the desired curve plotting routine */
 
void PG_plot_curve(dev, x, y, n, info, l)
   PG_device *dev;
   REAL *x, *y;
   int n;
   pcons *info;
   int l;
   {int lncol, lnsty, scatter, marker, start, type;
    int *ppt, *pst, *pmk, *psc, *plc, *pls;
    double lnwid, *plw;

    SC_assoc_info(info,
		  "PLOT-TYPE", &ppt,
		  "HIST-START", &pst,
		  "MARKER-INDEX", &pmk,
		  "SCATTER", &psc,
		  "LINE-COLOR", &plc,
		  "LINE-WIDTH", &plw,
		  "LINE-STYLE", &pls,
		  NULL);
    type    = (ppt != NULL) ? *ppt : CARTESIAN;
    start   = (pst != NULL) ? *pst : _PG_hist_start;
    marker  = (pmk != NULL) ? *pmk : 0;
    scatter = (psc != NULL) ? *psc : _PG_scatter_plot;
    lncol   = (plc != NULL) ? *plc : dev->BLUE;
    lnwid   = (plw != NULL) ? *plw : 0.0;
    lnsty   = (pls != NULL) ? *pls : SOLID;

    switch (type)
       {case INSEL :
	     PG_insel_plot(dev, x, y, n,
			   lncol, lnwid, lnsty, l);
	     break;

        case POLAR :
	     PG_polar_plot(dev, x, y, n, lncol, lnwid, lnsty,
			   scatter, marker, l);
	     break;

        case HISTOGRAM :
	     PG_histogram_plot(dev, x, y, n, lncol, lnwid, lnsty,
			       scatter, marker, start, l);
	     break;

        case SCATTER   :
        case CARTESIAN :
             PG_rect_plot(dev, x, y, n, lncol, lnwid, lnsty,
			  scatter, marker, l);
	     break;

        case ERROR_BAR :
             PG_error_plot(dev, x, y, n, lncol, lnwid, lnsty,
			   scatter, marker, l, info);
	     break;

        default :
	     break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INTERP_POINT - interpolate to point at window boundary
 *                  - take into account possible log scales
 */

static void _PG_interp_point(dev, yp, x, x1, y1, x2, y2)
   PG_device *dev;
   REAL *yp;
   double x, x1, y1, x2, y2;
   {REAL y;
    REAL xta, xtb, xtc, yta, ytb;
    
    y = 0.0;
    
    if (dev->ifxlog)
       {x1 = log10(ABS(x1) + SMALL);
        x2 = log10(ABS(x2) + SMALL);
        x  = log10(ABS(x)  + SMALL);};

    if (dev->ifylog)
       {PM_interp(y, x, x1, y1, x2, y2);
        if (y <= 0.0)
           {*yp = -HUGE;
            return;};

        y1 = log10(ABS(y1) + SMALL);
        y2 = log10(ABS(y2) + SMALL);};

    WtoS(dev, x1, y1);
    WtoS(dev, x2, y2);
    WtoS(dev, x, y);

    PM_interp(y, x, x1, y1, x2, y2);

    StoW(dev, x, y);

    if (dev->ifxlog)
       x = POW(10.0, x);

    if (dev->ifylog)
       y = POW(10.0, y);

    *yp = y;

    return;}

/*--------------------------------------------------------------------------*/

#if 0

/*--------------------------------------------------------------------------*/

/* PG_DRAW_DATA_IDS_ALT - draw the curve data ids */
 
static void PG_draw_data_ids_alt(dev, x, y, n, label, info)
   PG_device *dev;
   REAL *x, *y;
   int n;
   int label;
   pcons *info;
   {char mark[2];
    int i, j, k, j0, m, type, scatter;
    int imn, imx;
    int *ppt, *psc;
    REAL s, ds, r, wxmin, wxmax, wymin, wymax, *ln;
    REAL xp, yp;
    REAL x1, x2, y1, y2, s1, s2;
    REAL xta, xtb, xtc, yta, ytb;
    REAL dx, dy, dt;


    PG_set_clipping(dev, TRUE);
    PG_get_viewport_WC(dev, &wxmin, &wxmax, &wymin, &wymax);
 
    mark[0] = label;
    mark[1] = '\0';

    m = 6;
    r = 0.5*(1.0 + PM_random(1));
    k = n/m;

    if (dev->ifxlog)
       dx = POW(10.0, (log10(wxmax/wxmin))/((double) m));
    else
       dx = (wxmax - wxmin)/((double) m);

    SC_assoc_info(info,
		  "PLOT-TYPE", &ppt,
		  "SCATTER", &psc,
		  NULL);
    type    = (ppt != NULL) ? *ppt : CARTESIAN;
    scatter = (psc != NULL) ? *psc : FALSE;

    if (!scatter && ((type == CARTESIAN) || (type == HISTOGRAM)))
       {ln = FMAKE_N(REAL, n, "PG_DRAW_DATA_IDS_ALT:ln");

/* NOTE: the following assume that the arrays are sorted */
        for (imn = 0; imn < n; imn++)
	    {if ((wxmin <= x[imn]) && (wymin <= y[imn]))
                break;};

        for (imx = n-1; imx >= 0; imx--)
	    {if ((x[imx] <= wxmax) && (y[imx] <= wymax))
                break;};

/* find the lengths of each segment */
        x1 = x[0];
        y1 = y[0];
        for (i = 1; i < n; i++)
            {x2 = x[i];
             y2 = y[i];
	     ln[i] = PM_hypot(x2 - x1, y2 - y1);
	     x1 = x2;
             y1 = y2;};

/* integrate to find the total curve length */
        for (i = 1, s = 0.0; i < n; i++)
            {s += ln[i];
             ln[i] = s;};

/* normalize to find the fractional curve length */
        for (i = 1, s = 1.0/ln[n-1]; i < n; i++)
            ln[i] *= s;

	if ((dev->ifxlog) || (dev->ifylog))
	   {dx = wxmax - wxmin;
	    dy = wymax - wymin;
	    dt = max(dx, dy);
	    ds = log(dt);
	    ds = ABS(ds);
	    ds = min(ds, _PG_axis_n_decades);
	    s  = ln[imx]*POW(ds, -((double) m));}
	else
	   {if (imn == imx)
               {if (imn == 0)
                   imx++;
                else
		   imn--;};

	    ds = (ln[imx] - ln[imn])/((double) m);
	    s  = ln[imn] + r*ds;};

	for (i = 0; i < m; i++)
	    {j = PM_find_index(ln, s, n);

             if ((imn <= j) && (j <= imx))
                {if (type == HISTOGRAM || scatter)
                    {xp = x[j-1];
                     yp = y[j-1];}
                 else
                    {x1 = x[j-1];
                     y1 = y[j-1];
                     x2 = x[j];
                     y2 = y[j];
                     s1 = ln[j-1];
                     s2 = ln[j];

                     if (x1 < wxmin)
                        {PM_interp(y1, wxmin, x1, y1, x2, y2);
                         x1 = wxmin;}
                     if (x2 > wxmax)
                        {PM_interp(y2, wxmax, x1, y1, x2, y2);
                         x2 = wxmax;}

		     xp = x1 + (x2 - x1)*(s - s1)/(s2 - s1 + SMALL);
		     if ((dev->ifxlog) || (dev->ifylog))
			s *= ds;
                     else
		        s += ds;

                     _PG_interp_point(dev, &yp, xp, x1, y1, x2, y2);};

                 if ((wxmin < xp) && (xp < wxmax) &&
                     (wymin < yp) && (yp < wymax))
                    PG_write_WC(dev, xp, yp, mark);};};

	SFREE(ln);}

    else if (type == POLAR)
       {REAL tc, rc, xc, yc;

	j0 = ((REAL) k)*r;
	j0 = abs(j0);

        for (i = 0; i < m; i++)
            {j  = j0 + k*i;
             tc = x[j];
             rc = y[j];
             xc = rc*cos(tc);
             yc = rc*sin(tc);
             if (((wxmin <= xc) && (xc <= wxmax)) &&
                 ((wymin <= yc) && (yc <= wymax)))
                PG_write_WC(dev, xc, yc, mark);};};
 
    PG_set_clipping(dev, FALSE);
 
    return;}
 
/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* PG_DRAW_DATA_IDS - draw the curve data ids */
 
void PG_draw_data_ids(dev, x, y, n, label, info)
   PG_device *dev;
   REAL *x, *y;
   int n;
   int label;
   pcons *info;
   {char mark[2];
    int i, j, k, j0, m, type, scatter;
    int *ppt, *psc;
    REAL r, wxmin, wxmax, wymin, wymax;
    REAL xmn, xmx, ymn, ymx;
    REAL dx, x0, xp, yp;
    REAL x1, x2, y1, y2;
    REAL xta, xtb, xtc, yta, ytb;

    PG_set_clipping(dev, TRUE);
    PG_get_viewport_WC(dev, &wxmin, &wxmax, &wymin, &wymax);
    if (wxmin > wxmax)
       {r = wxmin;
	wxmin = wxmax;
	wxmax = r;};
    if (wymin > wymax)
       {r = wymin;
	wymin = wymax;
	wymax = r;};
    PM_maxmin(x, &xmn, &xmx, n);
    PM_maxmin(y, &ymn, &ymx, n);
    xmn = max(wxmin, xmn);
    ymn = max(wymin, ymn);
    xmx = min(wxmax, xmx);
    ymx = min(wymax, ymx);
    if (ymn == ymx)
       {ymn -= 1.0;
	ymx += 1.0;};

    mark[0] = label;
    mark[1] = '\0';

    m = 6;
    r = 0.5*(1.0 + PM_random(1));

    if (dev->ifxlog)
       {x0  = xmx/(POW(10.0, (double) _PG_axis_n_decades));
        xmn = max(xmn, x0);
	dx  = POW(10.0, (log10(xmx/xmn))/((double) m));
        x0  = xmn*POW(dx, r);}
    else
       {dx = (xmx - xmn)/((double) m);
        x0 = xmn + r*dx;}

    SC_assoc_info(info,
		  "PLOT-TYPE", &ppt,
		  "SCATTER", &psc,
		  NULL);
    type    = (ppt != NULL) ? *ppt : CARTESIAN;
    scatter = (psc != NULL) ? *psc : FALSE;

    if ((type == CARTESIAN) || (type == SCATTER) || (type == HISTOGRAM))
       {for (i = 0; i < m; i++)
            {if (dev->ifxlog)
                xp = x0*POW(dx, i);
             else
                xp = x0 + i*dx;

             j = PM_find_index(x, xp, n);
             if ((0 < j) && (j < n))
                {if (type == HISTOGRAM || scatter)
                    {xp = x[j-1];
                     yp = y[j-1];}
                 else
                    {x1 = x[j-1];
                     y1 = y[j-1];
                     x2 = x[j];
                     y2 = y[j];

                     if (x1 < xmn)
                        {PM_interp(y1, xmn, x1, y1, x2, y2);
                         x1 = xmn;}
                     if (x2 > xmx)
                        {PM_interp(y2, xmx, x1, y1, x2, y2);
                         x2 = xmx;}

                     _PG_interp_point(dev, &yp, xp, x1, y1, x2, y2);};

                 if ((xmn < xp) && (xp < xmx) &&
                     (ymn < yp) && (yp < ymx))
                    PG_write_WC(dev, xp, yp, mark);};};}

    else if (type == POLAR)
       {REAL tc, rc, xc, yc;

	k  = n/m;
	j0 = ((REAL) k)*r;
	j0 = abs(j0);

        for (i = 0; i < m; i++)
            {j  = j0 + k*i;
             tc = x[j];
             rc = y[j];
             xc = rc*cos(tc);
             yc = rc*sin(tc);
             if (((xmn <= xc) && (xc <= xmx)) &&
                 ((ymn <= yc) && (yc <= ymx)))
                PG_write_WC(dev, xc, yc, mark);};};
 
    PG_set_clipping(dev, FALSE);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_ERROR_PLOT - user interface to draw a scatter plot with error bars */

static void PG_error_plot(dev, x, y, n, lncol, lnwid, lnsty,
			  scatter, marker, l, info)
   PG_device *dev;
   REAL *x, *y;
   int n, lncol;
   double lnwid;
   int lnsty, scatter, marker, l;
   pcons *info;
   {int i, tn;
    REAL *tx, *ty, *px, *py;
    REAL x0, x1, x2, y0, y1, y2, dx, dy;
    REAL xmn, xmx, ymn, ymx;
    REAL *xp, *xm, *yp, *ym;

    SC_assoc_info(info,
		  "DY-PLUS", &yp,
		  "DY-MINUS", &ym,
		  "DX-PLUS", &xp,
		  "DX-MINUS", &xm,
		  NULL);

    if (l == 0)
       {PG_set_limits(dev, x, y, n, CARTESIAN);
        PG_axis(dev, CARTESIAN);};
 
    PG_set_clipping(dev, TRUE);
    PG_set_color_line(dev, lncol, TRUE);
    PG_set_line_width(dev, lnwid);
    PG_set_line_style(dev, lnsty);
 
    tn = 12*n*(((xp != NULL) || (xm != NULL)) +
               ((yp != NULL) || (ym != NULL)));
    tx = px = FMAKE_N(REAL, tn, "PG_ERROR_PLOT:tx");
    ty = py = FMAKE_N(REAL, tn, "PG_ERROR_PLOT:ty");

    if (xm == NULL)
       xm = xp;

    if (ym == NULL)
       ym = yp;

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
    dx = _PG_error_cap_size*(xmx - xmn);
    dy = _PG_error_cap_size*(ymx - ymn);

    for (i = 0, tn = 0; i < n; i++)
        {x0 = x[i];
         y0 = y[i];
         if (xp != NULL)
	    {x1 = x0 - xm[i];
	     x2 = x0 + xp[i];

	     *px++ = x1;
	     *py++ = y0;
	     *px++ = x2;
	     *py++ = y0;
             tn++;
             if (_PG_error_cap_size > 0.0)
                {y1 = y0 + dy;
                 y2 = y0 - dy;

		 *px++ = x1;
		 *py++ = y1;
		 *px++ = x1;
		 *py++ = y2;
		 tn++;

		 *px++ = x2;
		 *py++ = y1;
		 *px++ = x2;
		 *py++ = y2;
		 tn++;};};

	 if (yp != NULL)
	    {y1 = y0 - ym[i];
	     y2 = y0 + yp[i];

	     *px++ = x0;
	     *py++ = y1;
	     *px++ = x0;
	     *py++ = y2;
             tn++;
             if (_PG_error_cap_size > 0.0)
                {x1 = x0 + dx;
                 x2 = x0 - dx;

		 *px++ = x1;
		 *py++ = y1;
		 *px++ = x2;
		 *py++ = y1;
		 tn++;

		 *px++ = x1;
		 *py++ = y2;
		 *px++ = x2;
		 *py++ = y2;
		 tn++;};};};

    PG_draw_disjoint_polyline_2(dev, tx, ty, tn, FALSE, WORLDC);
 
    SFREE(tx);
    SFREE(ty);

    PG_set_clipping(dev, FALSE);
    PG_set_line_width(dev, 0.0);
    PG_set_line_style(dev, SOLID);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_RECT_PLOT - make a Cartesian plot */

void PG_rect_plot(dev, x, y, n, lncol, lnwid, lnsty, scatter, marker, l)
   PG_device *dev;
   REAL *x, *y;
   int n, lncol;
   double lnwid;
   int lnsty, scatter, marker, l;
   {int tn;
    REAL *tx, *ty;
    
    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% Begin CURVE\n%\n");
 
    if (l == 0)
       {PG_set_limits(dev, x, y, n, CARTESIAN);
        PG_axis(dev, CARTESIAN);};
 
    PG_set_clipping(dev, TRUE);
    PG_set_color_line(dev, lncol, TRUE);
    PG_set_line_width(dev, lnwid);
    PG_set_line_style(dev, lnsty);
 
/* Worst case scenario is every line between adjacent points
 * intersects domain/range boundary rectangle twice.
 */
    tn = 2.0*n;
    tx = FMAKE_N(REAL, tn, "PG_RECT_PLOT:tx");
    ty = FMAKE_N(REAL, tn, "PG_RECT_PLOT:ty");

    PG_clip_data(dev, tx, ty, &tn, x, y, n);

/* plot lines unless scatter set */
    if (scatter != 1)
       PG_draw_polyline(dev, tx, ty, tn, FALSE);
    else
       PG_draw_markers(dev, tn, tx, ty, marker);
 
    SFREE(tx);
    SFREE(ty);

    PG_set_clipping(dev, FALSE);
    PG_set_line_width(dev, 0.0);
    PG_set_line_style(dev, SOLID);
 
    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% End CURVE\n%\n");
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_HISTOGRAM_PLOT - make a Cartesian histogram plot */

void PG_histogram_plot(dev, x, y, n, lncol, lnwid, lnsty, scatter,
		       marker, start, l)
   PG_device *dev;
   REAL *x, *y;
   int n, lncol;
   double lnwid;
   int lnsty, scatter, marker, start, l;
   {int i, j, nl, nm, tn;
    REAL *xl, *yl, *py, *tx, *ty;

    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% Begin CURVE\n%\n");
 
    if (l == 0)
       {PG_set_limits(dev, x, y, n, HISTOGRAM);
        PG_axis(dev, CARTESIAN);};
 
    PG_set_clipping(dev, TRUE);
    PG_set_color_line(dev, lncol, TRUE);
    PG_set_line_width(dev, lnwid);
    PG_set_line_style(dev, lnsty);
 
/* transform data to histogram form */
    nm = n - 1;
    nl = 2*n - 2;
    xl = FMAKE_N(REAL, nl, "PG_HISTOGRAM_PLOT:xl");
    yl = FMAKE_N(REAL, nl, "PG_HISTOGRAM_PLOT:yl");

    if (start < 2)
       {py    = y + start;
        xl[0] = x[0];
        yl[0] = py[0];
        for (i = 1, j = 1; i < nm; i++, j += 2)
            {xl[j] = x[i];
             yl[j] = yl[j-1];

             xl[j+1] = x[i];
             yl[j+1] = py[i];};
        xl[j] = x[i];
        yl[j] = py[i-1];}

    else if (start == 2)
       {py    = y + 1;
        xl[0] = x[0];
        yl[0] = 0.5*(y[0] + y[1]);
        for (i = 1, j = 1; i < nm; i++, j += 2)
            {xl[j] = x[i];
             yl[j] = 0.5*(y[i-1] + y[i]);

             xl[j+1] = x[i];
             yl[j+1] = 0.5*(y[i] + y[i+1]);};
        xl[j] = x[i];
        yl[j] = 0.5*(y[i-1] + y[i]);};

/* Worst case scenario is every line between adjacent points
 * intersects viewport boundary rectangle twice.
 */
    tn = 2*nl;
    tx = FMAKE_N(REAL, tn, "PG_HISTOGRAM_PLOT:tx");
    ty = FMAKE_N(REAL, tn, "PG_HISTOGRAM_PLOT:ty");

    PG_clip_data(dev, tx, ty, &tn, xl, yl, nl);

/* plot lines unless scatter set */
    PG_move_gr_abs(dev, tx[0], ty[0]);

    if (scatter != 1)
       PG_draw_polyline(dev, tx, ty, tn, FALSE);
    else
       PG_draw_markers(dev, tn, tx, ty, marker);
 
    SFREE(tx);
    SFREE(ty);

    SFREE(xl);
    SFREE(yl);

    PG_set_clipping(dev, FALSE);
    PG_set_line_width(dev, 0.0);
    PG_set_line_style(dev, SOLID);
 
    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% End CURVE\n%\n");
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_INSEL_PLOT - make an Inselberg plot */
 
void PG_insel_plot(dev, x, y, n, lncol, lnwid, lnsty, l)
   PG_device *dev;
   REAL *x, *y;
   int n, lncol;
   double lnwid;
   int lnsty, l;
   {int i, ndim;
    REAL r, o, xc, yc, xp1, xp2, xsep;
    REAL wxmin, wxmax, wymin, wymax, wxext;
    REAL xmin, xmax, ymin, ymax, xext, yext;

    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% Begin CURVE\n%\n");
 
    if (l == 0)
       {PG_set_limits(dev, x, y, n, INSEL);
        PG_axis(dev, INSEL);};
 
    PG_set_clipping(dev, TRUE);
    PG_set_color_line(dev, lncol, TRUE);
    PG_set_line_width(dev, lnwid);
    PG_set_line_style(dev, lnsty);
 
/* get window and viewport limits */
    PG_get_viewport_WC(dev, &wxmin, &wxmax, &wymin, &wymax);
    wxext = wxmax - wxmin;

/* scale the x axis so the "domain" is compressed and centered */
    PG_get_bound(dev, &xmin, &xmax, &ymin, &ymax);
    xext = xmax - xmin;
    yext = ymax - ymin;

    r    = yext/xext;
    o    = ymin - r*xmin;
    ndim = 2;
    xsep = wxext/((REAL) (++ndim));
    xp1  = wxmin + xsep;
    xp2  = wxmin + 2.0*xsep;
    for (i = 0; i < n; i++)
        {xc = x[i];
         yc = y[i];
         if (((wxmin <= xc) && (xc <= wxmax)) &&
             ((wymin <= yc) && (yc <= wymax)))
            {xc = r*xc + o;
             PG_draw_line(dev, xp1, xc, xp2, yc);};};
 
    PG_set_clipping(dev, FALSE);
    PG_set_line_width(dev, 0.0);
    PG_set_line_style(dev, SOLID);
 
    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% End CURVE\n%\n");
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_POLAR_PLOT - make a polar plot */
 
void PG_polar_plot(dev, x, y, n, lncol, lnwid, lnsty, scatter, marker, l)
   PG_device *dev;
   REAL *x, *y;
   int n, lncol;
   double lnwid;
   int lnsty, scatter, marker, l;
   {int i;
    REAL xc, yc, rc, tc, xco, yco;
    REAL wxmin, wxmax, wymin, wymax;
    REAL xmin, xmax, ymin, ymax;

    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% Begin CURVE\n%\n");
 
    if (l == 0)
       {PG_set_limits(dev, x, y, n, POLAR);
        PG_axis(dev, POLAR);};
 
    PG_set_clipping(dev, TRUE);
    PG_set_color_line(dev, lncol, TRUE);
    PG_set_line_width(dev, lnwid);
    PG_set_line_style(dev, lnsty);
 
/* get window and viewport limits */
    PG_get_viewport_WC(dev, &wxmin, &wxmax, &wymin, &wymax);

/* scale the x axis so the "domain" is compressed and centered */
    PG_get_bound(dev, &xmin, &xmax, &ymin, &ymax);

    if (scatter)
       {REAL *xd, *yd;
       int nm;

        xd = FMAKE_N(REAL, n, "PG_POLAR_PLOT:xd");
        yd = FMAKE_N(REAL, n, "PG_POLAR_PLOT:yd");
        nm = 0;
        for (i = 1; i < n; i++)
            {tc = x[i];
             rc = y[i];
             xc = rc*cos(tc);
             yc = rc*sin(tc);
             if (((wxmin <= xc) && (xc <= wxmax)) &&
                 ((wymin <= yc) && (yc <= wymax)))
                {xd[nm] = xc;
                 yd[nm] = yc;
                 nm++;};};
        PG_draw_markers(dev, nm, xd, yd, marker);
        SFREE(xd);
        SFREE(yd);}

    else
       {tc  = x[0];
        rc  = y[0];
        xco = rc*cos(tc);
        yco = rc*sin(tc);
        for (i = 1; i < n; i++)
            {tc = x[i];
             rc = y[i];
             xc = rc*cos(tc);
             yc = rc*sin(tc);
             if (((wxmin <= xc) && (xc <= wxmax)) &&
                 ((wymin <= yc) && (yc <= wymax)))
                PG_draw_line(dev, xco, yco, xc, yc);
             xco = xc;
             yco = yc;};};
 
    PG_set_clipping(dev, FALSE);
    PG_set_line_width(dev, 0.0);
    PG_set_line_style(dev, SOLID);
 
    if (POSTSCRIPT_DEVICE(dev))
       io_printf(dev->file, "%s", "\n%\n% End CURVE\n%\n");
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

