/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	NJN/JRV/RBD
 *
 *	$Id: scan.c,v 6.1 96/11/23 22:53:59 nevin Rel $
 *
 *	Function:	- partial reduce from 0 to rank i, at each rank i
 *	Accepts:	- send buffer
 *			- receive buffer
 *			- count of elements
 *			- type of elements
 *			- operation to perform
 *			- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

int
MPI_Scan(sbuf, rbuf, count, dtype, op, comm)

void			*sbuf;
void			*rbuf;
int			count;
MPI_Datatype		dtype;
MPI_Op			op;
MPI_Comm		comm;

{
	int		size;			/* group size */
	int		rank;			/* my rank */
	int		err;			/* error code */
	char		*tmpbuf = 0;		/* temporary buffer */
	char		*origin;		/* data origin in temp buffer */
	MPI_Status	stat;			/* recv status */

	lam_initerr();
	lam_setfunc(BLKMPISCAN);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return (lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if (op == MPI_OP_NULL) {
		return(lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_OP, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPISCAN));
/*
 * Check for zero count case.
 */
	if (count == 0) {
		LAM_TRACE(lam_tr_cffend(BLKMPISCAN, -1, comm, dtype, count));

		lam_resetfunc(BLKMPISCAN);
		return(MPI_SUCCESS);
	}
/*
 * Initialize.
 */
	MPI_Comm_rank(comm, &rank);
	MPI_Comm_size(comm, &size);
/*
 * Switch to collective communicator.
 */
	lam_mkcoll(comm);
/*
 * If I'm rank 0, initialize the recv buffer.
 */
	if (rank == 0) {
		err = lam_dtsndrcv(sbuf, count, dtype,
				rbuf, count, dtype, BLKMPISCAN, comm);
		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCAN, err));
		}
	}
/*
 * Otherwise receive previous buffer and reduce.
 */
	else {
		if (!op->op_commute) {
/*
 * Allocate a temporary buffer.
 */
			err = lam_dtbuffer(dtype, count, &tmpbuf, &origin);
			if (err != MPI_SUCCESS) {
				lam_mkpt(comm);
				return(lam_errfunc(comm, BLKMPISCAN, err));
			}
/*
 * Copy the send buffer into the receive buffer.
 */
			err = lam_dtsndrcv(sbuf, count, dtype, rbuf,
					count, dtype, BLKMPISCAN, comm);
			
			if (err != MPI_SUCCESS) {
				if (tmpbuf) free(tmpbuf);
				lam_mkpt(comm);
				return(lam_errfunc(comm, BLKMPISCAN, err));
			}
			
			err = MPI_Recv(origin, count, dtype,
					rank - 1, BLKMPISCAN, comm, &stat);
		}
		else {
			origin = sbuf;
			
			err = MPI_Recv(rbuf, count, dtype,
					rank - 1, BLKMPISCAN, comm, &stat);
		}

		if (err != MPI_SUCCESS) {
			if (tmpbuf) free(tmpbuf);
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCAN, err));
		}

		if (op->op_f77dtype >= 0) {
			(op->op_func)(origin, rbuf, &count, &(op->op_f77dtype));
		} else {
			(op->op_func)(origin, rbuf, &count, &dtype);
		}

		if (tmpbuf) free(tmpbuf);
	}
/*
 * Send result to next process.
 */
	if (rank < (size - 1)) {
		err = MPI_Send(rbuf, count, dtype, rank + 1, BLKMPISCAN, comm);
		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCAN, err));
		}

	}

	lam_mkpt(comm);
	LAM_TRACE(lam_tr_cffend(BLKMPISCAN, -1, comm, dtype, count));
	lam_resetfunc(BLKMPISCAN);
	return(MPI_SUCCESS);
}
