/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: iprobe.c,v 6.1 96/11/23 22:52:40 nevin Rel $
 *
 *	Function:	- non-blocking check of incoming messages
 *			- message is not received
 *	Accepts:	- source rank
 *			- message tag
 *			- communicator
 *			- flag (returned value)
 *			- status (returned value)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>
#include <terror.h>

/*
 * external functions
 */
extern void		lam_initerr();
extern void		lam_resetfunc();
extern void		lam_setfunc();
extern void		lam_tr_msg();
extern void		_mpi_req_add();
extern void		_mpi_req_blkclr();
extern void		_mpi_req_blkset();
extern void		_mpi_req_rem();
extern int		lam_errfunc();
extern int		lam_mkerr();
extern int		lam_tr_incff();
extern int		_mpi_req_advance();
extern int		_mpi_req_build();
extern int		_mpi_req_destroy();
extern int		_mpi_req_start();
extern int		_mpi_req_end();
extern int		_rpi_c2c_iprobe();
extern int		_rpi_lamd_iprobe();

int
MPI_Iprobe(src, tag, comm, flag, stat)

int			src;
int			tag;
MPI_Comm		comm;
int			*flag;
MPI_Status		*stat;

{
	struct _req	request_storage;	/* request structure */
	MPI_Request	req;			/* request */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	int		r;
	double		startt;			/* start time */

	lam_initerr();
/*
 * Check additional arguments.
 */
	if ((flag == 0) || (stat == 0)) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Handle the trivial case.
 */
	if (src == MPI_PROC_NULL) {
		stat->MPI_ERROR = MPI_SUCCESS;
		stat->MPI_SOURCE = MPI_PROC_NULL;
		stat->MPI_TAG = MPI_ANY_TAG;
		stat->st_count = 0;
		stat->st_nelem = 0;
		stat->st_length = 0;
		*flag = 1;
		return(MPI_SUCCESS);
	}

	lam_setfunc(BLKMPIIPROBE);

	if ((fl_trace = LAM_TRACE_TOP())) startt = MPI_Wtime();
/*
 * Advance the system to give the probe a better chance of success.
 */
	_mpi_req_blkclr();

	err = _mpi_req_advance();
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	req = &request_storage;
	err = _mpi_req_build((char *) 0, 0, MPI_BYTE, src, tag, comm,
			LAM_RQIPROBE, &req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	err = _mpi_req_start(req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	r = RPI_SPLIT(_rpi_lamd_iprobe, _rpi_c2c_iprobe, (req));
	if (r < 0) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_INTERN, errno)));
	}

	err = _mpi_req_end(req);
	if (err != MPI_SUCCESS)
			return(lam_errfunc(comm, BLKMPIIPROBE, err));
/*
 * Copy the status info if synchronized.
 */
	if (r == 1) {
		*flag = 1;
		*stat = req->rq_status;
	} else {
		*flag = 0;
	}

	err = _mpi_req_destroy(&req);
	if (err != MPI_SUCCESS)
			return(lam_errfunc(comm, BLKMPIIPROBE, err));
/*
 * Generate a run time trace.
 */
	if (fl_trace) {
		lam_tr_msg(TRTNOIO, startt, LAM_S2US(MPI_Wtime() - startt),
				0, src, tag, comm, 0, 0, 0, 0, 0,
				LAM_RQIPROBE);
	}

	lam_resetfunc(BLKMPIIPROBE);
	return(MPI_SUCCESS);
}
