/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: finalize.c,v 6.1 96/11/23 22:51:49 nevin Rel $
 *
 *	Function:	- terminate the MPI session
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <lam_config.h>

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#include <terror.h>
#include <trreq.h>
#include <typical.h>
#include <t_types.h>

/*
 * external functions
 */
extern void		_cio_cleanup();
extern void		lam_initerr();
extern void		lam_nukecids();
extern void		lam_nukefunc();
extern void		lam_nukekeys();
extern void		lam_nukeprocs();
extern void		lam_nuketrace();
extern void		lam_resetfunc();
extern void		lam_setfunc();
extern void		lam_tr_cffstart();
extern void             _mpi_req_get();
extern int		lam_errfunc();
extern int              _mpi_req_getn();
extern int		_rpi_c2c_finalize();
extern int		_rpi_lamd_finalize();
extern void		lam_kexit();

/*
 * local functions
 */
static int		waitall();
static void		free_comm();
static void		free_dtype();
static void		free_env();
static void		free_errhdl();
static void		free_op();
static void		free_rdtype();

int
MPI_Finalize()

{
	int		err;			/* error code */
	int4		id;			/* name publshing ID */
	
	lam_initerr();
	lam_setfunc(BLKMPIFINAL);

	LAM_TRACE(lam_tr_cffstart(BLKMPIFINAL));
/*
 * Block until all pending requests are done.
 */
	err = waitall();
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIFINAL, err));
	}
/*
 * RPI specific finalization.
 */
	if (RPI_SPLIT(_rpi_lamd_finalize, _rpi_c2c_finalize, (0))) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIFINAL,
					lam_mkerr(MPI_ERR_INTERN, errno)));
	}
/*
 * Cleanup.
 */
	id = (int4) ((lam_myproc->p_gps.gps_pid << 16)
			| lam_myproc->p_gps.gps_node);
	lam_rtrnamesweep(0, id);
	
	free_env();
	free_rdtype();
	free_comm();
	free_op();
	free_dtype();
	free_errhdl();

	LAM_TRACE(lam_tr_cffend(BLKMPIFINAL, -1, 0, 0, 0));

	lam_tr_off();
	lam_nuketrace();
 	lam_nukeprocs();
	lam_flinit = 0;
	lam_flfinal = 1;
	lam_kexit(0);

	lam_resetfunc(BLKMPIFINAL);
	lam_nukefunc();
	
	return(MPI_SUCCESS);
}

/*
 *	waitall
 *
 *	Function:	- wait till all pending requests are done
 *	Returns:	- MPI_SUCCESS or error code
 */
static int
waitall()

{
	int		err;			/* error code */
	int		nreqs;			/* # pending requests */
	MPI_Request	*reqs;			/* request array */
	MPI_Status	*stats;			/* status array */

	nreqs = _mpi_req_getn();
	if (nreqs == 0) return(MPI_SUCCESS);

	reqs = (MPI_Request *) malloc((unsigned) nreqs * sizeof(MPI_Request));
	stats = (MPI_Status *) malloc((unsigned) nreqs * sizeof(MPI_Status));

	if ((reqs == 0) || (stats == 0)) {
		if (reqs) free((char *) reqs);
		if (stats) free((char *) stats);
		return(lam_mkerr(MPI_ERR_OTHER, errno));
	}

	_mpi_req_get(nreqs, reqs);

	err = MPI_Waitall(nreqs, reqs, stats);

	free((char *) reqs);
	free((char *) stats);

	return(err);
}

/*
 *	free_errhdl
 *
 *	Function:	- deallocate pre-defined error handles
 */
static void
free_errhdl()

{
/*
 * Empty stub, nothing to free.
 */
}

/*
 *	free_comm
 *
 *	Function:	- deallocate pre-defined communicators
 */
static void
free_comm()

{
/*
 * Free the "world" communicator.
 */
	lam_rmcid(MPI_COMM_WORLD->c_contextid);
	free((char *) MPI_COMM_WORLD->c_group);
/*
 * Free the "self" communicator.
 */
	lam_rmcid(MPI_COMM_SELF->c_contextid);
	free((char *) MPI_COMM_SELF->c_group);
/*
 * Free the "parent" communicator.
 */
	lam_rmcid(MPI_COMM_PARENT->c_contextid);
	if (MPI_COMM_PARENT->c_rgroup != MPI_GROUP_EMPTY) {
		free((char *) MPI_COMM_PARENT->c_rgroup);
	}
	
	lam_nukecids();
}

/*
 *	free_dtype
 *
 *	Function:	- deallocate basic (intrinsic) datatypes
 */
static void
free_dtype()

{
/*
 * Empty stub, nothing to free.
 */
}

/*
 *	free_rdtype
 *
 *	Function:	- deallocate reduction datatypes
 */
static void
free_rdtype()

{
	MPI_Datatype	dtype;

	dtype = MPI_2INT;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_2FLOAT; 
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_2DOUBLE;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_FLOAT_INT;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_DOUBLE_INT;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_LONG_DOUBLE_INT;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_LONG_INT;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_SHORT_INT;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);

	dtype = MPI_F_2INTEGER;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_F_2REAL;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
	dtype = MPI_F_2DOUBLE_PRECISION;
	dtype->dt_flags &= ~LAM_PREDEF; MPI_Type_free(&dtype);
}

/*
 *	free_op
 *
 *	Function:	- deallocate intrinsic reduction operations
 */
static void
free_op()

{
/*
 * Empty stub, nothing to free.
 */
}

/*
 *	free_env
 *
 *	Function:	- deallocate environment attributes
 */
static void
free_env()

{
	int		key;

	MPI_Attr_delete(MPI_COMM_WORLD, MPI_TAG_UB);
	MPI_Attr_delete(MPI_COMM_WORLD, MPI_HOST);
	MPI_Attr_delete(MPI_COMM_WORLD, MPI_IO);

	key = MPI_TAG_UB; MPI_Keyval_free(&key);
	key = MPI_HOST; MPI_Keyval_free(&key);
	key = MPI_IO; MPI_Keyval_free(&key);

	lam_nukekeys();
}
