/*
   PXKTextFieldCell.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February-March 1997
   A completely rewritten version of the original source by Scott Christley.
   
   This file is part of the GNUstep GUI X/DPS Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include "config.h"

#include <Foundation/NSAutoreleasePool.h>
#include <Foundation/NSString.h>
#include <Foundation/NSException.h>

#include <AppKit/NSColor.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSCStringText.h>
#include <gnustep/xdps/PXKCell.h>
#include <gnustep/xdps/PXKTextFieldCell.h>
#include <gnustep/xdps/PXKWindow.h>
#include <gnustep/xdps/PXKDPSContext.h>

typedef struct {
  /* Text info */
  float xOffsetOfFirstChar; /* Relative to the origin of cell's frame */
  BOOL textHasBeenChanged;

  /* The cursor position */
  int cursorPositionInString; /* The index of character in string the cursor is
				 positioned. An insert operation in string is
				 done at this index. */
  float xOffsetOfCursor; /* Relative to the origin of cell's frame */
  BOOL cursorPositionHasChanged;
  BOOL cursorIsVisible;

  /* Selection */
  NSRange selectedChars;
} PXKTextFieldCellPrivate;

#define xOffsetOfFirstChar \
    (((PXKTextFieldCellPrivate*)be_tfc_reserved)->xOffsetOfFirstChar)
#define textHasBeenChanged \
    (((PXKTextFieldCellPrivate*)be_tfc_reserved)->textHasBeenChanged)

#define cursorPositionInString \
    (((PXKTextFieldCellPrivate*)be_tfc_reserved)->cursorPositionInString)
#define xOffsetOfCursor \
    (((PXKTextFieldCellPrivate*)be_tfc_reserved)->xOffsetOfCursor)
#define cursorPositionHasChanged \
    (((PXKTextFieldCellPrivate*)be_tfc_reserved)->cursorPositionHasChanged)
#define cursorIsVisible \
    (((PXKTextFieldCellPrivate*)be_tfc_reserved)->cursorIsVisible)

#define selectedChars \
    (((PXKTextFieldCellPrivate*)be_tfc_reserved)->selectedChars)

@implementation PXKTextFieldCell

- (void)createBackendStructure
{
  be_tfc_reserved = malloc (sizeof (PXKTextFieldCellPrivate));
  xOffsetOfFirstChar = 0.0;
  textHasBeenChanged = YES;
  cursorPositionInString = 0;
  xOffsetOfCursor = 0.0;
  cursorIsVisible = NO;
  selectedChars = NSMakeRange (0, 0);
}

- initTextCell:(NSString *)aString
{
  /* Allocate the backend structure */
  [self createBackendStructure];

  return [super initTextCell:aString];
}

- (void)dealloc
{
  free (be_tfc_reserved);
  [super dealloc];
}

- copyWithZone:(NSZone *)zone
{
  id c;
  c = [super copyWithZone: zone];

  [c createBackendStructure];
  return c;
}

- (void)setStringValue:(NSString*)string
{
  if (!be_tfc_reserved)
    [self createBackendStructure];

  textHasBeenChanged = YES;
  cursorPositionInString = -1;
  cursorPositionHasChanged = YES;
  [super setStringValue:string];
}

- (void)setAlignment: (NSTextAlignment)mode
{
  if (!be_tfc_reserved)
    [self createBackendStructure];

  textHasBeenChanged = YES;
  [super setAlignment: mode];
}

- (void)drawInteriorWithFrame:(NSRect)frame inView:(NSView*)controlView
{
  float textHeight;
  NSFont* textFont;
  float y;

  // Save last view drawn to
  control_view = controlView;

  // We apply a clipping rectangle so save the graphics state
  PSgsave();

  /* Translate the coordinate system to frame's origin */
  PStranslate (frame.origin.x + xDist, frame.origin.y + yDist);
  frame.origin = NSZeroPoint;
  frame.size.width -= 2 * xDist;
  frame.size.height -= 2 * yDist;
  PSrectclip (0, 0, frame.size.width, frame.size.height);

  [[self textColor] set];
  textFont = [self font];
  [textFont set];
  textHeight = [textFont pointSize];

  /* Determine the y position of text */
  y = (frame.size.height - textHeight) / 2;

  if (textHasBeenChanged)
    [self _computeTextPosition:frame];
  if (cursorPositionHasChanged)
    [self _adjustCursorPosition:frame];

  PSmoveto (xOffsetOfFirstChar, y - [textFont descender]);
  if ([[self stringValue] cStringLength])
    PSshow ([[self stringValue] cString]);

  /* Show the cursor */
  if (cursorIsVisible) {
    PSsetlinewidth (0);
    PSmoveto (xOffsetOfCursor, y);
    PSrlineto (0, textHeight);
    PSstroke();
  }

  PSgrestore();
}

- (void)drawWithFrame:(NSRect)cellFrame
	       inView:(NSView*)controlView
{
  // We apply a clipping rectangle so save the graphics state
  PSgsave();

  /* Clear the cell frame */
  if ([self drawsBackground])
    {
      [[self backgroundColor] set];
      NSRectFill(cellFrame);
    }

  /* Now draw the border if needed. */
  if ([self isBordered]) {
    if ([self isBezeled]) {
      NSDrawBezel (cellFrame, cellFrame);
      cellFrame.origin.x += 2;
      cellFrame.origin.y += 2;
      cellFrame.size.width -= 4;
      cellFrame.size.height -= 4;
    }
    else {
      NSFrameRect (cellFrame);
      cellFrame.origin.x += 1;
      cellFrame.origin.y += 1;
      cellFrame.size.width -= 2;
      cellFrame.size.height -= 2;
    }
  }

  [self drawInteriorWithFrame:cellFrame inView:controlView];

  PSgrestore();
}

/* Compute the original position of text string in cell. This takes into
   consideration the alignment of text. Position of cursor is adjusted if the
   cell is scrollable and it is outside cell's frame rect. */
- (void)_computeTextPosition:(NSRect)frame
{
  NSFont* textFont = [self font];
  float textWidth = [textFont widthOfString:[self stringValue]];

  /* Determine the horizontal position of text */
  switch ([self alignment]) {
    case NSLeftTextAlignment:
	/* ignore the justified and natural alignments */
    case NSJustifiedTextAlignment:
    case NSNaturalTextAlignment:
      xOffsetOfFirstChar = frame.origin.x;
      break;
    case NSRightTextAlignment:
      if (textWidth < frame.size.width)
	xOffsetOfFirstChar = frame.origin.x + frame.size.width - textWidth;
      else
	xOffsetOfFirstChar = frame.origin.x;
      break;
    case NSCenterTextAlignment:
      if (textWidth < frame.size.width)
	xOffsetOfFirstChar = frame.origin.x
			     + (frame.size.width - textWidth) / 2;
      else
	xOffsetOfFirstChar = frame.origin.x;
      break;
  }

  textHasBeenChanged = NO;
}

- (void)_adjustCursorPosition:(NSRect)frame
{
  // Force a recomputation of the cursor location
  if (cursorPositionInString < 0)
    [self _setCursorLocation: NSZeroPoint];

  /* If text is scrollable, be sure the cursor is into the cell frame. */
  if ([self isScrollable]) {
    float offset = 0;

    if (xOffsetOfCursor < frame.origin.x) {
      offset = frame.origin.x - xOffsetOfCursor;
      xOffsetOfFirstChar += offset;
      xOffsetOfCursor += offset;
    }
    else if (xOffsetOfCursor > frame.origin.x + frame.size.width) {
      offset = xOffsetOfCursor - (frame.origin.x + frame.size.width);
      xOffsetOfFirstChar -= offset;
      xOffsetOfCursor -= offset;
    }
  }

  cursorPositionHasChanged = NO;
}

- (void)_setCursorVisibility:(BOOL)isVisible
{
  cursorIsVisible = isVisible;
}

- (void)_setCursorLocation:(NSPoint)point
{
  NSString* text = [self stringValue];
  NSFont* textFont = [self font];
  float textWidth = [textFont widthOfString:[self stringValue]];

  // Text info must be computed first
  if (textHasBeenChanged)
    return;
  //NSAssert (!textHasBeenChanged, @"text info has not been computed yet");

  /* Adjust the mouse position to take into consideration the border */
  if ([self isBordered]) {
    if ([self isBezeled])
      point.x -= 2;
    else point.x -= 1;
  }
  point.x -= xDist;

  /* Determine the position of cursor inside the text string so that it is in
     between two characters. In addition compute the x offset of cursor. */
  if (point.x <= xOffsetOfFirstChar) {
    cursorPositionInString = 0;
    xOffsetOfCursor = xOffsetOfFirstChar;
  }
  else if (point.x >= xOffsetOfFirstChar + textWidth) {
    cursorPositionInString = [text length];
    xOffsetOfCursor = xOffsetOfFirstChar + textWidth;
  }
  else {
    NSAutoreleasePool* pool = [NSAutoreleasePool new];
    NSFont* textFont = [self font];
    float maximumAdvancement = [textFont maximumAdvancement].width;
    float width = 0;
    NSString* substring;
    int lower, upper, midpoint = 0; /* Positions in the text string */
    BOOL found = NO;

    /* Adjust the horizontal position of cursor to be relative to the beginning
       of text position. */
    point.x -= xOffsetOfFirstChar;

    /* Use a binary search to determine where is positioned the cursor. The
       initial values of lower and upper are determined based on the maximum
       and respectively minimum advancement of the font (the maximum and
       minimun width of glyphs). */ 
    lower = point.x / maximumAdvancement;
    upper = [text length];

    while (upper >= lower) {
      midpoint = (lower + upper) / 2;
      substring = [text substringToIndex:midpoint];
      width = [textFont widthOfString:substring];
      if (width < point.x)
	lower = midpoint + 1;
      else if (width > point.x)
	upper = midpoint - 1;
      else {
	found = YES;
	break;
      }
    }

    /* We need the width of substring up to but not including midpoint to be
       less than cursor's position. The while statement below should be
       executed only once, so an if would be enough, but who knows... */
    while (width > point.x) {
      unichar c = [text characterAtIndex:--midpoint];
      width -= [textFont advancementForGlyph:c].width;
    }

    /* If `found' is YES then the cursor position lies exactly in between two
       characters. Otherwise the click has happened on the character at index
       'midpoint' in string. We have to determine if the click occured in
       the left or in the right half of the character. */
    if (found) {
      cursorPositionInString = midpoint;
      xOffsetOfCursor = xOffsetOfFirstChar + width;
    }
    else {
      float nextWidth;
      unichar c = [text characterAtIndex:midpoint];

      /* Compute the width of the substring with an additional character. Then
         compare the horizontal position of cursor with width and nextWidth to
	 determine the cursor position. */
      nextWidth = width + [textFont advancementForGlyph:c].width;

      NSAssert (width < point.x && point.x < nextWidth,
		@"cursor's position is not in between the two widths");

      if (nextWidth - point.x > point.x - width) {
	/* The click has happened in the first half of character */
	cursorPositionInString = midpoint;
	xOffsetOfCursor = xOffsetOfFirstChar + width;
      }
      else {
	cursorPositionInString = midpoint + 1;
	xOffsetOfCursor = xOffsetOfFirstChar + nextWidth;
      }
    }

    [pool release];
  }
}

- (void)deleteAfterCursor
{
  if (cursorPositionInString < [[self stringValue] length]) {
    NSMutableString* mText = [[[self stringValue] mutableCopy] autorelease];
    NSRange range = NSMakeRange(cursorPositionInString, 1);
  
    [mText deleteCharactersInRange:range];
    [super setStringValue:mText];
  }
}

- (void)moveCursorToLeft
{
  NSSize advancement;

  if (cursorPositionInString) {
    cursorPositionInString--;
    advancement = [[self font] advancementForGlyph:
	  [[self stringValue] characterAtIndex:cursorPositionInString]];
    xOffsetOfCursor -= advancement.width;
    cursorPositionHasChanged = YES;
    [super setStringValue:[self stringValue]];
  }
}

- (void)moveCursorToRight
{
  NSSize advancement;

  if (cursorPositionInString < [[self stringValue] length]) {
    advancement = [[self font] advancementForGlyph:
	  [[self stringValue] characterAtIndex:cursorPositionInString]];
    xOffsetOfCursor += advancement.width;
    cursorPositionInString++;
    cursorPositionHasChanged = YES;
    [super setStringValue:[self stringValue]];
  }
}

/* TODO: Change this implementation when 16-bit character strings will be
   available. This is just a hack to make the text field work. */
- (void)_handleKeyEvent:(NSEvent*)keyEvent
{
  NSMutableString* mText;
  NSString* keys = [keyEvent characters];
  unsigned short keyCode = [keyEvent keyCode];

  if (!keyCode) {
    /* The key is a normal key; insert its character in the string value. */
    NSString* substring;
    NSFont* font = [self font];

    mText = [[[self stringValue] mutableCopy] autorelease];
    [mText insertString:keys atIndex:cursorPositionInString];
    cursorPositionInString += [keys length];
    substring = [mText substringToIndex:cursorPositionInString];
    xOffsetOfCursor = xOffsetOfFirstChar + [font widthOfString:substring];
    cursorPositionHasChanged = YES;

    /* Call the super's method since our method will also modify the
       'textHasBeenChanged' thus making the string to be displayed in the
       default position. */
    [super setStringValue:mText];
    return;
  }

  /* The key is a special key; handle it below */
  switch (keyCode) {
    case NSLeftArrowFunctionKey:
      [self moveCursorToLeft];
      break;

    case NSRightArrowFunctionKey:
      [self moveCursorToRight];
      break;

    case NSBackspaceKey:
      [self moveCursorToLeft];
      [self deleteAfterCursor];
      break;

    case NSDeleteFunctionKey:
      [self deleteAfterCursor];
      break;
  }
}

@end
