
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "../gb.h"

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
#include "../graphics/table.xpm"

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */
static GbWidget gbwidget;

static gchar *Rows = "GtkTable::rows";
static gchar *Columns = "GtkTable::columns";
static gchar *Homogeneous = "GtkTable::homogeneous";
static gchar *RowSpacing = "GtkTable::row_spacing";
static gchar *ColSpacing = "GtkTable::column_spacing";


static void show_table_dialog (GbWidgetNewData * data);
static void on_table_dialog_ok (GtkWidget * widget, GbWidgetNewData * data);
static void on_table_dialog_destroy (GtkWidget * widget, GbWidgetNewData * data);

static void update_table_size (GtkWidget * table, gint rows, gint cols);

static void gb_table_insert_row_before (GtkWidget * menuitem,
					GtkWidget * widget);
static void gb_table_insert_row_after (GtkWidget * menuitem,
				       GtkWidget * widget);
static void gb_table_insert_column_before (GtkWidget * menuitem,
					   GtkWidget * widget);
static void gb_table_insert_column_after (GtkWidget * menuitem,
					  GtkWidget * widget);
static void gb_table_insert_row_or_col (GtkWidget * table, gint row, gint col);
static void gb_table_delete_row (GtkWidget * menuitem, GtkWidget * widget);
static void gb_table_delete_column (GtkWidget * menuitem, GtkWidget * widget);

/******
 * NOTE: To use these functions you need to uncomment them AND add a pointer
 * to the function in the GbWidget struct at the end of this file.
 ******/

/*
 * Creates a new GtkWidget of class GtkTable, performing any specialized
 * initialization needed for the widget to work correctly in this environment.
 * If a dialog box is used to initialize the widget, return NULL from this
 * function, and call data->callback with your new widget when it is done.
 * If the widget needs a special destroy handler, add a signal here.
 */
GtkWidget *
gb_table_new (GbWidgetNewData * data)
{
  GtkWidget *new_widget;
  gint rows, cols;

  if (data->action == GB_LOADING)
    {
      rows = load_int (data->loading_data, Rows);
      cols = load_int (data->loading_data, Columns);
      new_widget = gtk_table_new (rows, cols, FALSE);
      return new_widget;
    }
  else
    {
      show_table_dialog (data);
      return NULL;
    }
}


static void
show_table_dialog (GbWidgetNewData * data)
{
  GtkWidget *window, *vbox, *separator, *hbbox, *label, *spinbutton;
  GtkWidget *button, *table;
  GtkObject *adjustment;

  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);
  gtk_window_set_title (GTK_WINDOW (window), _("New table"));

  gtk_signal_connect_object (GTK_OBJECT (window), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (on_table_dialog_destroy),
		      data);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  /*gtk_container_border_width (GTK_CONTAINER (vbox), 10); */
  gtk_widget_show (vbox);

  /* Rows label & entry */
  table = gtk_table_new (2, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 5);
  gtk_container_border_width (GTK_CONTAINER (table), 4);
  gtk_widget_show (table);

  label = gtk_label_new (_("Number of rows:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL,
		    0, 0);
  gtk_widget_show (label);

  adjustment = gtk_adjustment_new (3, 1, 100, 1, 10, 10);
  spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
  /* save pointer to entry so we can find it easily in the OK handler */
  gtk_object_set_data (GTK_OBJECT (window), "spinbutton", spinbutton);
  gtk_table_attach (GTK_TABLE (table), spinbutton, 1, 2, 0, 1,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 4, 4);
  gtk_widget_set_usize (spinbutton, 50, -1);
  gtk_widget_grab_focus (spinbutton);
  gtk_widget_show (spinbutton);

  /* Columns label & entry */
  label = gtk_label_new (_("Number of columns:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL,
		    0, 0);
  gtk_widget_show (label);

  adjustment = gtk_adjustment_new (3, 1, 100, 1, 10, 10);
  spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
  /* save pointer to entry so we can find it easily in the OK handler */
  gtk_object_set_data (GTK_OBJECT (window), "spinbutton2", spinbutton);
  gtk_table_attach (GTK_TABLE (table), spinbutton, 1, 2, 1, 2,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 4, 4);
  gtk_widget_set_usize (spinbutton, 50, -1);
  gtk_widget_show (spinbutton);

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, TRUE, TRUE, 5);
  gtk_widget_show (separator);

  hbbox = gtk_hbutton_box_new ();
  gtk_box_pack_start (GTK_BOX (vbox), hbbox, TRUE, TRUE, 5);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (hbbox);

  button = gtk_button_new_with_label (_("OK"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_table_dialog_ok),
		      data);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));

  gtk_widget_show (window);
  gtk_grab_add (window);
}


static void
on_table_dialog_ok (GtkWidget * widget, GbWidgetNewData * data)
{
  GtkWidget *new_widget, *spinbutton, *window;
  gint rows, cols, row, col;

  window = gtk_widget_get_toplevel (widget);

  /* Only call callback if placeholder/fixed widget is still there */
  if (gb_widget_can_finish_new (data))
    {
      spinbutton = gtk_object_get_data (GTK_OBJECT (window), "spinbutton");
      g_return_if_fail (spinbutton != NULL);
      rows = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinbutton));

      spinbutton = gtk_object_get_data (GTK_OBJECT (window), "spinbutton2");
      g_return_if_fail (spinbutton != NULL);
      cols = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinbutton));

      new_widget = gtk_table_new (rows, cols, FALSE);
      for (row = 0; row < rows; row++)
	{
	  for (col = 0; col < cols; col++)
	    {
	      gtk_table_attach_defaults (GTK_TABLE (new_widget),
					 editor_new_placeholder (),
					 col, col + 1, row, row + 1);
	    }
	}
      gb_widget_initialize (new_widget, data);
      (*data->callback) (new_widget, data);
    }
  gtk_widget_destroy (window);
}


static void
on_table_dialog_destroy (GtkWidget * widget, GbWidgetNewData * data)
{
  gb_widget_free_new_data (data);
  gtk_grab_remove (widget);
}



/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_table_create_properties (GtkWidget * widget, GbWidgetCreateArgData * data)
{
  property_add_int_range (Rows, _("Rows:"),
			  _("The number of rows in the table"),
			  1, 1000, 1, 10, 1);
  property_add_int_range (Columns, _("Columns:"),
			  _("The number of columns in the table"),
			  1, 1000, 1, 10, 1);
  property_add_bool (Homogeneous, _("Homogeneous"),
		     _("If the children should all be the same size"));
  property_add_int_range (RowSpacing, _("Row Spacing:"),
			  _("The space between each row"),
			  0, 1000, 1, 10, 1);
  property_add_int_range (ColSpacing, _("Col Spacing:"),
			  _("The space between each column"),
			  0, 1000, 1, 10, 1);
}



/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_table_get_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{
  gb_widget_output_int (data, Rows, GTK_TABLE (widget)->nrows);
  gb_widget_output_int (data, Columns, GTK_TABLE (widget)->ncols);
  gb_widget_output_bool (data, Homogeneous, GTK_TABLE (widget)->homogeneous);
  gb_widget_output_int (data, RowSpacing, GTK_TABLE (widget)->row_spacing);
  gb_widget_output_int (data, ColSpacing, GTK_TABLE (widget)->column_spacing);
}



/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_table_set_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  gboolean homogeneous, myApply;
  gint rows, cols, row_spacing, column_spacing;

  rows = gb_widget_input_int (data, Rows);
  myApply = data->apply;
  cols = gb_widget_input_int (data, Columns);
  if (myApply || data->apply)
    {
      update_table_size (widget, rows, cols);
    }

  homogeneous = gb_widget_input_bool (data, Homogeneous);
  if (data->apply)
    {
      GTK_TABLE (widget)->homogeneous = homogeneous;
      gtk_widget_queue_resize (widget);
    }

  row_spacing = gb_widget_input_int (data, RowSpacing);
  if (data->apply)
    gtk_table_set_row_spacings (GTK_TABLE (widget), row_spacing);

  column_spacing = gb_widget_input_int (data, ColSpacing);
  if (data->apply)
    gtk_table_set_col_spacings (GTK_TABLE (widget), column_spacing);
}


/* This changes the size of the table to the given dimensions. Placeholders
   are added in empty cells, and widgets are destroyed if they fall outside
   the new dimensions.
   NOTE: this changes the table data structures, which we would prefer not to
   do, but there is no alternative. */
static void
update_table_size (GtkWidget * table, gint rows, gint cols)
{
  GList *children;
  GtkTableChild *child;

  g_return_if_fail (rows > 0);
  g_return_if_fail (cols > 0);

  if (GTK_TABLE (table)->nrows == rows && GTK_TABLE (table)->ncols == cols)
    return;

  gtk_container_block_resize (GTK_CONTAINER (table));

  children = GTK_TABLE (table)->children;
  while (children)
    {
      child = children->data;
      children = children->next;

      /* Remove the widget if it doesn't fit into the new dimensions,
         or crop it if it extends past the edges. */
      if (child->left_attach >= cols || child->top_attach >= rows)
	{
	  gtk_container_remove (GTK_CONTAINER (table), child->widget);
	}
      else
	{
	  if (child->right_attach > cols)
	    child->right_attach = cols;
	  if (child->bottom_attach > rows)
	    child->bottom_attach = rows;
	}
    }

  /* This is in ../gbwidget.c. It will expand the dimensions if necessary. */
  gb_widget_update_table_placeholders (table, rows, cols);

  /* This bit is especially naughty. It makes sure the table shrinks,
     so we don't get extra spacings on the end. */
  GTK_TABLE (table)->nrows = rows;
  GTK_TABLE (table)->ncols = cols;

  gtk_container_unblock_resize (GTK_CONTAINER (table));
  /* We clear the selection since it isn't displayed properly. */
  editor_clear_selection (NULL);
}


/*
 * Adds menu items to a context menu which is just about to appear!
 * Add commands to aid in editing a GtkTable, with signals pointing to
 * other functions in this file.
 */
static void
gb_table_create_popup_menu (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;

  g_return_if_fail (data->child != NULL);

  menuitem = gtk_menu_item_new_with_label (_("Insert Row Before"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_table_insert_row_before),
		      data->child);

  menuitem = gtk_menu_item_new_with_label (_("Insert Row After"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_table_insert_row_after),
		      data->child);

  menuitem = gtk_menu_item_new_with_label (_("Insert Column Before"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_table_insert_column_before),
		      data->child);

  menuitem = gtk_menu_item_new_with_label (_("Insert Column After"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_table_insert_column_after),
		      data->child);

  menuitem = gtk_menu_item_new_with_label (_("Delete Row"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_table_delete_row), data->child);

  menuitem = gtk_menu_item_new_with_label (_("Delete Column"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_table_delete_column), data->child);
}


static void
gb_table_insert_row_before (GtkWidget * menuitem, GtkWidget * widget)
{
  GtkWidget *table;
  GtkTableChild *tchild;

  table = widget->parent;
  g_return_if_fail (GTK_IS_TABLE (table));
  tchild = gb_table_find_child (GTK_TABLE (table), widget);
  g_return_if_fail (tchild != NULL);
  gb_table_insert_row_or_col (table, tchild->top_attach, -1);
}


static void
gb_table_insert_row_after (GtkWidget * menuitem, GtkWidget * widget)
{
  GtkWidget *table;
  GtkTableChild *tchild;

  table = widget->parent;
  g_return_if_fail (GTK_IS_TABLE (table));
  tchild = gb_table_find_child (GTK_TABLE (table), widget);
  g_return_if_fail (tchild != NULL);
  gb_table_insert_row_or_col (table, tchild->bottom_attach, -1);
}


static void
gb_table_insert_column_before (GtkWidget * menuitem, GtkWidget * widget)
{
  GtkWidget *table;
  GtkTableChild *tchild;

  table = widget->parent;
  g_return_if_fail (GTK_IS_TABLE (table));
  tchild = gb_table_find_child (GTK_TABLE (table), widget);
  g_return_if_fail (tchild != NULL);
  gb_table_insert_row_or_col (table, -1, tchild->left_attach);
}


static void
gb_table_insert_column_after (GtkWidget * menuitem, GtkWidget * widget)
{
  GtkWidget *table;
  GtkTableChild *tchild;

  table = widget->parent;
  g_return_if_fail (GTK_IS_TABLE (table));
  tchild = gb_table_find_child (GTK_TABLE (table), widget);
  g_return_if_fail (tchild != NULL);
  gb_table_insert_row_or_col (table, -1, tchild->right_attach);
}


/* This inserts a row or column into the table at the given position.
   Use -1 for the other unused argument. */
static void
gb_table_insert_row_or_col (GtkWidget * table, gint row, gint col)
{
  GtkTableChild *tchild;
  GList *child;
  GtkWidget *widget, *tmp_label;
  gint rows, cols;

  gtk_container_block_resize (GTK_CONTAINER (table));

  /* This relies on knowing the internals of GtkTable, so it is fast.
     First it adds a simple label at the bottom right of the new table size,
     to ensure that the table grows. Then it removes the label, and moves
     all the widgets down/right as appropriate, and adds any necessary
     placeholders. */
  tmp_label = gtk_label_new ("");
  rows = GTK_TABLE (table)->nrows + (row != -1 ? 1 : 0);
  cols = GTK_TABLE (table)->ncols + (col != -1 ? 1 : 0);
  gtk_table_attach_defaults (GTK_TABLE (table), tmp_label,
			     cols - 1, cols, rows - 1, rows);

  child = GTK_TABLE (table)->children;
  while (child)
    {
      tchild = child->data;
      child = child->next;
      widget = tchild->widget;

      if ((row != -1 && tchild->top_attach >= row)
	  || (col != -1 && tchild->left_attach >= col))
	{
	  /* If we're inserting a row, we move the widget down.
	     If we're inserting a col, we move the widget right. */
	  if (row != -1)
	    {
	      tchild->top_attach++;
	      tchild->bottom_attach++;
	    }
	  else
	    {
	      tchild->left_attach++;
	      tchild->right_attach++;
	    }
	}
    }

  /* Now remove the temporary label. */
  gtk_container_remove (GTK_CONTAINER (table), tmp_label);

  /* This fills any empty cells with placeholders. */
  gb_widget_update_table_placeholders (table, -1, -1);
  gtk_container_unblock_resize (GTK_CONTAINER (table));

  /* If the tables properties are currently shown, update rows/cols. */
  if (property_get_widget () == table)
    {
      property_set_auto_apply (FALSE);
      if (row != -1)
	property_set_int (Rows, GTK_TABLE (table)->nrows);
      else
	property_set_int (Columns, GTK_TABLE (table)->ncols);
      property_set_auto_apply (TRUE);
    }
}


static void
gb_table_delete_row (GtkWidget * menuitem, GtkWidget * widget)
{
  GtkWidget *table;
  GtkTableChild *tchild;
  GList *children;
  guint16 nrows, ncols, row;

  table = widget->parent;
  nrows = GTK_TABLE (table)->nrows - 1;
  if (nrows == 0)
    return;
  ncols = GTK_TABLE (table)->ncols;
  tchild = gb_table_find_child (GTK_TABLE (table), widget);
  g_return_if_fail (tchild != NULL);
  row = tchild->top_attach;

  children = GTK_TABLE (table)->children;
  while (children)
    {
      tchild = children->data;
      children = children->next;
      if (tchild->top_attach == row && tchild->bottom_attach == row + 1)
	{
	  gtk_container_remove (GTK_CONTAINER (table), tchild->widget);
	}
      else if (tchild->top_attach <= row && tchild->bottom_attach > row)
	{
	  tchild->bottom_attach -= 1;
	}
      else if (tchild->top_attach > row)
	{
	  tchild->top_attach -= 1;
	  tchild->bottom_attach -= 1;
	}
    }
  update_table_size (table, nrows, ncols);

  if (property_get_widget () == table)
    {
      property_set_auto_apply (FALSE);
      property_set_int (Rows, nrows);
      property_set_auto_apply (TRUE);
    }
}


static void
gb_table_delete_column (GtkWidget * menuitem, GtkWidget * widget)
{
  GtkWidget *table;
  GtkTableChild *tchild;
  GList *children;
  guint16 nrows, ncols, col;

  table = widget->parent;
  nrows = GTK_TABLE (table)->nrows;
  ncols = GTK_TABLE (table)->ncols - 1;
  if (ncols == 0)
    return;
  tchild = gb_table_find_child (GTK_TABLE (table), widget);
  g_return_if_fail (tchild != NULL);
  col = tchild->left_attach;

  children = GTK_TABLE (table)->children;
  while (children)
    {
      tchild = children->data;
      children = children->next;
      if (tchild->left_attach == col && tchild->right_attach == col + 1)
	{
	  gtk_container_remove (GTK_CONTAINER (table), tchild->widget);
	}
      else if (tchild->left_attach <= col && tchild->right_attach > col)
	{
	  tchild->right_attach -= 1;
	}
      else if (tchild->left_attach > col)
	{
	  tchild->left_attach -= 1;
	  tchild->right_attach -= 1;
	}
    }
  update_table_size (table, nrows, ncols);

  if (property_get_widget () == table)
    {
      property_set_auto_apply (FALSE);
      property_set_int (Columns, ncols);
      property_set_auto_apply (TRUE);
    }
}


/*
 * Writes the source code needed to create this widget.
 * You have to output everything necessary to create the widget here, though
 * there are some convenience functions to help.
 */
static void
gb_table_write_source (GtkWidget * widget, GbWidgetWriteSourceData * data)
{
  if (data->create_widget)
    {
      source_add (data, "  %s = gtk_table_new (%i, %i, %s);\n", data->wname,
		  GTK_TABLE (widget)->nrows, GTK_TABLE (widget)->ncols,
		  GTK_TABLE (widget)->homogeneous ? "TRUE" : "FALSE");
    }

  gb_widget_write_standard_source (widget, data);

  if (GTK_TABLE (widget)->row_spacing != 0)
    {
      source_add (data, "  gtk_table_set_row_spacings (GTK_TABLE (%s), %i);\n",
		  data->wname, GTK_TABLE (widget)->row_spacing);
    }
  if (GTK_TABLE (widget)->column_spacing != 0)
    {
      source_add (data, "  gtk_table_set_col_spacings (GTK_TABLE (%s), %i);\n",
		  data->wname, GTK_TABLE (widget)->column_spacing);
    }
}



/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
GbWidget *
gb_table_init ()
{
  /* Initialise the GTK type */
  gtk_table_get_type ();

  /* Initialize the GbWidget structure */
  gb_widget_init_struct (&gbwidget);

  /* Fill in the pixmap struct & tooltip */
  gbwidget.pixmap_struct = table_xpm;
  gbwidget.tooltip = _("Table");

  /* Fill in any functions that this GbWidget has */
  gbwidget.gb_widget_new = gb_table_new;
  gbwidget.gb_widget_create_properties = gb_table_create_properties;
  gbwidget.gb_widget_get_properties = gb_table_get_properties;
  gbwidget.gb_widget_set_properties = gb_table_set_properties;
  gbwidget.gb_widget_write_source = gb_table_write_source;
  gbwidget.gb_widget_create_popup_menu = gb_table_create_popup_menu;

  return &gbwidget;
}
