/*

This is a heavily modified version of dbz!

Many of the features have been removed.

structure of NL index:

* hash file contains following data for each entry:

** offset in nodelist file
** number of nodelist file
** hub
** net number
** node number

* hash file does not contain:

** zone number

Can easily be looked up in a table. I assume there are only a few
different zone numbers:

*** organization of zone number listing:

nlnum, offset (from, to) -> zone number

** point number

Can easily be looked up in the nodelist


original copyright notices follow:

dbz.c  V3.2

Copyright 1988 Jon Zeeff (zeeff@b-tech.ann-arbor.mi.us)
You can use this code in any manner, as long as you leave my name on it
and don't hold me responsible for any problems with it.

Hacked on by gdb@ninja.UUCP (David Butler); Sun Jun  5 00:27:08 CDT 1988

Various improvments + INCORE by moraes@ai.toronto.edu (Mark Moraes)

Major reworking by Henry Spencer as part of the C News project.

Minor lint and CodeCenter (Saber) fluff removal by Rich $alz (March, 1991).
Non-portable CloseOnExec() calls added by Rich $alz (September, 1991).
Added "writethrough" and tagmask calculation code from
<rob@violet.berkeley.edu> and <leres@ee.lbl.gov> by Rich $alz (December, 1992).
Merged in MMAP code by David Robinson, formerly <david@elroy.jpl.nasa.gov>
now <david.robinson@sun.com> (January, 1993).

These routines replace dbm as used by the usenet news software
(it's not a full dbm replacement by any means).  It's fast and
simple.  It contains no AT&T code.

In general, dbz's files are 1/20 the size of dbm's.  Lookup performance
is somewhat better, while file creation is spectacularly faster.

*/
#ifdef USE_DBZ

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>

#include "dbz.h"


/* Generic pointer, used by memcpy, malloc, etc. */
/* =()<typedef @@<POINTER>@@ *POINTER;>()= */
typedef char *POINTER;


/*
 * #ifdef index.  "LIA" = "leave it alone unless you know what you're doing".
 *
 * DBZDEBUG	enable debugging
 * NPAGBUF	size of .pag buffer, in longs (LIA)
 * MAXRUN	length of run which shifts to next table (see below) (LIA)
 */
/* SUPPRESS 530 *//* Empty body for statement */
/* SUPPRESS 701 on free *//* Conflicting declaration */
#undef DBZDEBUG

#ifndef SEEK_SET
#define	SEEK_SET	0
#endif

extern unsigned long crc32ccitt(char *, int);

static int dbzversion = 3;	/* for validating .dir file format */

/*
 * The dbz database exploits the fact that when news stores a <key,value>
 * tuple, the `value' part is a seek offset into a text file, pointing to
 * a copy of the `key' part.  This avoids the need to store a copy of
 * the key in the dbz files.  However, the text file *must* exist and be
 * consistent with the dbz files, or things will fail.
 *
 * The basic format of the database is a simple hash table containing the
 * values.  A value is stored by indexing into the table using a hash value
 * computed from the key; collisions are resolved by linear probing (just
 * search forward for an empty slot, wrapping around to the beginning of
 * the table if necessary).  Linear probing is a performance disaster when
 * the table starts to get full, so a complication is introduced.  The
 * database is actually one *or more* tables, stored sequentially in the
 * .pag file, and the length of linear-probe sequences is limited.  The
 * search (for an existing item or an empty slot) always starts in the
 * first table, and whenever MAXRUN probes have been done in table N,
 * probing continues in table N+1.  This behaves reasonably well even in
 * cases of massive overflow.  There are some other small complications
 * added, see comments below.
 *
 * The table size is fixed for any particular database, but is determined
 * dynamically when a database is rebuilt.  The strategy is to try to pick
 * the size so the first table will be no more than 2/3 full, that being
 * slightly before the point where performance starts to degrade.  (It is
 * desirable to be a bit conservative because the overflow strategy tends
 * to produce files with holes in them, which is a nuisance.)
 */

/*
 * ANSI C says an offset into a file is a long, not an off_t, for some
 * reason.  This actually does simplify life a bit, but it's still nice
 * to have a distinctive name for it.  Beware, this is just for readability,
 * don't try to change this.
 */
#define	of_t	long

/*
 * We assume that unused areas of a binary file are zeros, and that the
 * bit pattern of `(of_t)0' is all zeros.  The alternative is rather
 * painful file initialization.  Note that okayvalue(), if OVERFLOW is
 * defined, knows what value of an offset would cause overflow.
 */
#define	VACANT	NULL

/*
 * A new, from-scratch database, not built as a rebuild of an old one,
 * needs to know table size.  Normally the user supplies this info, but
 * there have to be defaults.
 */
#ifndef DEFSIZE
#define	DEFSIZE	120011		/* 300007 might be better */
#endif

/*
 * We read configuration info from the .dir file into this structure,
 * so we can avoid wired-in assumptions for an existing database.
 *
 * Among the info is a record of recent peak usages, so that a new table
 * size can be chosen intelligently when rebuilding.  10 is a good
 * number of usages to keep, since news displays marked fluctuations
 * in volume on a 7-day cycle.
 */
struct dbzconfig {
	int olddbz;		/* .dir file empty but .pag not? */
	int tsize;		/* table size */
	int valuesize;		/* size of table values, == SOF */
};
static struct dbzconfig conf;
static int getconf();
static long getno();
static int putconf();


/* #define	SOF	8	(sizeof(of_t)) */
/* static int SOF = 8; */
#define SOF	12
#define MAX_FETCH_SIZE	10

#define NEXTFLAG_OFF	5
#define NEXTFLAG_MASK	0x80

/*
 * Stdio buffer for .pag reads.  Buffering more than about 16 does not help
 * significantly at the densities we try to maintain, and the much larger
 * buffers that most stdios default to are much more expensive to fill.
 * With small buffers, stdio is performance-competitive with raw read(),
 * and it's much more portable.
 */
#ifndef NPAGBUF
#define	NPAGBUF	16
#endif
#ifdef _IOFBF
static char pagbuf[NPAGBUF][SOF];	/* only needed if !NOBUFFER && _IOFBF */
#endif

/*
 * Data structure for recording info about searches.
 */
struct searcher {
	of_t place;		/* current location in file */
	int tabno;		/* which table we're in */
	int run;		/* how long we'll stay in this table */
#		ifndef MAXRUN
#		define	MAXRUN	100
#		endif
	long hash;		/* the key's hash code (for optimization) */
	int seen;		/* have we examined current location? */
	int aborted;		/* has i/o error aborted search? */
};
static void start();
#define	FRESH	((struct searcher *)NULL)
static char *search();
static int set();

/*
 * Arguably the searcher struct for a given routine ought to be local to
 * it, but a fetch() is very often immediately followed by a store(), and
 * in some circumstances it is a useful performance win to remember where
 * the fetch() completed.  So we use a global struct and remember whether
 * it is current.
 */
static struct searcher srch;
static struct searcher *prevp;	/* &srch or FRESH */

/*
 * The double parentheses needed to make this work are ugly, but the
 * alternative (under most compilers) is to pack around 2K of unused
 * strings -- there's just no way to get rid of them.
 */
#ifdef DBZDEBUG
static int debug = 1;			/* controlled by dbzdebug() */
#define DEBUG(args) if (debug) { (void) printf args ; } else
#else
#define	DEBUG(args)	;
#endif

extern int atoi();
extern long atol();

/* misc. forwards */
static int isprime();

/* file-naming stuff */
static char dir[] = ".dir";
static char pag[] = ".pag";
static char *enstring();

/* central data structures */
static FILE *dirf;		/* descriptor for .dir file */
static int dirronly;		/* dirf open read-only? */
static FILE *pagf = NULL;	/* descriptor for .pag file */
static of_t pagpos;		/* posn in pagf; only search may set != -1 */
static int pagronly;		/* pagf open read-only? */
static FILE *bufpagf;		/* well-buffered pagf, for incore rewrite */
static int written;		/* has a store() been done? */

static int (*dbzcheckfunc)() = NULL;


/*
 *  Mark a file close-on-exec so that it doesn't get shared with our
 *  children.  Ignore any error codes.
 */
void CloseOnExec(int fd, int flag)
{
  int oerrno;

  oerrno = errno;
  (void)fcntl(fd, F_SETFD, flag ? 1 : 0);
  errno = oerrno;
}


void dbzcheck(int (*newfunc)())
{
  dbzcheckfunc = newfunc;
}


/*
 - dbzfresh - set up a new database, no historical info
 * name ... base name; .dir and .pag must exist
 * size ... table size (0 means default)
 * 0 success, -1 failure
 */
int dbzfresh(char *name, long size)
{
  register char *fn;
  struct dbzconfig c;
  register FILE *f;

  if (pagf != NULL)
  {
    DEBUG(("dbzfresh: database already open\n"));
    return(-1);
  }
  if (size != 0 && size < 2)
  {
    DEBUG(("dbzfresh: preposterous size (%ld)\n", size));
    return(-1);
  }

  /* get default configuration */
  if (getconf((FILE *)NULL, (FILE *)NULL, &c) < 0)
    return(-1);		/* "can't happen" */

  /* and mess with it as specified */
  if (size != 0)
    c.tsize = size;

  /* write it out */
  fn = enstring(name, dir);
  if (fn == NULL)
    return(-1);
  f = fopen(fn, "w");
  free((POINTER)fn);
  if (f == NULL)
  {
    DEBUG(("dbzfresh: unable to write config\n"));
    return(-1);
  }
  if (putconf(f, &c) < 0)
  {
    (void) fclose(f);
    return(-1);
  }
  if (fclose(f) == EOF)
  {
    DEBUG(("dbzfresh: fclose failure\n"));
    return(-1);
  }

  /* create/truncate .pag */
  fn = enstring(name, pag);
  if (fn == NULL)
    return(-1);
  f = fopen(fn, "w");
  free((POINTER)fn);
  if (f == NULL)
  {
    DEBUG(("dbzfresh: unable to create/truncate .pag file\n"));
    return(-1);
  }
  else
  {
    (void) fclose(f);
  }

  /* and punt to dbminit for the hard work */
  return(dbminit(name));
}

/*
 - dbzsize - what's a good table size to hold this many entries?
 * content ... 0 means what's the default
 */
long dbzsize(long contents)
{
  long n;

  if (contents <= 0)	/* foulup or default inquiry */
  {
    DEBUG(("dbzsize: preposterous input (%ld)\n", contents));
    return(DEFSIZE);
  }

  n = (contents/2)*3;	/* try to keep table at most 2/3 full */
  if (!(n & 1))		/* make it odd */
    n++;

  DEBUG(("dbzsize: tentative size %ld\n", n));
  while (!isprime(n))	/* and look for a prime */
    n += 2;
  DEBUG(("dbzsize: final size %ld\n", n));

  return(n);
}

/*
 - isprime - is a number prime?
 *
 * This is not a terribly efficient approach.
 */
static int isprime(long x)
{
  static int quick[] = { 2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 0 };
  register int *ip;
  register long div;
  register long stop;

  /* hit the first few primes quickly to eliminate easy ones */
  /* this incidentally prevents ridiculously small tables */
  for (ip = quick; (div = *ip) != 0; ip++)
    if (x%div == 0)
    {
      DEBUG(("isprime: quick result on %ld\n", (long)x));
      return(0);
    }

  /* approximate square root of x */
  for (stop = x; x/stop < stop; stop >>= 1)
    continue;
  stop <<= 1;

  /* try odd numbers up to stop */
  for (div = *--ip; div < stop; div += 2)
    if (x%div == 0)
      return(0);

  return(1);
}

/*
 - dbminit - open a database, creating it (using defaults) if necessary
 *
 * We try to leave errno set plausibly, to the extent that underlying
 * functions permit this, since many people consult it if dbminit() fails.
 */
int 				/* 0 success, -1 failure */
dbminit(char *name)
{
  size_t s;
  char *dirfname;
  char *pagfname;

  if (pagf != NULL)
  {
    DEBUG(("dbminit: dbminit already called once\n"));
    errno = 0;
    return(-1);
  }

  /* open the .dir file */
  dirfname = enstring(name, dir);
  if (dirfname == NULL)
    return(-1);

  dirf = fopen(dirfname, "r+");
  if (dirf == NULL)
  {
    dirf = fopen(dirfname, "r");
    dirronly = 1;
  }
  else
    dirronly = 0;
  free((POINTER)dirfname);
  if (dirf == NULL)
  {
    DEBUG(("dbminit: can't open .dir file\n"));
    return(-1);
  }
  CloseOnExec((int)fileno(dirf), 1);

  /* open the .pag file */
  pagfname = enstring(name, pag);
  if (pagfname == NULL)
  {
    (void) fclose(dirf);
    return(-1);
  }
  pagf = fopen(pagfname, "r+b");
  if (pagf == NULL)
  {
    pagf = fopen(pagfname, "rb");
    if (pagf == NULL)
    {
      DEBUG(("dbminit: .pag open failed\n"));
      (void) fclose(dirf);
      free((POINTER)pagfname);
      return(-1);
    }
    pagronly = 1;
  }
  else if (dirronly)
    pagronly = 1;
  else
    pagronly = 0;
  if (pagf != NULL)
    CloseOnExec((int)fileno(pagf), 1);
#ifdef _IOFBF
  (void) setvbuf(pagf, (char *)pagbuf, _IOFBF, sizeof(pagbuf));
#endif
  pagpos = -1;
  /* don't free pagfname, need it below */

  /* pick up configuration */
  if (getconf(dirf, pagf, &conf) < 0)
  {
    DEBUG(("dbminit: getconf failure\n"));
    (void) fclose(pagf);
    (void) fclose(dirf);
    free((POINTER)pagfname);
    pagf = NULL;
    errno = EDOM;	/* kind of a kludge, but very portable */
    return(-1);
  }

  /* get first table into core, if it looks desirable and feasible */
  s = (size_t)conf.tsize * SOF;
  bufpagf = NULL;
  free((POINTER)pagfname);

  /* misc. setup */
  written = 0;
  prevp = FRESH;
  DEBUG(("dbminit: succeeded\n"));
  return(0);
}

/*
 - enstring - concatenate two strings into a malloced area
 */
static char *			/* NULL if malloc fails */
enstring(char *s1, char *s2)
{
  char *p;

  p = malloc((size_t)strlen(s1) + (size_t)strlen(s2) + 1);
  if (p != NULL)
  {
    (void) strcpy(p, s1);
    (void) strcat(p, s2);
  }
  else
  {
    DEBUG(("enstring(%s, %s) out of memory\n", s1, s2));
  }
  return(p);
}

/*
 - dbmclose - close a database
 */
int dbmclose()
{
  int ret = 0;

  if (pagf == NULL)
  {
    DEBUG(("dbmclose: not opened!\n"));
    return(-1);
  }

  if (fclose(pagf) == EOF)
  {
    DEBUG(("dbmclose: fclose(pagf) failed\n"));
    ret = -1;
  }
  /* pagf = basef; */		/* ensure valid pointer; dbzsync checks it */
  if (dbzsync() < 0)
    ret = -1;
  if (bufpagf != NULL && fclose(bufpagf) == EOF) {
    DEBUG(("dbmclose: fclose(bufpagf) failed\n"));
    ret = -1;
  }
  pagf = NULL;
  if (fclose(dirf) == EOF)
  {
    DEBUG(("dbmclose: fclose(dirf) failed\n"));
    ret = -1;
  }

  DEBUG(("dbmclose: %s\n", (ret == 0) ? "succeeded" : "failed"));
  return(ret);
}

/*
 - dbzsync - push all in-core data out to disk
 */
int dbzsync()
{
  int ret = 0;

  if (pagf == NULL)
  {
    DEBUG(("dbzsync: not opened!\n"));
    return(-1);
  }
  if (!written)
    return(0);

  if (!conf.olddbz)
    if (putconf(dirf, &conf) < 0)
      ret = -1;

  DEBUG(("dbzsync: %s\n", (ret == 0) ? "succeeded" : "failed"));
  return(ret);
}

/*
 - dbzcancel - cancel writing of in-core data
 * Mostly for use from child processes.
 * Note that we don't need to futz around with stdio buffers, because we
 * always fflush them immediately anyway and so they never have stale data.
 */
int dbzcancel()
{
  if (pagf == NULL) {
    DEBUG(("dbzcancel: not opened!\n"));
    return(-1);
  }

  written = 0;
  return(0);
}

/*
 - fetch - get an entry from the database
 *
 * Disgusting fine point, in the name of backward compatibility:  if the
 * last character of "key" is a NUL, that character is (effectively) not
 * part of the comparison against the stored keys.
 */
datum				/* dptr NULL, dsize 0 means failure */
fetch(datum key)
{
  static char fetch_val[MAX_FETCH_SIZE * SOF];
  char *key_ptr;		/* return value points here */
  datum output;
  size_t keysize;

  DEBUG(("fetch: (%s)\n", key.dptr));
  output.dptr = NULL;
  output.dsize = 0;
  prevp = FRESH;

  /* Key is supposed to be less than DBZMAXKEY */
  keysize = key.dsize;
  if (keysize >= DBZMAXKEY)
  {
    keysize = DBZMAXKEY;
    DEBUG(("keysize is %d - truncated to %d\n", key.dsize, DBZMAXKEY));
  }

  if (pagf == NULL)
  {
    DEBUG(("fetch: database not open!\n"));
    return(output);
  }

  start(&srch, &key, FRESH);
  while ((key_ptr = search(&srch)) != NULL)
  {
    DEBUG(("got 0x%lx\n", *key_ptr));

    /* printf("%08lx %08lx %08lx\n", *((unsigned long *) key_ptr + 0), *((unsigned long *) key_ptr + 1), *((unsigned long *) key_ptr + 2)); */


    /* check if entry has been found */
    if ((dbzcheckfunc == NULL) || ((*dbzcheckfunc)(key.dptr, key_ptr)))
    {
      /* we found it */
      if (output.dsize + SOF <= MAX_FETCH_SIZE * SOF)
      {
	memcpy(fetch_val + output.dsize, key_ptr, SOF);
	output.dptr = fetch_val;
	output.dsize += SOF;
      }
      else
      {
	/* TODO: where should this memory be freed... */
	if (output.dptr != fetch_val)
	{
	  output.dptr = realloc(output.dptr, output.dsize + SOF);
	}
	else
	{
	  output.dptr = malloc(output.dsize + SOF);
	  memcpy(output.dptr, fetch_val, MAX_FETCH_SIZE * SOF);
	}
	memcpy(output.dptr + output.dsize, key_ptr, SOF);
	output.dsize += SOF;
      }
      DEBUG(("fetch: successful\n"));

      if ((key_ptr[NEXTFLAG_OFF] & NEXTFLAG_MASK) == 0)
      {
	return(output);
      }
      else
      {
	/* printf("fetch: multiple entry\n"); */
	/* reset the next-flag in the output */
	*((char *) output.dptr + output.dsize - SOF + NEXTFLAG_OFF) &= ~NEXTFLAG_MASK;
      }
    }
  }

  if (output.dptr == NULL)
  {
    /* we didn't find it */
    DEBUG(("fetch: failed\n"));
    prevp = &srch;			/* remember where we stopped */
  }

  return(output);
}

/*
 - store - add an entry to the database
 */
int				/* 0 success, -1 failure */
store(datum key, datum data)
{
  static char value[SOF];

  if (pagf == NULL)
  {
    DEBUG(("store: database not open!\n"));
    return(-1);
  }
  if (pagronly)
  {
    DEBUG(("store: database open read-only\n"));
    return(-1);
  }
  if (data.dsize % SOF)
  {
    DEBUG(("store: value size wrong (%d)\n", data.dsize));
    return(-1);
  }
  if (key.dsize >= DBZMAXKEY)
  {
    DEBUG(("store: key size too big (%d)\n", key.dsize));
    return(-1);
  }

  /* copy the value in to ensure alignment */
  (void) memcpy((POINTER)value, (POINTER)data.dptr, SOF);
  DEBUG(("store: (%s, %ld)\n", key.dptr, *(long *) value));

  /* if entry should be added */
  if (data.dsize > SOF)
  {
    char *key_ptr;		/* return value points here */

    /* printf("store: multiple entry\n"); */

    start(&srch, &key, FRESH);
    while ((key_ptr = search(&srch)) != NULL)
    {
      DEBUG(("got 0x%lx\n", *key_ptr));

      /* check if entry has been found */
      if ((dbzcheckfunc == NULL) || ((*dbzcheckfunc)(key.dptr, key_ptr)))
      {
	if ((key_ptr[NEXTFLAG_OFF] & NEXTFLAG_MASK) == 0)
	  break;
      }
    }

    /* set flag that another entry is available */
    key_ptr[NEXTFLAG_OFF] |= NEXTFLAG_MASK;
    written = 1;
    set(&srch, key_ptr);

    prevp = &srch;
  }

  /* find the place, exploiting previous search if possible */
  start(&srch, &key, prevp);
  while (search(&srch) != NULL)
    continue;

  prevp = FRESH;
  written = 1;
  return(set(&srch, value));
}

/*
 - getconf - get configuration from .dir file
 */
static int			/* 0 success, -1 failure */
getconf(df, pf, cp)
register FILE *df;		/* NULL means just give me the default */
register FILE *pf;		/* NULL means don't care about .pag */
register struct dbzconfig *cp;
{
  int c;
  int err = 0;

  c = (df != NULL) ? getc(df) : EOF;
  if (c == EOF)		/* empty file, no configuration known */
  {
    cp->olddbz = 0;
    if (df != NULL && pf != NULL && getc(pf) != EOF)
      cp->olddbz = 1;
    cp->tsize = DEFSIZE;
    cp->valuesize = SOF;
    DEBUG(("getconf: defaults (%ld)\n", cp->tsize));
    return(0);
  }
  (void) ungetc(c, df);

  /* first line, the vital stuff */
  if (getc(df) != 'd' || getc(df) != 'b' || getc(df) != 'z')
    err = -1;
  if (getno(df, &err) != dbzversion)
    err = -1;
  cp->tsize = getno(df, &err);
  cp->valuesize = getno(df, &err);
  if (cp->valuesize != SOF) {
    DEBUG(("getconf: wrong of_t size (%d)\n", cp->valuesize));
    err = -1;
    cp->valuesize = SOF;	/* to protect the loops below */
  }
  if (getc(df) != '\n')
    err = -1;
#ifdef DBZDEBUG
  DEBUG(("size %ld ", cp->tsize));
  DEBUG(("\n"));
#endif

  if (err < 0)
  {
    DEBUG(("getconf error\n"));
    return(-1);
  }
  return(0);
}

/*
 - getno - get a long
 */
static long getno(FILE *f, int *ep)
{
  char *p;
#	define	MAXN	50
  char getbuf[MAXN];
  int c;

  while ((c = getc(f)) == ' ')
    continue;
  if (c == EOF || c == '\n') {
    DEBUG(("getno: missing number\n"));
    *ep = -1;
    return(0);
  }
  p = getbuf;
  *p++ = c;
  while ((c = getc(f)) != EOF && c != '\n' && c != ' ')
    if (p < &getbuf[MAXN-1])
      *p++ = c;
  if (c == EOF)
  {
    DEBUG(("getno: EOF\n"));
    *ep = -1;
  } else
    (void) ungetc(c, f);
  *p = '\0';

  if (strspn(getbuf, "-1234567890") != strlen(getbuf)) {
    DEBUG(("getno: `%s' non-numeric\n", getbuf));
    *ep = -1;
  }
  return(atol(getbuf));
}

/*
 - putconf - write configuration to .dir file
 */
static int			/* 0 success, -1 failure */
putconf(FILE *f, struct dbzconfig *cp)
{
  int ret = 0;

  if (fseek(f, (of_t)0, SEEK_SET) != 0)
  {
    DEBUG(("fseek failure in putconf\n"));
    ret = -1;
  }
  (void) fprintf(f, "dbz %d %d %d", dbzversion, cp->tsize, cp->valuesize);
  (void) fprintf(f, "\n");

  (void) fflush(f);
  if (ferror(f))
    ret = -1;

  DEBUG(("putconf status %d\n", ret));
  return(ret);
}

/*
 - start - set up to start or restart a search
 */
static void start(struct searcher *sp, datum *kp, struct searcher *osp)
{
  unsigned long h;

  h = crc32ccitt(kp->dptr, kp->dsize);
  if (osp != FRESH && osp->hash == h)
  {
    if (sp != osp)
      *sp = *osp;
    DEBUG(("search restarted\n"));
  }
  else
  {
    sp->hash = h;
    sp->place = h % conf.tsize;
    sp->tabno = 0;
    sp->run = (conf.olddbz) ? conf.tsize : MAXRUN;
    sp->aborted = 0;
  }
  sp->seen = 0;
}

/*
 - search - conduct part of a search
 */
static char *search(struct searcher *sp)
{
  of_t dest;
  static char val[SOF];		/* buffer for value (can't fread register) */
  char *value;
  of_t place;

  if (sp->aborted)
    return(NULL);

  for (;;)
  {
    /* determine location to be examined */
    place = sp->place;
    if (sp->seen)
    {
      /* go to next location */
      if (--sp->run <= 0)
      {
	sp->tabno++;
	sp->run = MAXRUN;
      }
      place = (place+1)%conf.tsize + sp->tabno*conf.tsize;
      sp->place = place;
    }
    else
      sp->seen = 1;	/* now looking at current location */
    DEBUG(("search @ %ld\n", place));

    /* seek, if necessary */
    dest = place * SOF;
    if (pagpos != dest)
    {
      if (fseek(pagf, dest, SEEK_SET) != 0) {
	DEBUG(("search: seek failed\n"));
	pagpos = -1;
	sp->aborted = 1;
	return(NULL);
      }
      pagpos = dest;
    }

    /* read it */
    if (fread((POINTER)val, SOF, 1, pagf) == 1)
      value = val;
    else if (ferror(pagf))
    {
      DEBUG(("search: read failed\n"));
      pagpos = -1;
      sp->aborted = 1;
      return(NULL);
    }
    else
      value = VACANT;

    /* and finish up */
    pagpos += SOF;

    /* vacant slot is always cause to return */
    if ((value == VACANT) || (!memcmp(value, "\0\0\0\0\0\0\0\0\0\0\0", SOF)))
    {
      DEBUG(("search: empty slot\n"));
      return(NULL);
    };

    DEBUG(("got 0x%lx\n", value));
    return(value);
  }
  /* NOTREACHED */
}

/*
 - set - store a value into a location previously found by search
 */
static int			/* 0 success, -1 failure */
set(struct searcher *sp, char *value)
{
  of_t place = sp->place;

  if (sp->aborted)
    return(-1);

  /* seek to spot */
  pagpos = -1;		/* invalidate position memory */
  if (fseek(pagf, (of_t)(place * SOF), SEEK_SET) != 0)
  {
    DEBUG(("set: seek failed\n"));
    sp->aborted = 1;
    return(-1);
  }

  /* write in data */
  if (fwrite((POINTER)value, SOF, 1, pagf) != 1)
  {
    DEBUG(("set: write failed\n"));
    sp->aborted = 1;
    return(-1);
  }
  /* fflush improves robustness, and buffer re-use is rare anyway */
  if (fflush(pagf) == EOF)
  {
    DEBUG(("set: fflush failed\n"));
    sp->aborted = 1;
    return(-1);
  }

  DEBUG(("set: succeeded\n"));
  return(0);
}

/*
 - dbzgetdirf - get dir file
 */
FILE *dbzgetdirf(void)
{
  return dirf;
}


/*
 - dbzdebug - control dbz debugging at run time
 */
#ifdef DBZDEBUG
int				/* old value */
dbzdebug(value)
int value;
{
  int old = debug;

  debug = value;
  return(old);
}
#endif

#endif
