/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#define MVMEBugEntry			15		/* Trap #15 to call MVMEBug */

#define MB_OUTCHR			0x0020
#define MB_OUTSTR			0x0021
#define MB_WRITD			0x0028
#define MB_INCHR			0x0000
#define MB_INSTAT			0x0001
#define MB_DSKRD			0x0010
#define MB_SYMBOLTA			0x0130
#define MB_SYMBOLTD			0x0131
#define MB_RETURN			0x0063
#define MB_RTC_READ			0x0053

/*--------------------------------------------------------------------------*/
/* Disk Read command block
 */
typedef struct {
	unsigned char	clun,
					dlun;
	unsigned short	status;
	unsigned long	address,
					block;
	unsigned short	count;
	unsigned char	flags,
					mod;
} DSKRD_CMND;

/*--------------------------------------------------------------------------*/

/*
        32 bit word at  0x14 block number of start of kernel
        16 bit word at  0x18 number of blocks to load
        32 bit word at  0x1E boot load address
        32 bit word at 0x90 to contain 0 (Media config area start block)
        Locations 0xF8 to 0xFF to contain "MOTOROLA"
*/

#define MAX_LDR_FRAGS	5

typedef struct {
	unsigned long sp, pc;				/* 00								*/
	char		filler1[12];			/* 00								*/
	unsigned long boot_start_block;		/* 14 bootfile start block addr		*/
	unsigned short boot_block_count;	/* 18 number of blocks in file		*/
	char		filler2[4];				/* 1a								*/
	unsigned long boot_load_addr;		/* 1e 								*/
	char		filler3[14];			/* 22								*/
	char		boot_code[0x60];		/* 30								*/
	unsigned long zero;					/* 90								*/
	char		filler4[12];			/* 94								*/
	DSKRD_CMND	cdb[MAX_LDR_FRAGS];		/* a0								*/
	char		boot_id[16];			/* f0 id string "Linux68kMOTOROLA"	*/
} BOOTBLOCK;

#define BOOT_BLOCK_ID  "Linux68kMOTOROLA"

/*--------------------------------------------------------------------------*/
/* Call MVMEBug putchar subroutine.
 */

static
__inline__
void
MVMEBug_putchar
(	int		c
)
{	register unsigned long	d0 asm("d0") = (unsigned long) c;

	asm volatile (
		"	move.b	%%d0,-(%%sp)
			trap	#15
			dc.w	0x0020
		"
		: /* no outputs */
		: "r" (d0)
		: "d0"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug putstr subroutine.
 */

static
__inline__
void
MVMEBug_putstr
(	const char		*str,
	const char		*end
)
{	register void			*a0 asm("a0") = (void *) str;
	register void			*a1 asm("a1") = (void *) end;

	asm volatile (
		"	move.l	%%a1,-(%%sp)
			move.l	%%a0,-(%%sp)
			trap	#15
			dc.w	0x0021
		"
		: /* no outputs */
		: "r" (a0), "r" (a1)
		: "a0", "a1"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug print subroutine.
 */

static
__inline__
void
MVMEBug_print
(	const char		*str,
	void			*vap
)
{	register void			*a0 asm("a0") = (void *) str;
	register void			*a1 asm("a1") = (void *) vap;

	asm volatile (
		"	move.l	%%a1,-(%%sp)
			move.l	%%a0,-(%%sp)
			trap	#15
			dc.w	0x0028
		"
		: /* no outputs */
		: "r" (a0), "r" (a1)
		: "a0", "a1"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug getchar subroutine.
 */

static
__inline__
int
MVMEBug_getchar
(	void
)
{	register unsigned long	d0 asm("d0") = 0;

	asm volatile (
		"	subq.l	#2,%%sp
			trap	#15
			dc.w	0x0000
			move.b	(%%sp)+,%%d0
		"
		: "=r" (d0)
		: "r" (d0)
		: "d0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug input status subroutine.
 */

static
__inline__
int
MVMEBug_getchar_status
(	void
)
{	register unsigned long	d0 asm("d0");

	asm volatile (
		"	move.l	#0,-(%%sp)
			trap	#15
			dc.w	0x0001
			beq.s	1f
			move.l	#1,(%%sp)
		1:
			move.l	(%%sp)+,%%d0
		"
		: "=r" (d0)
		: /* no inputs */
		: "d0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug RTC read subroutine.
 */

static
__inline__
void
MVMEBug_rtc_read
(	void			*buf
)
{	register void *			d0 asm("d0") = (void *) buf;

	asm volatile (
		"	move.l	%%d0,-(%%sp)
			trap	#15
			dc.w	0x0053
		"
		: /* no outputs */
		: "r" (d0)
		: "d0"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug disk_read subroutine.
 */

static
__inline__
void
MVMEBug_disk_read
(	void			*cmnd
)
{	register void *			d0 asm("d0") = (void *) cmnd;

	asm volatile (
		"	move.l	%%d0,-(%%sp)
			trap	#15
			dc.w	0x0010
		"
		: /* no outputs */
		: "r" (d0)
		: "d0", "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug.
 */

static
__inline__
void
MVMEBug_enter
(	void
)
{
	asm volatile (
		"	trap	#15
			dc.w	0x0063
		"
		: /* No outputs */
		: /* No inputs  */
		: "d0", "d1", "d2", "d3", "d4", "d5", "d6", "d7",
		  "a0", "a1", "a2", "a3", "a4", "a5", "a6", "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug BRD_ID subroutine.
 */

static
__inline__
void *
MVMEBug_brd_id
(	void
)
{	register void *			d0 asm("d0");

	asm volatile (
		"	subq.l	#4,%%sp
			trap	#15
			dc.w	0x0070
			move.l	(%%sp)+,%%d0
		"
		: "=r" (d0)
		: /* no inputs */
		: "d0"
	);

	return d0;
}

/*-----------------------------< end of file >------------------------------*/
