/*
 *  VME Linux/m68k Loader
 *
 *  (c) Copyright 1997 by Nick Holgate and others.
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING for more details.
 *
 *  The definitions in this file were derived from kernel version 2.1.90
 *  bootinfo.h and setup.h. They were brought together here to enable
 *  compilation without version 2.1.xx kernel sources.
 */

/*--------------------------------------------------------------------------*/
/*  Bootinfo definitions
 *
 *  This is an easily parsable and extendable structure containing all
 *  information to be passed from the bootstrap to the kernel.
 *
 *  This way I hope to keep all future changes back/forewards compatible.
 *  Thus, keep your fingers crossed...
 *
 *  This structure is copied right after the kernel bss by the bootstrap
 *  routine.
 */

struct bi_record {
    unsigned short tag;			/* tag ID */
    unsigned short size;		/* size of record (in bytes) */
    unsigned long data[0];		/* data */
};

/*--------------------------------------------------------------------------*/
/*  Tag Definitions
 *
 *  Machine independent tags start counting from 0x0000
 *  Machine dependent tags start counting from 0x8000
 */

#define BI_LAST			0x0000	/* last record (sentinel) */
#define BI_MACHTYPE		0x0001	/* machine type (u_long) */
#define BI_CPUTYPE		0x0002	/* cpu type (u_long) */
#define BI_FPUTYPE		0x0003	/* fpu type (u_long) */
#define BI_MMUTYPE		0x0004	/* mmu type (u_long) */
#define BI_MEMCHUNK		0x0005	/* memory chunk address and size */
								/* (struct mem_info) */
#define BI_RAMDISK		0x0006	/* ramdisk address and size */
								/* (struct mem_info) */
#define BI_COMMAND_LINE	0x0007	/* kernel command line parameters */
								/* (string) */

/*--------------------------------------------------------------------------*/
/* Stuff for bootinfo interface versioning
 *
 * At the start of kernel code, a 'struct bootversion' is located.
 * bootstrap checks for a matching version of the interface before booting
 * a kernel, to avoid user confusion if kernel and bootstrap don't work
 * together :-)
 *
 * If incompatible changes are made to the bootinfo interface, the major
 * number below should be stepped (and the minor reset to 0) for the
 * appropriate machine. If a change is backward-compatible, the minor
 * should be stepped. "Backwards-compatible" means that booting will work,
 * but certain features may not.
 */

#define BOOTINFOV_MAGIC				0x4249561A	/* 'BIV^Z' */
#define MK_BI_VERSION(major,minor)	(((major)<<16)+(minor))
#define BI_VERSION_MAJOR(v)			(((v) >> 16) & 0xffff)
#define BI_VERSION_MINOR(v)			((v) & 0xffff)

struct bootversion {
    unsigned short branch;
    unsigned long magic;
    struct {
		unsigned long machtype;
		unsigned long version;
    } machversions[0];
};

#define MVME16x_BOOTI_VERSION  MK_BI_VERSION( 2, 0 )
#define BVME6000_BOOTI_VERSION MK_BI_VERSION( 2, 0 )

/*--------------------------------------------------------------------------*/
/*  Linux/m68k Architectures
 */

#define MACH_AMIGA    1
#define MACH_ATARI    2
#define MACH_MAC      3
#define MACH_APOLLO   4
#define MACH_SUN3     5
#define MACH_MVME147  6
#define MACH_MVME16x  7
#define MACH_BVME6000 8

/*--------------------------------------------------------------------------*/
/*  CPU, FPU and MMU types
 *
 *  Note: we may rely on the following equalities:
 *
 *      CPU_68020 == MMU_68851
 *      CPU_68030 == MMU_68030
 *      CPU_68040 == FPU_68040 == MMU_68040
 *      CPU_68060 == FPU_68060 == MMU_68060
 */

#define CPUB_68020     0
#define CPUB_68030     1
#define CPUB_68040     2
#define CPUB_68060     3

#define CPU_68020      (1<<CPUB_68020)
#define CPU_68030      (1<<CPUB_68030)
#define CPU_68040      (1<<CPUB_68040)
#define CPU_68060      (1<<CPUB_68060)

#define FPUB_68881     0
#define FPUB_68882     1
#define FPUB_68040     2                       /* Internal FPU */
#define FPUB_68060     3                       /* Internal FPU */
#define FPUB_SUNFPA    4                       /* Sun-3 FPA */

#define FPU_68881      (1<<FPUB_68881)
#define FPU_68882      (1<<FPUB_68882)
#define FPU_68040      (1<<FPUB_68040)
#define FPU_68060      (1<<FPUB_68060)
#define FPU_SUNFPA     (1<<FPUB_SUNFPA)

#define MMUB_68851     0
#define MMUB_68030     1                       /* Internal MMU */
#define MMUB_68040     2                       /* Internal MMU */
#define MMUB_68060     3                       /* Internal MMU */
#define MMUB_APOLLO    4                       /* Custom Apollo */
#define MMUB_SUN3      5                       /* Custom Sun-3 */

#define MMU_68851      (1<<MMUB_68851)
#define MMU_68030      (1<<MMUB_68030)
#define MMU_68040      (1<<MMUB_68040)
#define MMU_68060      (1<<MMUB_68060)
#define MMU_SUN3       (1<<MMUB_SUN3)
#define MMU_APOLLO     (1<<MMUB_APOLLO)

/*--------------------------------------------------------------------------*/
/*  Miscellaneous
 */

#define NUM_MEMINFO	(4)
#define CL_SIZE		(256)


struct mem_info {
	unsigned long addr;		/* physical address of memory chunk */
	unsigned long size;		/* length of memory chunk (in bytes) */
};

#ifdef BOOTINFO_COMPAT_1_0

/*--------------------------------------------------------------------------*/
/*  Backwards compatibility with bootinfo interface version 1.0
 */

#define COMPAT_MACH_MVME167				4	/* changed to 7 for 2.1.xx		*/
#define COMPAT_MACH_MVME162				5	/* changed to 7 for 2.1.xx		*/
#define COMPAT_MACH_BVME6000			6	/* changed to 8 for 2.1.xx		*/
#define COMPAT_BVME6000_BOOTI_VERSION	MK_BI_VERSION( 1, 0 )
#define COMPAT_MVME167_BOOTI_VERSION	MK_BI_VERSION( 1, 0 )
#define COMPAT_MVME162_BOOTI_VERSION	MK_BI_VERSION( 1, 0 )

struct compat_mem_info {
    unsigned long addr;
    unsigned long size;
};

#define COMPAT_NUM_MEMINFO  4

#define COMPAT_CPUB_68020 0
#define COMPAT_CPUB_68030 1
#define COMPAT_CPUB_68040 2
#define COMPAT_CPUB_68060 3
#define COMPAT_FPUB_68881 5
#define COMPAT_FPUB_68882 6
#define COMPAT_FPUB_68040 7
#define COMPAT_FPUB_68060 8

#define COMPAT_CPU_68020    (1<<COMPAT_CPUB_68020)
#define COMPAT_CPU_68030    (1<<COMPAT_CPUB_68030)
#define COMPAT_CPU_68040    (1<<COMPAT_CPUB_68040)
#define COMPAT_CPU_68060    (1<<COMPAT_CPUB_68060)
#define COMPAT_CPU_MASK     (31)
#define COMPAT_FPU_68881    (1<<COMPAT_FPUB_68881)
#define COMPAT_FPU_68882    (1<<COMPAT_FPUB_68882)
#define COMPAT_FPU_68040    (1<<COMPAT_FPUB_68040)
#define COMPAT_FPU_68060    (1<<COMPAT_FPUB_68060)
#define COMPAT_FPU_MASK     (0xfe0)

#define COMPAT_CL_SIZE      (256)

struct compat_bootinfo {
    unsigned long machtype;
    unsigned long cputype;
    struct compat_mem_info memory[COMPAT_NUM_MEMINFO];
    int num_memory;
    unsigned long ramdisk_size;
    unsigned long ramdisk_addr;
    char command_line[COMPAT_CL_SIZE];
};

#endif /* BOOTINFO_COMPAT_1_0 */

/*-----------------------------< end of file >------------------------------*/
