#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <glibtop.h>
#include <glibtop/xmalloc.h>
#include <glibtop/union.h>

#include <gnome.h>

#include "properties.h"

#include "fsusage.h"
#include "graph.h"
#include "proc.h"
#include "global.h"

#define PROC_CMD_LEN 40

struct _FsUsageProcInfo {
	char          *cmd;
	unsigned long  value, percent;
};

typedef struct _FsUsageProcInfo FsUsageProcInfo;

struct _SelectedFsCbData
{
	GtkWidget *button;
	gint index;
};

typedef struct _SelectedFsCbData SelectedFsCbData;

enum _FieldType {
	TOTAL = 0,
	USED,
	FREE
};

typedef enum _FieldType FieldType;

static void fsusage_type_set (GtkWidget *, gpointer);

GnomeUIInfo fsUsageMenu [] = {
  {GNOME_APP_UI_ITEM, N_("Total"), NULL, fsusage_type_set, (gpointer)TOTAL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Used"), NULL, fsusage_type_set, (gpointer)USED, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Free"), NULL, fsusage_type_set, (gpointer)FREE, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  GNOMEUIINFO_END
};

static GtkWidget *sw;
static Graph     *graph;
static GtkWidget *f;

/* static proc_t **mount_list = NULL; */
static glibtop_mountentry *mount_list = NULL;
static FsUsageProcInfo *fsusage_data = NULL;
static gint fsusage_number = 0;
static unsigned long value_total;
static gint cfg_run;
static gint run_tag = -1;
static FieldType ftype = TOTAL;
static gchar *graph_head;
static gchar *graph_tail;

static void     fsusage_handler (GtkNotebook *notebook, GtkNotebookPage *page, GTopPageSignal s);
static gpointer fsusage_data_fn (GraphCmd cmd, gpointer data);
static void     fsusage_properties_init (GTopPropertiesHook *hook, GtkWidget *win);
static void     fsusage_properties_update (GTopPropertiesHook *hook);
static void     fsusage_properties_load (GTopPropertiesHook *hook);
static void     fsusage_properties_save (GTopPropertiesHook *hook);
static gint     fsusage_update ();
static void     fsusage_init ();
static void     fsusage_start_stop (gint start);

#define GTOP_NUM_FSTYPES	7

static const gchar *gtop_fstype_names [GTOP_NUM_FSTYPES]  = {
	N_("ufs"),
	N_("nfs"),
	N_("msdos"),
	N_("iso9660"),
	N_("ext2fs"),
	N_("minix"),
	N_("other")
};

static const gchar *gtop_fstype_labels [GTOP_NUM_FSTYPES] = {
	N_("User Filesystem (FreeBSD)"),
	N_("Network Filesystem"),
	N_("MSDOS Filesystem"),
	N_("CDROM Filesystem"),
	N_("Second Extended Filesystem (Linux)"),
	N_("Minix Filesystem (Linux)"),
	N_("Other Filesystems")
};

GTopPropertiesHook FsUsagePropertiesHook = {
	fsusage_properties_init,
	NULL,
	fsusage_properties_update,
	fsusage_properties_load,
	fsusage_properties_save,
	NULL
};

void *
addFsUsageView ()
{
	GtkWidget *gw;

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_border_width (GTK_CONTAINER (sw), GNOME_PAD_SMALL);
	gtk_widget_set_name (sw, "FsUsageGraph");

	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (f), GTK_SHADOW_NONE);
	gtk_container_border_width (GTK_CONTAINER (f), GNOME_PAD_SMALL << 1);
 
	graph = graph_new (fsusage_data_fn);
	gw = graph_widget (graph);
	graph_colors_set (graph, graph_default_colors, GRAPH_DEFAULT_COLORS);

	gtk_container_add (GTK_CONTAINER (f), gw);
	gtk_container_add (GTK_CONTAINER (sw), f);

	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), sw,
				  gtk_label_new (_("Filesystems")));

	gtk_widget_show (gw);
	gtk_widget_show (f);
	gtk_widget_show (sw);

	fsusage_init ();

	return fsusage_handler;
}

static void
fsusage_cfg_save ()
{
	/* printf ("proview_cfg_save\n"); */

	//gnome_config_set_int ("/gtop/procview/run", cfg_run);
	//gnome_config_set_int ("/gtop/procview/summary", !cfg_summary);
	/*gnome_config_set_int ("/gtop/procview/sort_field", cfg_sf);*/
	/*gnome_config_set_int ("/gtop/procview/sort_order", !cfg_order);*/

	gnome_config_set_int ("/gtop/fsusage/field_type", ftype);
	gnome_config_set_int ("/gtop/fsusage/run", cfg_run);

	gnome_config_sync ();
}

static void
fsusage_time_cb (GtkWidget *w, gpointer gp)
{
	cfg_run = !cfg_run;
	fsusage_start_stop (cfg_run);
}

static void
fsusage_handler (GtkNotebook *notebook, GtkNotebookPage *page, GTopPageSignal s)
{
	/*printf ("procview handler %x %x\n", page->child, vbox);*/

	/* check if it is for me */
	if (page->child == sw) {

		switch (s) {

		case GTOP_PAGE_CFG_SAVE:

			fsusage_cfg_save ();
			break;

		case GTOP_PAGE_ENTER:

			/*printf ("procview page entered\n");*/

			gtop_time_cb = fsusage_time_cb;
			fsusage_update ();
			fsusage_start_stop (cfg_run);

			gnome_app_menu_show (GNOME_APP(window),
					     GTOP_MENU_FSUSAGE, -1);

			break;

		case GTOP_PAGE_LEAVE:

			/*printf ("procview page leaved\n");*/

			gtop_time_cb = NULL;
			fsusage_start_stop (0);

			gnome_app_menu_hide (GNOME_APP(window),
					     GTOP_MENU_FSUSAGE, -1);
			break;

		default:
			break;
		}
	}
}

static void
fsusage_type_set (GtkWidget *w, gpointer p)
{
	GTopPropFsMode fsmode = gtop_properties.fsmode;

	ftype = (FieldType)p;

	switch (fsmode) {
	case GTOP_FSMODE_SUBLOCKS:
		switch (ftype) {
		case TOTAL:
			graph_head = _("Total Filesystem Sizes "
				       "(including reserved blocks)");
			graph_tail = _("Sum of Total Sizes: %ldk");
			break;
		case USED:
			graph_head = _("Used Space on Filesystems "
				       "(including reserved blocks)");
			graph_tail = _("Sum of Uses Space: %ldk");
			break;
		case FREE:
			graph_head = _("Free Space on Filesystems "
				       "(including reserved blocks)");
			graph_tail = _("Sum of Free Space: %ldk");
			break;
		}
		break;
	case GTOP_FSMODE_BLOCKS:
		switch (ftype) {
		case TOTAL:
			graph_head = _("Total Filesystem Sizes");
			graph_tail = _("Sum of Total Sizes: %ldk");
			break;
		case USED:
			graph_head = _("Used Space on Filesystems");
			graph_tail = _("Sum of Uses Space: %ldk");
			break;
		case FREE:
			graph_head = _("Free Space on Filesystems");
			graph_tail = _("Sum of Free Space: %ldk");
			break;
		}
		break;
	case GTOP_FSMODE_INODES:
		switch (ftype) {
		case TOTAL:
			graph_head = _("Total Number of Inodes");
			graph_tail = _("Sum of Total Number of Inodes: %ld");
			break;
		case USED:
			graph_head = _("Used Inodes on Filesystems");
			graph_tail = _("Sum of Used Inodes: %ld");
			break;
		case FREE:
			graph_head = _("Free Inodes on Filesystems");
			graph_tail = _("Sum of Free Inodes: %ld");
			break;
		}
		break;
	}

	fsusage_update ();
}

static gpointer
fsusage_data_fn (GraphCmd cmd, gpointer data)
{
	FsUsageProcInfo *info = data;
	GTopPropFsMode fsmode = gtop_properties.fsmode;
	static gchar buf [256];
	static gchar tail [256];

	switch (cmd) {

	case GRAPH_FIRST:

		return (fsusage_data) ? (fsusage_data [0].cmd) ? fsusage_data : NULL : NULL;

	case GRAPH_NEXT:

		return (info [1].cmd) ? info+1 : NULL;

	case GRAPH_VALUE:

		return (gpointer)(info->value);

	case GRAPH_LABEL:

		switch (fsmode) {
		case GTOP_FSMODE_SUBLOCKS:
		case GTOP_FSMODE_BLOCKS:
			sprintf (buf, "%21s (%3d%% free): %14ldk",
				 info->cmd, info->percent, info->value);
			break;
		case GTOP_FSMODE_INODES:
			sprintf (buf, "%21s (%3d%% free) : %14ld",
				 info->cmd, info->percent, info->value);
			break;
		}

		return buf;

	case GRAPH_HEAD:

		return graph_head;

	case GRAPH_TAIL:

		sprintf (tail, graph_tail,
			 value_total >> 10);

		return tail;

	default:

		return NULL;
	}
}

static int
fsusage_cmd_cmp (const void *i1,
		  const void *i2)
{
	return strcmp (((FsUsageProcInfo *)i1)->cmd,
		       ((FsUsageProcInfo *)i2)->cmd);
}

static gint
fsusage_update ()
{
	FsUsageProcInfo *ti;
	ProcInfo info;
	gint n = 0, i, j, k = 0, l;
	unsigned long value, percent;
	char *cmd;
	static gchar bts [256];
	glibtop_mountlist mountlist;
	GTopPropFsMode fsmode;
	glong selected_fs_mask;
	gint selected_fs;

	/* printf ("fsusage_update\n"); */

	if (mount_list)
		glibtop_free (mount_list);

	mount_list = glibtop_get_mountlist (&mountlist, 0);

	n = mountlist.number;

	/* temporary info */

	ti = g_new (FsUsageProcInfo, n);

	value = 0; /* keep gcc happy */
	value_total = 0;

	fsmode = gtop_properties.fsmode;
	selected_fs = gtop_properties.selected_fs;
	selected_fs_mask = gtop_properties.selected_fs_mask;

	for (i=0, k=0; i<n; i++) {
		glibtop_fsusage fsusage;

		if (selected_fs) {
			gchar *type = mount_list [i].type ?
				mount_list [i].type : "other";
			gint selected = 0;

			for (j = 0; j < GTOP_NUM_FSTYPES; j++) {
				const gchar *name = gtop_fstype_names [j];

				if (!(selected_fs_mask & (1 << j)))
					continue;

				if (!strcmp (name, "other"))
					selected = 1;

				if (!strcmp (type, name))
					selected = 1;
			}

			if (!selected)
				continue;
		}

		glibtop_get_fsusage (&fsusage, mount_list [i].mountdir);

		switch (fsmode) {
		case GTOP_FSMODE_SUBLOCKS:
			switch (ftype) {
			case TOTAL:
				value = fsusage.blocks;
				break;
			case USED:
				value = fsusage.blocks - fsusage.bfree;
				break;
			case FREE:
				value = fsusage.bfree;
				break;
			}
			percent = fsusage.blocks ?
				(fsusage.bfree * 100 / fsusage.blocks) : 0;
			break;
		case GTOP_FSMODE_BLOCKS:
			switch (ftype) {
			case TOTAL:
				value = fsusage.blocks;
				break;
			case USED:
				value = fsusage.blocks - fsusage.bavail;
				break;
			case FREE:
				value = fsusage.bavail;
				break;
			}
			percent = fsusage.blocks ?
				(fsusage.bavail * 100 / fsusage.blocks) : 0;
			break;
		case GTOP_FSMODE_INODES:
			switch (ftype) {
			case TOTAL:
				value = fsusage.files;
				break;
			case USED:
				value = fsusage.files - fsusage.ffree;
				break;
			case FREE:
				value = fsusage.ffree;
				break;
			}
			percent = fsusage.files ?
				(fsusage.ffree * 100 / fsusage.files) : 0;
			break;
		}

		value >>= 1;
		if (value == 0)
			continue;

		ti [k].cmd     = glibtop_strdup (mount_list [i].mountdir);
		ti [k].percent = percent;
		ti [k].value   = value;
		value_total   += value << 10;
		k++;
	}

	n = k;

	if (!n) { /* avoids some trouble. */
		ti [0].cmd     = glibtop_strdup ("unknown");
		ti [0].percent = 0;
		ti [0].value   = 1;
		value_total    = 1;
		n = 1;
	}

	/*
	 * sort info by cmd
	 * we need same processes cmd grouped
	 *
	 */

	qsort (ti, n, sizeof (FsUsageProcInfo), fsusage_cmd_cmp);

	if (fsusage_data) {
		for (i = 0; i < fsusage_number; i++)
			glibtop_free ((fsusage_data+i)->cmd);
		g_free (fsusage_data);
	}

	fsusage_number = n;
	fsusage_data = g_new (FsUsageProcInfo, n+1);

	memset (fsusage_data, 0, sizeof (FsUsageProcInfo) * (n+1));
	
	memcpy (fsusage_data, ti, sizeof (FsUsageProcInfo) * n);

	g_free (ti);
	
	graph_minimal_height_set
	  (graph, sw->allocation.height - 8*GNOME_PAD_SMALL);
	graph_update (graph);

	return TRUE;
}

static void
fsusage_start_stop (gint start)
{
	/* printf ("fsusage_start_stop %d\n", start); */

	if (!start) {
		if (run_tag != -1) {
			gtk_timeout_remove (run_tag);
			run_tag = -1;
		}

	} else {

		if (run_tag != -1)
			return;

		fsusage_update ();
		run_tag = gtk_timeout_add (3000, fsusage_update, NULL);
	}

	gnome_stock_set_icon ((GnomeStock*) (TBC (RUN)->icon),
			      (start) ? GNOME_STOCK_PIXMAP_TIMER :
			      GNOME_STOCK_PIXMAP_TIMER_STOP);

	gtk_widget_queue_draw (GTK_WIDGET (TBC (RUN)->widget));
}

static void
fsusage_init ()
{
	gboolean def;

	ftype   = gnome_config_get_int_with_default
		("/gtop/fsusage/field_type=0", &def);
	cfg_run = gnome_config_get_int ("/gtop/fsusage/run=1");
	
	if (def)
		ftype = TOTAL;

	fsusage_type_set (NULL, (gpointer)ftype);
}

static void
fsusage_properties_update (GTopPropertiesHook *hook)
{
	fsusage_type_set (NULL, (gpointer)ftype);
	fsusage_update ();
}

static void
selected_fs_cb (GtkWidget *widget, GtkWidget *button)
{
	gtop_temp_properties.selected_fs = GTK_TOGGLE_BUTTON (button)->active;
	gtop_properties_changed ();
}

static void
selected_fsmask_cb (GtkWidget *widget, SelectedFsCbData *cb_data)
{
	if (GTK_TOGGLE_BUTTON (cb_data->button)->active)
		gtop_temp_properties.selected_fs_mask |= (1 << cb_data->index);
	else
		gtop_temp_properties.selected_fs_mask &= ~(1 << cb_data->index);
	gtop_properties_changed ();
}

static void
radio_sublocks_cb (GtkWidget *widget, GtkWidget *button)
{
	if (!GTK_TOGGLE_BUTTON (button)->active)
		return;

	gtop_temp_properties.fsmode = GTOP_FSMODE_SUBLOCKS;
	gtop_properties_changed ();
}

static void
radio_blocks_cb (GtkWidget *widget, GtkWidget *button)
{
	if (!GTK_TOGGLE_BUTTON (button)->active)
		return;

	gtop_temp_properties.fsmode = GTOP_FSMODE_BLOCKS;
	gtop_properties_changed ();
}

static void
radio_inodes_cb (GtkWidget *widget, GtkWidget *button)
{
	if (!GTK_TOGGLE_BUTTON (button)->active)
		return;

	gtop_temp_properties.fsmode = GTOP_FSMODE_INODES;
	gtop_properties_changed ();
}

static void
fsusage_properties_init (GTopPropertiesHook *hook, GtkWidget *win)
{
	GtkWidget *vb, *vbox, *frame, *label, *button, *spin, *table;
	GSList *group;
	gint i;

	vb = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vb), GNOME_PAD_SMALL);

	frame = gtk_frame_new (_("Select information to show"));
	table = gtk_table_new (3, 1, TRUE);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD << 2);

	button = gtk_radio_button_new_with_label
		(NULL, _("Show 1k-blocks reserved blocks"));
	if (gtop_temp_properties.fsmode == GTOP_FSMODE_SUBLOCKS)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
	gtk_signal_connect
		(GTK_OBJECT (button), "toggled", radio_sublocks_cb, button);
	gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 1, 0, 1);

	group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	button = gtk_radio_button_new_with_label (group, _("Show 1k-blocks"));
	if (gtop_temp_properties.fsmode == GTOP_FSMODE_BLOCKS)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
	gtk_signal_connect
		(GTK_OBJECT (button), "toggled", radio_blocks_cb, button);
	gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 1, 1, 2);

	group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	button = gtk_radio_button_new_with_label (group, _("Show inodes"));
	if (gtop_temp_properties.fsmode == GTOP_FSMODE_INODES)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
	gtk_signal_connect
		(GTK_OBJECT (button), "toggled", radio_inodes_cb, button);
	gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 1, 2, 3);

	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vb), frame, FALSE, TRUE, 0);

	frame = gtk_frame_new ("");
	table = gtk_table_new (GTOP_NUM_FSTYPES+2, 8, FALSE);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD << 2);

	label = gtk_check_button_new_with_label
		(_("Only show selected filesystems"));
	if (gtop_temp_properties.selected_fs)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (label), TRUE);
	gtk_signal_connect
		(GTK_OBJECT (label), "toggled", selected_fs_cb, label);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 8, 0, 1);

	for (i = 0; i < GTOP_NUM_FSTYPES; i++) {
		SelectedFsCbData *cb_data;

		label = gtk_check_button_new_with_label
			(_(gtop_fstype_names [i]));

		if (gtop_temp_properties.selected_fs_mask & (1 << i))
			gtk_toggle_button_set_state
				(GTK_TOGGLE_BUTTON (label), TRUE);

		cb_data = g_new (SelectedFsCbData, 1);

		cb_data->index = i;
		cb_data->button = label;

		gtk_signal_connect (GTK_OBJECT (label), "toggled",
				    selected_fsmask_cb, cb_data);

		gtk_table_attach_defaults
			(GTK_TABLE (table), label, 1, 2, i+1, i+2);

		label = gtk_label_new (_(gtop_fstype_labels [i]));
		gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
		gtk_table_attach_defaults
			(GTK_TABLE (table), label, 2, 3, i+1, i+2);
	}

	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vb), frame, FALSE, TRUE, 0);

	gtk_notebook_append_page
	  (GTK_NOTEBOOK (GNOME_PROPERTY_BOX (win)->notebook),
	   vb, gtk_label_new (_("Filesystems")));
}

static void
fsusage_properties_load (GTopPropertiesHook *hook)
{
	gtop_properties.fsmode = gnome_config_get_int ("gtop/fsusage/fsmode=1");

	gtop_properties.selected_fs =
		gnome_config_get_int ("gtop/fsusage/selected_fs=0");

	gtop_properties.selected_fs_mask =
		gnome_config_get_int ("gtop/fsusage/selected_fs_mask=0");
}

static void
fsusage_properties_save (GTopPropertiesHook *hook)
{
	gnome_config_set_int
		("gtop/fsusage/fsmode", gtop_properties.fsmode);

	gnome_config_set_int
		("gtop/fsusage/selected_fs", gtop_properties.selected_fs);

	gnome_config_set_int ("gtop/fsusage/selected_fs_mask",
			      gtop_properties.selected_fs_mask);
}
