/*
 * $Id: plugins.c,v 1.1 1998/09/13 19:13:24 gregm Exp $
 * GXSNMP -- An snmp management application
 * Copyright (C) 1998 Gregory McLean
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * plugin.c -- Support for dynamically-loaded plugin components.
 * Originally from gnumeric.
 * Orignal author Tom Dyas (tdyas@romulus.rutgers.edu)
 * Embraced and extended by Gregory McLean <gregm@comstar.net>
 * Yeah I'm a theif :)
 */
#include <unistd.h>
#include <dirent.h>
#include <glib.h>
#include <gmodule.h>
#include <gnome.h>
#include <string.h>
#include "plugins.h"

GList *plugin_list = NULL;

PluginData *
plugin_load (gchar *modfile)
{
  PluginData   *data;
  
  g_return_val_if_fail (modfile != NULL, NULL);

  g_print ("Loading plugin '%s'.\n", modfile);

  data = g_new0 (PluginData, 1);

  if (!data)
    {
      g_warning ("Plugin memory allocation error loading: `%s`\n", modfile);
      return NULL;
    }
  data->handle = g_module_open (modfile, 0);
  if (!data->handle)
    {
      g_warning ("Unable to open module file: `%s`\n", g_module_error ());
      g_free (data);
      return NULL;
    }
  if (!g_module_symbol (data->handle, "init_plugin",
			(gpointer *)&data->init_plugin))
    {
      g_warning ("Bad plugin `%s`\nIt must contain a init_plugin function\n",
		 modfile);
      goto error;
    }
  if (!g_module_symbol (data->handle, "cleanup_plugin",
			(gpointer *)&data->cleanup_plugin))
    {
      g_warning ("Bad plugin `%s`\nIt must contain a cleanup_plugin "
		 "function\n", modfile);
      goto error;
    }
  plugin_list = g_list_append (plugin_list, data);
  return data;

 error:
  g_module_close (data->handle);
  g_free (data);
  return NULL;
}

void
plugin_unload (struct PluginData *pd)
{
  g_return_if_fail (pd != NULL);

  g_print ("Attempting to unload plugin %s\n", g_module_name (pd->handle));
  if (pd->refcount > 0)
    {
      g_warning ("unload_plugin: Refcount is positive, cannot unload plugin "
		 "%s\n",g_module_name (pd->handle));
      return;
    }
  if (pd->cleanup_plugin)
    pd->cleanup_plugin (pd);
  plugin_list = g_list_remove (plugin_list, pd);

  g_module_close (pd->handle);
  g_free (pd);
}

static void
plugin_load_plugins_in_dir (char *directory)
{
  DIR           *d;
  struct dirent *e;
  
  if ((d = opendir (directory)) == NULL)
    return;
  
  while ((e = readdir (d)) != NULL)
    {
      if (strncmp (e->d_name + strlen (e->d_name) - 3, ".so", 3) == 0)
	{
	  char *plugin_name;
	  
	  plugin_name = g_copy_strings (directory, e->d_name, NULL);
	  plugin_load (plugin_name);
	  
	  g_free (plugin_name);
	}
    }
  closedir (d);
}

void
plugins_init(void)
{
  char *plugin_dir;
  char *home_dir = getenv ("HOME");
  
  if (!g_module_supported())
    return;
  
  g_print ("plugins_init()\n");
  
  /* Load the user plugins */
  plugin_dir = g_copy_strings (home_dir ? home_dir : "", 
			       "/.gxsnmp/plugins/", NULL);
  plugin_load_plugins_in_dir (plugin_dir);
  g_free (plugin_dir);
  
  /* Load the system plugins */
  plugin_dir = gnome_unconditional_libdir_file ("gxsnmp/plugins/");
  plugin_load_plugins_in_dir (plugin_dir);
  g_free (plugin_dir);
}

/* EOF */
      
