#include <string.h>
#include <sys/types.h>
#include <asm/page.h>
#include <sys/swap.h>
#include <sys/mount.h>
#include <stdio.h>
#include "dinstall.h"
#include "../busybox/internal.h"
#include INCLINGUA

extern int mount_one( char *blockDevice, char *directory, char *filesystemType,
  unsigned long flags, char *string_flags, int noMtab, int fake);

#ifdef _TESTING_
extern int reboot_system(void);
extern void initScreen (char *);
extern void finishScreen(void);
#endif

static struct fdisk_disk *
select_drive(void) {
  struct fdisk_disk *disk,**dlist;
  struct d_choices *choices;
  int rs, items=0;

  choices=malloc(5*sizeof(struct d_choices));
  dlist=malloc(5*sizeof(struct fdisk_disk *));

  /* shell version used try_open here. Why ? */
  
  if (NULL == fdisk_disks) {
    problemBox(MSG_NO_HARD_DISK_L,MSG_NO_HARD_DISK);
    return(NULL);
  }
  
  disk = fdisk_disks;
  while (disk) {
    choices[items].string=disk->name;
    choices[items].tag=NULL;
    choices[items].state=0;
    dlist[items]=disk;
    items++;
    if ((items%5)==0) {
      choices = realloc(choices,(items+5)*sizeof(struct d_choices));
      dlist = realloc(dlist,(items+5)*sizeof(struct fdisk_disk *));
    }
    disk = disk->next;
  }
  
  rs = menuBox(MSG_SELECT_HD, MSG_SELECT_DISK_DRIVE, choices, items, 1);
  if (rs != -1)
    disk = dlist[rs];
  else
    disk = NULL;

  free(dlist);
  free(choices);
  return(disk);
}

int partition_disk (void) {
  struct fdisk_disk *d;
  struct fdisk_partition *p;
  char *mounts, *swaps, *tmpbuf; 
  int root_on_disk=0, status;
  struct stat statbuf;

  sync();
  mounts = swaps = NULL;

  if (! (d = select_drive())) {
    return(1);
  }

  p = d->partitions;

  while (p) {
    if (p->in_use) {
      if (fdisk_fstype_of(p->type) == FSTYPE_SWAP) {
	if (swaps) {
	  addtolist(&swaps,p->name);
        } else {
	  swaps = strdup(p->name);
        }
      } else {
	if (mounts) {
	  addtolist(&mounts,p->name);
        } else {
          mounts = strdup(p->name);
        }
      }  
    }
    p=p->next_by_disk;
  }
  prtbuf[0]='\0';
  if (mounts)
    sprintf(prtbuf,MSG_REPARTITION_MOUNTED ,mounts,d->name,d->name);
  if (swaps) {
    tmpbuf=(char *)malloc(strlen(swaps)+2*strlen(d->name)+320);
    sprintf(tmpbuf,MSG_REPARTITION_SWAP,swaps,d->name,d->name);
    strcat(prtbuf,tmpbuf);
    free(tmpbuf);
  }
  if ((mounts)||(swaps)){
    strcat(prtbuf,MSG_REPARTITION_HD);
    if (!(yesNoBox(prtbuf,MSG_REPARTITION_HD))) return (1);
  }
  if (mounts) {
    p = mounted_partitions;
    while (p) {
      if (strstr(mounts,p->name)) {
	if (do_umount(p->name,0)) {
/* XXX should check errno ? */
	  sprintf(prtbuf, MSG_UMOUNT_FAILED_L2,p->name, p->name, p->name);
	  problemBox(prtbuf,MSG_UMOUNT_FAILED);
	  root_on_disk=1;
	}
      }
      p=p->next_in_use;
    }
    free (mounts);
  }
  if (swaps) {
    p = swapon_partitions;
    while (p) {
      if (strstr(swaps,p->name)) {
	struct FileInfo finfo;
	finfo.source = p->name;
	swapoff_fn(&finfo);
      }
      p=p->next_in_use;
    }
    free (swaps);
  }
  if ( NAME_ISEXE( "/sbin/cfdisk", &statbuf ) ) {
    setenv("LANG", LINGUA, 1);
    sprintf(prtbuf,"cfdisk %s",d->name);
    boxSuspend();
    status=system(prtbuf);
    boxResume();
    if (status != 0) {
      if (yesNoBox(MSG_INVALID_PT_L,MSG_INVALID_PT)!=0) {
        sprintf(prtbuf,"cfdisk -z %s",d->name);
        boxSuspend();
        status=system(prtbuf);
        boxResume();
      }
    }
  } else {
    sprintf(prtbuf,"fdisk %s",d->name);
    boxSuspend();
    status=system(prtbuf);
    boxResume();
  }
  if (root_on_disk) reboot_system();
  return 0;
}

static int mount_partition (struct fdisk_partition *partition) {
  char *mount_point,*real_mount_point, *prototype, *mode, *type;
  struct stat statbuf;
  int status;

  type = fdisk_fstype_name_of(partition->type);
  if (!type) {
    sprintf(prtbuf,MSG_INVALID_FS_L,partition->name,sysname(partition->type));
    problemBox(prtbuf,MSG_INVALID_FS);
    return 1;
  }
  if (! Root) {
    sprintf(prtbuf,MSG_MOUNT_ROOT_L,partition->name); 
    if (! yesNoBox(prtbuf,MSG_MOUNT_ROOT) ) return 1;
    mount_point=strdup("/");
    real_mount_point=strdup("/target");
  } else {
    if ( (! NAME_ISDIR( "/target/usr", &statbuf ) ) ||
         (! strcmp(Root->name,block_device("/target/usr",NULL))) ){
      prototype="/usr";
    } else if ( (! NAME_ISDIR( "/target/var", &statbuf ) ) ||
                (! strcmp(Root->name,block_device("/target/var",NULL))) ){ 
      prototype="/var";
    } else if ( (! NAME_ISDIR( "/target/home", &statbuf ) ) ||
                (! strcmp(Root->name,block_device("/target/home",NULL))) ){
      prototype="/home";
    } else {
      prototype="/";
    }

    sprintf(prtbuf,MSG_SELECT_MOUNT_POINT_L,partition->name);
    mount_point=inputBox(prtbuf,MSG_SELECT_MOUNT_POINT,prototype);
    if (NULL == mount_point) return 1;
    real_mount_point=(char *)malloc(8+strlen(mount_point));
    real_mount_point[0]='\0';
    strcat(real_mount_point,"/target");
    strcat(real_mount_point,mount_point);
  }

  if (! NAME_ISDIR( real_mount_point, &statbuf ) ) {
    struct FileInfo finfo;
    if (! strcmp(real_mount_point,"/target/tmp") ) {
      mode="1777";
    } else {
      mode="755";
    }
    finfo.source = real_mount_point;
    finfo.orWithMode = strtol(mode, 0, 010);
    finfo.makeParentDirectories = 1;
    if (! mkdir_fn(&finfo)) {
      chown(finfo.source, 0, 3); /* root_uid= 0 , sys_gid= 3 */
    }
  }
#ifdef NFSROOT
  if ( ! strcmp(type,"nfs") ) {
    /* try external mount program because the one provided in busybox cannot
     * deal with NFS drives */
    sprintf(prtbuf,"mount -t nfs %s %s",partition->name,real_mount_point);
    status = system(prtbuf);
  }
  else
#endif
    status = mount_one(partition->name,real_mount_point,type,0,"\0",0,0);
  if (status) {
    perrorBox(MSG_MOUNT_FAILED);
    free(real_mount_point);
    free(mount_point);
    return 1;
  }
  if (! strcmp(mount_point,"/") ) {
    Root=partition;
	check_pending_config();
  }
  free(mount_point);
  return 0;
}

static int get_cflag (const char *name) {
  struct stat statbuf;

  if (! NAME_ISEXE( "/sbin/badblocks", &statbuf ) ) {
/* 	not_on_lowmem "/sbin/badblocks not on lowmem disk" */
    return 0;
  }
  sprintf(prtbuf,MSG_SCAN_BADBLOCKS_L, name);
  return yesNoBox(prtbuf,MSG_SCAN_BADBLOCKS);
}

int init_linux (void) {
  struct fdisk_partition *p;
  int cflag,status;

  sync();
  p=select_not_mounted(MSG_SELECT_INITIALIZE ,MSG_SELECT_PARTITION,1,FSTYPE_EXT2);
  if (!p) return 1;
  cflag=get_cflag(p->name);
  sprintf(prtbuf,MSG_INITIALIZE_INFO,p->name);
  if(! yesNoBox(prtbuf,MSG_ARE_YOU_SURE) ) return 1;

  setenv("LANG", LINGUA, 1);
  sprintf(prtbuf,"mkfs.ext2 %s %s",( cflag ? "-c" : ""),p->name);
  boxSuspend();
  printf(CLEAR MSG_CREATING_FS);
  status=system(prtbuf);
  boxResume();
  if (status) {
/* FIXME	write_it_down "The filesystem was not created." */
    problemBox(MSG_FS_NOT_CREATED,MSG_PROBLEM);
    return 1;
  }
  return mount_partition(p);
}

int init_swap (void) {
  struct fdisk_partition *p;
  struct FileInfo finfo;
  int cflag,status;

  sync();
  p=select_not_mounted(MSG_SELECT_INITIALIZE_SWAP_L,MSG_SELECT_SWAP,1,FSTYPE_SWAP);
  if (!p) return 1;
  cflag=get_cflag(p->name); 
  sprintf(prtbuf,MSG_INITIALIZE_SWAP_INFO,p->name);
  if(! yesNoBox(prtbuf,MSG_ARE_YOU_SURE) ) return 1;

  sprintf(prtbuf,"mkswap %s %s",( cflag ? "-c" : ""),p->name);
  boxSuspend();
  printf(CLEAR MSG_INITIALIZING_SWAP);
  status=system(prtbuf);
  boxResume();
  if (status) {
/* FIXME        write_it_down "The swap partition could not be initialized." */
    problemBox(MSG_SWAP_NOT_CREATED,MSG_PROBLEM);
    return 1;
  }
  finfo.source = p->name;
  if (swapon_fn(&finfo)) {
    sprintf(prtbuf,MSG_SWAP_NOT_ACTIVED,p->name);
    perrorBox(prtbuf);
    return 1;
  }
  return 0;
}

int no_swap (void) {
  struct fdisk_partition *p;

  if ( yesNoBox( MSG_DO_WIHOUT_SWAP_L,MSG_DO_WIHOUT_SWAP )) {
    if (swapon_partitions) {
      p = swapon_partitions;
      while (p) {
        struct FileInfo finfo;
        finfo.source = p->name;
        swapoff_fn(&finfo);
        p=p->next_in_use;
      }
    }
    noSwap=1;
  }
  return 0;
}

int mount_any (void) {
  struct fdisk_partition *p;

  int ntypes = 5;
#ifdef NFSROOT
  if (is_network_configured())
	  ntypes++;
#endif
  p=select_not_mounted( MSG_SELECT_PARTITION_2MOUNT, MSG_SELECT_PARTITION, ntypes,
		FSTYPE_EXT2,FSTYPE_MINIX,FSTYPE_MSDOS,FSTYPE_AFFS,FSTYPE_HFS,FSTYPE_NFS);
  if (!p) return 1;
  return mount_partition(p);
}

static int unmount_partition (struct fdisk_partition *p) {

  if (do_umount(p->mount_point,0)) {
/* XXX should check errno ? */
    sprintf(prtbuf,MSG_UMOUNT_FAILED_L2, p->name, p->name, p->name);
    problemBox(prtbuf,MSG_UMOUNT_FAILED);
    return 1;
  }
  if ( p == Root ) Root=NULL;
  return 0;
}

int unmount_any (void) {
  struct fdisk_partition *p, **plist;
  struct d_choices *choices;
  int rs, items=0;

  choices=malloc(5*sizeof(struct d_choices));
  plist=malloc(5*sizeof(struct fdisk_partition *));

  p=mounted_partitions;
  if ((p == NULL) || 
      ((! strcmp(p->name,InstallationRootDevice)) && (p->next_in_use==NULL))) {
    problemBox(MSG_NO_MOUNTED_PARTITION,MSG_PROBLEM);
    return 1;
  }

  p = mounted_partitions;
  while (p) {
    if (strcmp(p->name,InstallationRootDevice)) {
      choices[items].string=p->name;
      choices[items].tag=NULL;
      choices[items].state=0;
      plist[items]=p;
      items++;
      if ((items%5)==0) {
        choices = realloc(choices,(items+5)*sizeof(struct d_choices));
        plist = realloc(plist,(items+5)*sizeof(struct fdisk_partition *));
      }
    }
    p = p->next_in_use;
  }

  rs = menuBox(MSG_SELECT_PARTITION_2UMOUNT, MSG_SELECT_PARTITION,
	choices, items, 1);
  if (rs != -1)
    p = plist[rs];
  else
    p = NULL;

  free(plist);
  free(choices);
  if (p == NULL) return 1;
  return unmount_partition(p);
}

int activate_swap (void) {
  struct fdisk_partition *p;
  struct FileInfo finfo;

  p=select_not_mounted(MSG_SELECT_ACTIVATE_SWAP_L ,MSG_SELECT_SWAP,1,FSTYPE_SWAP);
  if (!p) return 1;
  sprintf(prtbuf,MSG_ACTIVATE_INFO,p->name);
  if(! yesNoBox(prtbuf,MSG_ARE_YOU_SURE) ) return 1;
  finfo.source = p->name;
  if (swapon_fn(&finfo)) {
    sprintf(prtbuf,MSG_SWAP_NOT_ACTIVED,p->name);
    perrorBox(prtbuf);
    return 1;
  }
  return 0;
}

int view_partitions (void) {
  struct fdisk_disk *d;
  struct fdisk_partition *p;

  d = fdisk_disks;
  if (d) {
    strcpy(prtbuf,MSG_VIEW_PARTITIONS_TITLE);
    strcat(prtbuf,"\n");
    while (d) {
      p = d->partitions;
      if (p) {
        while (p) {
          sprintf(prtbuf+strlen(prtbuf),"%-10s  %-3s  %-20s  %-18s\n",p->name,
            (p->in_use)?MSG_YES:MSG_NO,sysname(p->type),
            (p->mount_point)? p->mount_point : " ");
          p = p->next_by_disk;
        }
      } else {
        sprintf(prtbuf+strlen(prtbuf),MSG_NO_PARTITIONS,d->name);
	strcat(prtbuf,"\n");
      }
      d = d->next;
      strcat(prtbuf,"\n");
    }
    problemBox(prtbuf, MSG_PARTITIONS_LIST);
    return 0;
  } else {
    problemBox(MSG_NO_HARD_DISK_L, MSG_NO_HARD_DISK);
    return 1;
  }
}

#ifdef _TESTING_
/*
 * To test, compile using: make partition_config_test
 */
char *real_kver = KVER;
char *drivers_image_filename;
char *kernel_image_filename;
int reboot_system(void) {printf ("Reboot the system\n"); return 0;}
int check_pending_config(void) {return 0;}
void main(void) {
  Root=calloc(1,sizeof(struct fdisk_partition));
  fdisk_disks=calloc(1,sizeof(struct fdisk_disk));
  Root->name=strdup("/dev/sda1");
  fdisk_disks->name=strdup("/dev/sda");

  initScreen("Partition config test program");
  partition_disk();

  /*
  if ( choose_medium() ) {
    fprintf(stderr,"\nFailed!\n");
  } else {
    fprintf(stderr,"\nArchive_Dir = %s\n",Archive_Dir);
  }  
  if ( choose_medium() ) {
    fprintf(stderr,"\nFailed!\n");
  } else {
    fprintf(stderr,"\nArchive_Dir = %s\n",Archive_Dir);
  }  
  */
  finishScreen();
}
#endif
