/*
 * apcd.c - Daemon for the APC Smart UPS
 *
 * Copyright (c) 1995 Pavel Korensky
 * Copyright (c) 2000 Koleszar Tibor (for the extensions)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL PAVEL KORENSKY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF PAVEL
 * KORENSKY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * PAVEL KORENSKY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND PAVEL KORENSKY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/*      
 * Version:
 *      
 * $Id: apcd.c,v 1.7 1995/11/07 12:40:03 root Exp root $
 *      
 *      
 * History:
 *      
 * Revision 1.8 1997/05/15
 * Overall code cleanup
 * Added signal handler for SIGHUP
 * Added writing of pid file
 * Make shutdown use init if slave
 * Rewrote shutdown for master
 * Nils Rennebarth, nils@nus.de
 * 
 * $Log: apcd.c,v $
 * Revision 1.7  1995/11/07  12:40:03  root
 * Version 0.5 Beta, uploaded to the sunsite
 *
 * Revision 1.6  1995/11/01  15:25:28  root
 * Several adaptations for clien/server - NOT FUNCTIONAL
 *
 * Revision 1.5  1995/05/23  07:25:08  root
 * First public ALPHA version
 *
 * Revision 1.4  1995/05/23  01:07:40  root
 * Parameters are on the command line, instead of config.h file
 *
 * Revision 1.3  1995/05/23  00:25:43  root
 * System shutdown with UPS switch off was added
 *
 * Revision 1.2  1995/05/21  21:10:56  root
 * Some small fixes
 *
 * Revision 1.1  1995/05/21  20:15:13  root
 * Initial revision
 *
 *
 *
 *
 *      
 */


#include "apcd.h"
#include "version.h"

#ifndef lint
static char *version = "$Id: apcd.c,v 1.8 2000/01/02 17:39:03 root Exp root $";
#endif

UPSINFO myUPS;
FILE *valfile;
FILE *UPSlogfile;
int slave;			/* TRUE iff slave  */
int master;			/* TRUE iff master this is for readability only */
int port;
int socketfd, newsocketfd;
char *use_port;
char *master_name;
char *logfilename;
char *slaves[MAX_SLAVES];
int num_slaves = 0;
int power_timer = 10;
int log_timer = 30;
int log_counter = 0;
volatile int killme = 0;
int notest = 0;
int alarmup = 0;
int alarmdown = 0;
int pending = 0;
int alarmcount = 0;
int wasmsg = 0;
int mastertimeout = 0;
int gottimeout = 0;
int battlow = 0;
int gotpowerok = 0;
int masterbatlow = 0;
int notifydown = 0;
int writelog = 0;
int min_batt_warn = 0;
float min_batt_load = 0.0;
int min_batt_ignore = 2;
volatile int counter = 0;
struct termios oldtio, newtio;


int 
main (int argc, char *argv[])
{
  int i;
  FILE *fp;
  char q, answer[MAXLINE];


  if (!strcmp(argv[1], "--version") || !strcmp(argv[1], "-v")) {
   printf("APC SmartUPS daemon version %s\n", VERSION);
   exit(1);
  }

  use_port = calloc (100, sizeof (char));
  master_name = calloc (100, sizeof (char));
  logfilename = calloc (100, sizeof (char));
  for (i = 0; i <= MAX_SLAVES; i++)
    slaves[i] = calloc (100, sizeof (char));

  if (parse_config ())
    {
      fprintf (stderr, "Configuration file is bad or missing\n");
      exit (1);
    }
  /* Initialize system log */
  openlog (DAEMONID, LOG_PID | LOG_NDELAY | LOG_PERROR, LOG_DAEMON);

  if (master)
    UPSlogfile = fopen (logfilename, "a");

  /* Become daemon */
  start_daemon ();
  signal_setup ();
  syslog (LOG_INFO, "Starting APC SmartUPS daemon version %s", VERSION);

  if (master)
    {
      syslog (LOG_INFO, "Master mode: port %s, timeout %d, logging interval %d seconds.",
	      use_port, power_timer, log_timer);
      syslog (LOG_INFO, "UPS statistics in %s", logfilename);
      for (i = 0; i < num_slaves; i++)
	syslog (LOG_INFO, "Slave: %s", slaves[i]);
    }
  else
    {
      syslog (LOG_INFO, "Slave mode. Master is %s", master_name);
    }
  if (master)
    setup_tty ();

  /* Create PID file */
  if ((fp = fopen (PIDFILENAME, "w")) != NULL)
    {
      fprintf (fp, "%d\n", getpid ());
      fclose (fp);
    }
  else
    {
      fprintf (stderr, "apcd: warning: cannot open pidfile %s\n",
	       PIDFILENAME);
      perror (DAEMONID);
    }

  /* Open socket for network communication */
  if (slave)
    {
      if ((socketfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
	  syslog (LOG_ERR, "Can't open stream socket");
	}
    }
  if (master)
    {
      prepare_master ();
      if (!notest)
	{
	  q = 'A';
	  write (port, &q, 1);
	  getline (port, answer);
	}
      while (!killme)
	{
	  do_master ();
	}
    }
  else
    {
      prepare_slave ();
      while (!killme)
	{
	  do_slave ();
	}
    }
  if (killme == 1)
    {
      syslog (LOG_INFO, "Ending apcd version %s", VERSION);
      if (master)
	{
	  tcsetattr (port, TCSANOW, &oldtio);
	  close (port);
	  for (i = 0; i < num_slaves; i++)
	    close (slavesocket[i]);
	  fclose (UPSlogfile);
	}
      else
	{
	  close (socketfd);
	}
      closelog ();
      unlink (PIDFILENAME);
    }
  if (killme == 2)
    {
      if (master)
	send_to_slaves (-2);
      mesall ("SYSTEM IS GOING DOWN NOW !!!\n");
      if (master)
	{
	  for (i = 0; i < num_slaves; i++)
	    close (slavesocket[i]);
	}
      else
	{
	  close (socketfd);
	}
      do_shutdown ();
    }
  return (0);
}

void 
do_slave ()
{
  char msg[100];

  if (get_master_message (newsocketfd) != 0)
    {
      return;
    }
  if (gottimeout)
    {
      gottimeout = 0;
      sprintf (msg, "Power failure, system will go down in %d minutes.\n", mastertimeout);
      syslog (LOG_INFO, msg);
      if (mastertimeout == 0)
	{
	  go_down ();
	}
      else
	{
	  mesall (msg);
	  wasmsg = 1;
	}
    }
  if (gotpowerok)
    {
      gotpowerok = 0;
      syslog (LOG_INFO, "Power restored, shutdown cancelled");
      if (wasmsg)
	{
	  mesall ("Power restored, shutdown cancelled.\n");
	  wasmsg = 0;
	}
    }
  if (masterbatlow)
    go_down_batt ();
}

void 
do_master ()
{
  char msg[100];

  fillUPS (port, &myUPS);
  if (writelog)
    {
      log_UPS_status ();
      writelog = 0;
    }
  if (pending)
    {
      if (notifydown)
	{
	  send_to_slaves (alarmcount);
	  notifydown = 0;
	  sprintf (msg, "Power failure, system will go down in %d minutes.\n", alarmcount / 60);
	  mesall (msg);
	  wasmsg = 1;
	}
      if (alarmcount == 0)
	{
	  pending = 0;
	}
      if (min_batt_warn)
	{
	  syslog (LOG_WARNING, "Battery Level lower than %f", min_batt_load);
	  go_down_batt ();
	  min_batt_warn = 0;
	}
    }
  if (alarmup)
    {
      if (!pending)
	{
	  alarmcount = power_timer * 60;
	  pending = 1;
	}
      alarmup = 0;
    }
  if (alarmdown)
    {
      if (wasmsg)
	{
	  send_to_slaves (-1);
	  mesall ("Power restored, shutdown cancelled\n");
	  wasmsg = 0;
	}
      alarmcount = 0;
      pending = 0;
      alarmdown = 0;
    }
  if (battlow)
    go_down_batt ();
}

/* Setup of the communication port. Hope it will work
 */

void 
setup_tty ()
{
  port = open (use_port, O_RDWR | O_NOCTTY);
  if (port < 0)
    {
      syslog (LOG_ERR, "Unable to open port %s", use_port);
      exit (-1);
    }
  tcgetattr (port, &oldtio);	/* Save old settings */
  newtio.c_cflag = DEFAULT_SPEED | CS8 | CLOCAL | CREAD;
  newtio.c_iflag = IGNPAR;	/* Ignore errors, raw input */
  newtio.c_oflag = 0;		/* Raw output */
  newtio.c_lflag = 0;		/* No local echo */
  newtio.c_cc[VMIN] = 1;
  newtio.c_cc[VTIME] = 0;
  tcflush (port, TCIFLUSH);
  tcsetattr (port, TCSANOW, &newtio);
}


/* Become a daemon, release stdin, stdout etc.
 */

void 
start_daemon ()
{
  int pid;

  close (0);
  close (1);
  close (2);
  if ((pid = fork ()) < 0)
    {
      syslog (LOG_ERR, "Unable to fork");
      exit (1);
    }
  if (pid != 0)
    exit (0);
};


/* Setup various signal handlers. Code here is adapted from diald program
 * which is (c) Eric Schenk.
 */

void 
signal_setup ()
{
  struct itimerval itv;

  itv.it_interval.tv_usec = 0;
  itv.it_interval.tv_sec = 1;
  itv.it_value.tv_usec = 0;
  itv.it_value.tv_sec = 1;
  signal (SIGINT, sig_intr);
  signal (SIGTERM, sig_term);
  signal (SIGALRM, sig_alarm);
  siginterrupt (SIGALRM, FALSE);
  siginterrupt (SIGINT, FALSE);
  siginterrupt (SIGTERM, FALSE);
  setitimer (ITIMER_REAL, &itv, NULL);
  if (master)
    {
      signal (SIGHUP, sig_hup);
      signal (SIGUSR1, dump_status);
    }

}

void 
sig_intr (int sig)
{
  syslog (LOG_INFO, "SIGINTR Termination requested");
  killme = 1;
  signal (sig, sig_intr);
}

void 
sig_term (int sig)
{
  int i;

  syslog (LOG_INFO, "SIGTERM Termination requested");
  syslog (LOG_INFO, "Ending apcd version %s", VERSION);
  if (master)
    {
      tcsetattr (port, TCSANOW, &oldtio);
      close (port);
      for (i = 0; i < num_slaves; i++)
	close (slavesocket[i]);
      fclose (UPSlogfile);
    }
  else
    {
      close (socketfd);
      close (newsocketfd);
    }
  closelog ();
  unlink (PIDFILENAME);
  exit (0);
}

void 
sig_hup (int sig)
{
  if (master)
    {
      fclose (UPSlogfile);
      UPSlogfile = fopen (logfilename, "a");
    }
  signal (sig, sig_hup);
}

/* This one is called every second */
void 
sig_alarm (int sig)
{
  if (pending)
    {
      alarmcount--;
      if (alarmcount == 0)
	{
	  go_down ();
	}
      else if (alarmcount % 60 == 0)
	{
	  notifydown = 1;
	}
    }
  if (counter > 0)
    {
      counter--;
    }
  else
    {
      counter = log_timer - 1;
      writelog = 1;
    }
  signal (sig, sig_alarm);
}

void 
dump_status (int sig)
{
  valfile = fopen ("/tmp/upsstat", "w");
  fprintf (valfile, "ULINE:%.1f\n", myUPS.LineVoltage);
  fprintf (valfile, "MLINE:%.1f\n", myUPS.LineMax);
  fprintf (valfile, "NLINE:%.1f\n", myUPS.LineMin);
  fprintf (valfile, "FLINE:%.1f\n", myUPS.LineFreq);
  fprintf (valfile, "VOUTP:%.1f\n", myUPS.OutputVoltage);
  fprintf (valfile, "LOUTP:%.1f\n", myUPS.UPSLoad);
  fprintf (valfile, "BOUTP:%.1f\n", myUPS.BattVoltage);
  fprintf (valfile, "BCHAR:%.1f\n", myUPS.BatLoad);
  fprintf (valfile, "BFAIL:%d\n", battlow);
  fprintf (valfile, "UTEMP:%.1f\n", myUPS.UPSTemp);
  if (pending)
    {
      fprintf (valfile, "UBATT:1\n");
      fprintf (valfile, "UPOWR:0\n");
    }
  else
    {
      fprintf (valfile, "UBATT:0\n");
      fprintf (valfile, "UPOWR:1\n");
    };
  fprintf (valfile, "UTST1:NONE\n");
  fprintf (valfile, "UTST2:NONE\n");
  fprintf (valfile, "UTST3:NONE\n");
  fprintf (valfile, "UTST4:NONE\n");
  fprintf (valfile, "UTST5:NONE\n");
  fprintf (valfile, "UTST6:NONE\n");
  fprintf (valfile, "UTST7:NONE\n");
  fprintf (valfile, "UTST8:NONE\n");
  fprintf (valfile, "UTST9:NONE\n");
  fprintf (valfile, "UTST0:NONE\n");
  fprintf (valfile, "UUSR1:NONE\n");
  fprintf (valfile, "UUSR2:NONE\n");
  fprintf (valfile, "UUSR3:NONE\n");
  fprintf (valfile, "UUSR4:NONE\n");
  fprintf (valfile, "UUSR5:NONE\n");
  fprintf (valfile, "UUSR6:NONE\n");
  fprintf (valfile, "UUSR7:NONE\n");
  fprintf (valfile, "UUSR8:NONE\n");
  fprintf (valfile, "UUSR9:NONE\n");
  fprintf (valfile, "UUSR0:NONE\n");
  fclose (valfile);
}


void 
go_down ()
{
  syslog (LOG_INFO, "System is going down - power failure");
  killme = 2;
}

void 
go_down_batt ()
{
  syslog (LOG_INFO, "System is going down - battery low");
  killme = 2;
}


int 
getline (int fd, char *s)
{
  int i, j, ending;
  char c;
  sigset_t block_alarm;
  fd_set read_timeout_fds;
  struct timeval read_timeout_tv;


  i = 0;
  ending = 0;
  sigemptyset (&block_alarm);
  sigaddset (&block_alarm, SIGALRM);

  while (!ending)
    {
      sigprocmask (SIG_BLOCK, &block_alarm, NULL);
      FD_ZERO (&read_timeout_fds);
      FD_SET (0, &read_timeout_fds);
      read_timeout_tv.tv_sec = 5;
      read_timeout_tv.tv_usec = 0;

      if (select (fd + 1, &read_timeout_fds, NULL, NULL, &read_timeout_tv))
	{

	  read (fd, &c, 1);

	}
      else
	{
	  syslog (LOG_INFO, "No connection between PC and SmartUPS. Exiting.");
	  syslog (LOG_INFO, "Ending apcd version %s", VERSION);
	  if (master)
	    {
	      tcsetattr (port, TCSANOW, &oldtio);
	      close (port);
	      for (j = 0; j < num_slaves; j++)
		close (slavesocket[j]);
	      fclose (UPSlogfile);
	    }
	  else
	    close (socketfd);

	  closelog ();
	  unlink (PIDFILENAME);

	}

      sigprocmask (SIG_UNBLOCK, &block_alarm, NULL);
      switch (c)
	{
	case UPS_ON_BATT:
	  syslog (LOG_INFO, "UPS is going on battery");
	  alarmup = 1;
	  break;
	case UPS_ON_LINE:
	  syslog (LOG_INFO, "UPS is going on-line");
	  alarmdown = 1;
	  break;
	case BATT_LOW:
	  battlow = 1;
	  break;
	case BATT_OK:
	  battlow = 0;
	  break;
	case '\n':
	  ending = 1;
	  break;
	default:
	  s[i++] = c;
	  break;
	}
    }
  s[i] = '\0';
  return (0);
}

int 
fillUPS (int fd, UPSINFO * ups)
{
  char answer[MAXLINE];
  char q;

  q = 'Y';
  write (fd, &q, 1);
  getline (fd, answer);

  q = BATT_FULL;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->BatLoad = atof (answer);
  /* At startup, BatLoad is sometimes read incorrect. Prevent
     erroneus reboots */
  if (ups->BatLoad < min_batt_load)
    {
      if (min_batt_ignore)
	{
	  min_batt_ignore--;
	}
      else if (!min_batt_warn)
	{
	  min_batt_warn = 1;
	}
    }

  q = UPS_LINE_MIN;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->LineMin = atof (answer);

  q = UPS_LINE_MAX;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->LineMax = atof (answer);

  q = UPS_LOAD;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->UPSLoad = atof (answer);

  q = LINE_FREQ;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->LineFreq = atof (answer);

  q = LINE_VOLTAGE;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->LineVoltage = atof (answer);

  q = OUTPUT_VOLTAGE;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->OutputVoltage = atof (answer);

  q = UPS_TEMP;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->UPSTemp = atof (answer);

  q = BATT_VOLTAGE;
  write (fd, &q, 1);
  getline (fd, answer);
  ups->BattVoltage = atof (answer);

  q = UPS_STATUS;
  write (fd, &q, 1);
  getline (fd, answer);
//  ups->Status = atoi (answer);
 ups->Status = strtol(answer, NULL, 16);
 
  return (0);
}


/* mesusr() and mesall() function are actually parts of shutdown source
 * modified for our special case.  I am using them for sending messages
 * before shutdown
 */

void 
mesusr (char *mess, struct utmp *ut)
{
  int fd;
  char term[40] =
  {'/', 'd', 'e', 'v', '/', 0};

  (void) strncat (term, ut->ut_line, sizeof (ut->ut_line));
  if ((fd = open (term, O_RDWR | O_NONBLOCK)) < 0)
    return;
  write (fd, "\n\n", 2);
  write (fd, DAEMONMES, strlen (DAEMONMES));
  write (fd, ": ", 2);
  write (fd, mess, strlen (mess));
  close (fd);
}

void 
mesall (char *mess)
{
  struct utmp *ut;
  utmpname (_PATH_UTMP);
  setutent ();
  ut = getutent ();
  while ((ut = getutent ()))
    {
      if (ut->ut_type == USER_PROCESS)
	mesusr (mess, ut);
    }
  endutent ();
}


/*

 * From here, there are parts of the source from shutdown.c which
 * is a part of linux-utils-2.1
 *
 */

void write_wtmp (), unmount_disks (), unmount_disks_ourselves ();



void
do_shutdown ()
{
  char *args[8];
  int argp = 0;
  int i;
  char a;

  if (slave == 1)
    {				/* a slave can call init to do a regular shutdown */
      /* Create the arguments for init. */
      args[argp++] = INIT;
      args[argp++] = DOWN_LEVEL;
      args[argp] = (char *) NULL;
      /* Now execute init to change runlevel. */
      execv (INIT, args);

      /* Oops - failed. */
      fprintf (stderr, "\rapcd: cannot execute %s\r\n", INIT);
      exit (1);
    }
  /*
   * the master needs to shutdown the UPS as the very last action
   * so we need to do everything ourselves. This is stolen from
   * shutdown.c in sysvinit-2.71 from Miquel van Smoorenburg,
   * miquels@drinkel.cistron.nl
   */


  /* First close all files. */
  for (i = 0; i < 3; i++)
    if (!isatty (i))
      close (i);
  for (i = 3; i < 20; i++)
    close (i);
  close (255);

  /* idle init. */
  if (kill (1, SIGTSTP) < 0)
    {
      fprintf (stderr, "%s: warning: can't idle init.\r\n",
	       DAEMONID);
    }

  /* Kill all processes. */
  fprintf (stderr, "%s: sending all processes the TERM signal...\r\n",
	   DAEMONID);
  (void) kill (-1, SIGTERM);
  sleep (5);
  fprintf (stderr, "%s: sending all processes the KILL signal.\r\n",
	   DAEMONID);
  (void) kill (-1, SIGKILL);

  sleep (1);			/* Give init the chance to collect zombies. */
  setenv ("PATH", "/bin:/sbin:/usr/bin:/usr/sbin", 1);

  /* Record the fact that we're going down */
  write_wtmp ();

  /* This is for those who have quota installed. */
  system ("accton 2>/dev/null");
  system ("quotaoff -a 2>/dev/null");

  fprintf (stderr, "%s: turning off swap\r\n", DAEMONID);
  system ("swapoff -a");
  fprintf (stderr, "%s: unmounting all file systems\r\n", DAEMONID);
  system ("umount -a");

  sleep (1);
  /* shutdown UPS */
  a = 'S';
  write (port, &a, 1);
  sleep (9999);			/* Wait for UPS switch off */
  init_reboot (BMAGIC_HALT);	/* Just for sure :-) */
  /* NOTREACHED */
  exit (0);			/* to quiet gcc */
}


void
write_wtmp ()
{
  /* write in wtmp that we are dying */
  int fd;
  struct utmp ut;

  memset ((char *) &ut, 0, sizeof (ut));
  strcpy (ut.ut_line, "~~");
  memcpy (ut.ut_name, "shutdown", sizeof (ut.ut_name));

  time (&ut.ut_time);
  ut.ut_type = RUN_LVL;

  if ((fd = open (_PATH_WTMP, O_WRONLY | O_APPEND)) > 0)
    {
      write (fd, (char *) &ut, sizeof (ut));
      close (fd);
    }
}

void
unmount_disks ()
{
  /* better to use umount directly because it may be smarter than us */

  int pid;
  int result;
  int status;

  sync ();
  if ((pid = fork ()) < 0)
    {
      printf ("Cannot fork for umount, trying manually.\n");
      unmount_disks_ourselves ();
      return;
    }
  if (!pid)
    {
      execl (_PATH_UMOUNT, UMOUNT_ARGS, NULL);
      printf ("Cannot exec %s, trying umount.\n", _PATH_UMOUNT);
      execlp ("umount", UMOUNT_ARGS, NULL);
      printf ("Cannot exec umount, trying manually.\n");
      unmount_disks_ourselves ();
      exit (0);
    }
  while ((result = wait (&status)) != -1 && result != pid)
    ;
  if (result == -1 || status)
    {
      printf ("Running umount failed, trying manually.\n");
      unmount_disks_ourselves ();
    }
}

void
unmount_disks_ourselves ()
{
  /* unmount all disks */

  FILE *mtab;
  struct mntent *mnt;
  char *mntlist[128];
  int i;
  int n;
  char *filesys;

  sync ();
  if (!(mtab = setmntent (_PATH_MTAB, "r")))
    {
      printf ("Cannot open %s.\n", _PATH_MTAB);
      return;
    }
  n = 0;
  while (n < 100 && (mnt = getmntent (mtab)))
    {
      mntlist[n++] = strdup (mnt->mnt_fsname[0] == '/' ?
			     mnt->mnt_fsname : mnt->mnt_dir);
    }
  endmntent (mtab);

  /* we are careful to do this in reverse order of the mtab file */

  for (i = n - 1; i >= 0; i--)
    {
      filesys = mntlist[i];
#ifdef DEBUGGING
      printf ("umount %s\n", filesys);
#else
      if (umount (mntlist[i]) < 0)
	printf ("Couldn't umount %s\n", filesys);
#endif
    }
}

void 
log_UPS_status ()
{
  char msg[100];
  time_t nowtime;

  time (&nowtime);
  strftime (msg, 100, "%c", localtime (&nowtime));
  fprintf (UPSlogfile, "%s APC: ", msg);
  fprintf (UPSlogfile, "%.1f %.1f %.1f ", myUPS.BatLoad, myUPS.LineMin, myUPS.LineMax);
  fprintf (UPSlogfile, "%.1f %.1f %.1f ", myUPS.UPSLoad, myUPS.LineFreq, myUPS.LineVoltage);
  fprintf (UPSlogfile, "%.1f %.1f %.1f\n", myUPS.OutputVoltage, myUPS.UPSTemp, myUPS.BattVoltage);
  fflush (UPSlogfile);
}


/*
   Local Variables:
   c-basic-offset: 8
   tab-width: 8
   End:
 */
