// --------------------------------------------------------------------------
// CONFIGURATION FILE EVALUATION
// --------------------------------------------------------------------------

#include "ConfigFile.h"

// Keywords.
static const char key_Frequency[] = "Frequency=";
static const char key_Channels[] = "Channels=";
static const char key_BitsPerSample[] = "BitsPerSample=";
static const char key_MaxFrags[] = "MaxFrags=";
static const char key_FragSize[] = "FragSize=";
static const char key_BufSize[] = "BufSize=";

static const char key_ForceSongSpeed[] = "ForceSongSpeed=";
static const char key_ClockSpeed[] = "ClockSpeed=";
static const char key_MemoryMode[] = "MemoryMode=";
static const char key_MOS8580[] = "MOS8580=";
static const char key_MeasuredMasterVolume[] = "MeasuredMasterVolume=";
static const char key_UseFilter[] = "UseFilter=";
static const char key_FilterFs[] = "FilterFs=";
static const char key_FilterFm[] = "FilterFm=";
static const char key_FilterFt[] = "FilterFt=";

static const char key_Mixer[] = "Mixer=";
static const char key_AutoPanning[] = "AutoPanning=";
static const char key_Voice1Vol[] = "Voice1Vol=";
static const char key_Voice2Vol[] = "Voice2Vol=";
static const char key_Voice3Vol[] = "Voice3Vol=";
static const char key_Voice4Vol[] = "Voice4Vol=";
static const char key_PanPosLeftHQ[] = "PanPosLeftHQ=";
static const char key_PanPosRightHQ[] = "PanPosRightHQ=";
static const char key_PanPosLeftFP[] = "PanPosLeftFP=";
static const char key_PanPosRightFP[] = "PanPosRightFP=";

static const char key_Yes[] = "Yes";
static const char key_No[] = "No";
static const char key_PAL[] = "PAL";
static const char key_NTSC[] = "NTSC";
static const char key_FullBankSwitching[] = "FullBankSwitching";
static const char key_TransparentROM[] = "TransparentROM";
static const char key_PlaySID[] = "PlaySID";
static const char key_Normal[] = "Normal";
static const char key_Centered[] = "Centered";
static const char key_StereoSurround[] = "StereoSurround";
static const char key_HQ[] = "HighQuality";
static const char key_FP[] = "FullPanning";

static const char key_Usage[] = "Usage=";

static const char key_Directory[] = "Directory=";
static const char key_NameFilter[] = "NameFilter=";

static const char key_HVSCroot[] = "HVSCroot=";
static const char key_STILshowGlobal[] = "STILshowGlobal=";
static const char key_STILshowEntries[] = "STILshowEntries=";
static const char key_HVSCbugEntries[] = "HVSCbugEntries=";
static const char key_STILallOrSong[] = "STILallOrSong=";
static const char key_STILautoResize[] = "STILautoResize=";
static const char key_STILmaxHeight[] = "STILmaxHeight=";
static const char key_StilOpen[] = "StilOpen=";  // case!

static const char key_MainPosX[] = "MainPosX=";
static const char key_MainPosY[] = "MainPosY=";
static const char key_DirPosX[] = "DirPosX=";
static const char key_DirPosY[] = "DirPosY=";
static const char key_DirExtX[] = "DirExtX=";
static const char key_DirExtY[] = "DirExtY=";
static const char key_DirOpen[] = "DirOpen=";

ConfigFile::ConfigFile(const char* fileName) : TextFile(fileName)
{
    const char* homeDir = getenv("HOME");
    if (homeDir != NULL)
    {
        configPath = homeDir;
    }
    else
    {
        configPath = "/var/tmp";
    }
    configPath += "/";
    configPath += fileName;
#ifdef WB_DEBUG
    cout << configPath << endl;
#endif
}

const AudioConfig& ConfigFile::getAudioConfig() const
{
    return myAudioConfig;
}

const emuConfig& ConfigFile::getEmuConfig() const
{
    return myEmuConfig;
}

const HVSCconfig& ConfigFile::getHVSCconfig() const
{
    return myHVSCconfig;
}

const VariousConfig& ConfigFile::getVariousConfig() const
{
    return myConfig;
}

void ConfigFile::setAudioConfig(const AudioConfig& inAudioConfig)
{
    myAudioConfig = inAudioConfig;
}

void ConfigFile::setEmuConfig(const emuConfig& inEmuConfig)
{
    myEmuConfig = inEmuConfig;
}

void ConfigFile::setHVSCconfig(const HVSCconfig& inHVSCconfig)
{
    myHVSCconfig = inHVSCconfig;
}

void ConfigFile::setVariousConfig(const VariousConfig& inConfig)
{
    myConfig = inConfig;
}

// -------------------------------------------------------------------- Input

void readMergedPanPosLeft(voiceVol panLevels[4], const udword allPanPosLeft)
{
    panLevels[0].l = (allPanPosLeft>>24)&255;
    panLevels[1].l = (allPanPosLeft>>16)&255;
    panLevels[2].l = (allPanPosLeft>>8)&255;
    panLevels[3].l = allPanPosLeft&255;
}

void readMergedPanPosRight(voiceVol panLevels[4], const udword allPanPosRight)
{
    panLevels[0].r = (allPanPosRight>>24)&255;
    panLevels[1].r = (allPanPosRight>>16)&255;
    panLevels[2].r = (allPanPosRight>>8)&255;
    panLevels[3].r = allPanPosRight&255;
}

bool ConfigFile::load()
{
    open(configPath);
    while (status && !endOfFile())  // line-by-line loop
    {
        readNextLine();
        // Skip blank and comment lines.
        while (status && !endOfFile() && isBlank() || isComment())
        {
            readNextLine();
        };
        // Evaluate line.
#ifdef WB_DEBUG
        cout << "Line " << retLineNum() << ", " << retLineLen() << ": ";
        cout << retLineBuf() << endl;
        cout << "ParseBuf: " << retParseBuf() << endl;
#endif
        
        // Audio settings.
        if (isKey(key_Frequency))
            myAudioConfig.frequency = atoi(retCurParseBuf());
        else if (isKey(key_Channels))
        {
            myAudioConfig.channels = atoi(retCurParseBuf());
        }
        else if (isKey(key_BitsPerSample))
        {
            myAudioConfig.precision = atoi(retCurParseBuf());
        }
        else if (isKey(key_BufSize))
        {
            myAudioConfig.bufSize = atoi(retCurParseBuf());
        }
        else if (isKey(key_MaxFrags))
        {
            myAudioConfig.maxFrags = atoi(retCurParseBuf());
        }
        else if (isKey(key_FragSize))
        {
            myAudioConfig.fragSize = atoi(retCurParseBuf());
        }
        
        // Emulator settings.
        else if (isKey(key_ClockSpeed))
        {
            if (isKey(key_PAL))
                myEmuConfig.clockSpeed = SIDTUNE_CLOCK_PAL;
            else
                myEmuConfig.clockSpeed = SIDTUNE_CLOCK_NTSC;
        }
        else if (isKey(key_ForceSongSpeed))
        {
            myEmuConfig.forceSongSpeed = isKey(key_Yes);
        }
        else if (isKey(key_MemoryMode))
        {
            if (isKey(key_FullBankSwitching,false))
                myEmuConfig.memoryMode = MPU_BANK_SWITCHING;
            else if (isKey(key_TransparentROM,false))
                myEmuConfig.memoryMode = MPU_TRANSPARENT_ROM;
            else
                myEmuConfig.memoryMode = MPU_PLAYSID_ENVIRONMENT;
        }
        else if (isKey(key_MOS8580))
        {
            myEmuConfig.mos8580 = isKey(key_Yes);
        }
        else if (isKey(key_MeasuredMasterVolume))
        {
            myEmuConfig.measuredVolume = isKey(key_Yes);
        }
        
        // Filter settings.
        else if (isKey(key_UseFilter))
            myEmuConfig.emulateFilter = isKey(key_Yes);
        else if (isKey(key_FilterFs))
            myEmuConfig.filterFs = atof(retCurParseBuf());
        else if (isKey(key_FilterFm))
            myEmuConfig.filterFm = atof(retCurParseBuf());
        else if (isKey(key_FilterFt))
            myEmuConfig.filterFt = atof(retCurParseBuf());
        
        // HVSC/STIL.
        else if (isKey(key_HVSCroot))
            myHVSCconfig.hvscRootPath.setPath(retLineBuf()+strlen(key_HVSCroot));
        else if (isKey(key_STILshowGlobal))
            myHVSCconfig.showGlobalComments = isKey(key_Yes);
        else if (isKey(key_STILshowEntries))
            myHVSCconfig.showStilEntries = isKey(key_Yes);
        else if (isKey(key_HVSCbugEntries))
            myHVSCconfig.showBugListEntries = isKey(key_Yes);
        else if (isKey(key_STILallOrSong))
        {
            myHVSCconfig.showForFile = isKey(key_Yes);
            myHVSCconfig.showForCurrentSong = !myHVSCconfig.showForFile;
        }
        else if (isKey(key_STILautoResize))
            myHVSCconfig.autoResize = isKey(key_Yes);
        else if (isKey(key_STILmaxHeight))
            myHVSCconfig.maxHeight = retCurParseBuf();  // QString copies
        
        else if (isKey(key_StilOpen))
        {
            myConfig.enableSTILview = isKey(key_Yes);
        }
        
        // Listening mileage.
        else if (isKey(key_Usage))
            myConfig.usage = (udword)atol(retCurParseBuf());
        
        // Mixer settings.
        else if (isKey(key_Mixer))
        {
            if (isKey(key_HQ,false))
                myEmuConfig.volumeControl = SIDEMU_VOLCONTROL;
            else if (isKey(key_FP,false))
                myEmuConfig.volumeControl = SIDEMU_FULLPANNING;
            else if (isKey(key_StereoSurround,false))
                myEmuConfig.volumeControl = SIDEMU_STEREOSURROUND;
            else
                myEmuConfig.volumeControl = SIDEMU_NONE;
        }
        else if (isKey(key_AutoPanning))
        {
            if (isKey(key_Centered))
                myEmuConfig.autoPanning = SIDEMU_CENTEREDAUTOPANNING;
            else
                myEmuConfig.autoPanning = SIDEMU_NONE;
        }
        else if (isKey(key_Voice1Vol))
            myConfig.myMixerConfig.volTotal[0] = atoi(retCurParseBuf());
        else if (isKey(key_Voice2Vol))
            myConfig.myMixerConfig.volTotal[1] = atoi(retCurParseBuf());
        else if (isKey(key_Voice3Vol))
            myConfig.myMixerConfig.volTotal[2] = atoi(retCurParseBuf());
        else if (isKey(key_Voice4Vol))
            myConfig.myMixerConfig.volTotal[3] = atoi(retCurParseBuf());
        
        else if (isKey(key_PanPosLeftHQ))
            readMergedPanPosLeft(myConfig.myMixerConfig.volHQ,atol(retCurParseBuf()));
        else if (isKey(key_PanPosRightHQ))
            readMergedPanPosRight(myConfig.myMixerConfig.volHQ,atol(retCurParseBuf()));
        else if (isKey(key_PanPosLeftFP))
            readMergedPanPosLeft(myConfig.myMixerConfig.volFP,atol(retCurParseBuf()));
        else if (isKey(key_PanPosRightFP))
            readMergedPanPosRight(myConfig.myMixerConfig.volFP,atol(retCurParseBuf()));
    
        // Window geometries.
        // Main dialog.
        else if (isKey(key_MainPosX))
            myConfig.geomMainDlg.x = atoi(retCurParseBuf());
        else if (isKey(key_MainPosY))
            myConfig.geomMainDlg.y = atoi(retCurParseBuf());
        // File dialog.
        else if (isKey(key_DirPosX))
            myConfig.geomDirDlg.x = atoi(retCurParseBuf());
        else if (isKey(key_DirPosY))
            myConfig.geomDirDlg.y = atoi(retCurParseBuf());
        else if (isKey(key_DirExtX))
            myConfig.geomDirDlg.w = atoi(retCurParseBuf());
        else if (isKey(key_DirExtY))
            myConfig.geomDirDlg.h = atoi(retCurParseBuf());
        else if (isKey(key_DirOpen))
            myConfig.showDirDlg = isKey(key_Yes);
        // Sidtunes directory.
        else if (isKey(key_Directory))
            myConfig.currentDir = retLineBuf()+strlen(key_Directory);  // QString copies
        else if (isKey(key_NameFilter))
            myConfig.nameFilter = retCurParseBuf();  // QString copies
        
    };

    (myAudioConfig.channels == 1) ? (myEmuConfig.channels = SIDEMU_MONO)
        : (myEmuConfig.channels = SIDEMU_STEREO);
    (myAudioConfig.precision == 8) ? (myEmuConfig.bitsPerSample = SIDEMU_8BIT)
        : (myEmuConfig.bitsPerSample = SIDEMU_16BIT);
    
    close();
    return status;
}

// ------------------------------------------------------------------- Output

void writeMergedPanPosLeft(ofstream& toFile, const voiceVol panLevels[4])
{
    long int allPanPosLeft;
    allPanPosLeft = (panLevels[0].l<<24)+(panLevels[1].l<<16)+
        (panLevels[2].l<<8)+(panLevels[3].l);
    toFile << allPanPosLeft << endl;
}

void writeMergedPanPosRight(ofstream& toFile, const voiceVol panLevels[4])
{
    long int allPanPosRight;
    allPanPosRight = (panLevels[0].r<<24)+(panLevels[1].r<<16)+
        (panLevels[2].r<<8)+(panLevels[3].r);
    toFile << allPanPosRight << endl;
}

void writeBoolkey(ofstream& toFile, const char* key, const bool flag)
{
    toFile << key;
    (flag) ? (toFile << key_Yes << endl) : (toFile << key_No << endl);
}

bool ConfigFile::save()
{
    bool wasSuccess = false;

#if defined(HAVE_IOS_BIN)
    ofstream toFile(configPath,ios::out|ios::bin|ios::trunc);
#else
    ofstream toFile(configPath,ios::out|ios::binary|ios::trunc);
#endif
    if (toFile)
    {
        toFile 
            << "# ~/.sidplayrc" << endl
            << "#" << endl
            << "# SIDPLAY/X11 configuration file." << endl
            << "#" << endl
            << "# DO NOT edit manually." << endl
            << endl;

        // Sidtunes directory.
        toFile << key_Directory << myConfig.currentDir << endl;
        toFile << key_NameFilter << myConfig.nameFilter << endl;
        
        // HVSC/STIL.
        toFile << key_HVSCroot << myHVSCconfig.hvscRootPath.path() << endl;
        writeBoolkey(toFile,key_STILshowGlobal,myHVSCconfig.showGlobalComments);
        writeBoolkey(toFile,key_STILshowEntries,myHVSCconfig.showStilEntries);
        writeBoolkey(toFile,key_HVSCbugEntries,myHVSCconfig.showBugListEntries);
        writeBoolkey(toFile,key_STILallOrSong,myHVSCconfig.showForFile);
        writeBoolkey(toFile,key_STILautoResize,myHVSCconfig.autoResize);
        toFile << key_STILmaxHeight << myHVSCconfig.maxHeight << endl;

        writeBoolkey(toFile,key_StilOpen,myConfig.enableSTILview);

        // Listening mileage.
        toFile
            << key_Usage << (myConfig.usage) << endl
            << endl;
        
        toFile << "# Audio quality settings." << endl
            << key_Frequency << myAudioConfig.frequency << endl
            << key_Channels << myAudioConfig.channels << endl
            << key_BitsPerSample << myAudioConfig.precision << endl
            << key_BufSize << myAudioConfig.bufSize << endl
            << key_MaxFrags << myAudioConfig.maxFrags << endl
            << key_FragSize << myAudioConfig.fragSize << endl
            << endl;

        toFile << "# Emulator settings." << endl;

        toFile << key_ClockSpeed;
        if (myEmuConfig.clockSpeed == SIDTUNE_CLOCK_PAL)
            toFile << key_PAL << endl;
        else  // if (myEmuConfig.clockSpeed == SIDTUNE_CLOCK_NTSC)
            toFile << key_NTSC << endl;

        writeBoolkey(toFile,key_ForceSongSpeed,myEmuConfig.forceSongSpeed);
        
        toFile << key_MemoryMode;
        if (myEmuConfig.memoryMode == MPU_BANK_SWITCHING)
            toFile << key_FullBankSwitching << endl;
        else if (myEmuConfig.memoryMode == MPU_TRANSPARENT_ROM)
            toFile << key_TransparentROM << endl;
        else if (myEmuConfig.memoryMode == MPU_PLAYSID_ENVIRONMENT)
            toFile << key_PlaySID << endl;

        writeBoolkey(toFile,key_MOS8580,myEmuConfig.mos8580);
        writeBoolkey(toFile,key_MeasuredMasterVolume,myEmuConfig.measuredVolume);
        writeBoolkey(toFile,key_UseFilter,myEmuConfig.emulateFilter);
        toFile << key_FilterFs << myEmuConfig.filterFs << endl;
        toFile << key_FilterFm << myEmuConfig.filterFm << endl;
        toFile << key_FilterFt << myEmuConfig.filterFt << endl
            << endl;

        toFile << "# Mixer settings." << endl;

        toFile << key_Mixer;
        if (myEmuConfig.volumeControl == SIDEMU_VOLCONTROL)
            toFile << key_HQ << endl;
        else if (myEmuConfig.volumeControl == SIDEMU_FULLPANNING)
            toFile << key_FP << endl;
        else if (myEmuConfig.volumeControl == SIDEMU_STEREOSURROUND)
            toFile << key_StereoSurround << endl;
        else
            toFile << key_No << endl;
            
        toFile << key_AutoPanning;
        if (myEmuConfig.autoPanning == SIDEMU_CENTEREDAUTOPANNING)
            toFile << key_Centered;
        else
            toFile << key_No;
        toFile << endl;
        
        toFile << key_Voice1Vol << (int)myConfig.myMixerConfig.volTotal[0] << endl;
        toFile << key_Voice2Vol << (int)myConfig.myMixerConfig.volTotal[1] << endl;
        toFile << key_Voice3Vol << (int)myConfig.myMixerConfig.volTotal[2] << endl;
        toFile << key_Voice4Vol << (int)myConfig.myMixerConfig.volTotal[3] << endl;

        toFile << key_PanPosLeftHQ;
        writeMergedPanPosLeft(toFile,myConfig.myMixerConfig.volHQ);
        toFile << key_PanPosRightHQ;
        writeMergedPanPosRight(toFile,myConfig.myMixerConfig.volHQ);
        toFile << key_PanPosLeftFP;
        writeMergedPanPosLeft(toFile,myConfig.myMixerConfig.volFP);
        toFile << key_PanPosRightFP;
        writeMergedPanPosRight(toFile,myConfig.myMixerConfig.volFP);
        toFile << endl;

        toFile << "# Window geometries." << endl;
        toFile
            << key_MainPosX << myConfig.geomMainDlg.x << endl
            << key_MainPosY << myConfig.geomMainDlg.y << endl
            << key_DirPosX << myConfig.geomDirDlg.x << endl
            << key_DirPosY << myConfig.geomDirDlg.y << endl
            << key_DirExtX << myConfig.geomDirDlg.w << endl
            << key_DirExtY << myConfig.geomDirDlg.h << endl;
        writeBoolkey(toFile,key_DirOpen,myConfig.showDirDlg);
        toFile << endl;
        
        toFile.close();
        wasSuccess = toFile;
    }
    
    return wasSuccess;
}

void ConfigFile::print() const
{
    ;
}
